<?php
/**
 * OpenAI Vision API Provider
 * 
 * @package CM_Auto_Alt_Text
 * @version 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * OpenAI GPT-4 Vision API Provider
 */
class CM_Auto_Alt_Text_OpenAI_Provider extends CM_Auto_Alt_Text_Abstract_Provider {

    /**
     * Get provider ID
     */
    public function get_id() {
        return 'openai';
    }

    /**
     * Get provider display name
     */
    public function get_display_name() {
        return 'OpenAI GPT-4 Vision';
    }

    /**
     * Get provider description
     */
    public function get_description() {
        return 'OpenAI GPT-4 Vision provides highly accurate and contextual image descriptions. Best quality but higher cost.';
    }

    /**
     * Get default configuration
     */
    protected function get_default_config() {
        return [
            'enabled' => false,
            'api_key' => '',
            'priority' => 2,
            'cost_per_call' => 0.01, // Approximate cost
            'max_file_size' => 20 * 1024 * 1024, // 20MB
            'max_width' => 2048,
            'max_height' => 2048,
            'supported_types' => [IMAGETYPE_JPEG, IMAGETYPE_PNG, IMAGETYPE_WEBP, IMAGETYPE_GIF],
            'timeout' => 30,
            'endpoint' => 'https://api.openai.com/v1/chat/completions',
            'model' => 'gpt-4o', // Updated to latest model (October 2024)
            'max_tokens' => 300,
            'temperature' => 0.3,
        ];
    }

    /**
     * Generate alt text for an image
     */
    public function generate_alt_text($image_path, $options = []) {
        $this->validate_image($image_path);
        
        if (empty($this->config['api_key'])) {
            throw new Exception('OpenAI API key not configured');
        }

        $start_time = microtime(true);
        
        // Encode image to base64
        $image_data = file_get_contents($image_path);
        $base64_image = base64_encode($image_data);
        $mime_type = mime_content_type($image_path);
        
        // Create prompt based on options
        $prompt = $this->create_prompt($options);
        
        // Prepare request
        $headers = [
            'Authorization' => 'Bearer ' . $this->config['api_key'],
            'Content-Type' => 'application/json',
        ];

        $body = [
            'model' => $this->config['model'],
            'messages' => [
                [
                    'role' => 'user',
                    'content' => [
                        [
                            'type' => 'text',
                            'text' => $prompt,
                        ],
                        [
                            'type' => 'image_url',
                            'image_url' => [
                                'url' => 'data:' . $mime_type . ';base64,' . $base64_image,
                                'detail' => 'auto',
                            ],
                        ],
                    ],
                ],
            ],
            'max_tokens' => $this->config['max_tokens'],
            'temperature' => $this->config['temperature'],
        ];

        // Make request
        $response = $this->make_request($this->config['endpoint'], [
            'method' => 'POST',
            'headers' => $headers,
            'body' => json_encode($body),
            'timeout' => $this->config['timeout'],
        ]);

        $response_time = microtime(true) - $start_time;
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if (!is_array($data) || !isset($data['choices'][0]['message']['content'])) {
            $error_msg = isset($data['error']['message']) ? sanitize_text_field($data['error']['message']) : 'Invalid API response';
            throw new Exception('OpenAI API error: ' . esc_html($error_msg));
        }

        $description = $data['choices'][0]['message']['content'];
        
        // Process the alt text
        $alt_text = $this->process_alt_text($description, $options);
        
        if (empty($alt_text)) {
            throw new Exception('Generated alt text did not meet quality requirements');
        }

        // Calculate quality score
        $quality_score = $this->calculate_quality_score($alt_text, $options);

        // Calculate actual cost
        $tokens_used = $data['usage']['total_tokens'] ?? 0;
        $actual_cost = $this->calculate_cost($tokens_used);

        // Update stats
        $this->update_stats([
            'success' => true,
            'response_time' => $response_time,
            'cost' => $actual_cost,
        ]);

        return [
            'alt_text' => $alt_text,
            'quality_score' => $quality_score,
            'cost' => $actual_cost,
            'response_time' => $response_time,
            'provider' => $this->get_id(),
            'raw_response' => $description,
            'tokens_used' => $tokens_used,
        ];
    }

    /**
     * Create prompt based on options
     */
    private function create_prompt($options) {
        $style = $options['style'] ?? 'descriptive';
        $max_length = $options['max_length'] ?? 125;
        $context = $options['context'] ?? [];
        
        $prompt = "Generate alt text for this image. ";
        
        switch ($style) {
            case 'seo':
                $prompt .= "Focus on SEO-friendly descriptions that would help with search engine optimization. ";
                break;
            case 'accessible':
                $prompt .= "Focus on accessibility for screen readers and visually impaired users. ";
                break;
            case 'descriptive':
            default:
                $prompt .= "Provide a clear, descriptive alternative text. ";
                break;
        }

        $prompt .= "The description should be concise (maximum {$max_length} characters), ";
        $prompt .= "specific, and help users understand what's in the image. ";
        $prompt .= "Avoid starting with 'Image of' or 'Photo of'. ";
        
        // Add context if available
        if (!empty($context['post_type'])) {
            if ($context['post_type'] === 'product') {
                $prompt .= "This appears to be a product image. ";
            } elseif ($context['post_type'] === 'post') {
                $prompt .= "This is from a blog post. ";
            }
        }
        
        if (!empty($context['categories'])) {
            $categories = implode(', ', array_slice($context['categories'], 0, 3));
            $prompt .= "Content categories: {$categories}. ";
        }
        
        if (isset($context['is_featured']) && $context['is_featured']) {
            $prompt .= "This is a featured image. ";
        }

        $prompt .= "Return only the alt text, no additional commentary.";
        
        return $prompt;
    }

    /**
     * Calculate cost based on tokens used
     */
    private function calculate_cost($tokens) {
        // GPT-4 Vision pricing (approximate)
        $cost_per_1k_tokens = 0.01; // $0.01 per 1K tokens
        return ($tokens / 1000) * $cost_per_1k_tokens;
    }

    /**
     * Test provider connection
     */
    public function test_connection() {
        if (empty($this->config['api_key'])) {
            throw new Exception('API key not configured');
        }

        // Create a simple test image
        $test_image = imagecreate(100, 100);
        $white = imagecolorallocate($test_image, 255, 255, 255);
        $black = imagecolorallocate($test_image, 0, 0, 0);
        imagefill($test_image, 0, 0, $white);
        imagestring($test_image, 3, 10, 40, 'TEST', $black);
        
        $temp_file = tempnam(sys_get_temp_dir(), 'openai_test_') . '.png';
        imagepng($test_image, $temp_file);
        imagedestroy($test_image);

        try {
            $result = $this->generate_alt_text($temp_file, ['min_length' => 1, 'max_length' => 50]);
            wp_delete_file($temp_file);
            return true;
        } catch (Exception $e) {
            wp_delete_file($temp_file);
            throw $e;
        }
    }

    /**
     * Get configuration fields for admin interface
     */
    public function get_config_fields() {
        return [
            'api_key' => [
                'type' => 'text',
                'label' => 'API Key',
                'description' => 'Your OpenAI API key. Get one at platform.openai.com',
                'required' => true,
                'placeholder' => 'sk-...',
            ],
            'enabled' => [
                'type' => 'checkbox',
                'label' => 'Enable Provider',
                'description' => 'Enable OpenAI GPT-4 Vision for alt text generation',
                'default' => false,
            ],
            'priority' => [
                'type' => 'number',
                'label' => 'Priority',
                'description' => 'Provider priority (lower numbers = higher priority)',
                'default' => 2,
                'min' => 1,
                'max' => 100,
            ],
            'model' => [
                'type' => 'select',
                'label' => 'Model',
                'description' => 'OpenAI model to use for vision analysis',
                'default' => 'gpt-4o',
                'options' => [
                    'gpt-4o' => 'GPT-4o (Latest, Recommended)',
                    'gpt-4o-mini' => 'GPT-4o Mini (Cheaper)',
                    'gpt-4-turbo' => 'GPT-4 Turbo with Vision',
                    'gpt-4-vision-preview' => 'GPT-4 Vision Preview (Deprecated)',
                ],
            ],
            'max_tokens' => [
                'type' => 'number',
                'label' => 'Max Tokens',
                'description' => 'Maximum tokens for response',
                'default' => 300,
                'min' => 50,
                'max' => 500,
            ],
            'temperature' => [
                'type' => 'number',
                'label' => 'Temperature',
                'description' => 'Creativity level (0.0-1.0, lower = more consistent)',
                'default' => 0.3,
                'min' => 0,
                'max' => 1,
                'step' => 0.1,
            ],
            'timeout' => [
                'type' => 'number',
                'label' => 'Timeout (seconds)',
                'description' => 'Maximum time to wait for API response',
                'default' => 30,
                'min' => 10,
                'max' => 60,
            ],
        ];
    }

    /**
     * Get configuration value
     */
    public function get_config_value($key, $default = null) {
        return isset($this->config[$key]) ? $this->config[$key] : $default;
    }

    /**
     * Get provider limitations
     */
    public function get_limitations() {
        return [
            'max_file_size' => '20MB',
            'max_dimensions' => '2048x2048 (recommended)',
            'supported_formats' => 'JPEG, PNG, WebP, GIF',
            'rate_limit' => '5000 requests/hour',
            'batch_processing' => false,
        ];
    }

    /**
     * Get pricing information
     */
    public function get_pricing() {
        return [
            'model' => 'Pay per token',
            'cost_structure' => [
                'input_tokens' => '$0.01 per 1K tokens',
                'output_tokens' => '$0.03 per 1K tokens',
                'image_tokens' => 'Variable based on size and detail',
            ],
            'estimated_cost_per_call' => '$0.01 - $0.05',
        ];
    }

    /**
     * Check if provider supports batch processing
     */
    public function supports_batch() {
        return false;
    }

    /**
     * Get supported image types
     */
    public function get_supported_types() {
        return [
            'image/jpeg' => 'JPEG',
            'image/png' => 'PNG',
            'image/webp' => 'WebP',
            'image/gif' => 'GIF (non-animated)',
        ];
    }
}