<?php
/**
 * Google Vision API Provider
 * 
 * @package CM_Auto_Alt_Text
 * @version 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Google Cloud Vision API Provider
 */
class CM_Auto_Alt_Text_Google_Vision_Provider extends CM_Auto_Alt_Text_Abstract_Provider {

    /**
     * Get provider ID
     */
    public function get_id() {
        return 'google_vision';
    }

    /**
     * Get provider display name
     */
    public function get_display_name() {
        return 'Google Cloud Vision';
    }

    /**
     * Get provider description
     */
    public function get_description() {
        return 'Google Cloud Vision API provides accurate image analysis with good performance and competitive pricing.';
    }

    /**
     * Get default configuration
     */
    protected function get_default_config() {
        return [
            'enabled' => false,
            'api_key' => '',
            'priority' => 3,
            'cost_per_call' => 0.0015, // $1.50 per 1000 requests
            'max_file_size' => 20 * 1024 * 1024, // 20MB
            'max_width' => 5000,
            'max_height' => 5000,
            'supported_types' => [IMAGETYPE_JPEG, IMAGETYPE_PNG, IMAGETYPE_WEBP, IMAGETYPE_GIF, IMAGETYPE_BMP],
            'timeout' => 30,
            'endpoint' => 'https://vision.googleapis.com/v1/images:annotate',
        ];
    }

    /**
     * Generate alt text for an image
     */
    public function generate_alt_text($image_path, $options = []) {
        $this->validate_image($image_path);
        
        if (empty($this->config['api_key'])) {
            throw new Exception('Google Vision API key not configured');
        }

        $start_time = microtime(true);
        
        // Encode image to base64
        $image_data = file_get_contents($image_path);
        $base64_image = base64_encode($image_data);
        
        // Prepare request
        $headers = [
            'Content-Type' => 'application/json',
        ];

        $body = [
            'requests' => [
                [
                    'image' => [
                        'content' => $base64_image,
                    ],
                    'features' => [
                        [
                            'type' => 'LABEL_DETECTION',
                            'maxResults' => 10,
                        ],
                        [
                            'type' => 'OBJECT_LOCALIZATION',
                            'maxResults' => 10,
                        ],
                        [
                            'type' => 'TEXT_DETECTION',
                            'maxResults' => 5,
                        ],
                    ],
                ],
            ],
        ];

        $url = $this->config['endpoint'] . '?key=' . urlencode($this->config['api_key']);

        // Make request
        $response = $this->make_request($url, [
            'method' => 'POST',
            'headers' => $headers,
            'body' => json_encode($body),
            'timeout' => $this->config['timeout'],
        ]);

        $response_time = microtime(true) - $start_time;
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if (!is_array($data) || !isset($data['responses'][0])) {
            $error_msg = isset($data['error']['message']) ? sanitize_text_field($data['error']['message']) : 'Invalid API response';
            throw new Exception('Google Vision API error: ' . esc_html($error_msg));
        }

        $response_data = $data['responses'][0];
        
        if (isset($response_data['error'])) {
            throw new Exception('Google Vision API error: ' . esc_html(sanitize_text_field($response_data['error']['message'])));
        }

        // Process the response to create alt text
        $description = $this->process_google_response($response_data, $options);
        
        if (empty($description)) {
            throw new Exception('Could not generate description from API response');
        }

        // Process the alt text
        $alt_text = $this->process_alt_text($description, $options);
        
        if (empty($alt_text)) {
            throw new Exception('Generated alt text did not meet quality requirements');
        }

        // Calculate quality score
        $quality_score = $this->calculate_quality_score($alt_text, $options);

        // Update stats
        $this->update_stats([
            'success' => true,
            'response_time' => $response_time,
            'cost' => $this->config['cost_per_call'],
        ]);

        return [
            'alt_text' => $alt_text,
            'quality_score' => $quality_score,
            'cost' => $this->config['cost_per_call'],
            'response_time' => $response_time,
            'provider' => $this->get_id(),
            'raw_response' => $response_data,
        ];
    }

    /**
     * Process Google Vision API response
     */
    private function process_google_response($response_data, $options) {
        $elements = [];
        
        // Get object localizations (most specific)
        if (isset($response_data['localizedObjectAnnotations'])) {
            foreach ($response_data['localizedObjectAnnotations'] as $object) {
                if ($object['score'] > 0.5) {
                    $elements[] = $object['name'];
                }
            }
        }
        
        // Get labels (general categories)
        if (isset($response_data['labelAnnotations']) && empty($elements)) {
            foreach ($response_data['labelAnnotations'] as $label) {
                if ($label['score'] > 0.7 && count($elements) < 3) {
                    $elements[] = $label['description'];
                }
            }
        }
        
        // Get text if detected
        $text_detected = '';
        if (isset($response_data['textAnnotations'][0])) {
            $text_detected = $response_data['textAnnotations'][0]['description'];
            $text_detected = trim(preg_replace('/\s+/', ' ', $text_detected));
        }
        
        // Construct description
        $description = '';
        
        if (!empty($elements)) {
            $context = $options['context'] ?? [];
            
            if (count($elements) === 1) {
                $description = ucfirst($elements[0]);
            } else {
                $main_element = array_shift($elements);
                $description = ucfirst($main_element);
                
                if (count($elements) > 0) {
                    $description .= ' with ' . implode(' and ', array_slice($elements, 0, 2));
                }
            }
            
            // Add text context if relevant and short
            if (!empty($text_detected) && strlen($text_detected) < 20) {
                $description .= " showing text '{$text_detected}'";
            }
            
            // Add context-specific information
            if (isset($context['post_type']) && $context['post_type'] === 'product') {
                $description = 'Product image of ' . lcfirst($description);
            }
        } elseif (!empty($text_detected)) {
            $description = "Image containing text: {$text_detected}";
        } else {
            throw new Exception('No recognizable content found in image');
        }
        
        return $description;
    }

    /**
     * Test provider connection
     */
    public function test_connection() {
        if (empty($this->config['api_key'])) {
            throw new Exception('API key not configured');
        }

        // Validate API key format (Google API keys typically start with AIza)
        $api_key = trim($this->config['api_key']);
        if (strlen($api_key) < 30) {
            throw new Exception('API key appears to be invalid (too short)');
        }

        // Make a minimal API request to verify the key works
        // Use a very small base64 encoded 1x1 red pixel PNG
        $tiny_png_base64 = 'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mP8z8DwHwAFBQIAX8jx0gAAAABJRU5ErkJggg==';

        $headers = [
            'Content-Type' => 'application/json',
        ];

        // Request only LABEL_DETECTION with maxResults: 1 to minimize cost
        $body = [
            'requests' => [
                [
                    'image' => [
                        'content' => $tiny_png_base64,
                    ],
                    'features' => [
                        [
                            'type' => 'LABEL_DETECTION',
                            'maxResults' => 1,
                        ],
                    ],
                ],
            ],
        ];

        $url = $this->config['endpoint'] . '?key=' . urlencode($api_key);

        try {
            $response = wp_remote_post($url, [
                'headers' => $headers,
                'body' => json_encode($body),
                'timeout' => 15,
            ]);

            if (is_wp_error($response)) {
                throw new Exception('Connection failed: ' . $response->get_error_message());
            }

            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            // Check for specific error codes
            if ($response_code === 403) {
                if (isset($data['error']['message'])) {
                    $error_msg = $data['error']['message'];
                    if (strpos($error_msg, 'API key not valid') !== false) {
                        throw new Exception('Invalid API key. Please check your Google Cloud Console.');
                    } elseif (strpos($error_msg, 'Cloud Vision API has not been used') !== false) {
                        throw new Exception('Google Cloud Vision API is not enabled. Please enable it in your Google Cloud Console at: console.cloud.google.com/apis/library/vision.googleapis.com');
                    } elseif (strpos($error_msg, 'quota') !== false) {
                        throw new Exception('API quota exceeded or billing not enabled. Check your Google Cloud Console.');
                    } else {
                        throw new Exception('Access denied (HTTP 403): ' . $error_msg);
                    }
                } else {
                    throw new Exception('Access denied (HTTP 403). Make sure the Vision API is enabled in Google Cloud Console.');
                }
            }

            if ($response_code === 400) {
                if (isset($data['error']['message'])) {
                    throw new Exception('Bad request: ' . $data['error']['message']);
                } else {
                    throw new Exception('Bad request (HTTP 400). Please check your API configuration.');
                }
            }

            if ($response_code !== 200) {
                $error_msg = isset($data['error']['message']) ? $data['error']['message'] : 'Unknown error';
                throw new Exception('API error (HTTP ' . $response_code . '): ' . $error_msg);
            }

            // Check if response is valid
            if (!is_array($data) || !isset($data['responses'])) {
                throw new Exception('Invalid API response structure');
            }

            // Success! The API key works
            return true;

        } catch (Exception $e) {
            // Re-throw with more context
            throw $e;
        }
    }

    /**
     * Get configuration fields for admin interface
     */
    public function get_config_fields() {
        return [
            'api_key' => [
                'type' => 'text',
                'label' => 'API Key',
                'description' => 'Your Google Cloud Vision API key. Get one at console.cloud.google.com',
                'required' => true,
                'placeholder' => 'Enter your Google Cloud API key',
            ],
            'enabled' => [
                'type' => 'checkbox',
                'label' => 'Enable Provider',
                'description' => 'Enable Google Cloud Vision for alt text generation',
                'default' => false,
            ],
            'priority' => [
                'type' => 'number',
                'label' => 'Priority',
                'description' => 'Provider priority (lower numbers = higher priority)',
                'default' => 3,
                'min' => 1,
                'max' => 100,
            ],
            'timeout' => [
                'type' => 'number',
                'label' => 'Timeout (seconds)',
                'description' => 'Maximum time to wait for API response',
                'default' => 30,
                'min' => 10,
                'max' => 60,
            ],
        ];
    }

    /**
     * Get configuration value
     */
    public function get_config_value($key, $default = null) {
        return isset($this->config[$key]) ? $this->config[$key] : $default;
    }

    /**
     * Get provider limitations
     */
    public function get_limitations() {
        return [
            'max_file_size' => '20MB',
            'max_dimensions' => '5000x5000',
            'supported_formats' => 'JPEG, PNG, WebP, GIF, BMP',
            'rate_limit' => '1800 requests/minute',
            'batch_processing' => true,
        ];
    }

    /**
     * Get pricing information
     */
    public function get_pricing() {
        return [
            'free_tier' => [
                'calls' => 1000,
                'period' => 'month',
                'cost' => 0,
            ],
            'paid_tier' => [
                'cost_per_1000' => 1.50,
                'features' => [
                    'LABEL_DETECTION' => '$1.50 per 1000',
                    'OBJECT_LOCALIZATION' => '$1.50 per 1000',
                    'TEXT_DETECTION' => '$1.50 per 1000',
                ],
            ],
        ];
    }

    /**
     * Check if provider supports batch processing
     */
    public function supports_batch() {
        return true;
    }

    /**
     * Get supported image types
     */
    public function get_supported_types() {
        return [
            'image/jpeg' => 'JPEG',
            'image/png' => 'PNG',
            'image/webp' => 'WebP',
            'image/gif' => 'GIF',
            'image/bmp' => 'BMP',
        ];
    }
}