<?php
/**
 * Cloudmersive AI Provider
 * 
 * @package CM_Auto_Alt_Text
 * @version 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cloudmersive Image Recognition API Provider
 */
class CM_Auto_Alt_Text_Cloudmersive_Provider extends CM_Auto_Alt_Text_Abstract_Provider {

    /**
     * Get provider ID
     */
    public function get_id() {
        return 'cloudmersive';
    }

    /**
     * Get provider display name
     */
    public function get_display_name() {
        return 'Cloudmersive Image Recognition';
    }

    /**
     * Get provider description
     */
    public function get_description() {
        return 'Cloudmersive offers 800 free API calls per month. Note: API structure is inconsistent - you may need to select "Premium" tier even on free accounts. Recommended as backup option.';
    }

    /**
     * Get default configuration
     */
    protected function get_default_config() {
        return [
            'enabled' => true,
            'api_key' => '',
            'priority' => 5, // Last priority due to messy API structure
            'cost_per_call' => 0.0, // Free tier
            'max_file_size' => 5 * 1024 * 1024, // 5MB
            'max_width' => 4096,
            'max_height' => 4096,
            'supported_types' => [IMAGETYPE_JPEG, IMAGETYPE_PNG, IMAGETYPE_GIF],
            'timeout' => 30,
            'account_tier' => 'free', // 'free' or 'premium'
            'endpoint_free' => 'https://api.cloudmersive.com/image/recognize/detect-objects',
            'endpoint_premium' => 'https://api.cloudmersive.com/image/recognize/describe',
        ];
    }

    /**
     * Generate alt text for an image
     */
    public function generate_alt_text($image_path, $options = []) {
        $this->validate_image($image_path);
        
        if (empty($this->config['api_key'])) {
            throw new Exception('Cloudmersive API key not configured');
        }

        $start_time = microtime(true);
        
        // Use tier-based endpoint selection (no fallback)
        $tier = isset($this->config['account_tier']) ? $this->config['account_tier'] : 'free';
        $endpoint = ($tier === 'premium') 
            ? $this->config['endpoint_premium']
            : $this->config['endpoint_free'];
        
        $use_objects_endpoint = ($tier === 'free');
        
        // Prepare multipart form data
        $boundary = wp_generate_password(24, false);
        $headers = [
            'Apikey' => trim($this->config['api_key']), // Use trim() not sanitize_text_field() to avoid key corruption
            'Content-Type' => 'multipart/form-data; boundary=' . $boundary,
        ];

        // Build multipart body
        $body = '--' . $boundary . "\r\n";
        $body .= 'Content-Disposition: form-data; name="imageFile"; filename="' . sanitize_file_name(basename($image_path)) . '"' . "\r\n";
        $body .= 'Content-Type: ' . mime_content_type($image_path) . "\r\n\r\n";
        $body .= file_get_contents($image_path) . "\r\n";
        $body .= '--' . $boundary . '--';

        // Make single API request based on account tier
        $response = wp_remote_post($endpoint, [
            'headers' => $headers,
            'body' => $body,
            'timeout' => $this->config['timeout'],
        ]);

        $response_time = microtime(true) - $start_time;
        
        // Check for HTTP errors
        if (is_wp_error($response)) {
            throw new Exception('HTTP request failed: ' . esc_html($response->get_error_message()));
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            $error_msg = "HTTP request failed with code: {$response_code}";
            if (!empty($response_body)) {
                // Handle HTML error pages (like 404 pages)
                if (strpos($response_body, '<!DOCTYPE html>') !== false) {
                    $error_msg .= " - Server returned HTML error page (endpoint not found)";
                } else {
                    $error_data = json_decode($response_body, true);
                    if (isset($error_data['Message'])) {
                        $error_msg .= " - " . sanitize_text_field($error_data['Message']);
                    } elseif (isset($error_data['message'])) {
                        $error_msg .= " - " . sanitize_text_field($error_data['message']);
                    } else {
                        $error_msg .= " - " . esc_html(substr($response_body, 0, 200));
                    }
                }
            }
            
            // Add tier-specific guidance
            if ($response_code === 404 || $response_code === 403) {
                $current_tier = isset($this->config['account_tier']) ? $this->config['account_tier'] : 'free';
                if ($current_tier === 'premium') {
                    $error_msg .= " (Premium endpoint not accessible. Try selecting 'Free' tier in settings.)";
                } else {
                    $error_msg .= " (Endpoint not found. Check API key validity.)";
                }
            } elseif ($response_code === 400) {
                $error_msg .= " (Bad request. Check image format and API key.)";
            }
            
            throw new Exception(esc_html($error_msg));
        }

        $data = json_decode($response_body, true);

        if (!is_array($data)) {
            throw new Exception('Invalid API response format. Response: ' . esc_html(substr($response_body, 0, 200)));
        }

        // Handle response based on endpoint used
        if ($use_objects_endpoint) {
            // Generate description from detected objects (free tier)
            $description = $this->generate_alt_text_from_objects($data, $options);
        } else {
            // Extract description directly (premium tier)
            $description = '';
            if (isset($data['BestOutcome']['Description'])) {
                $description = $data['BestOutcome']['Description'];
            }
        }
        
        if (empty($description)) {
            // For debugging - log the full response if WP_DEBUG is enabled
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.PHP.DevelopmentFunctions.error_log_print_r
                error_log('Cloudmersive API Response: ' . print_r($data, true));
            }
            if ($use_objects_endpoint) {
                throw new Exception('No objects detected or unable to generate description.');
            } else {
                throw new Exception('No description found in API response.');
            }
        }

        // Process the alt text
        $alt_text = $this->process_alt_text($description, $options);
        
        if (empty($alt_text)) {
            throw new Exception('Generated alt text did not meet quality requirements');
        }

        // Calculate quality score
        $quality_score = $this->calculate_quality_score($alt_text, $options);

        // Update stats
        $this->update_stats([
            'success' => true,
            'response_time' => $response_time,
            'cost' => $this->config['cost_per_call'],
        ]);

        return [
            'alt_text' => $alt_text,
            'quality_score' => $quality_score,
            'cost' => $this->config['cost_per_call'],
            'response_time' => $response_time,
            'provider' => $this->get_id(),
            'raw_response' => $description,
        ];
    }

    /**
     * Test provider connection
     */
    public function test_connection() {
        if (empty($this->config['api_key'])) {
            throw new Exception('API key not configured');
        }

        // Use tier-based endpoint for testing
        $tier = isset($this->config['account_tier']) ? $this->config['account_tier'] : 'free';
        $test_url = ($tier === 'premium') 
            ? $this->config['endpoint_premium']
            : $this->config['endpoint_free'];
        
        $boundary = wp_generate_password(24, false);
        $headers = [
            'Apikey' => trim($this->config['api_key']), // Use trim() not sanitize_text_field() to avoid key corruption
            'Content-Type' => 'multipart/form-data; boundary=' . $boundary,
        ];

        // Use a proper small test JPEG (100x100 blue square - known to work with Cloudmersive)
        $test_image_data = base64_decode('/9j/4AAQSkZJRgABAQEASABIAAD/2wBDAAMCAgMCAgMDAwMEAwMEBQgFBQQEBQoHBwYIDAoMDAsKCwsNDhIQDQ4RDgsLEBYQERMUFRUVDA8XGBYUGBIUFRT/2wBDAQMEBAUEBQkFBQkUDQsNFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBT/wAARCABkAGQDASIAAhEBAxEB/8QAHwAAAQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAAAgEDAwIEAwUFBAQAAAF9AQIDAAQRBRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkKFhcYGRolJicoKSo0NTY3ODk6Q0RFRkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqDhIWGh4iJipKTlJWWl5iZmqKjpKWmp6ipqrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl5ufo6erx8vP09fb3+Pn6/8QAHwEAAwEBAQEBAQEBAQAAAAAAAAECAwQFBgcICQoL/8QAtREAAgECBAQDBAcFBAQAAQJ3AAECAxEEBSExBhJBUQdhcRMiMoEIFEKRobHBCSMzUvAVYnLRChYkNOEl8RcYGRomJygpKjU2Nzg5OkNERUZHSElKU1RVVldYWVpjZGVmZ2hpanN0dXZ3eHl6goOEhYaHiImKkpOUlbaWmJmaoqOkpaanqKmqsrO0tba3uLm6wsPExcbHyMnK0tPU1dbX2Nna4uPk5ebn6Onq8vP09fb3+Pn6/9oADAMBAAIRAxEAPwD9/KKKKAP/2Q==');
        
        $body = '--' . $boundary . "\r\n";
        $body .= 'Content-Disposition: form-data; name="imageFile"; filename="test.jpg"' . "\r\n";
        $body .= 'Content-Type: image/jpeg' . "\r\n\r\n";
        $body .= $test_image_data . "\r\n";
        $body .= '--' . $boundary . '--';

        $response = wp_remote_post($test_url, [
            'headers' => $headers,
            'body' => $body,
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            throw new Exception('Connection test failed: ' . esc_html($response->get_error_message()));
        }

        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code !== 200) {
            $response_body = wp_remote_retrieve_body($response);
            $error_msg = "HTTP {$response_code}";
            if (!empty($response_body)) {
                // Handle HTML error responses (like 404 pages)
                if (strpos($response_body, '<!DOCTYPE html>') !== false) {
                    $error_msg .= ": Server returned HTML error page";
                } else {
                    $error_data = json_decode($response_body, true);
                    if (isset($error_data['Message'])) {
                        $error_msg .= ": " . sanitize_text_field($error_data['Message']);
                    } elseif (isset($error_data['message'])) {
                        $error_msg .= ": " . sanitize_text_field($error_data['message']);
                    } else {
                        $error_msg .= ": " . esc_html(substr($response_body, 0, 200));
                    }
                }
            }
            
            // Add tier-specific guidance
            if ($response_code === 404 || $response_code === 403) {
                $current_tier = isset($this->config['account_tier']) ? $this->config['account_tier'] : 'free';
                if ($current_tier === 'premium') {
                    $error_msg .= " (Premium endpoint not accessible. Your account may be free tier - try selecting 'Free' in settings)";
                } else {
                    $error_msg .= " (Free tier endpoint not accessible. Your API key may not have Image Recognition enabled. Get a free Image Recognition API key at https://account.cloudmersive.com/signup)";
                }
            } elseif ($response_code === 401) {
                $error_msg .= " (Invalid API key. Please check your key and ensure you copied it completely without spaces)";
            } elseif ($response_code === 500) {
                $error_msg .= " (Server error. This may indicate: invalid API key, API rate limit exceeded, or temporary API issue. Check your API key and quota at cloudmersive.com)";
            }
            
            throw new Exception('Connection test failed - ' . esc_html($error_msg));
        }

        return true;
    }

    /**
     * Get configuration fields for admin interface
     */
    public function get_config_fields() {
        return [
            'account_tier' => [
                'type' => 'select',
                'label' => 'Account Tier',
                'description' => 'Select your Cloudmersive account type. Free tier uses object detection (800 calls/month), Premium uses natural language descriptions.',
                'required' => true,
                'default' => 'free',
                'options' => [
                    ['label' => 'Free (Object Detection - 800 calls/month)', 'value' => 'free'],
                    ['label' => 'Premium (Natural Language Descriptions)', 'value' => 'premium'],
                ],
            ],
            'api_key' => [
                'type' => 'text',
                'label' => 'API Key',
                'description' => 'Your Cloudmersive API key. Get a free key at cloudmersive.com',
                'required' => true,
                'placeholder' => 'Enter your Cloudmersive API key',
            ],
            'enabled' => [
                'type' => 'checkbox',
                'label' => 'Enable Provider',
                'description' => 'Enable Cloudmersive for alt text generation',
                'default' => true,
            ],
            'priority' => [
                'type' => 'number',
                'label' => 'Priority',
                'description' => 'Provider execution order (1=first, 5=last). Note: Cloudmersive has messy API structure.',
                'default' => 5,
                'min' => 1,
                'max' => 10,
            ],
            'timeout' => [
                'type' => 'number',
                'label' => 'Timeout (seconds)',
                'description' => 'Maximum time to wait for API response',
                'default' => 30,
                'min' => 10,
                'max' => 60,
            ],
        ];
    }

    /**
     * Get configuration value
     */
    public function get_config_value($key, $default = null) {
        return isset($this->config[$key]) ? $this->config[$key] : $default;
    }

    /**
     * Get provider limitations
     */
    public function get_limitations() {
        return [
            'max_file_size' => '5MB',
            'max_dimensions' => '4096x4096',
            'supported_formats' => 'JPEG, PNG, GIF',
            'rate_limit' => '1 request/second (free tier)',
            'monthly_limit' => '800 requests/month (free tier)',
            'batch_processing' => false,
        ];
    }

    /**
     * Get pricing information
     */
    public function get_pricing() {
        return [
            'free_tier' => [
                'calls' => 800,
                'period' => 'month',
                'cost' => 0,
            ],
            'paid_tiers' => [
                [
                    'name' => 'Basic',
                    'calls' => 10000,
                    'period' => 'month',
                    'cost' => 9.99,
                ],
                [
                    'name' => 'Pro',
                    'calls' => 100000,
                    'period' => 'month',
                    'cost' => 49.99,
                ],
            ],
        ];
    }

    /**
     * Check if provider supports batch processing
     */
    public function supports_batch() {
        return false;
    }

    /**
     * Get supported image types
     */
    public function get_supported_types() {
        return [
            'image/jpeg' => 'JPEG',
            'image/png' => 'PNG',
            'image/gif' => 'GIF',
        ];
    }

    /**
     * Generate alt text from detected objects (free tier compatible)
     */
    private function generate_alt_text_from_objects($response_data, $options = []) {
        if (!isset($response_data['Objects']) || empty($response_data['Objects'])) {
            return '';
        }

        $objects = $response_data['Objects'];
        
        // Sort objects by confidence score (highest first)
        usort($objects, function($a, $b) {
            return ($b['Score'] ?? 0) <=> ($a['Score'] ?? 0);
        });

        $descriptions = [];
        $people_count = 0;
        $main_objects = [];

        // Process objects to create natural description
        foreach ($objects as $object) {
            $class_name = $object['ObjectClassName'] ?? '';
            $score = $object['Score'] ?? 0;
            
            // Only include objects with reasonable confidence (>= 30%)
            if ($score < 0.3) {
                continue;
            }

            if ($class_name === 'person') {
                $people_count++;
            } elseif (!empty($class_name) && !in_array($class_name, $main_objects)) {
                $main_objects[] = $class_name;
            }
        }

        // Build description
        $parts = [];
        
        if ($people_count > 0) {
            if ($people_count === 1) {
                $parts[] = 'a person';
            } else {
                $parts[] = $people_count . ' people';
            }
        }

        // Add main objects (limit to 3 most confident)
        $main_objects = array_slice($main_objects, 0, 3);
        if (!empty($main_objects)) {
            $parts = array_merge($parts, $main_objects);
        }

        if (empty($parts)) {
            return 'Image with various elements';
        }

        // Create natural language description
        if (count($parts) === 1) {
            return 'Image showing ' . $parts[0];
        } elseif (count($parts) === 2) {
            return 'Image showing ' . $parts[0] . ' and ' . $parts[1];
        } else {
            $last = array_pop($parts);
            return 'Image showing ' . implode(', ', $parts) . ' and ' . $last;
        }
    }
}