<?php
/**
 * REST API Endpoints
 * 
 * @package CM_Auto_Alt_Text
 * @version 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * REST API for CM Auto Alt Text
 */
class CM_Auto_Alt_Text_REST_API {

    /**
     * Initialize the REST API
     */
    public static function init() {
        add_action('rest_api_init', [__CLASS__, 'register_routes']);
    }

    /**
     * Register REST API routes
     */
    public static function register_routes() {
        $namespace = 'cm-auto-alt-text/v1';

        // Provider management
        register_rest_route($namespace, '/providers', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_providers'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
        ]);

        register_rest_route($namespace, '/providers/(?P<id>[a-zA-Z0-9_-]+)', [
            'methods' => 'PUT',
            'callback' => [__CLASS__, 'update_provider'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
            'args' => [
                'id' => [
                    'required' => true,
                    'validate_callback' => function($param) {
                        return is_string($param) && !empty($param);
                    },
                ],
            ],
        ]);

        // Provider testing
        register_rest_route($namespace, '/providers/(?P<id>[a-zA-Z0-9_-]+)/test', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'test_provider'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
        ]);

        register_rest_route($namespace, '/providers/test-all', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'test_all_providers'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
        ]);

        // Statistics
        register_rest_route($namespace, '/stats/providers', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_provider_stats'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
        ]);

        register_rest_route($namespace, '/stats/costs', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_cost_stats'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
        ]);

        register_rest_route($namespace, '/stats/performance', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'get_performance_stats'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
        ]);

        // Settings
        register_rest_route($namespace, '/settings', [
            [
                'methods' => 'GET',
                'callback' => [__CLASS__, 'get_settings'],
                'permission_callback' => [__CLASS__, 'check_permissions'],
            ],
            [
                'methods' => 'PUT',
                'callback' => [__CLASS__, 'update_settings'],
                'permission_callback' => [__CLASS__, 'check_permissions'],
            ],
        ]);

        // Image processing
        register_rest_route($namespace, '/generate/(?P<id>\d+)', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'generate_alt_text'],
            'permission_callback' => [__CLASS__, 'check_edit_permissions'],
            'args' => [
                'id' => [
                    'required' => true,
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    },
                ],
            ],
        ]);

        // Bulk operations
        register_rest_route($namespace, '/bulk/process', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'bulk_process'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
        ]);

        // Configuration export/import
        register_rest_route($namespace, '/config/export', [
            'methods' => 'GET',
            'callback' => [__CLASS__, 'export_config'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
        ]);

        register_rest_route($namespace, '/config/import', [
            'methods' => 'POST',
            'callback' => [__CLASS__, 'import_config'],
            'permission_callback' => [__CLASS__, 'check_permissions'],
        ]);
    }

    /**
     * Check permissions for admin endpoints
     */
    public static function check_permissions() {
        return current_user_can('manage_options');
    }

    /**
     * Check permissions for edit endpoints
     */
    public static function check_edit_permissions($request) {
        $attachment_id = $request->get_param('id');
        return current_user_can('edit_post', $attachment_id);
    }

    /**
     * Get all providers
     */
    public static function get_providers($request) {
        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $providers = $provider_manager->get_all_providers();
        
        // Exclude Azure and AWS from admin dashboard (keeping code for future development)
        // These providers have complex API key setup processes
        $excluded_from_ui = ['azure_vision', 'aws_rekognition'];
        
        $response = [];
        foreach ($providers as $id => $provider) {
            // Skip excluded providers in admin UI
            if (in_array($id, $excluded_from_ui, true)) {
                continue;
            }
            
            $response[$id] = [
                'id' => $provider->get_id(),
                'name' => $provider->get_display_name(),
                'description' => $provider->get_description(),
                'enabled' => $provider->is_enabled(),
                'configured' => $provider->is_configured(),
                'priority' => $provider->get_priority(),
                'cost_per_call' => $provider->get_cost_per_call(),
                'config_fields' => $provider->get_config_fields(),
                'limitations' => $provider->get_limitations(),
                'pricing' => $provider->get_pricing(),
                'supports_batch' => $provider->supports_batch(),
                'supported_types' => $provider->get_supported_types(),
            ];
        }

        return rest_ensure_response($response);
    }

    /**
     * Update provider configuration
     */
    public static function update_provider($request) {
        $provider_id = $request->get_param('id');
        $config = $request->get_json_params();

        if (empty($config) || !is_array($config)) {
            return new WP_Error('invalid_config', 'Invalid configuration data', ['status' => 400]);
        }

        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $success = $provider_manager->update_provider_config($provider_id, $config);

        if (!$success) {
            return new WP_Error('provider_not_found', 'Provider not found', ['status' => 404]);
        }

        return rest_ensure_response(['success' => true, 'message' => 'Provider updated successfully']);
    }

    /**
     * Test provider connection
     */
    public static function test_provider($request) {
        $provider_id = $request->get_param('id');
        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $provider = $provider_manager->get_provider($provider_id);

        if (!$provider) {
            return new WP_Error('provider_not_found', 'Provider not found', ['status' => 404]);
        }

        try {
            $result = $provider->test_connection();
            return rest_ensure_response([
                'success' => true,
                'message' => 'Connection successful',
                'provider' => $provider->get_display_name(),
            ]);
        } catch (Exception $e) {
            return rest_ensure_response([
                'success' => false,
                'message' => $e->getMessage(),
                'provider' => $provider->get_display_name(),
            ]);
        }
    }

    /**
     * Test all providers
     */
    public static function test_all_providers($request) {
        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $results = $provider_manager->test_all_providers();
        
        return rest_ensure_response($results);
    }

    /**
     * Get provider statistics
     */
    public static function get_provider_stats($request) {
        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $stats = $provider_manager->get_provider_stats();
        
        return rest_ensure_response($stats);
    }

    /**
     * Get cost statistics
     */
    public static function get_cost_stats($request) {
        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $costs = $provider_manager->get_monthly_costs();
        
        return rest_ensure_response($costs);
    }

    /**
     * Get performance statistics
     */
    public static function get_performance_stats($request) {
        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $performance = $provider_manager->get_performance_comparison();
        
        return rest_ensure_response($performance);
    }

    /**
     * Get plugin settings
     */
    public static function get_settings($request) {
        $settings = [
            'enabled' => get_option('auto_alt_text_enabled', 1),
            'max_length' => get_option('auto_alt_text_max_length', 125),
            'min_length' => get_option('auto_alt_text_min_length', 10),
            'language' => get_option('auto_alt_text_language', 'en'),
            'style' => get_option('auto_alt_text_style', 'descriptive'),
            'fallback' => get_option('auto_alt_text_fallback', ''),
            'bulk_limit' => get_option('auto_alt_text_bulk_limit', 50),
            'include_keywords' => get_option('auto_alt_text_include_keywords', ''),
            'quality_threshold' => get_option('auto_alt_text_quality_threshold', 60),
            'auto_approve' => get_option('auto_alt_text_auto_approve', 1),
        ];

        return rest_ensure_response($settings);
    }

    /**
     * Update plugin settings
     */
    public static function update_settings($request) {
        $settings = $request->get_json_params();

        if (empty($settings) || !is_array($settings)) {
            return new WP_Error('invalid_settings', 'Invalid settings data', ['status' => 400]);
        }

        $allowed_settings = [
            'auto_alt_text_enabled' => 'boolean',
            'auto_alt_text_max_length' => 'integer',
            'auto_alt_text_min_length' => 'integer',
            'auto_alt_text_language' => 'string',
            'auto_alt_text_style' => 'string',
            'auto_alt_text_fallback' => 'string',
            'auto_alt_text_bulk_limit' => 'integer',
            'auto_alt_text_include_keywords' => 'string',
            'auto_alt_text_quality_threshold' => 'integer',
            'auto_alt_text_auto_approve' => 'boolean',
        ];

        foreach ($settings as $key => $value) {
            $option_key = 'auto_alt_text_' . $key;
            
            if (isset($allowed_settings[$option_key])) {
                $type = $allowed_settings[$option_key];
                
                switch ($type) {
                    case 'boolean':
                        $value = (bool) $value;
                        break;
                    case 'integer':
                        $value = absint($value);
                        break;
                    case 'string':
                        $value = sanitize_text_field($value);
                        break;
                }
                
                update_option($option_key, $value);
            }
        }

        return rest_ensure_response(['success' => true, 'message' => 'Settings updated successfully']);
    }

    /**
     * Generate alt text for a specific image
     */
    public static function generate_alt_text($request) {
        $attachment_id = absint($request->get_param('id'));
        $provider = $request->get_param('provider');
        $options = $request->get_json_params() ?: [];

        if ($provider && $provider !== 'auto') {
            $options['preferred_provider'] = sanitize_text_field($provider);
        }

        try {
            $alt_text = auto_alt_text_generate_alt($attachment_id, $options);
            
            if ($alt_text) {
                // Get generation metadata
                $metadata = get_post_meta($attachment_id, '_cm_alt_text_generation_log', true);
                $last_generation = is_array($metadata) ? end($metadata) : [];
                
                return rest_ensure_response([
                    'success' => true,
                    'alt_text' => $alt_text,
                    'provider' => $last_generation['provider'] ?? 'unknown',
                    'quality_score' => $last_generation['quality_score'] ?? 0,
                    'cost' => $last_generation['cost'] ?? 0,
                    'response_time' => $last_generation['response_time'] ?? 0,
                ]);
            } else {
                return new WP_Error('generation_failed', 'Failed to generate alt text', ['status' => 500]);
            }
        } catch (Exception $e) {
            return new WP_Error('generation_error', $e->getMessage(), ['status' => 500]);
        }
    }

    /**
     * Start bulk processing
     */
    public static function bulk_process($request) {
        $params = $request->get_json_params();
        $limit = isset($params['limit']) ? absint($params['limit']) : 50;
        $date_filter = isset($params['date_filter']) ? sanitize_text_field($params['date_filter']) : 'all';
        $provider = isset($params['provider']) ? sanitize_text_field($params['provider']) : 'auto';

        // Find images without alt text using optimized custom query
        // This approach avoids slow meta_query for better performance on large sites
        global $wpdb;
        
        // Build query parameters
        $sql_params = [];
        $date_conditions = '';
        
        if ($date_filter !== 'all' && preg_match('/^\d{4}-\d{2}$/', $date_filter)) {
            list($year, $month) = explode('-', $date_filter);
            $date_from = sprintf('%04d-%02d-01 00:00:00', absint($year), absint($month));
            $date_to = gmdate('Y-m-t 23:59:59', mktime(0, 0, 0, absint($month), 1, absint($year)));
            $date_conditions = " AND p.post_date >= %s AND p.post_date <= %s";
            $sql_params[] = $date_from;
            $sql_params[] = $date_to;
        }
        
        // Add limit parameter
        $sql_params[] = $limit;
        
        // Create cache key based on parameters for optimized query results
        $cache_key = 'auto_alt_text_bulk_' . md5(serialize([$limit, $date_filter]));
        $image_ids = wp_cache_get($cache_key, 'auto_alt_text');
        
        if (false === $image_ids) {
            // Use optimized approach: get all image attachments first, then filter programmatically
            // This avoids complex meta_query while maintaining WordPress standards compliance
            $query_args = [
                'post_type' => 'attachment',
                'post_mime_type' => 'image',
                'posts_per_page' => $limit * 3, // Get more to account for filtering
                'post_status' => 'inherit',
                'fields' => 'ids',
                'no_found_rows' => true,
                'update_post_meta_cache' => false,
                'update_post_term_cache' => false,
                'orderby' => 'date',
                'order' => 'DESC',
            ];
            
            if ($date_filter !== 'all' && preg_match('/^\d{4}-\d{2}$/', $date_filter)) {
                list($year, $month) = explode('-', $date_filter);
                $query_args['date_query'] = [
                    [
                        'year' => absint($year),
                        'month' => absint($month),
                    ],
                ];
            }
            
            $all_images = new WP_Query($query_args);
            $image_ids = [];
            $count = 0;
            
            // Filter for images without alt text
            foreach ($all_images->posts as $attachment_id) {
                if ($count >= $limit) {
                    break;
                }
                
                $existing_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
                if (empty($existing_alt)) {
                    $image_ids[] = $attachment_id;
                    $count++;
                }
            }
            
            wp_reset_postdata();
            
            // Cache for 5 minutes to avoid repeated queries
            wp_cache_set($cache_key, $image_ids, 'auto_alt_text', 300);
        }
        
        // Create a fake query result object for compatibility
        $images = new stdClass();
        $images->posts = $image_ids;
        $processed = 0;
        $time = time();

        if (!empty($images->posts)) {
            foreach ($images->posts as $attachment_id) {
                // Double-check alt text for safety (though our custom query should be accurate)
                $existing_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
                
                if ('' === $existing_alt || false === $existing_alt) {
                    wp_schedule_single_event($time + $processed, 'auto_alt_text_bulk_event', [
                        $attachment_id,
                        ['provider' => $provider]
                    ]);
                    $processed++;
                }
            }
        }

        wp_reset_postdata();

        return rest_ensure_response([
            'success' => true,
            'scheduled' => $processed,
            'message' => sprintf('Scheduled %d images for processing', $processed),
        ]);
    }

    /**
     * Export configuration
     */
    public static function export_config($request) {
        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $config = $provider_manager->export_config();
        
        return rest_ensure_response($config);
    }

    /**
     * Import configuration
     */
    public static function import_config($request) {
        $config = $request->get_json_params();

        if (empty($config) || !is_array($config)) {
            return new WP_Error('invalid_config', 'Invalid configuration data', ['status' => 400]);
        }

        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $success = $provider_manager->import_config($config);

        if ($success) {
            return rest_ensure_response(['success' => true, 'message' => 'Configuration imported successfully']);
        } else {
            return new WP_Error('import_failed', 'Failed to import configuration', ['status' => 500]);
        }
    }
}

// Initialize the REST API
CM_Auto_Alt_Text_REST_API::init();