<?php
/**
 * Provider Manager Class
 * 
 * @package CM_Auto_Alt_Text
 * @version 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Manages all AI providers and their configuration
 */
class CM_Auto_Alt_Text_Provider_Manager {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Available providers
     */
    private $providers = [];
    
    /**
     * Provider configurations
     */
    private $config = [];

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->load_config();
        $this->register_providers();
    }

    /**
     * Load configuration from WordPress options
     */
    private function load_config() {
        $this->config = get_option('auto_alt_text_providers', []);
    }

    /**
     * Save configuration to WordPress options
     */
    public function save_config() {
        update_option('auto_alt_text_providers', $this->config);
    }

    /**
     * Register all available providers
     */
    private function register_providers() {
        // Cloudmersive Provider
        $this->register_provider('cloudmersive', 'CM_Auto_Alt_Text_Cloudmersive_Provider');
        
        // OpenAI Provider
        $this->register_provider('openai', 'CM_Auto_Alt_Text_OpenAI_Provider');
        
        // Google Vision Provider
        $this->register_provider('google_vision', 'CM_Auto_Alt_Text_Google_Vision_Provider');
        
        // Azure Vision Provider
        $this->register_provider('azure_vision', 'CM_Auto_Alt_Text_Azure_Vision_Provider');
        
        // AWS Rekognition Provider
        $this->register_provider('aws_rekognition', 'CM_Auto_Alt_Text_AWS_Rekognition_Provider');
    }

    /**
     * Register a single provider
     */
    private function register_provider($id, $class_name) {
        if (class_exists($class_name)) {
            $provider_config = $this->config[$id] ?? [];
            $this->providers[$id] = new $class_name($provider_config);
        }
    }

    /**
     * Get all registered providers
     */
    public function get_all_providers() {
        return $this->providers;
    }

    /**
     * Get enabled providers sorted by priority
     */
    public function get_enabled_providers() {
        $enabled = array_filter($this->providers, function($provider) {
            return $provider->is_enabled();
        });

        // Sort by priority (lower number = higher priority)
        uasort($enabled, function($a, $b) {
            return $a->get_priority() - $b->get_priority();
        });

        return $enabled;
    }

    /**
     * Get provider by ID
     */
    public function get_provider($id) {
        return isset($this->providers[$id]) ? $this->providers[$id] : null;
    }

    /**
     * Update provider configuration
     */
    public function update_provider_config($provider_id, $config) {
        if (isset($this->providers[$provider_id])) {
            $this->config[$provider_id] = array_merge(
                $this->config[$provider_id] ?? [],
                $config
            );
            $this->save_config();
            
            // Reload the provider with new config
            $class_name = get_class($this->providers[$provider_id]);
            $this->providers[$provider_id] = new $class_name($this->config[$provider_id]);
            
            return true;
        }
        return false;
    }

    /**
     * Enable/disable a provider
     */
    public function set_provider_enabled($provider_id, $enabled) {
        if (isset($this->providers[$provider_id])) {
            $this->config[$provider_id]['enabled'] = (bool) $enabled;
            $this->save_config();
            
            // Reload the provider
            $class_name = get_class($this->providers[$provider_id]);
            $this->providers[$provider_id] = new $class_name($this->config[$provider_id]);
            
            return true;
        }
        return false;
    }

    /**
     * Set provider priority
     */
    public function set_provider_priority($provider_id, $priority) {
        if (isset($this->providers[$provider_id])) {
            $this->config[$provider_id]['priority'] = intval($priority);
            $this->save_config();
            
            // Reload the provider
            $class_name = get_class($this->providers[$provider_id]);
            $this->providers[$provider_id] = new $class_name($this->config[$provider_id]);
            
            return true;
        }
        return false;
    }

    /**
     * Test all enabled providers
     */
    public function test_all_providers() {
        $results = [];
        
        foreach ($this->get_enabled_providers() as $id => $provider) {
            try {
                $result = $provider->test_connection();
                $results[$id] = [
                    'success' => true,
                    'message' => 'Connection successful',
                    'provider' => $provider->get_display_name(),
                ];
            } catch (Exception $e) {
                $results[$id] = [
                    'success' => false,
                    'message' => $e->getMessage(),
                    'provider' => $provider->get_display_name(),
                ];
            }
        }
        
        return $results;
    }

    /**
     * Get best provider for image type
     */
    public function get_best_provider($image_path, $options = []) {
        $enabled_providers = $this->get_enabled_providers();
        
        if (empty($enabled_providers)) {
            return null;
        }

        // If specific provider requested
        if (isset($options['preferred_provider'])) {
            $preferred = $this->get_provider($options['preferred_provider']);
            if ($preferred && $preferred->is_enabled()) {
                return $preferred;
            }
        }

        // Get image info for smart selection
        $image_info = getimagesize($image_path);
        $file_size = filesize($image_path);
        
        // Apply intelligent selection logic
        foreach ($enabled_providers as $provider) {
            // Check file size limits
            $max_size = $provider->get_config_value('max_file_size', 5 * 1024 * 1024);
            if ($file_size > $max_size) {
                continue;
            }

            // Check image dimensions if needed
            $max_width = $provider->get_config_value('max_width', 10000);
            $max_height = $provider->get_config_value('max_height', 10000);
            
            if ($image_info && ($image_info[0] > $max_width || $image_info[1] > $max_height)) {
                continue;
            }

            // This provider can handle the image
            return $provider;
        }

        // Fallback to first enabled provider
        return reset($enabled_providers);
    }

    /**
     * Get provider statistics summary
     */
    public function get_provider_stats() {
        $stats = [];
        
        foreach ($this->providers as $id => $provider) {
            $provider_stats = $provider->get_stats();
            $stats[$id] = [
                'name' => $provider->get_display_name(),
                'enabled' => $provider->is_enabled(),
                'configured' => $provider->is_configured(),
                'calls_made' => $provider_stats['calls_made'],
                'total_cost' => $provider_stats['total_cost'],
                'success_rate' => $provider_stats['success_rate'],
                'avg_response_time' => $provider_stats['avg_response_time'],
            ];
        }
        
        return $stats;
    }

    /**
     * Get monthly cost summary
     */
    public function get_monthly_costs() {
        $monthly_costs = get_option('auto_alt_text_monthly_costs', []);
        $current_month = gmdate('Y-m');
        
        $summary = [
            'current_month' => $current_month,
            'current_total' => 0,
            'current_calls' => 0,
            'providers' => [],
            'history' => [],
        ];

        // Current month summary
        if (isset($monthly_costs[$current_month])) {
            foreach ($monthly_costs[$current_month] as $provider_id => $data) {
                $summary['current_total'] += $data['cost'];
                $summary['current_calls'] += $data['calls'];
                $summary['providers'][$provider_id] = $data;
            }
        }

        // Historical data (last 6 months)
        $months = [];
        for ($i = 0; $i < 6; $i++) {
            $month = gmdate('Y-m', strtotime("-$i months"));
            $months[] = $month;
        }

        foreach ($months as $month) {
            if (isset($monthly_costs[$month])) {
                $month_total = 0;
                $month_calls = 0;
                
                foreach ($monthly_costs[$month] as $data) {
                    $month_total += $data['cost'];
                    $month_calls += $data['calls'];
                }
                
                $summary['history'][$month] = [
                    'total_cost' => $month_total,
                    'total_calls' => $month_calls,
                ];
            }
        }

        return $summary;
    }

    /**
     * Get provider performance comparison
     */
    public function get_performance_comparison() {
        $comparison = [];
        
        foreach ($this->providers as $id => $provider) {
            if (!$provider->is_enabled()) {
                continue;
            }
            
            $stats = $provider->get_stats();
            $comparison[$id] = [
                'name' => $provider->get_display_name(),
                'cost_per_call' => $provider->get_cost_per_call(),
                'success_rate' => $stats['success_rate'],
                'avg_response_time' => $stats['avg_response_time'],
                'total_calls' => $stats['calls_made'],
                'score' => $this->calculate_provider_score($stats, $provider->get_cost_per_call()),
            ];
        }

        // Sort by score (higher is better)
        uasort($comparison, function($a, $b) {
            return $b['score'] - $a['score'];
        });

        return $comparison;
    }

    /**
     * Calculate provider performance score
     */
    private function calculate_provider_score($stats, $cost_per_call) {
        $score = 0;
        
        // Success rate (0-40 points)
        $score += ($stats['success_rate'] / 100) * 40;
        
        // Response time (0-30 points) - faster is better
        $max_acceptable_time = 5; // 5 seconds
        if ($stats['avg_response_time'] <= $max_acceptable_time) {
            $score += (1 - ($stats['avg_response_time'] / $max_acceptable_time)) * 30;
        }
        
        // Cost efficiency (0-30 points) - lower cost is better
        $max_acceptable_cost = 0.01; // $0.01 per call
        if ($cost_per_call <= $max_acceptable_cost) {
            $score += (1 - ($cost_per_call / $max_acceptable_cost)) * 30;
        }
        
        return round($score, 2);
    }

    /**
     * Reset provider statistics
     */
    public function reset_provider_stats($provider_id = null) {
        if ($provider_id) {
            delete_option('auto_alt_text_provider_stats_' . $provider_id);
            if (isset($this->providers[$provider_id])) {
                $this->providers[$provider_id]->reset_stats();
            }
        } else {
            // Reset all providers
            foreach ($this->providers as $id => $provider) {
                delete_option('auto_alt_text_provider_stats_' . $id);
                $provider->reset_stats();
            }
            
            // Reset monthly costs
            delete_option('auto_alt_text_monthly_costs');
        }
    }

    /**
     * Export provider configuration
     */
    public function export_config() {
        return [
            'providers' => $this->config,
            'stats' => $this->get_provider_stats(),
            'monthly_costs' => get_option('auto_alt_text_monthly_costs', []),
            'exported_at' => current_time('mysql'),
            'plugin_version' => CM_AUTO_ALT_TEXT_VERSION,
        ];
    }

    /**
     * Import provider configuration
     */
    public function import_config($config_data) {
        if (!is_array($config_data) || !isset($config_data['providers'])) {
            return false;
        }

        // Validate the configuration
        foreach ($config_data['providers'] as $provider_id => $config) {
            if (!isset($this->providers[$provider_id])) {
                continue; // Skip unknown providers
            }
            
            $this->config[$provider_id] = $config;
        }

        $this->save_config();
        
        // Reload all providers
        $this->register_providers();
        
        return true;
    }
}