<?php
/**
 * Abstract AI Provider Class
 * 
 * @package CM_Auto_Alt_Text
 * @version 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Abstract base class for AI providers
 */
abstract class CM_Auto_Alt_Text_Abstract_Provider {
    
    /**
     * Provider configuration
     */
    protected $config = [];
    
    /**
     * Provider stats
     */
    protected $stats = [
        'calls_made' => 0,
        'total_cost' => 0,
        'success_rate' => 0,
        'avg_response_time' => 0,
    ];

    /**
     * Constructor
     */
    public function __construct($config = []) {
        $this->config = array_merge($this->get_default_config(), $config);
        $this->load_stats();
    }

    /**
     * Get provider ID (must be unique)
     */
    abstract public function get_id();

    /**
     * Get provider display name
     */
    abstract public function get_display_name();

    /**
     * Get provider description
     */
    abstract public function get_description();

    /**
     * Get default configuration
     */
    abstract protected function get_default_config();

    /**
     * Generate alt text for an image
     * 
     * @param string $image_path Path to the image file
     * @param array $options Generation options
     * @return array|false Result array with alt_text, quality_score, cost, etc.
     */
    abstract public function generate_alt_text($image_path, $options = []);

    /**
     * Test provider connection and API key
     */
    abstract public function test_connection();

    /**
     * Get configuration fields for admin interface
     */
    abstract public function get_config_fields();

    /**
     * Check if provider is enabled and configured
     */
    public function is_enabled() {
        return !empty($this->config['enabled']) && $this->is_configured();
    }

    /**
     * Check if provider is properly configured
     */
    public function is_configured() {
        return !empty($this->config['api_key']);
    }

    /**
     * Get provider name for display
     */
    public function get_name() {
        return $this->get_display_name();
    }

    /**
     * Get provider priority (lower = higher priority)
     */
    public function get_priority() {
        return isset($this->config['priority']) ? intval($this->config['priority']) : 999;
    }

    /**
     * Get cost per API call
     */
    public function get_cost_per_call() {
        return isset($this->config['cost_per_call']) ? floatval($this->config['cost_per_call']) : 0.0;
    }

    /**
     * Get provider statistics
     */
    public function get_stats() {
        return $this->stats;
    }

    /**
     * Update provider statistics
     */
    protected function update_stats($call_data) {
        $this->stats['calls_made']++;
        
        if (isset($call_data['cost'])) {
            $this->stats['total_cost'] += $call_data['cost'];
        }
        
        if (isset($call_data['success'])) {
            $total_calls = $this->stats['calls_made'];
            $previous_successes = ($this->stats['success_rate'] / 100) * ($total_calls - 1);
            $current_successes = $previous_successes + ($call_data['success'] ? 1 : 0);
            $this->stats['success_rate'] = ($current_successes / $total_calls) * 100;
        }
        
        if (isset($call_data['response_time'])) {
            $total_calls = $this->stats['calls_made'];
            $this->stats['avg_response_time'] = (
                ($this->stats['avg_response_time'] * ($total_calls - 1)) + $call_data['response_time']
            ) / $total_calls;
        }
        
        $this->save_stats();
    }

    /**
     * Load provider statistics
     */
    protected function load_stats() {
        $saved_stats = get_option('auto_alt_text_provider_stats_' . $this->get_id(), []);
        $this->stats = array_merge($this->stats, $saved_stats);
    }

    /**
     * Save provider statistics
     */
    protected function save_stats() {
        update_option('auto_alt_text_provider_stats_' . $this->get_id(), $this->stats);
    }

    /**
     * Validate image file
     */
    protected function validate_image($image_path) {
        if (!file_exists($image_path)) {
            throw new Exception('Image file does not exist: ' . esc_html(sanitize_text_field($image_path)));
        }

        $file_size = filesize($image_path);
        $max_size = $this->config['max_file_size'] ?? (5 * 1024 * 1024); // 5MB default
        
        if ($file_size > $max_size) {
            throw new Exception('Image file too large: ' . esc_html(intval($file_size)) . ' bytes (max: ' . esc_html(intval($max_size)) . ')');
        }

        $image_info = getimagesize($image_path);
        if (!$image_info) {
            throw new Exception('Invalid image file: ' . esc_html(sanitize_text_field($image_path)));
        }

        $supported_types = $this->config['supported_types'] ?? [IMAGETYPE_JPEG, IMAGETYPE_PNG, IMAGETYPE_GIF, IMAGETYPE_WEBP];
        if (!in_array($image_info[2], $supported_types)) {
            throw new Exception('Unsupported image type: ' . esc_html(sanitize_text_field($image_info['mime'])));
        }

        return true;
    }

    /**
     * Make HTTP request with timeout and error handling
     */
    protected function make_request($url, $args = []) {
        $start_time = microtime(true);
        
        $default_args = [
            'timeout' => 30,
            'user-agent' => 'CM-Auto-Alt-Text/' . CM_AUTO_ALT_TEXT_VERSION . '; ' . home_url(),
        ];
        
        $args = array_merge($default_args, $args);
        
        $response = wp_remote_request($url, $args);
        $response_time = microtime(true) - $start_time;
        
        if (is_wp_error($response)) {
            $this->update_stats([
                'success' => false,
                'response_time' => $response_time,
            ]);
            throw new Exception('HTTP request failed: ' . esc_html(sanitize_text_field($response->get_error_message())));
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code < 200 || $response_code >= 300) {
            $this->update_stats([
                'success' => false,
                'response_time' => $response_time,
            ]);
            throw new Exception('HTTP request failed with code: ' . esc_html(intval($response_code)));
        }
        
        $this->update_stats([
            'success' => true,
            'response_time' => $response_time,
        ]);
        
        return $response;
    }

    /**
     * Process alt text to meet requirements
     */
    protected function process_alt_text($raw_text, $options = []) {
        if (empty($raw_text)) {
            return '';
        }

        // Clean up the text
        $text = trim($raw_text);
        $text = preg_replace('/\s+/', ' ', $text); // Normalize whitespace
        $text = ucfirst($text); // Capitalize first letter
        
        // Remove trailing punctuation except periods
        $text = rtrim($text, '.,!?;:');
        
        // Ensure it doesn't end with a period unless it's a complete sentence
        if (str_word_count($text) > 5 && !preg_match('/[.!?]$/', $text)) {
            $text .= '.';
        }

        // Apply length constraints
        $max_length = $options['max_length'] ?? 125;
        $min_length = $options['min_length'] ?? 10;
        
        if (strlen($text) > $max_length) {
            $text = $this->truncate_text($text, $max_length);
        }
        
        if (strlen($text) < $min_length) {
            // Text too short, might not be descriptive enough
            return '';
        }

        // Apply style preferences
        $style = $options['style'] ?? 'descriptive';
        $text = $this->apply_style($text, $style, $options);

        return $text;
    }

    /**
     * Truncate text to specified length at word boundary
     */
    protected function truncate_text($text, $max_length) {
        if (strlen($text) <= $max_length) {
            return $text;
        }

        $truncated = substr($text, 0, $max_length);
        $last_space = strrpos($truncated, ' ');
        
        if ($last_space !== false && $last_space > ($max_length * 0.75)) {
            $truncated = substr($truncated, 0, $last_space);
        }
        
        return rtrim($truncated, '.,!?;:');
    }

    /**
     * Apply style preferences to alt text
     */
    protected function apply_style($text, $style, $options = []) {
        switch ($style) {
            case 'seo':
                return $this->apply_seo_style($text, $options);
            case 'accessible':
                return $this->apply_accessible_style($text, $options);
            case 'descriptive':
            default:
                return $text;
        }
    }

    /**
     * Apply SEO-focused styling
     */
    protected function apply_seo_style($text, $options) {
        $keywords = $options['include_keywords'] ?? '';
        if (!empty($keywords)) {
            $keyword_array = array_map('trim', explode(',', $keywords));
            foreach ($keyword_array as $keyword) {
                if (!empty($keyword) && stripos($text, $keyword) === false) {
                    // Try to naturally incorporate the keyword
                    $text = $this->incorporate_keyword($text, $keyword);
                    break; // Only add one keyword to avoid keyword stuffing
                }
            }
        }
        return $text;
    }

    /**
     * Apply accessibility-focused styling
     */
    protected function apply_accessible_style($text, $options) {
        // Ensure the text is descriptive and functional
        $context = $options['context'] ?? [];
        
        if (isset($context['is_featured']) && $context['is_featured']) {
            $text = 'Featured image: ' . $text;
        }
        
        return $text;
    }

    /**
     * Incorporate keyword naturally into text
     */
    protected function incorporate_keyword($text, $keyword) {
        // Simple implementation - can be enhanced
        if (strlen($text . ' featuring ' . $keyword) <= 125) {
            return $text . ' featuring ' . $keyword;
        }
        return $text;
    }

    /**
     * Calculate quality score for generated alt text
     */
    protected function calculate_quality_score($alt_text, $options = []) {
        $score = 0;
        $max_score = 100;

        // Length check (0-20 points)
        $length = strlen($alt_text);
        $min_length = $options['min_length'] ?? 10;
        $max_length = $options['max_length'] ?? 125;
        $optimal_length = ($min_length + $max_length) / 2;
        
        $length_score = 20 - (abs($length - $optimal_length) / $optimal_length) * 20;
        $score += max(0, $length_score);

        // Descriptiveness (0-30 points)
        $word_count = str_word_count($alt_text);
        if ($word_count >= 3) $score += 10;
        if ($word_count >= 5) $score += 10;
        if ($word_count >= 8) $score += 10;

        // Avoid generic terms (-10 points each)
        $generic_terms = ['image', 'photo', 'picture', 'graphic'];
        foreach ($generic_terms as $term) {
            if (stripos($alt_text, $term) !== false) {
                $score -= 10;
            }
        }

        // Grammar and readability (0-20 points)
        if (ucfirst($alt_text) === $alt_text) $score += 5; // Starts with capital
        if (preg_match('/[.!?]$/', $alt_text)) $score += 5; // Ends with punctuation
        if (!preg_match('/\s{2,}/', $alt_text)) $score += 5; // No double spaces
        if (preg_match('/^[A-Za-z0-9\s.,!?;:\'-]+$/', $alt_text)) $score += 5; // Only valid characters

        // Context relevance (0-30 points)
        $context = $options['context'] ?? [];
        if (!empty($context['categories']) || !empty($context['tags'])) {
            $score += 15; // Has context
        }
        if (isset($context['is_featured']) && $context['is_featured']) {
            $score += 15; // Featured image context
        }

        return min($max_score, max(0, $score));
    }
}