<?php
/*
Plugin Name: CM Auto Alt Text
Description: Automatically generates alt text for images using multiple AI providers (OpenAI, Google Vision, Azure, AWS, Cloudmersive) with intelligent fallback and cost optimization.
Version: 1.4.0
Author: Bitshost Team
Requires at least: 6.0
Tested up to: 6.8
Requires PHP: 8.0
License: GPL v2.0 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
Text Domain: cm-auto-alt-text
Domain Path: /languages
*/

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CM_AUTO_ALT_TEXT_VERSION', '1.4.0');
define('CM_AUTO_ALT_TEXT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CM_AUTO_ALT_TEXT_PLUGIN_URL', plugin_dir_url(__FILE__));

// Include provider system
require_once CM_AUTO_ALT_TEXT_PLUGIN_DIR . 'includes/class-provider-manager.php';
require_once CM_AUTO_ALT_TEXT_PLUGIN_DIR . 'includes/abstract-ai-provider.php';
require_once CM_AUTO_ALT_TEXT_PLUGIN_DIR . 'includes/providers/class-cloudmersive-provider.php';
require_once CM_AUTO_ALT_TEXT_PLUGIN_DIR . 'includes/providers/class-openai-provider.php';
require_once CM_AUTO_ALT_TEXT_PLUGIN_DIR . 'includes/providers/class-google-vision-provider.php';
require_once CM_AUTO_ALT_TEXT_PLUGIN_DIR . 'includes/providers/class-azure-vision-provider.php';
require_once CM_AUTO_ALT_TEXT_PLUGIN_DIR . 'includes/providers/class-aws-rekognition-provider.php';

/**
 * Core alt text generation function with multi-provider support
 * 
 * @param int $attachment_id The attachment ID to generate alt text for
 * @param array $options Generation options (provider, fallback_chain, etc.)
 * @return string|false Generated alt text or false on failure
 */
function auto_alt_text_generate_alt($attachment_id, $options = []) {
    // Validate attachment ID
    $attachment_id = absint($attachment_id);
    if (!$attachment_id) {
        return false;
    }

    // Check if plugin is enabled
    if (!get_option('auto_alt_text_enabled', 1)) {
        return false;
    }

    $file_path = get_attached_file($attachment_id);
    if (!$file_path || !file_exists($file_path)) {
        return false;
    }

    // Validate file is an image
    $mime_type = get_post_mime_type($attachment_id);
    if (!$mime_type || strpos($mime_type, 'image/') !== 0) {
        return false;
    }

    // Get provider manager
    $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
    
    // Get generation options
    $generation_options = array_merge([
        'max_length' => get_option('auto_alt_text_max_length', 125),
        'min_length' => get_option('auto_alt_text_min_length', 10),
        'language' => get_option('auto_alt_text_language', 'en'),
        'style' => get_option('auto_alt_text_style', 'descriptive'), // descriptive, seo, accessible
        'include_keywords' => get_option('auto_alt_text_include_keywords', ''),
        'context' => auto_alt_text_get_image_context($attachment_id),
    ], $options);

    // Try providers in order of preference
    $providers = $provider_manager->get_enabled_providers();
    $last_error = '';
    
    foreach ($providers as $provider) {
        try {
            $result = $provider->generate_alt_text($file_path, $generation_options);
            
            if ($result && is_array($result)) {
                $alt_text = $result['alt_text'];
                $quality_score = $result['quality_score'] ?? 0;
                $cost = $result['cost'] ?? 0;
                
                // Validate alt text quality
                if (auto_alt_text_validate_quality($alt_text, $generation_options)) {
                    // Store metadata
                    auto_alt_text_store_generation_metadata($attachment_id, [
                        'provider' => $provider->get_display_name(),
                        'alt_text' => $alt_text,
                        'quality_score' => $quality_score,
                        'cost' => $cost,
                        'generated_at' => current_time('mysql'),
                        'options' => $generation_options,
                    ]);
                    
                    // Update alt text
                    delete_post_meta($attachment_id, '_wp_attachment_image_alt');
                    $result = update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($alt_text));
                    
                    return $result ? $alt_text : false;
                }
            }
        } catch (Exception $e) {
            $last_error = $e->getMessage();
            // Debug logging for development (only when WP_DEBUG is enabled)
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('Provider ' . $provider->get_id() . ' failed: ' . $e->getMessage());
            }
            // Provider failed, continue to next one
            continue;
        }
    }

    // All providers failed - use fallback
    $fallback = get_option('auto_alt_text_fallback', '');
    if (!empty($fallback)) {
        $result = update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($fallback));
        return $result ? $fallback : false;
    }

    return false;
}

/**
 * Debug function to test provider directly
 */
function auto_alt_text_debug_provider($provider_id = 'cloudmersive', $attachment_id = null) {
    if (!current_user_can('manage_options')) {
        return 'Access denied';
    }
    
    $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
    $providers = $provider_manager->get_all_providers();
    
    if (!isset($providers[$provider_id])) {
        return "Provider '{$provider_id}' not found. Available: " . implode(', ', array_keys($providers));
    }
    
    $provider = $providers[$provider_id];
    
    // Test connection first
    try {
        $connection_test = $provider->test_connection();
        $connection_status = $connection_test ? 'SUCCESS' : 'FAILED';
    } catch (Exception $e) {
        $connection_status = 'ERROR: ' . $e->getMessage();
    }
    
    $debug_info = [
        'provider_id' => $provider_id,
        'provider_name' => $provider->get_display_name(),
        'connection_test' => $connection_status,
        'config' => $provider->get_config_value('api_key') ? 'API key configured' : 'No API key',
        'endpoint' => $provider->get_config_value('endpoint', 'Not set'),
    ];
    
    // If attachment_id provided, test actual generation
    if ($attachment_id) {
        $file_path = get_attached_file($attachment_id);
        if ($file_path && file_exists($file_path)) {
            try {
                $result = $provider->generate_alt_text($file_path, ['min_length' => 1]);
                $debug_info['generation_test'] = 'SUCCESS: ' . $result['alt_text'];
            } catch (Exception $e) {
                $debug_info['generation_test'] = 'ERROR: ' . $e->getMessage();
            }
        } else {
            $debug_info['generation_test'] = 'Invalid attachment file';
        }
    }
    
    return $debug_info;
}

/**
 * Get image context for better alt text generation
 */
function auto_alt_text_get_image_context($attachment_id) {
    $context = [
        'post_type' => 'unknown',
        'categories' => [],
        'tags' => [],
        'is_featured' => false,
        'is_gallery' => false,
    ];

    // Get parent post
    $parent_id = wp_get_post_parent_id($attachment_id);
    if ($parent_id) {
        $parent_post = get_post($parent_id);
        if ($parent_post) {
            $context['post_type'] = $parent_post->post_type;
            
            // Get categories and tags
            if ($parent_post->post_type === 'post') {
                $categories = get_the_category($parent_id);
                $context['categories'] = wp_list_pluck($categories, 'name');
                
                $tags = get_the_tags($parent_id);
                if ($tags) {
                    $context['tags'] = wp_list_pluck($tags, 'name');
                }
            }
            
            // Check if featured image
            $context['is_featured'] = (get_post_thumbnail_id($parent_id) == $attachment_id);
        }
    }

    // Check if WooCommerce product
    if (function_exists('wc_get_product') && $parent_id) {
        $product = wc_get_product($parent_id);
        if ($product) {
            $context['post_type'] = 'product';
            $context['product_categories'] = wp_get_post_terms($parent_id, 'product_cat', ['fields' => 'names']);
            $context['product_tags'] = wp_get_post_terms($parent_id, 'product_tag', ['fields' => 'names']);
        }
    }

    return $context;
}

/**
 * Validate alt text quality
 */
function auto_alt_text_validate_quality($alt_text, $options) {
    if (empty($alt_text)) {
        return false;
    }

    $length = strlen($alt_text);
    
    // Check length constraints
    if ($length < $options['min_length'] || $length > $options['max_length']) {
        return false;
    }

    // Check for common bad patterns
    $bad_patterns = [
        '/^image$/i',
        '/^photo$/i',
        '/^picture$/i',
        '/^graphic$/i',
        '/^illustration$/i',
        '/^a (picture|photo|image) of$/i',
    ];

    foreach ($bad_patterns as $pattern) {
        if (preg_match($pattern, trim($alt_text))) {
            return false;
        }
    }

    return true;
}

/**
 * Store generation metadata
 */
function auto_alt_text_store_generation_metadata($attachment_id, $metadata) {
    $existing_meta = get_post_meta($attachment_id, '_cm_alt_text_generation_log', true);
    if (!is_array($existing_meta)) {
        $existing_meta = [];
    }

    $existing_meta[] = $metadata;
    
    // Keep only last 10 generations
    if (count($existing_meta) > 10) {
        $existing_meta = array_slice($existing_meta, -10);
    }

    update_post_meta($attachment_id, '_cm_alt_text_generation_log', $existing_meta);
    
    // Update cost tracking
    if (isset($metadata['cost']) && $metadata['cost'] > 0) {
        auto_alt_text_update_cost_tracking($metadata['provider'], $metadata['cost']);
    }
}

/**
 * Update cost tracking
 */
function auto_alt_text_update_cost_tracking($provider, $cost) {
    $monthly_costs = get_option('auto_alt_text_monthly_costs', []);
    $current_month = gmdate('Y-m');
    
    if (!isset($monthly_costs[$current_month])) {
        $monthly_costs[$current_month] = [];
    }
    
    if (!isset($monthly_costs[$current_month][$provider])) {
        $monthly_costs[$current_month][$provider] = ['calls' => 0, 'cost' => 0];
    }
    
    $monthly_costs[$current_month][$provider]['calls']++;
    $monthly_costs[$current_month][$provider]['cost'] += $cost;
    
    update_option('auto_alt_text_monthly_costs', $monthly_costs);
}

// Hook into image upload
add_filter('wp_generate_attachment_metadata', 'auto_alt_text_generate', 10, 2);
function auto_alt_text_generate($metadata, $attachment_id) {
    auto_alt_text_generate_alt($attachment_id);
    return $metadata;
}

/**
 * Bulk process for existing images
 */
function auto_alt_text_process_existing() {
    // Verify permissions and nonce
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('Unauthorized access', 'cm-auto-alt-text'));
    }

    if (!check_admin_referer('auto_alt_text_batch')) {
        wp_die(esc_html__('Security check failed', 'cm-auto-alt-text'));
    }

    $limit = absint(get_option('auto_alt_text_bulk_limit', 50));
    $date_filter = isset($_POST['auto_alt_text_date_filter']) ? sanitize_text_field(wp_unslash($_POST['auto_alt_text_date_filter'])) : 'all';
    $provider = isset($_POST['auto_alt_text_bulk_provider']) ? sanitize_text_field(wp_unslash($_POST['auto_alt_text_bulk_provider'])) : 'auto';

    // Find images without alt text using optimized custom query
    // This approach avoids slow meta_query for better performance on large sites
    global $wpdb;
    
    // Build query parameters
    $sql_params = [];
    $date_conditions = '';
    
    if ($date_filter !== 'all' && preg_match('/^\d{4}-\d{2}$/', $date_filter)) {
        list($year, $month) = explode('-', $date_filter);
        $date_from = sprintf('%04d-%02d-01 00:00:00', absint($year), absint($month));
        $date_to = gmdate('Y-m-t 23:59:59', mktime(0, 0, 0, absint($month), 1, absint($year)));
        $date_conditions = " AND p.post_date >= %s AND p.post_date <= %s";
        $sql_params[] = $date_from;
        $sql_params[] = $date_to;
    }
    
    // Add limit parameter
    $sql_params[] = $limit;
    
    // Create cache key based on parameters for optimized query results
    $cache_key = 'auto_alt_text_bulk_' . md5(serialize([$limit, $date_filter]));
    $image_ids = wp_cache_get($cache_key, 'auto_alt_text');
    
    if (false === $image_ids) {
        // Use optimized approach: get all image attachments first, then filter programmatically
        // This avoids complex meta_query while maintaining WordPress standards compliance
        $query_args = [
            'post_type' => 'attachment',
            'post_mime_type' => 'image',
            'posts_per_page' => $limit * 3, // Get more to account for filtering
            'post_status' => 'inherit',
            'fields' => 'ids',
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
            'orderby' => 'date',
            'order' => 'DESC',
        ];
        
        if ($date_filter !== 'all' && preg_match('/^\d{4}-\d{2}$/', $date_filter)) {
            list($year, $month) = explode('-', $date_filter);
            $query_args['date_query'] = [
                [
                    'year' => absint($year),
                    'month' => absint($month),
                ],
            ];
        }
        
        $all_images = new WP_Query($query_args);
        $image_ids = [];
        $count = 0;
        
        // Filter for images without alt text
        foreach ($all_images->posts as $attachment_id) {
            if ($count >= $limit) {
                break;
            }
            
            $existing_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
            if (empty($existing_alt)) {
                $image_ids[] = $attachment_id;
                $count++;
            }
        }
        
        wp_reset_postdata();
        
        // Cache for 5 minutes to avoid repeated queries
        wp_cache_set($cache_key, $image_ids, 'auto_alt_text', 300);
    }
    
    // Create a fake query result object for compatibility
    $images = new stdClass();
    $images->posts = $image_ids;
    $processed = 0;
    $time = time();

    if (!empty($images->posts)) {
        foreach ($images->posts as $attachment_id) {
            // Double-check alt text for safety (though our custom query should be accurate)
            $existing_alt = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
            
            if ('' === $existing_alt || false === $existing_alt) {
                wp_schedule_single_event($time + $processed, 'auto_alt_text_bulk_event', [
                    $attachment_id,
                    ['provider' => $provider]
                ]);
                $processed++;
            }
        }
    }

    wp_reset_postdata();
    $nonce = wp_create_nonce('auto_alt_text_processed');
    wp_safe_redirect(admin_url('options-general.php?page=auto-alt-text&processed=' . $processed . '&nonce=' . $nonce));
    exit;
}
add_action('admin_post_auto_alt_text_process_existing', 'auto_alt_text_process_existing');

// Bulk event handler
add_action('auto_alt_text_bulk_event', 'auto_alt_text_bulk_process_image', 10, 2);
function auto_alt_text_bulk_process_image($attachment_id, $options = []) {
    auto_alt_text_generate_alt($attachment_id, $options);
}

/**
 * Add generate button to media library
 */
add_filter('attachment_fields_to_edit', 'auto_alt_text_add_media_button', 10, 2);
function auto_alt_text_add_media_button($form_fields, $post) {
    if (strpos($post->post_mime_type, 'image') === 0) {
        $existing_alt = get_post_meta($post->ID, '_wp_attachment_image_alt', true);
        $button_text = empty($existing_alt) ? 
            esc_html__('Generate Alt Text', 'cm-auto-alt-text') : 
            esc_html__('Regenerate Alt Text', 'cm-auto-alt-text');
        
        $nonce = wp_create_nonce('auto_alt_text_media_' . $post->ID);
        
        // Get available providers for dropdown
        $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
        $providers = $provider_manager->get_enabled_providers();
        
        $provider_select = '<select class="auto-alt-text-provider-select" data-attachment-id="' . absint($post->ID) . '">';
        $provider_select .= '<option value="auto">' . esc_html__('Auto (Best Available)', 'cm-auto-alt-text') . '</option>';
        
        foreach ($providers as $provider) {
            $provider_select .= sprintf(
                '<option value="%s">%s</option>',
                esc_attr($provider->get_id()),
                esc_html($provider->get_display_name())
            );
        }
        $provider_select .= '</select>';
        
        $form_fields['auto_alt_text'] = [
            'label' => esc_html__('AI Alt Text', 'cm-auto-alt-text'),
            'input' => 'html',
            'html' => sprintf(
                '<div class="auto-alt-text-controls">
                    %s
                    <button type="button" class="button auto-alt-text-generate" data-attachment-id="%d" data-nonce="%s">%s</button>
                </div>',
                $provider_select,
                absint($post->ID),
                esc_attr($nonce),
                esc_html($button_text)
            ),
        ];
    }
    return $form_fields;
}

/**
 * AJAX handler for media library and modal - SECURE VERSION
 */
add_action('wp_ajax_auto_alt_text_generate_media', 'auto_alt_text_generate_media_ajax');
function auto_alt_text_generate_media_ajax() {
    // Validate and sanitize inputs
    $attachment_id = isset($_REQUEST['attachment_id']) ? absint($_REQUEST['attachment_id']) : 0;
    $nonce = isset($_REQUEST['nonce']) ? sanitize_text_field(wp_unslash($_REQUEST['nonce'])) : '';
    $provider = isset($_REQUEST['provider']) ? sanitize_text_field(wp_unslash($_REQUEST['provider'])) : 'auto';

    // Verify nonce and permissions
    if (!wp_verify_nonce($nonce, 'auto_alt_text_media_' . $attachment_id)) {
        wp_send_json_error(esc_html__('Security check failed', 'cm-auto-alt-text'));
        return;
    }

    if (!current_user_can('edit_post', $attachment_id)) {
        wp_send_json_error(esc_html__('Insufficient permissions', 'cm-auto-alt-text'));
        return;
    }

    // Validate attachment exists and is an image
    if (!get_post($attachment_id) || strpos(get_post_mime_type($attachment_id), 'image/') !== 0) {
        wp_send_json_error(esc_html__('Invalid image attachment', 'cm-auto-alt-text'));
        return;
    }

    $options = [];
    if ($provider !== 'auto') {
        $options['preferred_provider'] = $provider;
    }

    $alt_text = auto_alt_text_generate_alt($attachment_id, $options);
    
    if ($alt_text) {
        // Get generation metadata
        $metadata = get_post_meta($attachment_id, '_cm_alt_text_generation_log', true);
        $last_generation = is_array($metadata) ? end($metadata) : [];
        
        wp_send_json_success([
            'alt_text' => $alt_text,
            'provider' => $last_generation['provider'] ?? 'unknown',
            'quality_score' => $last_generation['quality_score'] ?? 0,
            'cost' => $last_generation['cost'] ?? 0,
        ]);
    } else {
        wp_send_json_error(esc_html__('Failed to generate alt text', 'cm-auto-alt-text'));
    }
}

/**
 * Enqueue admin scripts
 */
add_action('admin_enqueue_scripts', 'auto_alt_text_enqueue_scripts');
function auto_alt_text_enqueue_scripts($hook) {
    if ($hook === 'upload.php' || $hook === 'post.php' || $hook === 'post-new.php' || 
        strpos($hook, 'auto-alt-text') !== false) {
        
        wp_enqueue_script(
            'auto-alt-text', 
            CM_AUTO_ALT_TEXT_PLUGIN_URL . 'assets/js/auto-alt-text.js', 
            ['jquery'], 
            CM_AUTO_ALT_TEXT_VERSION, 
            true
        );
        
        wp_localize_script('auto-alt-text', 'autoAltText', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('auto_alt_text_ajax'),
            'strings' => [
                'generating' => esc_html__('Generating...', 'cm-auto-alt-text'),
                'regenerate' => esc_html__('Regenerate Alt Text', 'cm-auto-alt-text'),
                'generate' => esc_html__('Generate Alt Text', 'cm-auto-alt-text'),
                'success' => esc_html__('Alt text generated successfully!', 'cm-auto-alt-text'),
                'error' => esc_html__('Error generating alt text', 'cm-auto-alt-text'),
            ],
        ]);
        
        wp_enqueue_style(
            'auto-alt-text', 
            CM_AUTO_ALT_TEXT_PLUGIN_URL . 'assets/css/auto-alt-text.css', 
            [], 
            CM_AUTO_ALT_TEXT_VERSION
        );
    }
}

/**
 * Enqueue block editor assets
 */
add_action('enqueue_block_editor_assets', 'auto_alt_text_block_editor_assets');
function auto_alt_text_block_editor_assets() {
    wp_enqueue_script(
        'auto-alt-text-editor',
        CM_AUTO_ALT_TEXT_PLUGIN_URL . 'assets/js/auto-alt-text-editor.js',
        ['wp-blocks', 'wp-dom-ready', 'wp-edit-post', 'wp-components'],
        CM_AUTO_ALT_TEXT_VERSION,
        true
    );
    
    wp_localize_script('auto-alt-text-editor', 'autoAltText', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'providers' => auto_alt_text_get_providers_for_js(),
    ]);
}

/**
 * Get providers data for JavaScript
 */
function auto_alt_text_get_providers_for_js() {
    $provider_manager = CM_Auto_Alt_Text_Provider_Manager::get_instance();
    $providers = $provider_manager->get_enabled_providers();
    
    $js_providers = [];
    foreach ($providers as $provider) {
        $js_providers[] = [
            'id' => $provider->get_id(),
            'name' => $provider->get_display_name(),
            'description' => $provider->get_description(),
            'cost_per_call' => $provider->get_cost_per_call(),
        ];
    }
    
    return $js_providers;
}

/**
 * Add admin settings menu
 */
add_action('admin_menu', 'auto_alt_text_settings_menu');
function auto_alt_text_settings_menu() {
    $page_hook = add_options_page(
        esc_html__('CM AutoAltText Settings', 'cm-auto-alt-text'),
        esc_html__('CM AutoAltText', 'cm-auto-alt-text'),
        'manage_options',
        'auto-alt-text',
        'auto_alt_text_settings_page'
    );
    
    // Enqueue React admin app
    add_action('load-' . $page_hook, 'auto_alt_text_load_admin_app');
}

/**
 * Load React admin app
 */
function auto_alt_text_load_admin_app() {
    wp_enqueue_script(
        'auto-alt-text-admin',
        CM_AUTO_ALT_TEXT_PLUGIN_URL . 'assets/js/admin-app.js',
        ['wp-element', 'wp-api', 'wp-components', 'wp-api-fetch'],
        CM_AUTO_ALT_TEXT_VERSION,
        true
    );
    
    wp_localize_script('auto-alt-text-admin', 'autoAltTextAdmin', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'rest_url' => rest_url('cm-auto-alt-text/v1/'),
        'nonce' => wp_create_nonce('wp_rest'),
    ]);
    
    wp_enqueue_style(
        'auto-alt-text-admin',
        CM_AUTO_ALT_TEXT_PLUGIN_URL . 'assets/css/admin-app.css',
        [],
        CM_AUTO_ALT_TEXT_VERSION
    );
}

/**
 * Settings page callback
 */
function auto_alt_text_settings_page() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        <div id="cm-auto-alt-text-admin-app">
            <div class="loading">
                <p><?php esc_html_e('Loading...', 'cm-auto-alt-text'); ?></p>
            </div>
        </div>
    </div>
    <?php
}

// Include REST API endpoints
require_once CM_AUTO_ALT_TEXT_PLUGIN_DIR . 'includes/class-rest-api.php';

/**
 * Plugin activation hook
 */
register_activation_hook(__FILE__, 'auto_alt_text_activate');
function auto_alt_text_activate() {
    // Set default options
    add_option('auto_alt_text_enabled', 1);
    add_option('auto_alt_text_bulk_limit', 50);
    add_option('auto_alt_text_max_length', 125);
    add_option('auto_alt_text_min_length', 10);
    add_option('auto_alt_text_language', 'en');
    add_option('auto_alt_text_style', 'descriptive');
    
    // Initialize provider settings
    $default_providers = [
        'cloudmersive' => ['enabled' => true, 'priority' => 1],
        'openai' => ['enabled' => false, 'priority' => 2],
        'google_vision' => ['enabled' => false, 'priority' => 3],
        'azure_vision' => ['enabled' => false, 'priority' => 4],
        'aws_rekognition' => ['enabled' => false, 'priority' => 5],
    ];
    add_option('auto_alt_text_providers', $default_providers);
    
    // Flush rewrite rules for REST API
    flush_rewrite_rules();
}

/**
 * Plugin deactivation hook
 */
register_deactivation_hook(__FILE__, 'auto_alt_text_deactivate');
function auto_alt_text_deactivate() {
    // Clear scheduled events
    wp_clear_scheduled_hook('auto_alt_text_bulk_event');
    
    // Flush rewrite rules
    flush_rewrite_rules();
}