/**
 * CM Auto Alt Text - Frontend JavaScript
 * 
 * @package CM_Auto_Alt_Text
 * @version 1.4.0
 */

jQuery(document).ready(function($) {
    'use strict';

    // Handle provider selection change
    $(document).on('change', '.auto-alt-text-provider-select', function() {
        var $select = $(this);
        var $button = $select.siblings('.auto-alt-text-generate');
        var provider = $select.val();
        
        // Update button data attribute
        $button.data('provider', provider);
        
        // Update button text based on provider
        if (provider === 'auto') {
            $button.find('.provider-info').text('');
        } else {
            var providerName = $select.find('option:selected').text();
            $button.find('.provider-info').text(' (' + providerName + ')');
        }
    });

    // Handle generate button clicks
    $(document).on('click', '.auto-alt-text-generate', function(e) {
        e.preventDefault();
        
        var $button = $(this);
        var attachmentId = parseInt($button.data('attachment-id'), 10);
        var nonce = $button.data('nonce');
        var provider = $button.data('provider') || 'auto';
        
        // Get provider from select if available
        var $providerSelect = $button.siblings('.auto-alt-text-provider-select');
        if ($providerSelect.length) {
            provider = $providerSelect.val();
        }
        
        // Validate data
        if (!attachmentId || !nonce) {
            showMessage('error', autoAltText.strings.error + ': Invalid data');
            return;
        }
        
        // Show loading state
        var originalText = $button.text();
        $button.prop('disabled', true)
               .html('<span class="spinner"></span>' + autoAltText.strings.generating);

        $.ajax({
            url: autoAltText.ajax_url,
            method: 'POST',
            data: {
                action: 'auto_alt_text_generate_media',
                attachment_id: attachmentId,
                nonce: nonce,
                provider: provider
            },
            timeout: 60000, // 60 second timeout for AI processing
            success: function(response) {
                if (response.success && response.data && response.data.alt_text) {
                    $button.text(autoAltText.strings.regenerate);
                    
                    // Show success message with provider info
                    var message = autoAltText.strings.success;
                    if (response.data.provider) {
                        message += ' (Provider: ' + response.data.provider + ')';
                    }
                    if (response.data.quality_score) {
                        message += ' (Quality: ' + Math.round(response.data.quality_score) + '%)';
                    }
                    
                    showMessage('success', message);
                    
                    // Update various alt text fields
                    updateAltTextField(attachmentId, response.data.alt_text);
                    
                    // Update cost display if available
                    if (response.data.cost > 0) {
                        showCostInfo(response.data.cost, response.data.provider);
                    }
                    
                } else {
                    var errorMsg = response.data || autoAltText.strings.error;
                    showMessage('error', autoAltText.strings.error + ': ' + errorMsg);
                    $button.text(originalText);
                }
            },
            error: function(xhr, status, error) {
                var errorMsg = autoAltText.strings.error;
                
                if (status === 'timeout') {
                    errorMsg = 'Request timed out. AI processing may take longer for complex images.';
                } else if (xhr.responseJSON && xhr.responseJSON.data) {
                    errorMsg = xhr.responseJSON.data;
                } else if (error) {
                    errorMsg = error;
                }
                
                showMessage('error', autoAltText.strings.error + ': ' + errorMsg);
                $button.text(originalText);
            },
            complete: function() {
                $button.prop('disabled', false);
            }
        });
    });

    /**
     * Update alt text field based on current context
     */
    function updateAltTextField(attachmentId, altText) {
        var $altField;
        var updated = false;
        
        // Try different alt text field selectors
        var selectors = [
            '#attachment-details-alt-text',
            '#attachment-details-two-column-alt-text',
            '.media-modal input[name="attachments[' + attachmentId + '][alt]"]',
            '#attachment_alt'
        ];
        
        for (var i = 0; i < selectors.length; i++) {
            $altField = $(selectors[i]);
            if ($altField.length) {
                $altField.val(altText).trigger('change');
                updated = true;
                break;
            }
        }
        
        // Sync with WordPress media frame
        syncWithMediaFrame(altText);
        
        if (!updated) {
            console.log('CM Auto Alt Text: No alt text field found for attachment ID ' + attachmentId);
        }
    }

    /**
     * Sync alt text with WordPress media frame
     */
    function syncWithMediaFrame(altText) {
        if (typeof wp !== 'undefined' && wp.media && wp.media.frame) {
            try {
                var selection = wp.media.frame.state().get('selection');
                if (selection && selection.first) {
                    selection.first().set('alt', altText);
                }
                
                var content = wp.media.frame.content.get();
                if (content && content.model) {
                    content.model.set('alt', altText);
                }
            } catch (error) {
                console.log('CM Auto Alt Text: Could not sync with media frame:', error);
            }
        }
    }

    /**
     * Show message to user
     */
    function showMessage(type, message) {
        // Try to use WordPress notices if available
        if (typeof wp !== 'undefined' && wp.data && wp.data.dispatch) {
            try {
                if (type === 'success') {
                    wp.data.dispatch('core/notices').createSuccessNotice(message, {
                        type: 'snackbar',
                        isDismissible: true
                    });
                } else {
                    wp.data.dispatch('core/notices').createErrorNotice(message, {
                        type: 'snackbar',
                        isDismissible: true
                    });
                }
                return;
            } catch (error) {
                // Fallback to alert
            }
        }
        
        // Fallback to alert
        alert(message);
    }

    /**
     * Show cost information
     */
    function showCostInfo(cost, provider) {
        if (cost > 0) {
            var costMessage = 'API call cost: $' + cost.toFixed(4) + ' (' + provider + ')';
            console.log('CM Auto Alt Text: ' + costMessage);
            
            // Show in admin bar if available
            var $adminBar = $('#wpadminbar');
            if ($adminBar.length) {
                var $costDisplay = $('<div class="cm-alt-text-cost">' + costMessage + '</div>');
                $costDisplay.css({
                    'position': 'fixed',
                    'top': '32px',
                    'right': '20px',
                    'background': '#23282d',
                    'color': '#fff',
                    'padding': '5px 10px',
                    'border-radius': '3px',
                    'font-size': '12px',
                    'z-index': 99999
                });
                
                $('body').append($costDisplay);
                
                setTimeout(function() {
                    $costDisplay.fadeOut(500, function() {
                        $costDisplay.remove();
                    });
                }, 3000);
            }
        }
    }

    /**
     * Handle dismissible notices
     */
    $(document).on('click', '.notice-dismiss', function() {
        var $notice = $(this).closest('.notice');
        var dismissible = $notice.data('dismissible');
        
        if (dismissible && autoAltText.nonce) {
            $.ajax({
                url: autoAltText.ajax_url,
                method: 'POST',
                data: {
                    action: 'auto_alt_text_dismiss_notice',
                    notice: dismissible,
                    _ajax_nonce: autoAltText.nonce
                }
            });
        }
    });

    /**
     * Initialize provider selection UI
     */
    function initProviderSelectionUI() {
        $('.auto-alt-text-provider-select').each(function() {
            var $select = $(this);
            var $button = $select.siblings('.auto-alt-text-generate');
            
            // Style the select
            $select.css({
                'margin-right': '10px',
                'min-width': '150px'
            });
            
            // Add provider info span to button
            if (!$button.find('.provider-info').length) {
                $button.append('<span class="provider-info"></span>');
            }
        });
    }

    // Initialize on page load
    initProviderSelectionUI();
    
    // Re-initialize when media modal opens
    $(document).on('DOMNodeInserted', function(e) {
        if ($(e.target).hasClass('media-modal') || $(e.target).find('.media-modal').length) {
            setTimeout(initProviderSelectionUI, 100);
        }
    });
});