/**
 * CM Auto Alt Text - React Admin Interface
 * 
 * @package CM_Auto_Alt_Text
 * @version 1.4.0
 */

const { createElement: e, useState, useEffect } = wp.element;
const { TabPanel, Button, Card, CardHeader, CardBody, ToggleControl, TextControl, SelectControl, Notice } = wp.components;

const AdminApp = () => {
    const [providers, setProviders] = useState({});
    const [settings, setSettings] = useState({});
    const [stats, setStats] = useState({});
    const [loading, setLoading] = useState(true);
    const [saving, setSaving] = useState(false);
    const [notice, setNotice] = useState(null);

    // Load data on component mount
    useEffect(() => {
        loadData();
    }, []);

    const loadData = async () => {
        setLoading(true);
        try {
            const [providersRes, settingsRes, statsRes] = await Promise.all([
                wp.apiFetch({ path: '/cm-auto-alt-text/v1/providers' }),
                wp.apiFetch({ path: '/cm-auto-alt-text/v1/settings' }),
                wp.apiFetch({ path: '/cm-auto-alt-text/v1/stats/providers' })
            ]);
            
            setProviders(providersRes);
            setSettings(settingsRes);
            setStats(statsRes);
        } catch (error) {
            setNotice({ type: 'error', message: 'Failed to load data: ' + error.message });
        }
        setLoading(false);
    };

    const saveSettings = async () => {
        setSaving(true);
        try {
            await wp.apiFetch({
                path: '/cm-auto-alt-text/v1/settings',
                method: 'PUT',
                data: settings
            });
            setNotice({ type: 'success', message: 'Settings saved successfully!' });
        } catch (error) {
            setNotice({ type: 'error', message: 'Failed to save settings: ' + error.message });
        }
        setSaving(false);
    };

    const updateProvider = async (providerId, config) => {
        try {
            await wp.apiFetch({
                path: `/cm-auto-alt-text/v1/providers/${providerId}`,
                method: 'PUT',
                data: config
            });
            
            // Reload providers data to get updated state
            const providersRes = await wp.apiFetch({ path: '/cm-auto-alt-text/v1/providers' });
            setProviders(providersRes);
            
            setNotice({ type: 'success', message: 'Provider updated successfully!' });
            return true;
        } catch (error) {
            setNotice({ type: 'error', message: 'Failed to update provider: ' + error.message });
            return false;
        }
    };

    const testProvider = async (providerId) => {
        try {
            const result = await wp.apiFetch({
                path: `/cm-auto-alt-text/v1/providers/${providerId}/test`,
                method: 'POST'
            });
            
            const messageType = result.success ? 'success' : 'error';
            setNotice({ type: messageType, message: result.message });
        } catch (error) {
            setNotice({ type: 'error', message: 'Test failed: ' + error.message });
        }
    };

    const testAllProviders = async () => {
        try {
            const results = await wp.apiFetch({
                path: '/cm-auto-alt-text/v1/providers/test-all',
                method: 'POST'
            });
            
            const messages = Object.values(results).map(r => 
                `${r.provider}: ${r.success ? 'Success' : r.message}`
            ).join('\n');
            
            setNotice({ type: 'info', message: messages });
        } catch (error) {
            setNotice({ type: 'error', message: 'Test failed: ' + error.message });
        }
    };

    if (loading) {
        return e('div', { className: 'cm-alt-text-loading' }, 'Loading...');
    }

    const tabs = [
        {
            name: 'providers',
            title: 'AI Providers',
            content: e(ProvidersTab, { 
                providers, 
                updateProvider, 
                testProvider, 
                testAllProviders
            })
        },
        {
            name: 'settings',
            title: 'Settings',
            content: e(SettingsTab, { 
                settings, 
                setSettings, 
                saveSettings, 
                saving 
            })
        },
        {
            name: 'stats',
            title: 'Statistics',
            content: e(StatsTab, { stats })
        },
        {
            name: 'bulk',
            title: 'Bulk Processing',
            content: e(BulkTab, { providers })
        }
    ];

    return e('div', { className: 'cm-auto-alt-text-admin' }, [
        notice && e(Notice, {
            key: 'notice',
            status: notice.type,
            onRemove: () => setNotice(null),
            isDismissible: true
        }, notice.message),
        
        e(TabPanel, {
            key: 'tabs',
            tabs: tabs,
            onSelect: () => setNotice(null)
        }, (tab) => tab.content)
    ]);
};

const ProvidersTab = ({ providers, updateProvider, testProvider, testAllProviders }) => {
    // Sort providers by priority (lower number = higher priority)
    const sortedProviders = Object.entries(providers).sort((a, b) => {
        const priorityA = a[1].priority || 999;
        const priorityB = b[1].priority || 999;
        return priorityA - priorityB;
    });
    
    return e('div', { className: 'providers-tab' }, [
        e('div', { key: 'header', className: 'tab-header' }, [
            e('h2', {}, 'AI Provider Configuration'),
            e(Button, {
                isPrimary: true,
                onClick: testAllProviders
            }, 'Test All Providers')
        ]),
        
        e('div', { key: 'providers', className: 'providers-grid' }, 
            sortedProviders.map(([id, provider]) => 
                e(ProviderCard, {
                    key: id,
                    id: id,
                    provider: provider,
                    updateProvider: updateProvider,
                    testProvider: testProvider
                })
            )
        )
    ]);
};

const ProviderCard = ({ id, provider, updateProvider, testProvider }) => {
    // Each card has its own saving state
    const [saving, setSaving] = useState(false);
    const [showPasswords, setShowPasswords] = useState({});
    
    // Initialize config from provider data
    const [config, setConfig] = useState(() => {
        const initialConfig = {
            enabled: provider.enabled || false,
            priority: provider.priority || 999
        };
        
        // Add config field values from provider
        Object.entries(provider.config_fields || {}).forEach(([key, field]) => {
            if (key !== 'enabled' && key !== 'priority') {
                initialConfig[key] = field.default || '';
            }
        });
        
        return initialConfig;
    });
    
    // Update config when provider data changes (after save)
    useEffect(() => {
        const newConfig = {
            enabled: provider.enabled || false,
            priority: provider.priority || 999
        };
        
        Object.entries(provider.config_fields || {}).forEach(([key, field]) => {
            if (key !== 'enabled' && key !== 'priority') {
                newConfig[key] = field.default || '';
            }
        });
        
        setConfig(newConfig);
    }, [provider.enabled, provider.priority, JSON.stringify(provider.config_fields)]);

    const handleConfigChange = (key, value) => {
        setConfig(prev => ({ ...prev, [key]: value }));
    };

    const handleSave = async () => {
        setSaving(true);
        await updateProvider(id, config);
        setSaving(false);
    };
    
    const togglePasswordVisibility = (key) => {
        setShowPasswords(prev => ({ ...prev, [key]: !prev[key] }));
    };

    const statusIcon = provider.configured ? 
        (provider.enabled ? '🟢' : '🟡') : '🔴';

    return e(Card, { className: 'provider-card' }, [
        e(CardHeader, { key: 'header' }, [
            e('h3', {}, `${statusIcon} ${provider.name}`),
            e('p', { className: 'provider-description' }, provider.description)
        ]),
        
        e(CardBody, { key: 'body' }, [
            // Enabled toggle
            e(ToggleControl, {
                key: 'enabled-toggle',
                label: 'Enable Provider',
                checked: config.enabled === true,
                onChange: (value) => handleConfigChange('enabled', value)
            }),
            
            // Priority
            e(TextControl, {
                key: 'priority-field',
                label: 'Priority (1 = highest)',
                type: 'number',
                value: config.priority,
                onChange: (value) => handleConfigChange('priority', parseInt(value))
            }),
            
            // Dynamic config fields
            ...Object.entries(provider.config_fields).map(([key, field]) => {
                if (key === 'enabled' || key === 'priority') return null;
                
                switch (field.type) {
                    case 'text':
                        return e(TextControl, {
                            key: key,
                            label: field.label,
                            type: 'text',
                            value: config[key] || '',
                            placeholder: field.placeholder,
                            onChange: (value) => handleConfigChange(key, value)
                        });
                    
                    case 'password':
                        const savedValue = field.default || '';
                        const currentValue = config[key] || '';
                        const hasValue = savedValue.length > 0;
                        
                        // Show masked if: has saved value AND not showing password AND field not modified
                        const shouldMask = hasValue && !showPasswords[key] && currentValue === savedValue;
                        const displayValue = shouldMask ? '••••••••••••••••••••••••••••••' : currentValue;
                        
                        return e('div', { 
                            key: key, 
                            style: { position: 'relative', marginBottom: '16px' } 
                        }, [
                            e(TextControl, {
                                label: field.label,
                                type: showPasswords[key] ? 'text' : 'password',
                                value: displayValue,
                                placeholder: field.placeholder,
                                onChange: (value) => {
                                    // If user types in masked field, clear the mask
                                    if (shouldMask) {
                                        handleConfigChange(key, '');
                                        setTimeout(() => handleConfigChange(key, value), 0);
                                    } else {
                                        handleConfigChange(key, value);
                                    }
                                },
                                style: { paddingRight: '40px' }
                            }),
                            hasValue && e(Button, {
                                key: 'toggle-' + key,
                                isSmall: true,
                                icon: showPasswords[key] ? 'hidden' : 'visibility',
                                onClick: () => togglePasswordVisibility(key),
                                style: { 
                                    position: 'absolute', 
                                    right: '8px', 
                                    top: '28px',
                                    zIndex: 10
                                },
                                title: showPasswords[key] ? 'Hide API Key' : 'Show API Key'
                            })
                        ]);
                    
                    case 'select':
                        return e('div', { key: key }, [
                            e(SelectControl, {
                                label: field.label,
                                value: config[key] || field.default,
                                options: field.options,
                                onChange: (value) => handleConfigChange(key, value)
                            }),
                            field.description && e('p', { 
                                style: { fontSize: '12px', color: '#666', marginTop: '-10px', marginBottom: '15px' } 
                            }, field.description)
                        ]);
                    
                    case 'number':
                        return e(TextControl, {
                            key: key,
                            label: field.label,
                            type: 'number',
                            min: field.min,
                            max: field.max,
                            step: field.step,
                            value: config[key] || field.default,
                            onChange: (value) => handleConfigChange(key, value)
                        });
                    
                    default:
                        return null;
                }
            }).filter(Boolean),
            
            // Actions
            e('div', { key: 'actions', className: 'provider-actions' }, [
                e(Button, {
                    key: 'save',
                    isPrimary: true,
                    isBusy: saving,
                    onClick: handleSave,
                    disabled: saving
                }, saving ? 'Saving...' : 'Save'),
                
                e(Button, {
                    key: 'test',
                    isSecondary: true,
                    onClick: () => testProvider(id),
                    disabled: !provider.configured || saving
                }, 'Test Connection')
            ]),
            
            // Limitations and pricing info
            provider.limitations && e('div', { key: 'info', className: 'provider-info' }, [
                e('h4', {}, 'Limitations'),
                e('ul', {}, 
                    Object.entries(provider.limitations).map(([key, value]) =>
                        e('li', { key: key }, `${key}: ${value}`)
                    )
                )
            ])
        ])
    ]);
};

const SettingsTab = ({ settings, setSettings, saveSettings, saving }) => {
    const handleChange = (key, value) => {
        setSettings(prev => ({ ...prev, [key]: value }));
    };

    return e('div', { className: 'settings-tab' }, [
        e('h2', { key: 'title' }, 'Plugin Settings'),
        
        e(Card, { key: 'general' }, [
            e(CardHeader, {}, e('h3', {}, 'General Settings')),
            e(CardBody, {}, [
                e(ToggleControl, {
                    label: 'Enable automatic alt text generation',
                    checked: settings.enabled,
                    onChange: (value) => handleChange('enabled', value)
                }),
                
                e(TextControl, {
                    label: 'Fallback alt text',
                    value: settings.fallback || '',
                    placeholder: 'Default text when AI fails',
                    onChange: (value) => handleChange('fallback', value)
                })
            ])
        ]),
        
        e(Card, { key: 'quality' }, [
            e(CardHeader, {}, e('h3', {}, 'Quality Settings')),
            e(CardBody, {}, [
                e(TextControl, {
                    label: 'Minimum alt text length',
                    type: 'number',
                    value: settings.min_length || 10,
                    onChange: (value) => handleChange('min_length', parseInt(value))
                }),
                
                e(TextControl, {
                    label: 'Maximum alt text length',
                    type: 'number',
                    value: settings.max_length || 125,
                    onChange: (value) => handleChange('max_length', parseInt(value))
                }),
                
                e(SelectControl, {
                    label: 'Alt text style',
                    value: settings.style || 'descriptive',
                    options: [
                        { value: 'descriptive', label: 'Descriptive' },
                        { value: 'seo', label: 'SEO-focused' },
                        { value: 'accessible', label: 'Accessibility-focused' }
                    ],
                    onChange: (value) => handleChange('style', value)
                }),
                
                e(TextControl, {
                    label: 'SEO keywords (comma-separated)',
                    value: settings.include_keywords || '',
                    placeholder: 'keyword1, keyword2, keyword3',
                    onChange: (value) => handleChange('include_keywords', value)
                })
            ])
        ]),
        
        e('div', { key: 'actions', className: 'settings-actions' }, [
            e(Button, {
                isPrimary: true,
                isBusy: saving,
                onClick: saveSettings
            }, saving ? 'Saving...' : 'Save Settings')
        ])
    ]);
};

const StatsTab = ({ stats }) => {
    return e('div', { className: 'stats-tab' }, [
        e('h2', { key: 'title' }, 'Provider Statistics'),
        
        e('div', { key: 'stats', className: 'stats-grid' },
            Object.entries(stats).map(([id, stat]) =>
                e(Card, { key: id, className: 'stat-card' }, [
                    e(CardHeader, {}, e('h3', {}, stat.name)),
                    e(CardBody, {}, [
                        e('div', { className: 'stat-item' }, [
                            e('strong', {}, 'Status: '),
                            stat.enabled ? (stat.configured ? 'Active' : 'Configured') : 'Disabled'
                        ]),
                        e('div', { className: 'stat-item' }, [
                            e('strong', {}, 'API Calls: '),
                            stat.calls_made
                        ]),
                        e('div', { className: 'stat-item' }, [
                            e('strong', {}, 'Total Cost: '),
                            `$${stat.total_cost.toFixed(4)}`
                        ]),
                        e('div', { className: 'stat-item' }, [
                            e('strong', {}, 'Success Rate: '),
                            `${stat.success_rate.toFixed(1)}%`
                        ]),
                        e('div', { className: 'stat-item' }, [
                            e('strong', {}, 'Avg Response Time: '),
                            `${stat.avg_response_time.toFixed(2)}s`
                        ])
                    ])
                ])
            )
        )
    ]);
};

const BulkTab = ({ providers }) => {
    const [processing, setProcessing] = useState(false);
    const [limit, setLimit] = useState(50);
    const [provider, setProvider] = useState('auto');
    const [dateFilter, setDateFilter] = useState('all');

    const startBulkProcess = async () => {
        setProcessing(true);
        try {
            const result = await wp.apiFetch({
                path: '/cm-auto-alt-text/v1/bulk/process',
                method: 'POST',
                data: { limit, provider, date_filter: dateFilter }
            });
            
            alert(`Success! Scheduled ${result.scheduled} images for processing.`);
        } catch (error) {
            alert('Failed to start bulk processing: ' + error.message);
        }
        setProcessing(false);
    };

    const enabledProviders = Object.entries(providers)
        .filter(([id, provider]) => provider.enabled)
        .map(([id, provider]) => ({ value: id, label: provider.name }));

    return e('div', { className: 'bulk-tab' }, [
        e('h2', { key: 'title' }, 'Bulk Processing'),
        
        e(Card, { key: 'form' }, [
            e(CardHeader, {}, e('h3', {}, 'Process Existing Images')),
            e(CardBody, {}, [
                e(TextControl, {
                    label: 'Number of images to process',
                    type: 'number',
                    min: 1,
                    max: 800,
                    value: limit,
                    onChange: (value) => setLimit(parseInt(value))
                }),
                
                e(SelectControl, {
                    label: 'AI Provider',
                    value: provider,
                    options: [
                        { value: 'auto', label: 'Auto (Best Available)' },
                        ...enabledProviders
                    ],
                    onChange: setProvider
                }),
                
                e(SelectControl, {
                    label: 'Date Filter',
                    value: dateFilter,
                    options: [
                        { value: 'all', label: 'All Dates' },
                        { value: '2024-10', label: 'October 2024' },
                        { value: '2024-09', label: 'September 2024' },
                        { value: '2024-08', label: 'August 2024' }
                    ],
                    onChange: setDateFilter
                }),
                
                e('p', { className: 'description' }, 
                    'This will process images without alt text in the background at 1 image per second.'
                ),
                
                e(Button, {
                    isPrimary: true,
                    isBusy: processing,
                    onClick: startBulkProcess,
                    disabled: enabledProviders.length === 0
                }, processing ? 'Processing...' : 'Start Bulk Processing')
            ])
        ])
    ]);
};

// Mount the app
document.addEventListener('DOMContentLoaded', () => {
    const container = document.getElementById('cm-auto-alt-text-admin-app');
    if (container) {
        wp.element.render(e(AdminApp), container);
    }
});