<?php
/**
 * Class \ClipTransparentSDK
 *
 * @package  Ecomerciar\ClipTransparent\Helper\ClipSdk
 */

namespace Conexa\ClipTransparent\Sdk;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Conexa\ClipTransparent\Api\ClipTransparentApi;
use Conexa\ClipTransparent\Includes\Helper\Helper;

/**
 * Main Class Clip Transparent Sdk.
 */
class ClipTransparentSDK {

	/**
	 * Defines Clip API Key
	 *
	 * @var string $api_key
	 */
	private string $api_key;
	/**
	 * Defines Clip API Secret
	 *
	 * @var string $api_secret API Secret;
	 */
	private string $api_secret;
	/**
	 * Defines Clip API Token
	 *
	 * @var string $api_token API Token;
	 */
	private string $api_token;
	/**
	 * Defines Debug flag
	 *
	 * @var bool $debug Debug flag ;
	 */
	private bool $debug;

	const CLIPTRANSP_JSON = 'application/json';

	/**
	 * Constructor.
	 *
	 * @param string  $api_key Clip API Key.
	 * @param string  $api_secret Clip API Secret.
	 * @param boolean $debug Debug Switch.
	 */
	public function __construct(
		string $api_key,
		string $api_secret,
		bool $debug = false
	) {
		$this->api_key    = $api_key;
		$this->api_secret = $api_secret;
		$this->api        = new ClipTransparentApi(
			array(
				'api_key'    => $api_key,
				'api_secret' => $api_secret,
				'debug'      => $debug,
			)
		);
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
		$this->api_token = 'Basic ' . base64_encode( $api_key . ':' . $api_secret );
		$this->debug     = $debug;
	}



	/**
	 * Validate Key
	 *
	 * @return bool
	 */
	public function validate_key() {
		try {
			$res = $this->api->get(
				$this->api->get_validate_base_url() . '/apikey',
				array(),
				array(
					'Authorization' => $this->api_token,
					'content-type'  => self::CLIPTRANSP_JSON,
				)
			);

		} catch ( \Exception $e ) {
			Helper::log_error( __FUNCTION__ . ': ' . $e->getMessage() );
			return array();
		}
		if ( ! empty( $this->handle_response( $res, __FUNCTION__ )['query'] ) ) {
			return true;
		}
		return false;
	}



	/**
	 * Charge Payment
	 *
	 * @param int   $order_id ID for WC Order.
	 * @param array $data Data.
	 * @return array
	 */
	public function charge_payment( $order_id, $data ) {

		$order = wc_get_order( $order_id );

		$external_reference = 'WC-' . absint( $order_id );
		$installments_int   = absint( $data['clip_transparent_installments'] ?? 0 );

		// Sanitize prevention data.
		$prevention_data           = sanitize_text_field( $data['clip_transparent_prevention'] ?? '' );
		$prevention_parts          = explode( ',', $prevention_data );
		$session_id                = isset( $prevention_parts[0] ) ? sanitize_text_field( $prevention_parts[0] ) : '';
		$device_finger_print_token = isset( $prevention_parts[1] ) ? sanitize_text_field( $prevention_parts[1] ) : '';

		$billing_states  = WC()->countries->get_states( $order->get_billing_country() );
		$billing_state   = ! empty( $billing_states[ $order->get_billing_state() ] ) ? $billing_states[ $order->get_billing_state() ] : '';
		$billing_country = ! empty( WC()->countries->countries[ $order->get_billing_country() ] ) ? WC()->countries->countries[ $order->get_billing_country() ] : '';

		$shipping_states  = WC()->countries->get_states( $order->get_shipping_country() );
		$shipping_state   = ! empty( $shipping_states[ $order->get_shipping_state() ] ) ? $shipping_states[ $order->get_shipping_state() ] : '';
		$shipping_country = ! empty( WC()->countries->countries[ $order->get_shipping_country() ] ) ? WC()->countries->countries[ $order->get_shipping_country() ] : '';

		$data_to_send = array(
            'description'        => __( 'Compra en tienda', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ),
			'external_reference' => $external_reference,
			'type'               => 'openplatform',
			'entry_mode'         => 'manual',
			'customer'           => array(
				'first_name' => $order->get_billing_first_name(),
				'last_name'  => $order->get_billing_last_name(),
				'email'      => $order->get_billing_email(),
				'phone'      => $order->get_billing_phone(),
				'address'    => array(
					'postal_code'     => ! empty( $order->get_billing_postcode() ) ? $order->get_billing_postcode() : '',
					'street'          => ! empty( $order->get_billing_address_1() ) ? $order->get_billing_address_1() : '',
					'exterior_number' => '',
					'internal_number' => '',
					'city'            => ! empty( $order->get_billing_city() ) ? $order->get_billing_city() : '',
					'country'         => $billing_country,
					'state'           => $billing_state,
				),
			),
			'payment_method'     => array(
				'token' => sanitize_text_field( $data['clip_transparent_cardtoken'] ?? '' ),
			),
			'installments'       => $installments_int,
			'amount'             => strval( number_format( floatval( $order->get_total() ), 2, '.', '' ) ),
			'tip_amount'         => 0,
			'currency'           => $order->get_currency(),
			'country'            => 'MX',
			'webhook_url'        => get_site_url() . '/wc-api/wc-clip-transparent',
			'prevention_data'    => array(
				'source'                    => 'plugin',
				'session_id'                => $session_id,
				'user_agent'                => htmlspecialchars_decode( $device_finger_print_token ),
				'device_finger_print_token' => $session_id,
			),
			'metadata'           => array(
				'website'          => get_site_url(),
				'source'           => 'txo_woocommerce',
				'billing_address'  => array(
					'postal_code'     => ! empty( $order->get_billing_postcode() ) ? $order->get_billing_postcode() : '',
					'street'          => ! empty( $order->get_billing_address_1() ) ? $order->get_billing_address_1() : '',
					'exterior_number' => '',
					'internal_number' => '',
					'colony'          => '',
					'reference'       => '',
					'city'            => ! empty( $order->get_billing_city() ) ? $order->get_billing_city() : '',
					'country'         => $billing_country,
					'state'           => $billing_state,
				),
				'shipping_address' => array(
					'postal_code'     => ! empty( $order->get_shipping_postcode() ) ? $order->get_shipping_postcode() : '',
					'street'          => ! empty( $order->get_shipping_address_1() ) ? $order->get_shipping_address_1() : '',
					'exterior_number' => '',
					'internal_number' => '',
					'colony'          => '',
					'reference'       => '',
					'city'            => ! empty( $order->get_shipping_city() ) ? $order->get_shipping_city() : '',
					'country'         => $shipping_country,
					'state'           => $shipping_state,
				),

			),
		);
		try {
			$res = $this->api->post(
				$this->api->get_base_url() . '/payments',
				$data_to_send,
				array(
					'Authorization' => $this->api_token,
					'content-type'  => self::CLIPTRANSP_JSON,
				)
			);
		} catch ( \Exception $e ) {
			Helper::log_error( __FUNCTION__ . ': ' . $e->getMessage() );
			return false;
		}

		return $this->handle_response( $res, __FUNCTION__ );
	}

	/**
	 * Get charge_payment
	 *
	 * @param int $payment_id payment id.
	 * @return array
	 */
	public function get_charge_payment( $payment_id ) {
		try {
			$res = $this->api->get(
				$this->api->get_base_url() . '/payments/' . $payment_id,
				array(),
				array(
					'Authorization' => $this->api_token,
					'content-type'  => self::CLIPTRANSP_JSON,
				)
			);
		} catch ( \Exception $e ) {
			Helper::log_error( __FUNCTION__ . ': ' . $e->getMessage() );
			return array();
		}
		return $this->handle_response( $res, __FUNCTION__ );
	}


	/**
	 * Request Refund
	 *
	 * @param string $id Id for process refund.
	 * @param float  $amount Amount to be refunded.
	 * @param string $reason Refund Reason.
	 *
	 * @return array
	 */
	public function request_refund( string $id, float $amount, string $reason = '' ) {
		$order        = wc_get_order( $id );
		$receipt_no   = $order->get_meta( \ClipTransparent::CLIPTRANSP_META_ORDER_RECEIPT_NO );
		$data_to_send = array(
			'reference' => array(
				'type' => 'receipt',
				'id'   => $receipt_no,
			),
			'amount'    => floatval( number_format( floatval( $amount ), 2, '.', '' ) ),
			'reason'    => $reason,
		);
		try {

			$res = $this->api->post(
				\ClipTransparent::get_url( 'refund' ) . '/refunds/',
				$data_to_send,
				array(
					'content-type'  => self::CLIPTRANSP_JSON,
					'accept'        => 'application/vnd.com.payclip.v2+json',
					'x-api-key'     => $this->api_token,
					'Authorization' => $this->api_token,
				)
			);
		} catch ( \Exception $e ) {
			Helper::log_error( __FUNCTION__ . ': ' . $e->getMessage() );
			return array();
		}
		return $this->handle_response( $res, __FUNCTION__ );
	}


	/**
	 * Handle Response
	 *
	 * @param array  $response Response data.
	 * @param string $function_name Function function is calling from.
	 *
	 * @return array
	 */
	protected function handle_response(
		$response = array(),
		string $function_name = ''
	) {
		return $response;
	}
}
