<?php
/**
 * Class WC_ClipTransparent
 *
 * @package  Conexa\ClipTransparent\Gateway\WC_ClipTransparent
 */

namespace Conexa\ClipTransparent\Gateway;

use Conexa\ClipTransparent\Includes\Helper\Helper;
use Conexa\ClipTransparent\Sdk\ClipTransparentSdk;

defined( 'ABSPATH' ) || class_exists( '\WC_Payment_Gateway' ) || exit();

/**
 * Main Class Clip Transparent Payment.
 */
class WC_ClipTransparent extends \WC_Payment_Gateway {

	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {
		$this->id                 = \ClipTransparent::CLIPTRANSP_GATEWAY_ID;
		$this->has_fields         = true; // in case you need a custom credit card form.
        $this->method_title       = __( 'Clip Mexico Payments for WooCommerce (Transparent Checkout)', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
        $this->method_description = __( 'Accept all local Mexican and international card payments with Clip (Credit, Debit, Vale cards, Months without interest, Visa, MC, AMEX, Discover, Diners Club, Si Vale, Edenred, Carnet and more).', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );

		$this->icon = Helper::get_assets_folder_url() . '/img/logotype_clip_primary.svg';

		$this->sdk = new ClipTransparentSdk(
			Helper::get_option( 'api_key' ),
			Helper::get_option( 'api_secret' )
		);

		// gateways can support subscriptions, refunds, saved payment methods.
		$this->supports = array(
			'products',
		);

		// Define user set variables.
        $this->title        = __( 'Pay with debit and credit card', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
		$this->instructions = $this->get_option(
			$this->description,
			$this->method_description
		);

		// Load the settings.
		$this->init_form_fields();
		$this->init_settings();

		$this->supports[] = 'refunds';

		global $current_section;
		if ( \ClipTransparent::CLIPTRANSP_GATEWAY_ID === $current_section ) {

			$this->enqueue_settings_js();

		}

		add_action(
			'woocommerce_update_options_payment_gateways_' . $this->id,
			array(
				$this,
				'process_admin_options',
			)
		);
		add_action(
			'woocommerce_thankyou_' . $this->id,
			array(
				$this,
				'thankyou_page',
			)
		);
	}


	/**
	 * Enqueue admin styles and scripts for settings page
	 */
	private function enqueue_settings_js() {
		// Enqueue admin CSS.
		wp_enqueue_style(
			'clip-transparent-admin',
			Helper::get_assets_folder_url() . '/css/clip-transparent-admin.css',
			array(),
			\ClipTransparent::CLIPTRANSP_VERSION
		);

		// Enqueue inline script for logotype functionality.
		wp_add_inline_script(
			'jquery',
			'var url = "' . esc_url( Helper::get_assets_folder_url() . '/img/logotype_clip_primary.svg' ) . '";
			jQuery(document).ready(
				function( $ ) {
					$(".logotype_clip").remove();
					var image = new Image();
					image.src = url;
					image.classList.add("logotype_clip");
					$("h2", $(".wrap.woocommerce")).first().prepend(image);
				});'
		);
	}


	/**
	 * Initialize Gateway Settings Form Fields
	 */
	public function init_form_fields() {
		$this->form_fields = include 'settings.php';
	}

	/**
	 * You will need it if you want your custom credit card form
	 */
	public function payment_fields() {

		$data                = array();
		$data['id']          = $this->id;
		$data['description'] = $this->description;

		$api_key = Helper::get_option( 'api_key' );

		$mode_clip = ( \ClipTransparent::CLIPTRANSP_MODE === 'test' ) ? 'stage' : 'prod';
		$version  = \ClipTransparent::CLIPTRANSP_VERSION;

		$theme_options  = Helper::get_option( 'theme_options' );
		$locale_options = Helper::get_option( 'locale_options' );

		helper::get_template_part( 'page', 'payment', $data );

		wp_register_script(
			'clip-transparent-sdk',
			\ClipTransparent::get_url( 'sdk' ),
			array(),
			$version,
			true
		);
		wp_enqueue_script( 'clip-transparent-sdk' );

		wp_register_script(
			'clip-transparent-payment',
			Helper::get_assets_folder_url() . '/js/clip-transparent-payment.js',
			array( 'jquery', 'clip-transparent-sdk' ),
			$version,
			true
		);
		wp_enqueue_script( 'clip-transparent-payment' );

		wp_localize_script(
			'clip-transparent-payment',
			'clip_transparent_object',
			array(
				'nonce'          => wp_create_nonce( 'wc-clip_transparent' ),
				'admin_ajax'     => esc_url( admin_url( 'admin-ajax.php' ) ),
				'api_key'        => $api_key,
				'theme_options'  => $theme_options,
				'locale_options' => $locale_options,
				'mode'           => $mode_clip,
			)
		);

		wp_register_script(
			'clip-transparent-datadog-link',
			\ClipTransparent::get_url( 'datadog' ),
			array(),
			$version,
			true
		);
		wp_enqueue_script( 'clip-transparent-datadog-link' );

		wp_register_script(
			'clip-transparent-datadog',
			Helper::get_assets_folder_url() . '/js/clip-transparent-datadog.js',
			array( 'jquery', 'clip-transparent-datadog-link' ),
			$version,
			true
		);
		wp_enqueue_script( 'clip-transparent-datadog' );

		wp_localize_script(
			'clip-transparent-datadog',
			'clip_transparent_datadog',
			array(
				'mode'    => $mode_clip,
				'version' => $version,
			)
		);

		// Enqueue frontend CSS.
		wp_enqueue_style(
			'clip-transparent-frontend',
			Helper::get_assets_folder_url() . '/css/clip-transparent-frontend.css',
			array(),
			\ClipTransparent::CLIPTRANSP_VERSION
		);
	}

	/**
	 * Get error message based on error code.
	 *
	 * @param string $error_code Error code from payment gateway.
	 * @throws \Exception Exception message.
	 */
	protected function handle_errors( $error_code ) {
		$message = '';

		switch ( $error_code ) {

			// Front SDK errors.
			case 'CL2200':
                $message = __( 'Check the card details', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			case 'AI1300':
                $message = __( 'Try another card', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			case 'GATEWAY TIMEOUT':
                $message = __( 'Internal Server Error', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;

			// POST Payment API errors.
			case 'RE-ERI03':
                $message = __( 'Unfortunately, international cards are not accepted for this payment. Contact the business 🛍️ to enable international cards, or Use a card issued in Mexico 🇲🇽.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			case 'RE-ERI01':
                $message = __( 'Oops! 😮 Your payment was not made. Don\'t worry, try again following our suggestions; Use a different card, or Choose to pay cash at a nearby convenience store 🏪💸.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			case in_array( $error_code, array( 'RE-ISS11', 'RE-ISS12', 'RE-ISS13', 'RE-ISS15', 'RE-ISS16', 'RE-ISS17', 'RE-ISS18' ), true ):
                $message = __( 'Oops! 😮 Your payment was not made. Don\'t worry, try again following our suggestions; Use a different card, or Choose to pay cash at a nearby convenience store 🏪💸.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			case in_array( $error_code, array( 'RE-ISS06', 'RE-ISS07' ), true ):
                $message = __( 'I\'m sorry! 🙁 your payment could not be completed. Try another card, or Choose to pay cash at a nearby convenience store 🏪💸.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			case in_array( $error_code, array( 'RE-ERI02', 'RE-ERI03', 'RE-BIN01' ), true ):
                $message = __( 'I\'m sorry! 🙁 your payment could not be completed. Try another card, or Choose to pay cash at a nearby convenience store 🏪💸.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			case 'RE-ISS01':
                $message = __( 'Oops! 😕 it seems that your bank rejected the payment due to lack of funds 💵. Contact your bank to check your card limit and try again, or Choose to pay cash at a nearby convenience store 🏪💸.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			case in_array( $error_code, array( 'RE-ISS14', 'RE-ISS99' ), true ):
                $message = __( 'We\'re sorry, but we encountered a problem processing your payment. Check the status by updating your order or contacting the merchant and try payment again if necessary.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			case in_array( $error_code, array( 'RE-ISS02', 'RE-ISS03' ), true ):
                $message = __( 'It seems that your bank rejected the payment 🙁. Confirm your card information 💳 and try again, or Choose to pay cash at a nearby convenience store 🏪💸.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
			default:
                $message = __( 'We were unable to process your order, please try again.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
				break;
		}

		throw new \Exception( esc_html( $message ) );
	}



	/**
	 * We're processing the payments here.
	 *
	 * @param int $order_id Order id.
	 * @return array
	 * @throws \Exception Exception message.
	 */
	public function process_payment( $order_id ) {

		try {
			// Sanitize and validate nonce.
			$nonce_value = isset( $_REQUEST['woocommerce-process-checkout-nonce'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['woocommerce-process-checkout-nonce'] ) ) : '';
			if ( empty( $nonce_value ) || ! wp_verify_nonce( $nonce_value, 'woocommerce-process_checkout' ) ) {
				WC()->session->set( 'refresh_totals', true );
            throw new \Exception( __( 'We were unable to process your order, please try again.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ) );
			}

			// Sanitize and validate error parameter.
			if ( isset( $_POST['clip_transparent_error'] ) && ! empty( $_POST['clip_transparent_error'] ) ) {
				$error_code = sanitize_text_field( wp_unslash( $_POST['clip_transparent_error'] ) );
				$this->handle_errors( $error_code );
			}

			// Validate required card token.
			if ( empty( $_POST['clip_transparent_cardtoken'] ) ) {
            throw new \Exception( __( 'We were unable to process your order, please try again.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ) );
			}

			// Fetch order details.
			$order = wc_get_order( $order_id );

			// Sanitize all input data.
			$extradata = array(
				'clip_transparent_cardtoken'    => sanitize_text_field( wp_unslash( $_REQUEST['clip_transparent_cardtoken'] ?? '' ) ),
				'clip_transparent_installments' => sanitize_text_field( wp_unslash( $_REQUEST['clip_transparent_installments'] ?? '' ) ),
				'clip_transparent_prevention'   => sanitize_text_field( wp_unslash( $_REQUEST['clip_transparent_prevention'] ?? '' ) ),
			);

			// Validate required fields.
			if ( empty( $extradata['clip_transparent_cardtoken'] ) ) {
            throw new \Exception( __( 'We were unable to process your order, please try again.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ) );
			}

			$charge_payment = $this->sdk->charge_payment( $order_id, $extradata );
			$response       = $this->sdk->get_charge_payment( $charge_payment['id'] );
			return $this->handle_response( $order_id, $response );

		} catch ( \Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
		}
	}



	/**
	 * Handle response
	 *
	 * @param int   $order_id order id.
	 * @param array $data data to be processed.
	 *
	 * @return array
	 * @throws \Exception Exception message.
	 */
	protected function handle_response( int $order_id, array $data ) {

		if ( isset( $data['status'] ) ) {
			if ( 'pending' === $data['status'] && isset( $data['pending_action']['url'] ) ) {
				return $this->mark_order_as_waiting_3ds( $order_id, $data );
			} elseif ( 'approved' === $data['status'] ) {
				return $this->mark_order_as_paid( $order_id, $data );
			} elseif ( isset( $data['status_detail'] ) && isset( $data['status_detail']['code'] ) ) {
					$this->handle_errors( $data['status_detail']['code'] );
			}
		}

		if ( isset( $data['errors']['error_code'] ) ) {
			$this->handle_errors( $data['errors']['error_code'] );
		}

        throw new \Exception( esc_html__( 'We were unable to process your order, please try again.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ) );
	}


	/**
	 * Mark order as waiting 3ds.
	 *
	 * @param int   $order_id Order id.
	 * @param array $data Data to be processed.
	 * @return array
	 */
	protected function mark_order_as_waiting_3ds( int $order_id, array $data ) {

		$order = wc_get_order( $order_id );

		$order->update_meta_data(
			\ClipTransparent::CLIPTRANSP_META_ORDER_3DS_URL,
			$data['pending_action']['url']
		);

		$receipt_no = isset( $data['receipt_no'] ) ? $data['receipt_no'] : '';
		$order->update_meta_data(
			\ClipTransparent::CLIPTRANSP_META_ORDER_RECEIPT_NO,
			$receipt_no
		);
		$order->update_meta_data(
			\ClipTransparent::CLIPTRANSP_META_ORDER_TRANSACTION_ID,
			$data['id']
		);
		$order->update_meta_data(
			\ClipTransparent::CLIPTRANSP_META_ORDER_TRANSACTION_DATE,
			current_datetime()->format( 'Y-m-d H:i:s' )
		);

		$order->save();

		$payment_nonce = wp_create_nonce( \ClipTransparent::CLIPTRANSP_GATEWAY_ID );
		// Return to Pay order Page.
		return array(
			'result'   => 'success',
			'redirect' => add_query_arg( 'clip_transparent_nonce', $payment_nonce, add_query_arg( 'clip_transparent_3ds', true, $order->get_checkout_payment_url( true ) ) ),
		);
	}


	/**
	 * Mark order as paid.
	 *
	 * @param int   $order_id Order id.
	 * @param array $data Data to be processed.
	 * @return array
	 */
	protected function mark_order_as_paid( int $order_id, array $data ) {

		$order = wc_get_order( $order_id );
		$order->payment_complete();

		$receipt_no = isset( $data['receipt_no'] ) ? $data['receipt_no'] : '';
		if ( ! empty( $receipt_no ) ) {
			$order->add_order_note(
				sprintf(
					/* translators: %s: System Flag */
                    esc_html__( 'Clip Transparent - Approved Payment. Receipt: %1$s. ID %2$s', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ),
					$receipt_no,
					$data['id']
				)
			);
			$order->update_meta_data(
				\ClipTransparent::CLIPTRANSP_META_ORDER_RECEIPT_NO,
				$receipt_no
			);
		}
		$order->update_meta_data(
			\ClipTransparent::CLIPTRANSP_META_ORDER_TRANSACTION_ID,
			$data['id']
		);
		$order->update_meta_data(
			\ClipTransparent::CLIPTRANSP_META_ORDER_TRANSACTION_DATE,
			current_datetime()->format( 'Y-m-d H:i:s' )
		);

		$order->save();

		// Return thank you page redirect.
		return array(
			'result'   => 'success',
			'redirect' => $this->get_return_url( $order ),
		);
	}

	/**
	 * Output for the order received page.
	 *
	 * @param string $order_id Order Id.
	 */
	public function thankyou_page( $order_id ) {
		// Nothing to add, but required to avoid Warnings.
	}


	/**
	 * Process Refunds
	 *
	 * @param int    $order_id Order to Refund.
	 * @param float  $amount Amount to Refund.
	 * @param string $reason Reason for refund.
	 *
	 * @return bool
	 */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {

		if ( empty( $order_id ) ) {
			return false;
		}

		if ( empty( $amount ) ) {
			/* translators: %s: Order ID */
            return new \WP_Error( 'wc-order', sprintf( esc_html__( 'Clip Transparent: Skipping ZERO value refund for Order ID %s.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ), $order_id ) );
		}

		$order      = wc_get_order( $order_id );
		$order_data = $order->get_data();

		$response = $this->sdk->request_refund( $order_id, $amount, $reason );

		if ( isset( $response['code_message'] ) ) {
			$message = '';
			if ( 'AI1801' === $response['code_message'] ) {
                $message = __( 'Clip: The refund amount is greater that the original.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
			}
			if ( 'AI1802' === $response['code_message'] ) {
                $message = __( 'Clip: The refund amount, plus previous refunds, is greater than the original amount.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
			}
			if ( 'AI1803' === $response['code_message'] ) {
                $message = __( 'Clip: The refund date has expired.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
			}
			if ( 'AI1804' === $response['code_message'] ) {
                $message = __( 'Clip: Refund declined.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
			}
			if ( 'AI1805' === $response['code_message'] ) {
                $message = __( 'Clip: Refunds are disabled.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
			}
			if ( 'AI1806' === $response['code_message'] ) {
                $message = __( 'Clip: Refund is disabled for payments with MSI and MCI.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
			}
			if ( 'AI1807' === $response['code_message'] ) {
                $message = __( 'Clip: Refund in process for this transaction. Please try again later.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
			}
			if ( 'AI1400' === $response['code_message'] ) {
                $message = __( 'Clip: Insufficient funds to make the refund.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' );
			}
			return new \WP_Error( 'wc-order', esc_html( $message ) );
		} elseif ( ! isset( $response['id'] ) ) {
            return new \WP_Error( 'wc-order', esc_html__( 'Clip: Unauthorized.', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ) );
		} else {
			/* translators: %s: Refund ID */
            $order->add_order_note( sprintf( esc_html__( 'Clip: Refund requested. Id: %s .', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ), $response['id'] ) );
		}
		$order->save();

		return true;
	}
}
