<?php
/**
 * Plugin Name: Clip Transparent Checkout
 * Description: Accept all local Mexican and international card payments with Clip
 * Version: 2.0.0
 * Requires PHP: 7.0
 * Author: Conexa
 * Author URI: https://conexa.ai
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: clip-mexico-payments-for-ecommerce-transparent-checkout
 * Domain Path: /languages
 * WC requires at least: 5.4.1
 * WC tested up to: 5.4
 *
 * @package Conexa\ClipTransparent
 */

defined( 'ABSPATH' ) || exit;

use Conexa\ClipTransparent\Includes\Helper\Helper;

add_action( 'plugins_loaded', array( 'ClipTransparent', 'init' ) );
add_action( 'activated_plugin', array( 'ClipTransparent', 'activation' ) );

if ( ! class_exists( 'ClipTransparent' ) ) {
	/**
	 * Plugin's base Class
	 */
	class ClipTransparent {

	const CLIPTRANSP_MODE        = 'prod';
	const CLIPTRANSP_VERSION     = '2.0.0';
	const CLIPTRANSP_PLUGIN_NAME = 'clip-mexico-payments-for-ecommerce-transparent-checkout';
	const CLIPTRANSP_MAIN_FILE   = __FILE__;
	const CLIPTRANSP_MAIN_DIR    = __DIR__;
	const CLIPTRANSP_GATEWAY_ID  = 'wc_clip_transparent';

		// Urls for PROD.
		const CLIPTRANSP_BACKEND_URL  = 'https://api.payclip.com';
		const CLIPTRANSP_SDK_URL      = 'https://sdk.clip.mx/js/clip-sdk.js';
		const CLIPTRANSP_VALIDATE_URL = 'https://api-gw.payclip.com';
		const CLIPTRANSP_REFUND_URL   = 'https://api.payclip.com';
		const CLIPTRANSP_THREE_DS_URL = 'https://3ds.payclip.com';
		const CLIPTRANSP_DATADOG_URL  = 'https://tools.clip.mx/transparent/metrics';

		// Urls for STAGE.
		const CLIPTRANSP_BACKEND_URL_TST  = 'https://dev-api.payclip.com';
		const CLIPTRANSP_SDK_URL_TST      = 'https://sdk.clip.mx/js/clip-sdk.js';
		const CLIPTRANSP_VALIDATE_URL_TST = 'https://dev-api.clip.mx';
		const CLIPTRANSP_REFUND_URL_TST   = 'https://testapi-gw.payclip.com';
		const CLIPTRANSP_THREE_DS_URL_TST = 'https://dev-3ds.payclip.com';
		const CLIPTRANSP_DATADOG_URL_TST  = 'https://www.datadoghq-browser-agent.com/us1/v5/datadog-rum.js';


		const CLIPTRANSP_META_ORDER_3DS_URL          = '_CLIP_TRANSPARENT_3DS_URL';
		const CLIPTRANSP_META_ORDER_RECEIPT_NO       = '_CLIP_TRANSPARENT_RECEIPT_NO';
		const CLIPTRANSP_META_ORDER_TRANSACTION_ID   = '_CLIP_TRANSPARENT_TRANSACTION_ID';
		const CLIPTRANSP_META_ORDER_TRANSACTION_DATE = '_CLIP_TRANSPARENT_TRANSACTION_DATE';

		/**
		 * Inits our plugin
		 *
		 * @return bool
		 */
		public static function init() {
			if ( ! self::check_system() ) {
				return false;
			}

            spl_autoload_register(
				function ( $requested_class ) {
					// Plugin base Namespace.
					if ( strpos( $requested_class, 'ClipTransparent' ) === false
						&& ( strpos( $requested_class, 'Ecomerciar' ) === false
						&& strpos( $requested_class, 'Conexa' ) === false ) ) {
						return;
					}
					$requested_class = str_replace( '\\', '/', $requested_class );
					$parts           = explode( '/', $requested_class );
					$classname       = array_pop( $parts );

					$filename = $classname;
					$filename = str_replace( 'WooCommerce', 'Woocommerce', $filename );
					$filename = str_replace( 'WC_', 'Wc', $filename );
					$filename = str_replace( 'WC', 'Wc', $filename );
					$filename = preg_replace( '/([A-Z])/', '-$1', $filename );
					$filename = 'class' . $filename;
					$filename = strtolower( $filename );
					$folder   = strtolower( array_pop( $parts ) );
					if ( 'class-clip-transparent' === $filename ) {
						return;
					}
					$boiler   = ( strpos( $requested_class, 'Conexa/ClipTransparent/Includes' ) === false ) ? 'src' : 'includes';
					$filepath = plugin_dir_path( __FILE__ ) . $boiler . '/' . $folder . '/' . $filename . '.php';
					if ( file_exists( $filepath ) ) {
						require_once $filepath;
					}
				}
			);

			include_once __DIR__ . '/src/hooks.php';
			Helper::init( 'ClipTransparent' );
			return true;
		}
		/**
		 * Checks system requirements
		 *
		 * @return bool
		 */
		public static function check_system() {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
			$system = self::check_components();

			if ( $system['flag'] ) {
				deactivate_plugins( plugin_basename( __FILE__ ) );
				echo '<div class="notice notice-error is-dismissible">'
				. '<p>' .
					sprintf(
						/* translators: %s: System Flag */
                        esc_html__(
                            '<strong>%1$s</strong> Requiere al menos %2$s versión %3$s o superior.',
                            'clip-mexico-payments-for-ecommerce-transparent-checkout'
                        ),
						esc_html( self::PLUGIN_NAME ),
						esc_html( $system['flag'] ),
						esc_html( $system['version'] )
					) .
					'</p>'
				. '</div>';
				return false;
			}

			if ( ! class_exists( 'WooCommerce' ) ) {
				deactivate_plugins( plugin_basename( __FILE__ ) );
				echo '<div class="notice notice-error is-dismissible">'
				. '<p>' .
					sprintf(
						/* translators: %s: System Flag */
                        esc_html__(
                            'WooCommerce debe estar activo antes de usar <strong>%s</strong>',
                            'clip-mexico-payments-for-ecommerce-transparent-checkout'
                        ),
						esc_html( self::PLUGIN_NAME )
					) .
					'</p>'
				. '</div>';
				return false;
			}
			return true;
		}
		/**
		 * Check the components required for the plugin to work (PHP, WordPress and WooCommerce)
		 *
		 * @return array
		 */
		private static function check_components() {
			global $wp_version;
			$flag    = false;
			$version = false;

			if ( version_compare( PHP_VERSION, '7.0', '<' ) ) {
				$flag    = 'PHP';
				$version = '7.0';
			} elseif ( version_compare( $wp_version, '5.4', '<' ) ) {
				$flag    = 'WordPress';
				$version = '5.4';
			} elseif (
			! defined( 'WC_VERSION' ) ||
			version_compare( WC_VERSION, '3.8.0', '<' )
			) {
				$flag    = 'WooCommerce';
				$version = '3.8.0';
			}
			return array(
				'flag'    => $flag,
				'version' => $version,
			);
		}

		/**
		 * Getters
		 *
		 * @param string $name Name of Url required depending plugin mode.
		 * @param string $mode Mode of Plugin, 'test' or 'production'.
		 * @return string
		 */
		public static function get_url( string $name, string $mode = '' ) {
			$mode = empty( $mode ) ? self::CLIPTRANSP_MODE : $mode;
			$url  = '';
			if ( 'test' === strtolower( $mode ) ) {
				if ( 'backend' === strtolower( $name ) ) {
					$url = self::CLIPTRANSP_BACKEND_URL_TST;
				} elseif ( 'sdk' === strtolower( $name ) ) {
					$url = self::CLIPTRANSP_SDK_URL_TST;
				} elseif ( 'validate' === strtolower( $name ) ) {
					$url = self::CLIPTRANSP_VALIDATE_URL_TST;
				} elseif ( 'refund' === strtolower( $name ) ) {
					$url = self::CLIPTRANSP_REFUND_URL_TST;
				} elseif ( 'three_ds' === strtolower( $name ) ) {
					$url = self::CLIPTRANSP_THREE_DS_URL_TST;
				} elseif ( 'datadog' === strtolower( $name ) ) {
					$url = self::CLIPTRANSP_DATADOG_URL_TST;
				}
			} elseif ( 'backend' === strtolower( $name ) ) {
					$url = self::CLIPTRANSP_BACKEND_URL;
			} elseif ( 'sdk' === strtolower( $name ) ) {
				$url = self::CLIPTRANSP_SDK_URL;
			} elseif ( 'validate' === strtolower( $name ) ) {
				$url = self::CLIPTRANSP_VALIDATE_URL;
			} elseif ( 'refund' === strtolower( $name ) ) {
				$url = self::CLIPTRANSP_REFUND_URL;
			} elseif ( 'three_ds' === strtolower( $name ) ) {
				$url = self::CLIPTRANSP_THREE_DS_URL;
			} elseif ( 'datadog' === strtolower( $name ) ) {
				$url = self::CLIPTRANSP_DATADOG_URL;
			}
			return $url;
		}


		/**
		 * Create a link to the settings page, in the plugins page
		 *
		 * @param array $links Links for plugin.
		 * @return array
		 */
		public static function create_settings_link( array $links ) {
			$link =
				'<a href="' .
				esc_url(
					get_admin_url(
						null,
						'admin.php?page=wc-settings&tab=checkout&section=' . self::CLIPTRANSP_GATEWAY_ID
					)
				) .
				'">' .
                __( 'Settings', 'clip-mexico-payments-for-ecommerce-transparent-checkout' ) .
				'</a>';
			array_unshift( $links, $link );
			return $links;
		}

		/**
		 * Activation Plugin Actions
		 *
		 * @param string $plugin Plugin Name.
		 * @return bool
		 */
		public static function activation( $plugin ) {
			if ( plugin_basename( self::CLIPTRANSP_MAIN_FILE ) === $plugin ) {
				wp_safe_redirect(
					admin_url( 'admin.php?page=wc-settings&tab=checkout&section=' . self::CLIPTRANSP_GATEWAY_ID )
				);
				exit();
			}
			return false;
		}

		/**
		 * Adds our payment method to WooCommerce
		 *
		 * @param array $gateways Gateways setted on Woo.
		 * @return array
		 */
		public static function add_payment_method( $gateways ) {
			$gateways[] = '\Conexa\ClipTransparent\Gateway\WC_ClipTransparent';
			return $gateways;
		}
	}

	// --- HPOS WooCommerce Compatibility
	add_action(
		'before_woocommerce_init',
		function () {
			if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
			}
		}
	);
}
