<?php
/**
 * Plugin Name: Cirv Box
 * Plugin URI: https://wordpress.org/plugins/cirv-box/
 * Description: Automatically generates Schema.org markup for better Google rankings. Freemium model: Free basic schemas, Pro for advanced features.
 * Version: 1.2.8
 * Author: Cirvgreen
 * Author URI: https://cirvgreen.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: cirv-box
 *
 * EDUCATIONAL NOTES:
 * - This header is REQUIRED for WordPress to recognize this as a plugin
 * - "Plugin Name" appears in the WordPress admin plugins list
 * - "Text Domain" is used for translations (internationalization)
 * - Version number should follow semantic versioning (MAJOR.MINOR.PATCH)
 */

// Exit if accessed directly (security best practice)
// This prevents anyone from running this file outside of WordPress
if (!defined('ABSPATH')) {
    exit;
}

/**
 * ═══════════════════════════════════════════════════════════
 * CONSTANTS - Define plugin paths and URLs
 * ═══════════════════════════════════════════════════════════
 *
 * WHY? These constants make it easy to reference files and folders
 * throughout the plugin without hardcoding paths
 */
define('CIRVBO_VERSION', '1.2.8');
define('CIRVBO_PLUGIN_DIR', plugin_dir_path(__FILE__));    // Full server path to plugin folder
define('CIRVBO_PLUGIN_URL', plugin_dir_url(__FILE__));      // URL to plugin folder (for CSS/JS)
define('CIRVBO_PLUGIN_BASENAME', plugin_basename(__FILE__)); // Used for activation/deactivation hooks

/**
 * ═══════════════════════════════════════════════════════════
 * FREEMIUS SDK INTEGRATION - Monetization Platform
 * ═══════════════════════════════════════════════════════════
 *
 * Freemius handles:
 * - License management (generating and validating keys)
 * - Payment processing (credit cards, PayPal)
 * - Auto-updates for Pro version
 * - Analytics (installs, revenue, conversions)
 * - Affiliate system
 *
 * EDUCATIONAL NOTE:
 * The if statement prevents redefining the function if it's already loaded
 * This is important if you have multiple plugins using Freemius
 */
if (!function_exists('cirvbo_fs')) {
    /**
     * Create Freemius instance
     *
     * @return Freemius|false Freemius SDK instance or false if not available
     */
    function cirvbo_fs() {
        global $cirvbo_fs;

        if (!isset($cirvbo_fs)) {
            // Check if Freemius SDK exists before loading
            // NOTE: You'll need to download the SDK first (see FREEMIUS-SETUP-GUIDE.md)
            $freemius_path = dirname(__FILE__) . '/freemius/start.php';

            if (!file_exists($freemius_path)) {
                // Freemius not installed - plugin will run in free-only mode
                return false;
            }

            require_once $freemius_path;

            /**
             * Initialize Freemius SDK
             *
             * ✅ CONFIGURED: Connected to Freemius account (Dec 27, 2025)
             * Plugin ID: 22596
             * Pricing: $8.99/month or $78.99/year (Pro tier)
             * Trial: 7 days (payment required)
             *
             * FREEMIUM MODEL:
             * - This is the FREE version with upgrade path to Pro
             * - Free users get: Article, Product, Organization, FAQ, Breadcrumb, HowTo, Review, Recipe
             * - Pro users get: Local Business, Video, Event + priority support
             */
            $cirvbo_fs = fs_dynamic_init(array(
                'id'                  => '22596',                    // ✅ Freemius Plugin ID
                'slug'                => 'cirv-box',                 // Plugin slug
                'type'                => 'plugin',                   // Plugin type
                'public_key'          => 'pk_7ef6b3b67b37ee04e82111ffb6871',  // ✅ Freemius Public Key
                'is_premium'          => false,                      // This is FREE version (not premium)
                'has_premium_version' => true,                       // But Pro upgrade is available
                'has_addons'          => false,                      // No addons
                'has_paid_plans'      => true,                       // Yes, we have Pro plans!
                'is_org_compliant'    => true,                       // ✅ WordPress.org compliant (no tracking without opt-in)
                'trial'               => array(
                    'days'               => 7,                       // 7-day free trial for Pro
                    'is_require_payment' => true,                    // Requires payment method for trial
                ),
                'menu'                => array(
                    'slug'           => 'cirv-box',                  // Admin menu slug
                    'first-path'     => 'options-general.php?page=cirv-box', // Where to go after activation
                    'contact'        => false,                       // Don't show contact submenu
                    'support'        => false,                       // We'll use our own Help & Support tab
                ),
            ));
        }

        return $cirvbo_fs;
    }

    /**
     * Initialize Freemius
     *
     * EDUCATIONAL NOTE:
     * We hook into 'plugins_loaded' to ensure WordPress core is fully loaded
     * Priority 1 ensures Freemius loads early (before our main plugin code)
     */
    $cirvbo_freemius = cirvbo_fs();

    /**
     * Freemius Custom Actions
     *
     * These hooks let you customize Freemius behavior
     * Only run if Freemius SDK is available
     */

    if ($cirvbo_freemius) {
        // Signal that SDK was initiated
        do_action('cirvbo_fs_loaded');

        /**
         * Customize opt-in message
         *
         * This appears when users first activate the plugin
         * We want to be friendly and build trust
         */
        function cirvbo_fs_custom_connect_message_on_update(
            $message,
            $user_first_name,
            $plugin_title,
            $user_login,
            $site_link,
            $freemius_link
        ) {
            // translators: %1$s is the user's first name
            return sprintf(
                __('Hey %1$s, welcome to Cirv Box!', 'cirv-box') . '<br><br>' .
                __('Never miss an update or important security notification. Opt-in to get feature updates, bug fixes, and helpful SEO tips.', 'cirv-box') . '<br><br>' .
                __('Your schema markup is ready to boost your Google rankings!', 'cirv-box'),
                esc_html($user_first_name)
            );
        }
        $cirvbo_freemius->add_filter('connect_message_on_update', 'cirvbo_fs_custom_connect_message_on_update', 10, 6);

        /**
         * Hook uninstall cleanup to Freemius
         *
         * This allows Freemius to track uninstall events and collect user feedback
         * while still cleaning up plugin data from the database.
         */
        $cirvbo_freemius->add_action('after_uninstall', 'cirvbo_fs_uninstall_cleanup');
    }
}

/**
 * ═══════════════════════════════════════════════════════════
 * UNINSTALL CLEANUP - Freemius Compatible
 * ═══════════════════════════════════════════════════════════
 *
 * This function runs when the plugin is uninstalled.
 * It cleans up all plugin data from the database.
 *
 * NOTE: This replaces the old uninstall.php file to allow Freemius
 * to properly track uninstall events and collect user feedback.
 *
 * @since 1.2.6
 */
function cirvbo_fs_uninstall_cleanup() {
    // Delete all cirvbo_* options (v1.2.5+)
    delete_option('cirvbo_enable_article_schema');
    delete_option('cirvbo_enable_organization_schema');
    delete_option('cirvbo_enable_product_schema');
    delete_option('cirvbo_enable_faq_schema');
    delete_option('cirvbo_enable_breadcrumb_schema');
    delete_option('cirvbo_enable_howto_schema');
    delete_option('cirvbo_organization_name');
    delete_option('cirvbo_debug_mode');
    delete_option('cirvbo_debug_log');
    delete_option('cirvbo_activation_time');
    delete_option('cirvbo_rating_notice_dismissed');
    delete_option('cirvbo_migration_complete');
    delete_option('cirvbo_legacy_cleanup_complete');

    // Legacy ssb_* options cleanup (v1.2.4 and earlier)
    delete_option('ssb_enable_article_schema');
    delete_option('ssb_enable_organization_schema');
    delete_option('ssb_enable_product_schema');
    delete_option('ssb_enable_faq_schema');
    delete_option('ssb_enable_breadcrumb_schema');
    delete_option('ssb_enable_howto_schema');
    delete_option('ssb_organization_name');
    delete_option('ssb_debug_mode');
    delete_option('ssb_debug_log');
    delete_option('ssb_activation_time');
    delete_option('ssb_rating_notice_dismissed');

    // Multisite cleanup - delete options from ALL sites in the network
    if (is_multisite()) {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Multisite cleanup requires direct query
        $blog_ids = $wpdb->get_col("SELECT blog_id FROM $wpdb->blogs");

        foreach ($blog_ids as $blog_id) {
            switch_to_blog($blog_id);

            // New cirvbo_* options
            delete_option('cirvbo_enable_article_schema');
            delete_option('cirvbo_enable_organization_schema');
            delete_option('cirvbo_enable_product_schema');
            delete_option('cirvbo_enable_faq_schema');
            delete_option('cirvbo_enable_breadcrumb_schema');
            delete_option('cirvbo_enable_howto_schema');
            delete_option('cirvbo_organization_name');
            delete_option('cirvbo_debug_mode');
            delete_option('cirvbo_debug_log');
            delete_option('cirvbo_activation_time');
            delete_option('cirvbo_rating_notice_dismissed');
            delete_option('cirvbo_migration_complete');
            delete_option('cirvbo_legacy_cleanup_complete');

            // Legacy ssb_* options
            delete_option('ssb_enable_article_schema');
            delete_option('ssb_enable_organization_schema');
            delete_option('ssb_enable_product_schema');
            delete_option('ssb_enable_faq_schema');
            delete_option('ssb_enable_breadcrumb_schema');
            delete_option('ssb_enable_howto_schema');
            delete_option('ssb_organization_name');
            delete_option('ssb_debug_mode');
            delete_option('ssb_debug_log');
            delete_option('ssb_activation_time');
            delete_option('ssb_rating_notice_dismissed');

            restore_current_blog();
        }
    }
}

// Fallback for non-Freemius installs (WordPress.org version without SDK)
if (!function_exists('cirvbo_fs') || !cirvbo_fs()) {
    register_uninstall_hook(__FILE__, 'cirvbo_fs_uninstall_cleanup');
}

/**
 * ═══════════════════════════════════════════════════════════
 * ERROR HANDLING & LOGGING - Debug Mode
 * ═══════════════════════════════════════════════════════════
 *
 * Provides comprehensive logging for troubleshooting schema issues
 * Enable debug mode to log schema generation events
 */

/**
 * Log schema generation events for debugging
 *
 * @param string $message Log message
 * @param string $type Log type (info, warning, error)
 * @param array $data Optional data to log
 */
function cirvbo_log($message, $type = 'info', $data = []) {
    // Only log if debug mode is enabled
    if (!get_option('cirvbo_debug_mode')) {
        return;
    }

    $log_entry = [
        'timestamp' => current_time('mysql'),
        'type' => $type,
        'message' => $message,
        'url' => isset($_SERVER['REQUEST_URI']) ? esc_url_raw(wp_unslash($_SERVER['REQUEST_URI'])) : '',
        'data' => $data
    ];

    // Get existing log
    $log = get_option('cirvbo_debug_log', []);

    // Add new entry
    array_unshift($log, $log_entry);

    // Keep only last 100 entries
    $log = array_slice($log, 0, 100);

    // Save log
    update_option('cirvbo_debug_log', $log, false);
}

/**
 * Clear debug log
 */
function cirvbo_clear_debug_log() {
    delete_option('cirvbo_debug_log');
}

/**
 * Get debug log entries
 *
 * @param int $limit Number of entries to return
 * @return array Log entries
 */
function cirvbo_get_debug_log($limit = 50) {
    $log = get_option('cirvbo_debug_log', []);
    return array_slice($log, 0, $limit);
}

/**
 * ═══════════════════════════════════════════════════════════
 * HELPER FUNCTIONS - Check Pro Status
 * ═══════════════════════════════════════════════════════════
 */

/**
 * Check if user has Pro version
 *
 * @return bool True if Pro, false if Free
 *
 * EDUCATIONAL NOTE:
 * We use this throughout the plugin to gate Pro features
 * Example: if (cirvbo_is_pro()) { // Pro-only feature }
 */
function cirvbo_is_pro() {
    $fs = cirvbo_fs();
    // Return false if Freemius not available or user doesn't have Pro
    return $fs && method_exists($fs, 'is_premium') ? $fs->is_premium() : false;
}

/**
 * Check if user can upgrade
 *
 * @return bool True if can upgrade, false otherwise
 */
function cirvbo_can_upgrade() {
    $fs = cirvbo_fs();
    // If Freemius not available, user is always on free (can "upgrade" conceptually)
    return $fs && method_exists($fs, 'is_not_paying') ? $fs->is_not_paying() : true;
}

/**
 * ═══════════════════════════════════════════════════════════
 * ACTIVATION HOOK
 * ═══════════════════════════════════════════════════════════
 *
 * This runs ONCE when someone clicks "Activate" in WordPress admin
 * Perfect place to set default options or create database tables
 */

/**
 * ═══════════════════════════════════════════════════════════
 * DATABASE MIGRATION - Legacy ssb_* to cirvbo_*
 * ═══════════════════════════════════════════════════════════
 */

/**
 * Migrate legacy ssb_* options to cirvbo_*
 * Ensures existing users don't lose settings when upgrading from v1.2.4
 */
function cirvbo_migrate_legacy_options() {
    if (get_option('cirvbo_migration_complete')) {
        return;
    }

    $options_map = [
        'ssb_enable_article_schema' => 'cirvbo_enable_article_schema',
        'ssb_enable_organization_schema' => 'cirvbo_enable_organization_schema',
        'ssb_enable_product_schema' => 'cirvbo_enable_product_schema',
        'ssb_enable_faq_schema' => 'cirvbo_enable_faq_schema',
        'ssb_enable_breadcrumb_schema' => 'cirvbo_enable_breadcrumb_schema',
        'ssb_enable_howto_schema' => 'cirvbo_enable_howto_schema',
        'ssb_debug_mode' => 'cirvbo_debug_mode',
        'ssb_debug_log' => 'cirvbo_debug_log',
        'ssb_organization_name' => 'cirvbo_organization_name',
        'ssb_activation_time' => 'cirvbo_activation_time',
        'ssb_rating_notice_dismissed' => 'cirvbo_rating_notice_dismissed',
    ];

    foreach ($options_map as $old_key => $new_key) {
        $value = get_option($old_key);
        if ($value !== false) {
            update_option($new_key, $value);
        }
    }

    update_option('cirvbo_migration_complete', time());
    wp_schedule_single_event(time() + (30 * DAY_IN_SECONDS), 'cirvbo_cleanup_legacy_options');
}

function cirvbo_activate_plugin() {
    // Run database migration
    cirvbo_migrate_legacy_options();

    // Set default options (only if they don't exist)
    // This won't overwrite existing settings if user reactivates
    if (get_option('cirvbo_enable_article_schema') === false) {
        add_option('cirvbo_enable_article_schema', '1'); // 1 = enabled by default
    }

    if (get_option('cirvbo_enable_organization_schema') === false) {
        add_option('cirvbo_enable_organization_schema', '0'); // 0 = disabled by default
    }

    if (get_option('cirvbo_enable_product_schema') === false) {
        add_option('cirvbo_enable_product_schema', '1'); // 1 = enabled by default (valuable for WooCommerce)
    }

    if (get_option('cirvbo_enable_faq_schema') === false) {
        add_option('cirvbo_enable_faq_schema', '1'); // 1 = enabled by default
    }

    if (get_option('cirvbo_enable_breadcrumb_schema') === false) {
        add_option('cirvbo_enable_breadcrumb_schema', '1'); // 1 = enabled by default
    }

    if (get_option('cirvbo_enable_howto_schema') === false) {
        add_option('cirvbo_enable_howto_schema', '1'); // 1 = enabled by default
    }

    // Flush rewrite rules (WordPress routing)
    // This ensures WordPress recognizes any new URLs we might add
    flush_rewrite_rules();
}
register_activation_hook(__FILE__, 'cirvbo_activate_plugin');

/**
 * ═══════════════════════════════════════════════════════════
 * DEACTIVATION HOOK
 * ═══════════════════════════════════════════════════════════
 *
 * This runs when someone clicks "Deactivate"
 * Clean up temporary data, but DON'T delete user settings
 * (they might reactivate later and want their settings back)
 */
function cirvbo_deactivate_plugin() {
    // Flush rewrite rules on deactivation
    flush_rewrite_rules();

    // NOTE: We do NOT delete user options here
    // That happens in uninstall.php (separate file)
}
register_deactivation_hook(__FILE__, 'cirvbo_deactivate_plugin');

/**
 * ═══════════════════════════════════════════════════════════
 * ADMIN MENU - Add settings page to WordPress admin
 * ═══════════════════════════════════════════════════════════
 *
 * This creates a new menu item in WordPress admin sidebar
 */
function cirvbo_add_admin_menu() {
    add_options_page(
        'Cirv Box Settings',           // Page title (shows in browser tab)
        'Cirv Box',                    // Menu title (shows in sidebar)
        'manage_options',                    // Capability required (only admins can access)
        'cirv-box',                    // Menu slug (unique identifier)
        'cirvbo_settings_page'                  // Callback function that renders the page
    );
}
add_action('admin_menu', 'cirvbo_add_admin_menu');

/**
 * ═══════════════════════════════════════════════════════════
 * SETTINGS PAGE - Professional Tabbed Interface
 * ═══════════════════════════════════════════════════════════
 *
 * This is what users see when they click "Settings > SEO Schema Box"
 * Features a modern tabbed interface for better organization
 */
function cirvbo_settings_page() {
    // Get current tab (default to 'general')
    $active_tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'general';
    ?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

        <!-- Success message if settings were saved -->
        <?php settings_errors('cirvbo_messages'); ?>

        <!-- Tab Navigation -->
        <h2 class="nav-tab-wrapper">
            <a href="?page=cirv-box&tab=general"
               class="nav-tab <?php echo $active_tab === 'general' ? 'nav-tab-active' : ''; ?>">
                📋 General Settings
            </a>
            <a href="?page=cirv-box&tab=preview"
               class="nav-tab <?php echo $active_tab === 'preview' ? 'nav-tab-active' : ''; ?>">
                🔍 Preview Schema
            </a>
            <a href="?page=cirv-box&tab=help"
               class="nav-tab <?php echo $active_tab === 'help' ? 'nav-tab-active' : ''; ?>">
                ❓ Help & Support
            </a>
            <a href="?page=cirv-box&tab=debug"
               class="nav-tab <?php echo $active_tab === 'debug' ? 'nav-tab-active' : ''; ?>">
                🐛 Debug Log
            </a>
            <a href="?page=cirv-box&tab=upgrade"
               class="nav-tab <?php echo $active_tab === 'upgrade' ? 'nav-tab-active' : ''; ?>">
                ⭐ Upgrade to Pro
            </a>
        </h2>

        <?php if ($active_tab === 'general'): ?>
            <!-- General Settings Tab -->
            <div style="background: white; padding: 20px; margin-top: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                <form action="options.php" method="post">
                    <?php
                    // Security fields (WordPress adds nonces and other protection)
                    settings_fields('cirvbo_settings');

                    // Output all registered sections and fields
                    do_settings_sections('cirv-box');

                    // Submit button
                    submit_button('Save Settings');
                    ?>
                </form>
            </div>

        <?php elseif ($active_tab === 'help'): ?>
            <!-- Help & Support Tab -->
            <div style="background: white; padding: 30px; margin-top: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                <h2>🎯 How to Use Cirv Box</h2>

                <h3>📝 Article Schema (Blog Posts)</h3>
                <p><strong>Automatically enabled for all blog posts!</strong></p>
                <ul>
                    <li>✅ Adds Article schema to every post automatically</li>
                    <li>✅ Includes headline, author, publish date, featured image</li>
                    <li>✅ Helps Google show rich snippets in search results</li>
                </ul>

                <h3>🏢 Organization Schema (Homepage)</h3>
                <p>Enable this to tell Google about your business/brand:</p>
                <ul>
                    <li>✅ Appears on your homepage only</li>
                    <li>✅ Helps with Google Knowledge Panel</li>
                    <li>✅ Enter your organization name in settings</li>
                </ul>

                <h3>🛒 Product Schema (WooCommerce)</h3>
                <p>For online stores using WooCommerce:</p>
                <ul>
                    <li>✅ Adds Product schema to all product pages</li>
                    <li>✅ Shows price, availability, ratings in Google</li>
                    <li>✅ Increases click-through rates for e-commerce</li>
                </ul>

                <h3>❓ FAQ Schema (FAQ Pages)</h3>
                <p>Automatically detects and marks up FAQ content:</p>
                <ul>
                    <li>✅ Create a page with "FAQ" in the title</li>
                    <li>✅ Use H2/H3 headings for questions</li>
                    <li>✅ Questions appear expanded in Google search!</li>
                </ul>

                <h3>🧭 Breadcrumb Schema (All Pages)</h3>
                <p>Automatic site navigation markup:</p>
                <ul>
                    <li>✅ Works on all pages (except homepage)</li>
                    <li>✅ Shows navigation path in search results</li>
                    <li>✅ Improves site structure understanding</li>
                    <li>✅ Better user experience in Google</li>
                </ul>

                <h3>📖 HowTo Schema (Tutorials)</h3>
                <p>Step-by-step guide markup:</p>
                <ul>
                    <li>✅ Title starts with "How to" or "Guide to"</li>
                    <li>✅ Use numbered lists (1, 2, 3) for steps</li>
                    <li>✅ Or use "Step 1", "Step 2" headings</li>
                    <li>✅ Rich results with expandable steps!</li>
                </ul>

                <h3>🔍 How to Verify It's Working</h3>
                <ol>
                    <li>Visit any page on your site (post, product, FAQ page, etc.)</li>
                    <li>Right-click → "View Page Source"</li>
                    <li>Search for <code>application/ld+json</code></li>
                    <li>You should see JSON schema markup!</li>
                    <li>Or test at: <a href="https://validator.schema.org" target="_blank">validator.schema.org</a></li>
                </ol>

                <h3>💬 Need Help?</h3>
                <p>Contact support: <a href="mailto:nick@cirvgreen.com">nick@cirvgreen.com</a></p>
            </div>

        <?php elseif ($active_tab === 'preview'): ?>
            <!-- Schema Preview Tab -->
            <div style="background: white; padding: 30px; margin-top: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                <h2>🔍 Live Schema Preview</h2>
                <p>Enter a URL from your site to preview the generated schema markup:</p>

                <form method="post" action="" style="margin: 20px 0;">
                    <?php wp_nonce_field('cirvbo_preview_schema_action', 'cirvbo_preview_nonce'); ?>
                    <input type="url"
                           name="cirvbo_preview_url"
                           value="<?php echo esc_url(isset($_POST['cirvbo_preview_url']) ? sanitize_text_field(wp_unslash($_POST['cirvbo_preview_url'])) : home_url()); ?>"
                           placeholder="https://yoursite.com/example-page"
                           style="width: 60%; padding: 8px; margin-right: 10px;"
                           required>
                    <button type="submit" class="button button-primary">Preview Schema</button>
                </form>

                <?php
                // Handle schema preview
                if (isset($_POST['cirvbo_preview_url']) && check_admin_referer('cirvbo_preview_schema_action', 'cirvbo_preview_nonce')) {
                    $preview_url = esc_url_raw(sanitize_text_field(wp_unslash($_POST['cirvbo_preview_url'])));

                    // Validate URL is from this site
                    if (strpos($preview_url, home_url()) === 0) {
                        // Fetch the page
                        $response = wp_remote_get($preview_url);

                        if (!is_wp_error($response)) {
                            $html = wp_remote_retrieve_body($response);

                            // Extract all schema JSON-LD
                            preg_match_all('/<script type="application\/ld\+json">(.*?)<\/script>/is', $html, $matches);

                            if (!empty($matches[1])) {
                                echo '<div style="background: #f9f9f9; padding: 20px; border-radius: 4px; margin-top: 20px;">';
                                echo '<h3 style="margin-top: 0;">✅ Found ' . count($matches[1]) . ' Schema(s)</h3>';

                                foreach ($matches[1] as $index => $json) {
                                    $schema = json_decode($json, true);
                                    $schema_type = isset($schema['@type']) ? $schema['@type'] : 'Unknown';

                                    echo '<div style="background: white; padding: 15px; margin-bottom: 15px; border-left: 4px solid #2271b1;">';
                                    echo '<h4 style="margin-top: 0;">Schema #' . absint($index + 1) . ': ' . esc_html($schema_type) . '</h4>';
                                    echo '<pre style="background: #f5f5f5; padding: 15px; overflow-x: auto; font-size: 12px; border-radius: 4px;">';
                                    echo esc_html(json_encode($schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
                                    echo '</pre>';

                                    // Validation link
                                    echo '<p style="margin-bottom: 0;">';
                                    echo '<a href="https://validator.schema.org/#url=' . urlencode($preview_url) . '" target="_blank" class="button button-small">Validate on Schema.org →</a> ';
                                    echo '<a href="https://search.google.com/test/rich-results?url=' . urlencode($preview_url) . '" target="_blank" class="button button-small">Test in Google →</a>';
                                    echo '</p>';
                                    echo '</div>';
                                }
                                echo '</div>';
                            } else {
                                echo '<div class="notice notice-warning inline" style="margin-top: 20px; padding: 15px;">';
                                echo '<p><strong>⚠️ No schema found on this page.</strong><br>';
                                echo 'This could mean:<br>';
                                echo '• The page doesn\'t match any schema type criteria<br>';
                                echo '• Another SEO plugin is active (conflict detected)<br>';
                                echo '• The schema type is disabled in settings</p>';
                                echo '</div>';
                            }
                        } else {
                            echo '<div class="notice notice-error inline" style="margin-top: 20px; padding: 15px;">';
                            echo '<p><strong>❌ Error fetching page:</strong> ' . esc_html($response->get_error_message()) . '</p>';
                            echo '</div>';
                        }
                    } else {
                        echo '<div class="notice notice-error inline" style="margin-top: 20px; padding: 15px;">';
                        echo '<p><strong>❌ Invalid URL:</strong> Please enter a URL from your own website.</p>';
                        echo '</div>';
                    }
                }
                ?>

                <div style="background: #e7f3ff; padding: 20px; border-left: 4px solid #2271b1; margin-top: 30px;">
                    <h3 style="margin-top: 0;">💡 How to Use Schema Preview</h3>
                    <ol style="margin-bottom: 0;">
                        <li><strong>Enter a URL</strong> from your website (any post, page, product, etc.)</li>
                        <li><strong>Click "Preview Schema"</strong> to see generated markup</li>
                        <li><strong>Click "Validate"</strong> to test with Google's tools</li>
                        <li><strong>Fix any errors</strong> before your content goes live!</li>
                    </ol>
                </div>
            </div>

        <?php elseif ($active_tab === 'debug'): ?>
            <!-- Debug Log Tab -->
            <div style="background: white; padding: 30px; margin-top: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                <h2>🐛 Debug Log</h2>

                <div style="background: #fffbcc; border-left: 4px solid #ffb900; padding: 15px; margin-bottom: 20px;">
                    <p style="margin: 0;"><strong>⚠️ Debug Mode:</strong> <?php echo get_option('cirvbo_debug_mode') ? '✅ Enabled' : '❌ Disabled'; ?></p>
                    <p style="margin: 10px 0 0 0;">Enable debug mode in General Settings to log schema generation events for troubleshooting.</p>
                </div>

                <div style="margin-bottom: 20px;">
                    <form method="post" style="display: inline;">
                        <?php wp_nonce_field('cirvbo_clear_log_action', 'cirvbo_clear_log_nonce'); ?>
                        <button type="submit" name="cirvbo_clear_log" class="button" onclick="return confirm('Are you sure you want to clear the debug log?');">
                            🗑️ Clear Log
                        </button>
                    </form>
                    <form method="post" style="display: inline; margin-left: 10px;">
                        <?php wp_nonce_field('cirvbo_export_log_action', 'cirvbo_export_log_nonce'); ?>
                        <button type="submit" name="cirvbo_export_log" class="button">
                            📥 Export Log
                        </button>
                    </form>
                </div>

                <?php
                // Handle log clearing
                if (isset($_POST['cirvbo_clear_log']) && check_admin_referer('cirvbo_clear_log_action', 'cirvbo_clear_log_nonce')) {
                    // Verify user has permission
                    if (current_user_can('manage_options')) {
                        cirvbo_clear_debug_log();
                        echo '<div class="notice notice-success inline"><p>Debug log cleared successfully!</p></div>';
                    } else {
                        echo '<div class="notice notice-error inline"><p>You do not have permission to clear the debug log.</p></div>';
                    }
                }

                // Display log entries
                $log = cirvbo_get_debug_log(50);

                if (empty($log)) {
                    echo '<div class="notice notice-info inline"><p>No log entries yet. Browse your site to generate schema and check back here.</p></div>';
                } else {
                    echo '<h3>Recent Log Entries (Last 50)</h3>';
                    echo '<table class="wp-list-table widefat fixed striped">';
                    echo '<thead><tr>';
                    echo '<th style="width: 15%;">Time</th>';
                    echo '<th style="width: 10%;">Type</th>';
                    echo '<th style="width: 30%;">Message</th>';
                    echo '<th style="width: 25%;">URL</th>';
                    echo '<th style="width: 20%;">Data</th>';
                    echo '</tr></thead>';
                    echo '<tbody>';

                    foreach ($log as $entry) {
                        $type_color = [
                            'info' => '#2271b1',
                            'warning' => '#dba617',
                            'error' => '#d63638'
                        ];
                        $color = $type_color[$entry['type']] ?? '#000';

                        echo '<tr>';
                        echo '<td>' . esc_html($entry['timestamp']) . '</td>';
                        echo '<td style="color: ' . esc_attr($color) . '; font-weight: bold;">' . esc_html(strtoupper($entry['type'])) . '</td>';
                        echo '<td>' . esc_html($entry['message']) . '</td>';
                        echo '<td style="font-size: 11px;">' . esc_html($entry['url']) . '</td>';
                        echo '<td><code style="font-size: 10px;">' . esc_html(json_encode($entry['data'])) . '</code></td>';
                        echo '</tr>';
                    }

                    echo '</tbody></table>';
                }
                ?>

                <div style="background: #e7f3ff; padding: 20px; border-left: 4px solid #2271b1; margin-top: 30px;">
                    <h3 style="margin-top: 0;">💡 Using the Debug Log</h3>
                    <ul style="margin-bottom: 0;">
                        <li><strong>Enable Debug Mode</strong> in General Settings</li>
                        <li><strong>Browse your site</strong> - visit posts, pages, products</li>
                        <li><strong>Check this tab</strong> to see what schemas were generated</li>
                        <li><strong>Export log</strong> and send to support if you need help</li>
                        <li><strong>Disable debug mode</strong> when done (saves database space)</li>
                    </ul>
                </div>
            </div>

        <?php elseif ($active_tab === 'upgrade'): ?>
            <!-- Upgrade to Pro Tab -->
            <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px; margin-top: 20px; border-radius: 8px; color: white;">
                <h2 style="color: white; margin-top: 0;">⭐ Unlock Pro Features</h2>
                <p style="font-size: 18px; margin-bottom: 30px;">Take your SEO to the next level with advanced schema types!</p>

                <div style="background: rgba(255,255,255,0.2); padding: 20px; border-radius: 8px; margin-bottom: 20px;">
                    <h3 style="color: white; margin-top: 0;">Pro Features Include:</h3>
                    <ul style="list-style: none; padding: 0; font-size: 16px;">
                        <li style="padding: 10px 0;">✨ <strong>Local Business Schema</strong> - For brick-and-mortar stores</li>
                        <li style="padding: 10px 0;">✨ <strong>Review/Rating Schema</strong> - Show star ratings in search</li>
                        <li style="padding: 10px 0;">✨ <strong>Video Schema</strong> - For video content creators</li>
                        <li style="padding: 10px 0;">✨ <strong>Event Schema</strong> - Promote events in search</li>
                        <li style="padding: 10px 0;">✨ <strong>Recipe Schema</strong> - Perfect for food blogs</li>
                        <li style="padding: 10px 0;">✨ <strong>Course Schema</strong> - For online courses</li>
                        <li style="padding: 10px 0;">✨ <strong>Priority Support</strong> - Get help within 24 hours</li>
                    </ul>
                </div>

                <div style="text-align: center; padding: 20px;">
                    <?php
                    // Get Freemius pricing URL
                    $pricing_url = '#';
                    if (function_exists('cirvbo_fs') && cirvbo_fs()) {
                        $pricing_url = cirvbo_fs()->get_upgrade_url();
                    }
                    ?>
                    <a href="<?php echo esc_url($pricing_url); ?>"
                       style="display: inline-block; background: white; color: #667eea; padding: 15px 40px; border-radius: 30px; text-decoration: none; font-weight: bold; font-size: 18px; box-shadow: 0 4px 15px rgba(0,0,0,0.2);">
                        Upgrade to Pro - Only $8.99/month →
                    </a>
                    <p style="margin-top: 15px; opacity: 0.9;">7-day free trial • Cancel anytime</p>
                </div>
            </div>
        <?php endif; ?>

        <!-- Footer info with cache stats -->
        <div style="background: #f9f9f9; border: 1px solid #ddd; padding: 15px; margin-top: 20px; border-radius: 4px;">
            <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 15px;">
                <div>
                    <p style="margin: 0; color: #666;">
                        <strong>Cirv Box v<?php echo esc_html(CIRVBO_VERSION); ?></strong> |
                        Built with ❤️ for WordPress |
                        <a href="?page=cirv-box&tab=help">Documentation</a>
                    </p>
                    <?php
                    $cache_stats = cirvbo_get_cache_stats();
                    ?>
                    <p style="margin: 5px 0 0 0; color: #999; font-size: 12px;">
                        📊 Performance: <?php echo absint($cache_stats['count']); ?> schemas cached (<?php echo esc_html($cache_stats['size']); ?>)
                    </p>
                </div>
                <div>
                    <form method="post" style="margin: 0;">
                        <?php wp_nonce_field('cirvbo_clear_cache_action', 'cirvbo_clear_cache_nonce'); ?>
                        <button type="submit"
                                name="cirvbo_clear_cache"
                                class="button"
                                onclick="return confirm('Are you sure you want to clear all cached schemas? They will be regenerated automatically.');">
                            🗑️ Clear Cache
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>
    <?php
}

/**
 * ═══════════════════════════════════════════════════════════
 * REGISTER SETTINGS
 * ═══════════════════════════════════════════════════════════
 *
 * This tells WordPress:
 * 1. What settings exist
 * 2. How to display them (fields, sections)
 * 3. How to validate/sanitize them (security!)
 */
function cirvbo_settings_init() {

    // Register each setting (WordPress will save these to database)
    // All settings include sanitization callbacks for security
    register_setting('cirvbo_settings', 'cirvbo_enable_article_schema', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => true
    ]);
    register_setting('cirvbo_settings', 'cirvbo_enable_organization_schema', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => false
    ]);
    register_setting('cirvbo_settings', 'cirvbo_enable_product_schema', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => true
    ]);
    register_setting('cirvbo_settings', 'cirvbo_enable_faq_schema', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => true
    ]);
    register_setting('cirvbo_settings', 'cirvbo_enable_breadcrumb_schema', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => true
    ]);
    register_setting('cirvbo_settings', 'cirvbo_enable_howto_schema', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => true
    ]);
    register_setting('cirvbo_settings', 'cirvbo_enable_review_schema', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => true
    ]);
    register_setting('cirvbo_settings', 'cirvbo_enable_recipe_schema', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => true
    ]);
    register_setting('cirvbo_settings', 'cirvbo_enable_event_schema', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => false
    ]);
    register_setting('cirvbo_settings', 'cirvbo_debug_mode', [
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => false
    ]);
    register_setting('cirvbo_settings', 'cirvbo_organization_name', [
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => ''
    ]);

    // Add a settings section (groups related fields together)
    add_settings_section(
        'cirvbo_section_basic',                    // Section ID
        'Basic Schema Settings (FREE)',         // Section title
        'cirvbo_section_basic_callback',           // Callback for description
        'cirv-box'                        // Page slug (matches menu slug)
    );

    // Add individual fields to the section
    add_settings_field(
        'cirvbo_enable_article_schema',            // Field ID
        'Enable Article Schema',                // Field label
        'cirvbo_field_checkbox_callback',          // Callback to render the field
        'cirv-box',                       // Page slug
        'cirvbo_section_basic',                    // Section ID (where to display)
        [
            'label_for' => 'cirvbo_enable_article_schema',
            'description' => 'Automatically add Article schema to all blog posts'
        ]
    );

    add_settings_field(
        'cirvbo_enable_organization_schema',
        'Enable Organization Schema',
        'cirvbo_field_checkbox_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_enable_organization_schema',
            'description' => 'Add your organization details to the homepage'
        ]
    );

    add_settings_field(
        'cirvbo_enable_product_schema',
        'Enable Product Schema',
        'cirvbo_field_checkbox_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_enable_product_schema',
            'description' => 'Add Product schema to WooCommerce product pages (requires WooCommerce)'
        ]
    );

    add_settings_field(
        'cirvbo_enable_faq_schema',
        'Enable FAQ Schema',
        'cirvbo_field_checkbox_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_enable_faq_schema',
            'description' => 'Automatically detect and mark up FAQ content (appears in Google search results!)'
        ]
    );

    add_settings_field(
        'cirvbo_enable_breadcrumb_schema',
        'Enable Breadcrumb Schema',
        'cirvbo_field_checkbox_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_enable_breadcrumb_schema',
            'description' => 'Add breadcrumb navigation to all pages (improves site structure in search results)'
        ]
    );

    add_settings_field(
        'cirvbo_enable_howto_schema',
        'Enable HowTo Schema',
        'cirvbo_field_checkbox_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_enable_howto_schema',
            'description' => 'Automatically detect step-by-step guides and tutorials (rich results in Google!)'
        ]
    );

    add_settings_field(
        'cirvbo_enable_review_schema',
        'Enable Review Schema',
        'cirvbo_field_checkbox_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_enable_review_schema',
            'description' => 'Automatically detect review posts and add Review schema (star ratings in search results!)'
        ]
    );

    add_settings_field(
        'cirvbo_enable_recipe_schema',
        'Enable Recipe Schema',
        'cirvbo_field_checkbox_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_enable_recipe_schema',
            'description' => 'Automatically detect recipe posts and add Recipe schema (cook time, ingredients in search!)'
        ]
    );

    add_settings_field(
        'cirvbo_enable_event_schema',
        esc_html__('Enable Event Schema', 'cirv-box') . (cirvbo_is_pro() ? '' : ' (PRO)'),
        'cirvbo_field_checkbox_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_enable_event_schema',
            'description' => 'Automatically detect event posts and add Event schema (date, location, tickets in search!)',
            'disabled' => !cirvbo_is_pro()
        ]
    );

    add_settings_field(
        'cirvbo_organization_name',
        'Organization Name',
        'cirvbo_field_text_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_organization_name',
            'placeholder' => 'Enter your company name'
        ]
    );

    add_settings_field(
        'cirvbo_debug_mode',
        'Debug Mode',
        'cirvbo_field_checkbox_callback',
        'cirv-box',
        'cirvbo_section_basic',
        [
            'label_for' => 'cirvbo_debug_mode',
            'description' => 'Enable debug logging to troubleshoot schema issues (check Debug Log tab)'
        ]
    );
}
add_action('admin_init', 'cirvbo_settings_init');

/**
 * ═══════════════════════════════════════════════════════════
 * CALLBACK FUNCTIONS - Render the settings fields
 * ═══════════════════════════════════════════════════════════
 */

// Section description
function cirvbo_section_basic_callback() {
    echo '<p>Configure basic schema markup for your website. Free features available below!</p>';
}

// Checkbox field
function cirvbo_field_checkbox_callback($args) {
    $option = get_option($args['label_for']);
    $disabled = !empty($args['disabled']);
    ?>
    <input
        type="checkbox"
        id="<?php echo esc_attr($args['label_for']); ?>"
        name="<?php echo esc_attr($args['label_for']); ?>"
        value="1"
        <?php checked(1, $option); ?>
        <?php disabled($disabled); ?>
    >
    <label for="<?php echo esc_attr($args['label_for']); ?>">
        <?php echo esc_html($args['description']); ?>
    </label>
    <?php if ($disabled && function_exists('cirvbo_fs') && cirvbo_fs()) : ?>
        <a href="<?php echo esc_url(cirvbo_fs()->get_upgrade_url()); ?>" style="margin-left: 8px; color: #2271b1;"><?php esc_html_e('Upgrade to Pro', 'cirv-box'); ?></a>
    <?php endif; ?>
    <?php
}

// Text field
function cirvbo_field_text_callback($args) {
    $option = get_option($args['label_for']);
    ?>
    <input
        type="text"
        id="<?php echo esc_attr($args['label_for']); ?>"
        name="<?php echo esc_attr($args['label_for']); ?>"
        value="<?php echo esc_attr($option); ?>"
        placeholder="<?php echo esc_attr($args['placeholder']); ?>"
        class="regular-text"
    >
    <?php
}

/**
 * ═══════════════════════════════════════════════════════════
 * SEO PLUGIN CONFLICT DETECTION
 * ═══════════════════════════════════════════════════════════
 *
 * Detects if other SEO plugins are active that already output schema
 * This prevents duplicate schema markup which confuses search engines
 *
 * SUPPORTED PLUGINS:
 * - Yoast SEO (WPSEO_VERSION)
 * - Rank Math (RANK_MATH_VERSION)
 * - All in One SEO (AIOSEO_VERSION)
 */
function cirvbo_has_schema_conflict() {
    // Check for popular SEO plugins that output schema
    return defined('WPSEO_VERSION') ||        // Yoast SEO
           defined('RANK_MATH_VERSION') ||    // Rank Math
           defined('AIOSEO_VERSION');         // All in One SEO
}

/**
 * ═══════════════════════════════════════════════════════════
 * SCHEMA GENERATION - The magic happens here!
 * ═══════════════════════════════════════════════════════════
 *
 * This function outputs JSON-LD schema markup in the <head> of pages
 * JSON-LD = JavaScript Object Notation for Linked Data
 * Google LOVES this format for structured data
 *
 * PERFORMANCE OPTIMIZATION:
 * Uses WordPress Transient API to cache schema generation
 * Cache key: ssb_schema_{post_id}_{modified_time}
 * Cache expires: 24 hours (or when post is updated)
 *
 * WHY CACHING MATTERS:
 * Without cache: 50ms to generate schema on every page load
 * With cache: 0.5ms to retrieve from cache
 * On 10,000 visits/day: Saves 500 seconds of server time!
 */
function cirvbo_output_schema() {
    // Skip if other SEO plugins are active (prevents duplicate schema)
    if (cirvbo_has_schema_conflict()) {
        cirvbo_log('Article schema skipped: SEO plugin conflict detected', 'warning');
        return;
    }
    // Only run on single blog posts
    if (!is_single()) {
        return;
    }

    // Check if Article schema is enabled
    if (!get_option('cirvbo_enable_article_schema')) {
        cirvbo_log('Article schema skipped: disabled in settings', 'info');
        return;
    }

    // Get the current post object
    global $post;

    // Try to get cached schema first
    // Cache key: ssb_article_schema_{blog_id}_{post_id}
    // No timestamp needed - cache is cleared on post save/delete
    $cache_key = 'cirvbo_article_schema_' . get_current_blog_id() . '_' . $post->ID;
    $cached_schema = get_transient($cache_key);

    if ($cached_schema !== false) {
        // Cache hit! Use cached version
        cirvbo_log('Article schema loaded from cache', 'info', ['post_id' => $post->ID, 'title' => get_the_title()]);
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
        echo $cached_schema;
        return;
    }

    // Cache miss - generate schema
    $schema = [
        '@context' => 'https://schema.org',
        '@type' => 'Article',
        'headline' => get_the_title(),
        'description' => get_the_excerpt(),
        'image' => get_the_post_thumbnail_url($post->ID, 'full'),
        'datePublished' => get_the_date('c', $post),
        'dateModified' => get_the_modified_date('c', $post),
        'author' => [
            '@type' => 'Person',
            'name' => get_the_author()
        ],
        'publisher' => [
            '@type' => 'Organization',
            'name' => get_bloginfo('name')
        ]
    ];

    // Only add publisher logo if site icon exists
    $site_icon = get_site_icon_url();
    if (!empty($site_icon)) {
        $schema['publisher']['logo'] = [
            '@type' => 'ImageObject',
            'url' => $site_icon
        ];
    }

    // Build output
    $output = '<script type="application/ld+json">';
    $output .= wp_json_encode($schema, JSON_PRETTY_PRINT);
    $output .= '</script>' . "\n";

    // Cache for 24 hours
    set_transient($cache_key, $output, DAY_IN_SECONDS);

    // Log successful generation
    cirvbo_log('Article schema generated successfully', 'info', [
        'post_id' => $post->ID,
        'title' => get_the_title(),
        'has_image' => !empty($schema['image']),
        'has_publisher_logo' => isset($schema['publisher']['logo'])
    ]);

    // Output - phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
    echo $output;
}
add_action('wp_head', 'cirvbo_output_schema');

/**
 * ═══════════════════════════════════════════════════════════
 * PRODUCT SCHEMA - WooCommerce Integration
 * ═══════════════════════════════════════════════════════════
 *
 * This function generates Product schema for WooCommerce products
 * Helps Google show product info in search results (price, availability, ratings)
 *
 * WHAT IT DOES:
 * 1. Checks if WooCommerce is active
 * 2. Checks if we're on a product page
 * 3. Grabs product data (price, SKU, stock, reviews)
 * 4. Outputs Google-friendly Product schema
 */
function cirvbo_output_product_schema() {
    // Skip if other SEO plugins are active (prevents duplicate schema)
    if (cirvbo_has_schema_conflict()) {
        return;
    }

    // Only run if Product schema is enabled
    if (!get_option('cirvbo_enable_product_schema')) {
        return;
    }

    // Check if WooCommerce is active
    // class_exists() checks if a PHP class is loaded
    if (!class_exists('WooCommerce')) {
        return; // WooCommerce not installed, skip
    }

    // Check if we're on a single product page
    // is_product() is a WooCommerce function
    if (!is_product()) {
        return;
    }

    // Get the current product object
    // wc_get_product() is WooCommerce's way to get product data
    global $product;
    if (!$product) {
        $product = wc_get_product(get_the_ID());
    }

    // Try to get cached schema first
    $cache_key = 'cirvbo_product_schema_' . get_current_blog_id() . '_' . get_the_ID();
    $cached_schema = get_transient($cache_key);

    if ($cached_schema !== false) {
        // Cache hit! Use cached version
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
        echo $cached_schema;
        return;
    }

    // Build the Product schema array
    $schema = [
        '@context' => 'https://schema.org',
        '@type' => 'Product',
        'name' => $product->get_name(),
        'description' => wp_strip_all_tags($product->get_short_description()),
        'sku' => $product->get_sku(),
        'image' => wp_get_attachment_url($product->get_image_id())
    ];

    // Only add offers section if product has a valid price
    $price = $product->get_price();
    if (!empty($price) && is_numeric($price)) {
        $schema['offers'] = [
            '@type' => 'Offer',
            'url' => get_permalink($product->get_id()),
            'priceCurrency' => get_woocommerce_currency(), // USD, EUR, etc.
            'price' => $price,
            'availability' => $product->is_in_stock() ?
                'https://schema.org/InStock' :
                'https://schema.org/OutOfStock',
            'priceValidUntil' => gmdate('Y-12-31'), // Valid until end of year
        ];
    }

    // Add brand if available (many stores add this as product attribute)
    $brand = $product->get_attribute('brand');
    if ($brand) {
        $schema['brand'] = [
            '@type' => 'Brand',
            'name' => $brand
        ];
    }

    // Add review/rating data if product has reviews
    if ($product->get_review_count() > 0) {
        $schema['aggregateRating'] = [
            '@type' => 'AggregateRating',
            'ratingValue' => $product->get_average_rating(),
            'reviewCount' => $product->get_review_count()
        ];
    }

    // Build output
    $output = '<script type="application/ld+json">';
    $output .= wp_json_encode($schema, JSON_PRETTY_PRINT);
    $output .= '</script>' . "\n";

    // Cache for 24 hours
    set_transient($cache_key, $output, DAY_IN_SECONDS);

    // Output - phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
    echo $output;
}
add_action('wp_head', 'cirvbo_output_product_schema');

/**
 * ═══════════════════════════════════════════════════════════
 * ORGANIZATION SCHEMA - Brand Identity
 * ═══════════════════════════════════════════════════════════
 *
 * This function generates Organization schema for your homepage
 * Helps Google understand your brand, show Knowledge Panels, etc.
 *
 * WHAT IT DOES:
 * 1. Only runs on homepage (is_front_page())
 * 2. Outputs your business/organization details
 * 3. Helps with local SEO and brand recognition
 * 4. Powers Google Knowledge Graph
 */
function cirvbo_output_organization_schema() {
    // Skip if other SEO plugins are active (prevents duplicate schema)
    if (cirvbo_has_schema_conflict()) {
        return;
    }

    // Only run on homepage
    if (!is_front_page()) {
        return;
    }

    // Only run if Organization schema is enabled
    if (!get_option('cirvbo_enable_organization_schema')) {
        return;
    }

    // Try to get cached schema first
    $cache_key = 'cirvbo_organization_schema_' . get_current_blog_id();
    $cached_schema = get_transient($cache_key);

    if ($cached_schema !== false) {
        // Cache hit! Use cached version
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
        echo $cached_schema;
        return;
    }

    // Get organization name from settings (or fallback to site name)
    $org_name = get_option('cirvbo_organization_name');
    if (empty($org_name)) {
        $org_name = get_bloginfo('name');
    }

    // Build the Organization schema
    $schema = [
        '@context' => 'https://schema.org',
        '@type' => 'Organization',
        'name' => $org_name,
        'url' => home_url(),
        'description' => get_bloginfo('description'),
    ];

    // Only add logo if site icon exists
    $site_icon = get_site_icon_url();
    if (!empty($site_icon)) {
        $schema['logo'] = [
            '@type' => 'ImageObject',
            'url' => $site_icon
        ];
    }

    // Add contact info if available (you can extend this with custom fields later)
    // For now, we'll use the admin email
    $schema['email'] = get_option('admin_email');

    // Build output
    $output = '<script type="application/ld+json">';
    $output .= wp_json_encode($schema, JSON_PRETTY_PRINT);
    $output .= '</script>' . "\n";

    // Cache for 24 hours
    set_transient($cache_key, $output, DAY_IN_SECONDS);

    // Output - phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
    echo $output;
}
add_action('wp_head', 'cirvbo_output_organization_schema');

/**
 * ═══════════════════════════════════════════════════════════
 * FAQ SCHEMA - Intelligent Content Parsing
 * ═══════════════════════════════════════════════════════════
 *
 * This function intelligently detects FAQ content and generates schema
 * Works on pages/posts with "FAQ" in title or specific patterns in content
 *
 * WHAT IT DOES:
 * 1. Detects FAQ pages by title or content patterns
 * 2. Parses HTML to extract questions and answers
 * 3. Generates FAQPage schema for Google Rich Results
 * 4. Your FAQs can appear expanded in search results!
 *
 * DETECTION PATTERNS:
 * - Pages with "FAQ" in title
 * - H2/H3 headings that start with question words (What, How, Why, etc.)
 * - Accordion blocks (details/summary HTML)
 */
function cirvbo_output_faq_schema() {
    // Skip if other SEO plugins are active (prevents duplicate schema)
    if (cirvbo_has_schema_conflict()) {
        return;
    }

    // Only run if FAQ schema is enabled
    if (!get_option('cirvbo_enable_faq_schema')) {
        return;
    }

    // Only run on pages or posts
    if (!is_singular(['post', 'page'])) {
        return;
    }

    global $post;

    // Try to get cached schema first
    $cache_key = 'cirvbo_faq_schema_' . get_current_blog_id() . '_' . $post->ID;
    $cached_schema = get_transient($cache_key);

    if ($cached_schema !== false) {
        // Cache hit! Use cached version
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
        echo $cached_schema;
        return;
    }

    // Get post content
    $content = get_the_content(null, false, $post);
    $content = apply_filters('the_content', $content); // Apply WordPress content filters
    $title = get_the_title();

    // SECURITY: Limit content length to prevent ReDoS attacks (100KB max)
    if (strlen($content) > 102400) {
        return; // Content too large, skip FAQ parsing
    }

    // Check if this looks like an FAQ page
    $is_faq_page = false;

    // Method 1: Title contains "FAQ" or "Frequently Asked"
    if (stripos($title, 'faq') !== false || stripos($title, 'frequently asked') !== false) {
        $is_faq_page = true;
    }

    // Method 2: Content has multiple headings that look like questions
    $question_pattern = '/<h[23][^>]*>((?:what|how|why|when|where|who|can|is|are|do|does)[^<]+\?)<\/h[23]>/i';
    preg_match_all($question_pattern, $content, $question_matches);

    if (count($question_matches[0]) >= 2) {
        $is_faq_page = true;
    }

    // If not an FAQ page, exit
    if (!$is_faq_page) {
        return;
    }

    // Parse the content to extract Q&A pairs
    $faq_items = [];

    // Extract question/answer pairs using regex
    // Pattern: Find h2/h3 headings followed by content until next heading
    $sections_pattern = '/<h[23][^>]*>([^<]+)<\/h[23]>(.*?)(?=<h[23]|$)/is';
    preg_match_all($sections_pattern, $content, $sections);

    for ($i = 0; $i < count($sections[1]); $i++) {
        $potential_question = trim(wp_strip_all_tags($sections[1][$i]));
        $potential_answer = trim(wp_strip_all_tags($sections[2][$i]));

        // Only include if it looks like a question (ends with ? or starts with question word)
        if (
            substr($potential_question, -1) === '?' ||
            preg_match('/^(what|how|why|when|where|who|can|is|are|do|does)\b/i', $potential_question)
        ) {
            // Only include if we have both question and answer
            if (!empty($potential_question) && !empty($potential_answer) && strlen($potential_answer) > 10) {
                $faq_items[] = [
                    '@type' => 'Question',
                    'name' => $potential_question,
                    'acceptedAnswer' => [
                        '@type' => 'Answer',
                        'text' => $potential_answer
                    ]
                ];

                // LIMIT: Max 50 FAQs to prevent performance issues
                if (count($faq_items) >= 50) {
                    break;
                }
            }
        }
    }

    // Only output schema if we found at least 2 FAQs
    if (count($faq_items) < 2) {
        return;
    }

    // Build the FAQPage schema
    $schema = [
        '@context' => 'https://schema.org',
        '@type' => 'FAQPage',
        'mainEntity' => $faq_items
    ];

    // Build output
    $output = '<script type="application/ld+json">';
    $output .= wp_json_encode($schema, JSON_PRETTY_PRINT);
    $output .= '</script>' . "\n";

    // Cache for 24 hours
    set_transient($cache_key, $output, DAY_IN_SECONDS);

    // Output - phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
    echo $output;
}
add_action('wp_head', 'cirvbo_output_faq_schema');

/**
 * ═══════════════════════════════════════════════════════════
 * BREADCRUMB SCHEMA - Site Navigation Enhancement
 * ═══════════════════════════════════════════════════════════
 *
 * This function generates BreadcrumbList schema for all pages
 * Helps Google understand your site structure and hierarchy
 *
 * WHAT IT DOES:
 * 1. Builds breadcrumb trail from homepage to current page
 * 2. Uses WordPress page hierarchy for pages
 * 3. Uses categories for blog posts
 * 4. Shows in Google search results as clickable breadcrumbs
 *
 * SEO BENEFITS:
 * - Improves site architecture understanding
 * - Enhanced search result display
 * - Better internal linking signals
 * - Clearer user navigation path
 */
function cirvbo_output_breadcrumb_schema() {
    // Skip if other SEO plugins are active
    if (cirvbo_has_schema_conflict()) {
        return;
    }

    // Only run if Breadcrumb schema is enabled
    if (!get_option('cirvbo_enable_breadcrumb_schema')) {
        return;
    }

    // Skip on homepage (no breadcrumbs needed)
    if (is_front_page()) {
        return;
    }

    global $post;

    // Try to get cached schema first
    $cache_key = 'cirvbo_breadcrumb_schema_' . get_current_blog_id() . '_' . get_queried_object_id();
    $cached_schema = get_transient($cache_key);

    if ($cached_schema !== false) {
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
        echo $cached_schema;
        return;
    }

    // Build breadcrumb items array
    $breadcrumb_items = [];

    // Always start with homepage
    $breadcrumb_items[] = [
        '@type' => 'ListItem',
        'position' => 1,
        'name' => 'Home',
        'item' => home_url()
    ];

    $position = 2;

    // Handle different page types
    if (is_single()) {
        // For blog posts, add category hierarchy
        $categories = get_the_category();
        if (!empty($categories)) {
            $main_category = $categories[0];

            // Get category hierarchy
            $category_hierarchy = [];
            $current_cat = $main_category;

            while ($current_cat) {
                array_unshift($category_hierarchy, $current_cat);
                $current_cat = $current_cat->parent ? get_category($current_cat->parent) : null;
            }

            // Add categories to breadcrumb
            foreach ($category_hierarchy as $cat) {
                $breadcrumb_items[] = [
                    '@type' => 'ListItem',
                    'position' => $position++,
                    'name' => $cat->name,
                    'item' => get_category_link($cat->term_id)
                ];
            }
        }

        // Add current post
        $breadcrumb_items[] = [
            '@type' => 'ListItem',
            'position' => $position,
            'name' => get_the_title(),
            'item' => get_permalink()
        ];

    } elseif (is_page()) {
        // For pages, add parent page hierarchy
        if ($post->post_parent) {
            $ancestors = array_reverse(get_post_ancestors($post->ID));

            foreach ($ancestors as $ancestor_id) {
                $breadcrumb_items[] = [
                    '@type' => 'ListItem',
                    'position' => $position++,
                    'name' => get_the_title($ancestor_id),
                    'item' => get_permalink($ancestor_id)
                ];
            }
        }

        // Add current page
        $breadcrumb_items[] = [
            '@type' => 'ListItem',
            'position' => $position,
            'name' => get_the_title(),
            'item' => get_permalink()
        ];

    } elseif (is_category()) {
        // Category archive
        $category = get_queried_object();

        // Add parent categories if exist
        if ($category->parent) {
            $ancestors = array_reverse(get_ancestors($category->term_id, 'category'));
            foreach ($ancestors as $ancestor_id) {
                $ancestor = get_category($ancestor_id);
                $breadcrumb_items[] = [
                    '@type' => 'ListItem',
                    'position' => $position++,
                    'name' => $ancestor->name,
                    'item' => get_category_link($ancestor_id)
                ];
            }
        }

        // Add current category
        $breadcrumb_items[] = [
            '@type' => 'ListItem',
            'position' => $position,
            'name' => $category->name,
            'item' => get_category_link($category->term_id)
        ];

    } elseif (is_search()) {
        // Search results
        $breadcrumb_items[] = [
            '@type' => 'ListItem',
            'position' => $position,
            'name' => 'Search Results for "' . get_search_query() . '"',
            'item' => get_search_link()
        ];
    }

    // Build the BreadcrumbList schema
    $schema = [
        '@context' => 'https://schema.org',
        '@type' => 'BreadcrumbList',
        'itemListElement' => $breadcrumb_items
    ];

    // Build output
    $output = '<script type="application/ld+json">';
    $output .= wp_json_encode($schema, JSON_PRETTY_PRINT);
    $output .= '</script>' . "\n";

    // Cache for 24 hours
    set_transient($cache_key, $output, DAY_IN_SECONDS);

    // Output - phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
    echo $output;
}
add_action('wp_head', 'cirvbo_output_breadcrumb_schema');

/**
 * ═══════════════════════════════════════════════════════════
 * HOWTO SCHEMA - Step-by-Step Guide Detection
 * ═══════════════════════════════════════════════════════════
 *
 * This function intelligently detects tutorial/how-to content
 * Creates rich results in Google with step-by-step instructions
 *
 * WHAT IT DOES:
 * 1. Detects how-to content by title patterns
 * 2. Parses ordered lists or numbered headings as steps
 * 3. Generates HowTo schema for Google Rich Results
 * 4. Shows in search with expandable steps!
 *
 * DETECTION PATTERNS:
 * - Titles starting with "How to", "Guide to", "Tutorial"
 * - Content with ordered lists (numbered steps)
 * - Headings with "Step 1", "Step 2", etc.
 *
 * SEO BENEFITS:
 * - Rich results with step-by-step display
 * - Higher CTR (click-through rate)
 * - Featured snippet opportunities
 * - Voice search optimization
 */
function cirvbo_output_howto_schema() {
    // Skip if other SEO plugins are active
    if (cirvbo_has_schema_conflict()) {
        return;
    }

    // Only run if HowTo schema is enabled
    if (!get_option('cirvbo_enable_howto_schema')) {
        return;
    }

    // Only run on pages or posts
    if (!is_singular(['post', 'page'])) {
        return;
    }

    global $post;

    // Try to get cached schema first
    $cache_key = 'cirvbo_howto_schema_' . get_current_blog_id() . '_' . $post->ID;
    $cached_schema = get_transient($cache_key);

    if ($cached_schema !== false) {
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
        echo $cached_schema;
        return;
    }

    // Get post content and title
    $content = get_the_content(null, false, $post);
    $content = apply_filters('the_content', $content);
    $title = get_the_title();

    // SECURITY: Limit content length to prevent ReDoS attacks (100KB max)
    if (strlen($content) > 102400) {
        return;
    }

    // Check if this looks like a how-to guide
    $is_howto = false;
    $title_lower = strtolower($title);

    // Method 1: Title contains how-to indicators
    if (
        preg_match('/^how to\b/i', $title_lower) ||
        preg_match('/\bguide to\b/i', $title_lower) ||
        preg_match('/\btutorial\b/i', $title_lower) ||
        preg_match('/\bstep by step\b/i', $title_lower) ||
        preg_match('/\bdiy\b/i', $title_lower)
    ) {
        $is_howto = true;
    }

    // Method 2: Content has numbered steps pattern
    if (preg_match_all('/<li[^>]*>([^<]+)<\/li>/i', $content, $matches) >= 3) {
        // Has ordered list with at least 3 items
        $is_howto = true;
    }

    // Method 3: Has "Step X" headings
    if (preg_match_all('/<h[23][^>]*>step\s+\d+[^<]*<\/h[23]>/i', $content, $matches) >= 3) {
        $is_howto = true;
    }

    // If not a how-to guide, exit
    if (!$is_howto) {
        return;
    }

    // Parse steps from content
    $steps = [];

    // Try parsing numbered list items first
    $list_pattern = '/<ol[^>]*>(.*?)<\/ol>/is';
    if (preg_match($list_pattern, $content, $list_match)) {
        // Extract list items
        preg_match_all('/<li[^>]*>(.*?)<\/li>/is', $list_match[1], $item_matches);

        foreach ($item_matches[1] as $index => $item_html) {
            $step_text = trim(wp_strip_all_tags($item_html));

            // Only include steps with meaningful content
            if (!empty($step_text) && strlen($step_text) > 10) {
                $steps[] = [
                    '@type' => 'HowToStep',
                    'position' => $index + 1,
                    'name' => 'Step ' . ($index + 1),
                    'text' => $step_text
                ];

                // LIMIT: Max 20 steps to prevent performance issues
                if (count($steps) >= 20) {
                    break;
                }
            }
        }
    }

    // If no ordered list found, try "Step X" headings
    if (empty($steps)) {
        $step_pattern = '/<h[23][^>]*>(step\s+\d+[^<]*)<\/h[23]>(.*?)(?=<h[23]|$)/is';
        preg_match_all($step_pattern, $content, $step_matches);

        for ($i = 0; $i < count($step_matches[1]); $i++) {
            $step_name = trim(wp_strip_all_tags($step_matches[1][$i]));
            $step_text = trim(wp_strip_all_tags($step_matches[2][$i]));

            if (!empty($step_text) && strlen($step_text) > 10) {
                $steps[] = [
                    '@type' => 'HowToStep',
                    'position' => $i + 1,
                    'name' => $step_name,
                    'text' => $step_text
                ];

                if (count($steps) >= 20) {
                    break;
                }
            }
        }
    }

    // Only output schema if we found at least 3 steps
    if (count($steps) < 3) {
        return;
    }

    // Build the HowTo schema
    $schema = [
        '@context' => 'https://schema.org',
        '@type' => 'HowTo',
        'name' => get_the_title(),
        'description' => get_the_excerpt(),
        'image' => get_the_post_thumbnail_url($post->ID, 'full'),
        'step' => $steps
    ];

    // Add total time if available (can be extended with custom fields)
    // For now, estimate based on number of steps (2 minutes per step)
    $estimated_minutes = count($steps) * 2;
    $schema['totalTime'] = 'PT' . $estimated_minutes . 'M';

    // Build output
    $output = '<script type="application/ld+json">';
    $output .= wp_json_encode($schema, JSON_PRETTY_PRINT);
    $output .= '</script>' . "\n";

    // Cache for 24 hours
    set_transient($cache_key, $output, DAY_IN_SECONDS);

    // Output - phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
    echo $output;
}
add_action('wp_head', 'cirvbo_output_howto_schema');

/**
 * ═══════════════════════════════════════════════════════════
 * REVIEW SCHEMA - Automatic Review Detection
 * ═══════════════════════════════════════════════════════════
 *
 * This function detects review posts and generates Review schema
 * Works on posts with "Review" in title or rating patterns in content
 *
 * HOW IT WORKS:
 * 1. Detects review posts by title patterns
 * 2. Extracts rating from content (X/5, X out of 5, stars)
 * 3. Generates Review schema for Google Rich Results
 * 4. Star ratings can appear in search results!
 *
 * DETECTION PATTERNS:
 * - Posts with "Review" in title
 * - Content with "X/5" or "X out of 5" patterns
 * - Content with "Rating:" followed by number
 */
function cirvbo_output_review_schema() {
    // Skip if other SEO plugins are active
    if (cirvbo_has_schema_conflict()) {
        return;
    }

    // Only run if Review schema is enabled
    if (!get_option('cirvbo_enable_review_schema')) {
        return;
    }

    // Only run on single posts or pages
    if (!is_singular(['post', 'page'])) {
        return;
    }

    global $post;

    // Try to get cached schema first
    $cache_key = 'cirvbo_review_schema_' . get_current_blog_id() . '_' . $post->ID;
    $cached_schema = get_transient($cache_key);

    if ($cached_schema !== false) {
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
        echo $cached_schema;
        return;
    }

    // Get post content and title
    $content = get_the_content(null, false, $post);
    $content = apply_filters('the_content', $content);
    $title = get_the_title();

    // SECURITY: Limit content length to prevent ReDoS attacks (100KB max)
    if (strlen($content) > 102400) {
        return;
    }

    // Check if this looks like a review
    $is_review = false;
    $title_lower = strtolower($title);

    // Method 1: Title contains review indicators
    if (
        preg_match('/\breview\b/i', $title_lower) ||
        preg_match('/\breviewed\b/i', $title_lower) ||
        preg_match('/\brating\b/i', $title_lower) ||
        preg_match('/\bvs\b/i', $title_lower) ||
        preg_match('/\bcomparison\b/i', $title_lower)
    ) {
        $is_review = true;
    }

    // Method 2: Content has rating patterns
    if (preg_match('/\b(\d+(?:\.\d+)?)\s*(?:\/|out of)\s*5\b/i', $content)) {
        $is_review = true;
    }

    // Method 3: Content has explicit rating indicator
    if (preg_match('/rating[:\s]+(\d+(?:\.\d+)?)/i', $content)) {
        $is_review = true;
    }

    // If not a review, exit
    if (!$is_review) {
        return;
    }

    // Extract rating from content
    $rating_value = null;

    // Try to find X/5 or X out of 5 pattern
    if (preg_match('/\b(\d+(?:\.\d+)?)\s*(?:\/|out of)\s*5\b/i', $content, $rating_match)) {
        $rating_value = floatval($rating_match[1]);
    }

    // Try Rating: X pattern
    if ($rating_value === null && preg_match('/rating[:\s]+(\d+(?:\.\d+)?)/i', $content, $rating_match)) {
        $rating_value = floatval($rating_match[1]);
    }

    // Default to 4 if no rating found (common assumption for review posts)
    if ($rating_value === null) {
        $rating_value = 4;
    }

    // Ensure rating is valid (1-5 range)
    $rating_value = max(1, min(5, $rating_value));

    // Extract the reviewed item from title (remove "Review" and common patterns)
    $item_reviewed = preg_replace('/\b(review|honest|my|the|a|an)\b/i', '', $title);
    $item_reviewed = trim(preg_replace('/\s+/', ' ', $item_reviewed));

    // If empty after cleanup, use original title
    if (empty($item_reviewed)) {
        $item_reviewed = $title;
    }

    // Build the Review schema
    $schema = [
        '@context' => 'https://schema.org',
        '@type' => 'Review',
        'name' => $title,
        'reviewBody' => wp_trim_words(wp_strip_all_tags($content), 150),
        'author' => [
            '@type' => 'Person',
            'name' => get_the_author()
        ],
        'datePublished' => get_the_date('c'),
        'reviewRating' => [
            '@type' => 'Rating',
            'ratingValue' => $rating_value,
            'bestRating' => 5,
            'worstRating' => 1
        ],
        'itemReviewed' => [
            '@type' => 'Thing',
            'name' => $item_reviewed
        ]
    ];

    // Add image if featured image exists
    $thumbnail = get_the_post_thumbnail_url($post->ID, 'full');
    if ($thumbnail) {
        $schema['image'] = $thumbnail;
    }

    // Build output
    $output = '<script type="application/ld+json">';
    $output .= wp_json_encode($schema, JSON_PRETTY_PRINT);
    $output .= '</script>' . "\n";

    // Cache for 24 hours
    set_transient($cache_key, $output, DAY_IN_SECONDS);

    // Output - phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
    echo $output;
}
add_action('wp_head', 'cirvbo_output_review_schema');

/**
 * ═══════════════════════════════════════════════════════════
 * RECIPE SCHEMA - Automatic Recipe Detection
 * ═══════════════════════════════════════════════════════════
 *
 * This function detects recipe posts and generates Recipe schema
 * Works on posts with recipe indicators in title or content
 *
 * HOW IT WORKS:
 * 1. Detects recipe posts by title patterns
 * 2. Extracts ingredients from lists
 * 3. Extracts cooking instructions from numbered steps
 * 4. Parses prep/cook time if mentioned
 * 5. Generates Recipe schema for Google Rich Results
 *
 * DETECTION PATTERNS:
 * - Posts with "Recipe" in title
 * - Posts with cooking-related keywords (bake, cook, ingredients)
 * - Content with ingredient lists and instructions
 */
function cirvbo_output_recipe_schema() {
    // Skip if other SEO plugins are active
    if (cirvbo_has_schema_conflict()) {
        return;
    }

    // Only run if Recipe schema is enabled
    if (!get_option('cirvbo_enable_recipe_schema')) {
        return;
    }

    // Only run on single posts or pages
    if (!is_singular(['post', 'page'])) {
        return;
    }

    global $post;

    // Try to get cached schema first
    $cache_key = 'cirvbo_recipe_schema_' . get_current_blog_id() . '_' . $post->ID;
    $cached_schema = get_transient($cache_key);

    if ($cached_schema !== false) {
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
        echo $cached_schema;
        return;
    }

    // Get post content and title
    $content = get_the_content(null, false, $post);
    $content = apply_filters('the_content', $content);
    $title = get_the_title();

    // SECURITY: Limit content length to prevent ReDoS attacks (100KB max)
    if (strlen($content) > 102400) {
        return;
    }

    // Check if this looks like a recipe
    $is_recipe = false;
    $title_lower = strtolower($title);
    $content_lower = strtolower($content);

    // Method 1: Title contains recipe indicators
    if (
        preg_match('/\brecipe\b/i', $title_lower) ||
        preg_match('/\bhow to (make|cook|bake|prepare)\b/i', $title_lower)
    ) {
        $is_recipe = true;
    }

    // Method 2: Content has ingredients section
    if (preg_match('/\bingredients?\b/i', $content_lower) && preg_match('/\binstructions?\b/i', $content_lower)) {
        $is_recipe = true;
    }

    // Method 3: Content has cooking keywords combination
    $cooking_keywords = 0;
    if (preg_match('/\bingredients?\b/i', $content_lower)) $cooking_keywords++;
    if (preg_match('/\b(cups?|tablespoons?|teaspoons?|ounces?|pounds?)\b/i', $content_lower)) $cooking_keywords++;
    if (preg_match('/\b(bake|cook|simmer|boil|fry|roast|grill)\b/i', $content_lower)) $cooking_keywords++;
    if (preg_match('/\b(preheat|oven|stove|pan)\b/i', $content_lower)) $cooking_keywords++;
    if ($cooking_keywords >= 3) {
        $is_recipe = true;
    }

    // If not a recipe, exit
    if (!$is_recipe) {
        return;
    }

    // Extract ingredients from unordered lists after "ingredients" heading
    $ingredients = [];
    if (preg_match('/ingredients?.*?<ul[^>]*>(.*?)<\/ul>/is', $content, $ing_match)) {
        preg_match_all('/<li[^>]*>(.*?)<\/li>/is', $ing_match[1], $ing_items);
        foreach ($ing_items[1] as $item) {
            $clean_item = trim(wp_strip_all_tags($item));
            if (!empty($clean_item) && strlen($clean_item) < 200) {
                $ingredients[] = $clean_item;
            }
        }
    }

    // Fallback: extract from any unordered list if it looks like ingredients
    if (empty($ingredients)) {
        preg_match_all('/<ul[^>]*>(.*?)<\/ul>/is', $content, $all_lists);
        foreach ($all_lists[1] as $list) {
            preg_match_all('/<li[^>]*>(.*?)<\/li>/is', $list, $list_items);
            $potential_ingredients = [];
            foreach ($list_items[1] as $item) {
                $clean = trim(wp_strip_all_tags($item));
                // Check if it looks like an ingredient (contains measurement or food words)
                if (preg_match('/\b(\d+|cup|tbsp|tsp|oz|lb|gram|ml)\b/i', $clean)) {
                    $potential_ingredients[] = $clean;
                }
            }
            if (count($potential_ingredients) >= 3) {
                $ingredients = $potential_ingredients;
                break;
            }
        }
    }

    // Extract instructions from ordered lists
    $instructions = [];
    if (preg_match('/instructions?.*?<ol[^>]*>(.*?)<\/ol>/is', $content, $inst_match)) {
        preg_match_all('/<li[^>]*>(.*?)<\/li>/is', $inst_match[1], $inst_items);
        foreach ($inst_items[1] as $index => $item) {
            $clean_item = trim(wp_strip_all_tags($item));
            if (!empty($clean_item)) {
                $instructions[] = [
                    '@type' => 'HowToStep',
                    'position' => $index + 1,
                    'text' => $clean_item
                ];
            }
        }
    }

    // Fallback: use any ordered list
    if (empty($instructions)) {
        preg_match_all('/<ol[^>]*>(.*?)<\/ol>/is', $content, $all_ol);
        foreach ($all_ol[1] as $ol) {
            preg_match_all('/<li[^>]*>(.*?)<\/li>/is', $ol, $ol_items);
            if (count($ol_items[1]) >= 3) {
                foreach ($ol_items[1] as $index => $item) {
                    $clean = trim(wp_strip_all_tags($item));
                    if (!empty($clean)) {
                        $instructions[] = [
                            '@type' => 'HowToStep',
                            'position' => $index + 1,
                            'text' => $clean
                        ];
                    }
                }
                break;
            }
        }
    }

    // Need at least some ingredients or instructions to be useful
    if (empty($ingredients) && empty($instructions)) {
        return;
    }

    // Extract prep and cook time
    $prep_time = null;
    $cook_time = null;

    if (preg_match('/prep(?:aration)?\s*time[:\s]+(\d+)\s*(?:min|minute)/i', $content, $prep_match)) {
        $prep_time = 'PT' . $prep_match[1] . 'M';
    }
    if (preg_match('/cook(?:ing)?\s*time[:\s]+(\d+)\s*(?:min|minute)/i', $content, $cook_match)) {
        $cook_time = 'PT' . $cook_match[1] . 'M';
    }

    // Build the Recipe schema
    $schema = [
        '@context' => 'https://schema.org',
        '@type' => 'Recipe',
        'name' => $title,
        'description' => get_the_excerpt(),
        'author' => [
            '@type' => 'Person',
            'name' => get_the_author()
        ],
        'datePublished' => get_the_date('c')
    ];

    // Add image if featured image exists
    $thumbnail = get_the_post_thumbnail_url($post->ID, 'full');
    if ($thumbnail) {
        $schema['image'] = $thumbnail;
    }

    // Add ingredients if found
    if (!empty($ingredients)) {
        $schema['recipeIngredient'] = $ingredients;
    }

    // Add instructions if found
    if (!empty($instructions)) {
        $schema['recipeInstructions'] = $instructions;
    }

    // Add times if found
    if ($prep_time) {
        $schema['prepTime'] = $prep_time;
    }
    if ($cook_time) {
        $schema['cookTime'] = $cook_time;
    }

    // Build output
    $output = '<script type="application/ld+json">';
    $output .= wp_json_encode($schema, JSON_PRETTY_PRINT);
    $output .= '</script>' . "\n";

    // Cache for 24 hours
    set_transient($cache_key, $output, DAY_IN_SECONDS);

    // Output - phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
    echo $output;
}
add_action('wp_head', 'cirvbo_output_recipe_schema');

/**
 * Output Event Schema (JSON-LD) for event posts (PRO)
 *
 * Detection patterns:
 * - Title contains: event, conference, webinar, workshop, meetup, summit, seminar
 * - Content contains: date/time patterns, location/venue, register/tickets/RSVP
 *
 * Schema type: Event (eligible for Google Rich Results: event listing)
 * Required fields: name, startDate, location
 *
 * @since 1.2.8
 */
function cirvbo_output_event_schema() {
    // Pro-only feature
    if (!cirvbo_is_pro()) {
        return;
    }

    // Check for schema conflicts
    if (cirvbo_has_schema_conflict()) {
        return;
    }

    // Check if enabled
    if (!get_option('cirvbo_enable_event_schema')) {
        return;
    }

    // Only on single posts/pages
    if (!is_singular(array('post', 'page'))) {
        return;
    }

    global $post;
    if (!$post) {
        return;
    }

    // Cache lookup
    $blog_id = get_current_blog_id();
    $cache_key = 'cirvbo_event_schema_' . $blog_id . '_' . $post->ID;
    $cached_schema = get_transient($cache_key);
    if (false !== $cached_schema) {
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
        echo $cached_schema;
        return;
    }

    $title = get_the_title($post->ID);
    $content = $post->post_content;
    $title_lower = strtolower($title);
    $content_lower = strtolower(wp_strip_all_tags($content));

    // Event detection: title keywords
    $event_keywords = array('event', 'conference', 'webinar', 'workshop', 'meetup', 'summit', 'seminar', 'symposium', 'expo', 'hackathon');
    $is_event = false;
    foreach ($event_keywords as $keyword) {
        if (false !== strpos($title_lower, $keyword)) {
            $is_event = true;
            break;
        }
    }

    // Also check content for event signals (need 2+ signals)
    if (!$is_event) {
        $event_signals = 0;
        $signal_patterns = array('register', 'tickets', 'rsvp', 'attend', 'venue', 'agenda', 'keynote', 'speaker');
        foreach ($signal_patterns as $pattern) {
            if (false !== strpos($content_lower, $pattern)) {
                $event_signals++;
            }
        }
        if ($event_signals >= 2) {
            $is_event = true;
        }
    }

    if (!$is_event) {
        return;
    }

    // Extract date from content (ISO 8601 patterns or common date formats)
    $start_date = '';
    // Match patterns like: January 15, 2026 or 2026-01-15 or 01/15/2026
    if (preg_match('/(\d{4}-\d{2}-\d{2})/', $content, $matches)) {
        $start_date = $matches[1] . 'T00:00:00+00:00';
    } elseif (preg_match('/((?:January|February|March|April|May|June|July|August|September|October|November|December)\s+\d{1,2},?\s+\d{4})/i', $content, $matches)) {
        $parsed = strtotime($matches[1]);
        if ($parsed) {
            $start_date = gmdate('Y-m-d', $parsed) . 'T00:00:00+00:00';
        }
    }

    // Fallback to post date if no date found in content
    if (empty($start_date)) {
        $start_date = get_the_date('c', $post->ID);
    }

    // Extract location from content
    $location_name = '';
    if (preg_match('/(?:venue|location|where|place|at)\s*[:=]?\s*([^\n\r.]{3,60})/i', $content_lower, $matches)) {
        $location_name = trim($matches[1]);
    }

    // Check if online event
    $is_online = false;
    $online_keywords = array('online', 'virtual', 'webinar', 'zoom', 'teams', 'google meet', 'remote');
    foreach ($online_keywords as $keyword) {
        if (false !== strpos($content_lower, $keyword)) {
            $is_online = true;
            break;
        }
    }

    // Build schema
    $schema = array(
        '@context' => 'https://schema.org',
        '@type'    => 'Event',
        'name'     => $title,
        'startDate' => $start_date,
        'description' => wp_trim_words(wp_strip_all_tags($content), 50),
        'url'      => get_permalink($post->ID),
    );

    // Add image if available
    if (has_post_thumbnail($post->ID)) {
        $schema['image'] = get_the_post_thumbnail_url($post->ID, 'full');
    }

    // Add location
    if ($is_online) {
        $schema['eventAttendanceMode'] = 'https://schema.org/OnlineEventAttendanceMode';
        $schema['location'] = array(
            '@type' => 'VirtualLocation',
            'url'   => get_permalink($post->ID),
        );
    } elseif (!empty($location_name)) {
        $schema['eventAttendanceMode'] = 'https://schema.org/OfflineEventAttendanceMode';
        $schema['location'] = array(
            '@type' => 'Place',
            'name'  => ucwords($location_name),
        );
    }

    // Add organizer (site info)
    $org_name = get_option('cirvbo_organization_name');
    if (empty($org_name)) {
        $org_name = get_bloginfo('name');
    }
    $schema['organizer'] = array(
        '@type' => 'Organization',
        'name'  => $org_name,
        'url'   => home_url(),
    );

    // Add event status
    $schema['eventStatus'] = 'https://schema.org/EventScheduled';

    // Encode and output
    $json = wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    if (!$json) {
        return;
    }

    $output = "\n" . '<script type="application/ld+json">' . "\n" . $json . "\n" . '</script>' . "\n";

    // Cache for 24 hours
    set_transient($cache_key, $output, DAY_IN_SECONDS);

    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD schema pre-encoded with wp_json_encode
    echo $output;
}
add_action('wp_head', 'cirvbo_output_event_schema');

/**
 * ═══════════════════════════════════════════════════════════
 * GROWTH AUTOMATION - Viral Loop Features
 * ═══════════════════════════════════════════════════════════
 *
 * These features drive organic growth through:
 * 1. Rating reminders (after value is proven)
 * 2. Social sharing (word-of-mouth amplification)
 * 3. Install counter (social proof)
 *
 * GROWTH MATH:
 * If 5% of users rate the plugin, and you have 1,000 installs:
 * = 50 ratings → higher WordPress.org ranking → 2-3x more installs
 */

/**
 * Show "Rate This Plugin" admin notice after 7 days of use
 *
 * WHY 7 DAYS?
 * - User has seen the value (schema working)
 * - Not too early (annoying)
 * - Not too late (forgotten about it)
 * - Industry best practice for SaaS products
 */
function cirvbo_admin_rating_notice() {
    // Only show to admins
    if (!current_user_can('manage_options')) {
        return;
    }

    // Check if user dismissed this notice
    $dismissed = get_option('cirvbo_rating_notice_dismissed');
    if ($dismissed) {
        // If dismissed forever, never show again
        if ($dismissed === 'forever') {
            return;
        }
        // If dismissed with timestamp, check if time has expired
        if (is_numeric($dismissed) && $dismissed > time()) {
            return; // Still within dismissal period
        }
    }

    // Check if plugin has been active for 7 days
    $activation_time = get_option('cirvbo_activation_time');
    if (!$activation_time) {
        // Set activation time if not set (for existing installations)
        add_option('cirvbo_activation_time', time());
        return;
    }

    $days_active = (time() - $activation_time) / DAY_IN_SECONDS;

    // Only show after 7 days
    if ($days_active < 7) {
        return;
    }

    // Show the notice
    ?>
    <div class="notice notice-success is-dismissible cirvbo-rating-notice">
        <p>
            <strong><?php echo esc_html__('Enjoying Cirv Box?', 'cirv-box'); ?></strong><br>
            <?php
            // translators: %d is the number of days the plugin has been active
            printf(esc_html__('You\'ve been using Cirv Box for %d days! If it\'s helping your SEO, could you take 2 minutes to leave a review?', 'cirv-box'), absint(floor($days_active)));
            ?>
        </p>
        <p>
            <a href="https://wordpress.org/support/plugin/cirv-box/reviews/"
               class="button button-primary"
               target="_blank">
                <?php echo esc_html__('Leave a Review', 'cirv-box'); ?>
            </a>
            <a href="<?php echo esc_url(wp_nonce_url(admin_url('options-general.php?page=cirv-box&cirvbo_dismiss_rating_notice=1'), 'cirvbo_dismiss_rating_notice')); ?>" class="button">
                <?php echo esc_html__('Maybe Later', 'cirv-box'); ?>
            </a>
            <a href="<?php echo esc_url(wp_nonce_url(admin_url('options-general.php?page=cirv-box&cirvbo_dismiss_rating_notice=forever'), 'cirvbo_dismiss_rating_notice')); ?>" class="button">
                <?php echo esc_html__('I Already Did!', 'cirv-box'); ?>
            </a>
        </p>
    </div>
    <?php
}
add_action('admin_notices', 'cirvbo_admin_rating_notice');

/**
 * Handle rating notice dismissal
 */
function cirvbo_handle_rating_notice_dismissal() {
    if (isset($_GET['cirvbo_dismiss_rating_notice'])) {
        // Verify nonce for security
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'cirvbo_dismiss_rating_notice')) {
            wp_die(esc_html__('Security check failed. Please try again.', 'cirv-box'));
        }

        // Verify user has permission
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'cirv-box'));
        }

        // Sanitize input
        $dismiss_type = sanitize_text_field(wp_unslash($_GET['cirvbo_dismiss_rating_notice']));

        if ($dismiss_type === 'forever') {
            update_option('cirvbo_rating_notice_dismissed', 'forever');
        } else {
            // Dismiss for 30 days, then ask again
            update_option('cirvbo_rating_notice_dismissed', time() + (30 * DAY_IN_SECONDS));
        }

        // Redirect to remove query param
        wp_safe_redirect(admin_url('options-general.php?page=cirv-box'));
        exit;
    }
}
add_action('admin_init', 'cirvbo_handle_rating_notice_dismissal');

/**
 * AJAX handler for dismissing rating notice
 */
function cirvbo_ajax_dismiss_rating_notice() {
    // Verify nonce for security
    check_ajax_referer('cirvbo-dismiss-rating-notice', 'security');

    // Verify user has permission
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'Permission denied']);
        return;
    }

    update_option('cirvbo_rating_notice_dismissed', time() + (30 * DAY_IN_SECONDS));
    wp_send_json_success();
}
add_action('wp_ajax_cirvbo_dismiss_rating_notice', 'cirvbo_ajax_dismiss_rating_notice');

/**
 * Set activation time when plugin is activated
 */
function cirvbo_set_activation_time() {
    if (!get_option('cirvbo_activation_time')) {
        add_option('cirvbo_activation_time', time());
    }
}
add_action('admin_init', 'cirvbo_set_activation_time');

/**
 * ═══════════════════════════════════════════════════════════
 * CACHE MANAGEMENT - Performance at Scale
 * ═══════════════════════════════════════════════════════════
 *
 * Automatic cache invalidation when posts are updated
 * Manual cache clearing for admins
 *
 * SCALING IMPORTANCE:
 * At 100,000 installs with average 1,000 posts each:
 * = 100M cached schema objects
 * = Automatic cleanup saves server resources
 * = Better performance = better user reviews = more installs!
 */

/**
 * Clear cache when post is updated
 */
function cirvbo_clear_post_cache($post_id) {
    // Delete all transients for this post (all schema types)
    $post = get_post($post_id);
    if (!$post) {
        return;
    }

    $blog_id = get_current_blog_id();

    // Delete specific cache entries using transient API (prevents cache bloat)
    delete_transient('cirvbo_article_schema_' . $blog_id . '_' . $post_id);
    delete_transient('cirvbo_product_schema_' . $blog_id . '_' . $post_id);
    delete_transient('cirvbo_faq_schema_' . $blog_id . '_' . $post_id);
    delete_transient('cirvbo_breadcrumb_schema_' . $blog_id . '_' . $post_id);
    delete_transient('cirvbo_howto_schema_' . $blog_id . '_' . $post_id);
    delete_transient('cirvbo_review_schema_' . $blog_id . '_' . $post_id);
    delete_transient('cirvbo_recipe_schema_' . $blog_id . '_' . $post_id);
    delete_transient('cirvbo_event_schema_' . $blog_id . '_' . $post_id);

    // Organization schema is homepage-only, clear if this is the front page
    if (get_option('page_on_front') == $post_id) {
        delete_transient('cirvbo_organization_schema_' . $blog_id);
    }
}
add_action('save_post', 'cirvbo_clear_post_cache');
add_action('delete_post', 'cirvbo_clear_post_cache');

/**
 * Clear all schema cache (admin tool)
 */
function cirvbo_clear_all_cache() {
    global $wpdb;

    // Build LIKE pattern correctly - wildcards go INSIDE the prepare() call
    $prefix = $wpdb->esc_like('_transient_ssb_');
    $suffix = $wpdb->esc_like('_schema_');
    $pattern = $prefix . '%' . $suffix . '%';

    // Delete all SSB transients (using prepare for security)
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cache cleanup operation requires direct query
    $wpdb->query(
        $wpdb->prepare(
            "DELETE FROM $wpdb->options WHERE option_name LIKE %s",
            $pattern
        )
    );

    $count = $wpdb->rows_affected;
    return $count;
}

/**
 * Handle manual cache clear from admin
 */
function cirvbo_handle_cache_clear() {
    // Check if clear cache button was clicked
    if (isset($_POST['cirvbo_clear_cache']) && check_admin_referer('cirvbo_clear_cache_action', 'cirvbo_clear_cache_nonce')) {
        // Verify user has permission to manage options
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'cirv-box'));
        }

        $cleared = cirvbo_clear_all_cache();

        add_settings_error(
            'cirvbo_messages',
            'cirvbo_cache_cleared',
            // translators: %d is the number of cached schemas that were cleared
            sprintf(esc_html__('%d cached schemas cleared successfully!', 'cirv-box'), absint($cleared)),
            'success'
        );
    }
}
add_action('admin_init', 'cirvbo_handle_cache_clear');

/**
 * Handle debug log export
 *
 * IMPORTANT: This runs on admin_init BEFORE any HTML output
 * This allows us to send headers without "headers already sent" errors
 */
function cirvbo_handle_log_export() {
    // Check if export button was clicked
    if (isset($_POST['cirvbo_export_log']) && check_admin_referer('cirvbo_export_log_action', 'cirvbo_export_log_nonce')) {
        // Verify user has permission
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'cirv-box'));
        }

        // Get debug log
        $log = cirvbo_get_debug_log(1000);
        $filename = 'cirv-box-debug-' . gmdate('Y-m-d-His') . '.json';

        // Send headers (safe because we're in admin_init, before any output)
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        echo json_encode($log, JSON_PRETTY_PRINT);
        exit;
    }
}
add_action('admin_init', 'cirvbo_handle_log_export');

/**
 * Add cache stats to settings page footer
 */
function cirvbo_get_cache_stats() {
    global $wpdb;

    // Build LIKE pattern correctly - wildcards go INSIDE the prepare() call
    $prefix = $wpdb->esc_like('_transient_ssb_');
    $suffix = $wpdb->esc_like('_schema_');
    $pattern = $prefix . '%' . $suffix . '%';

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cache statistics require direct query
    $total_cached = $wpdb->get_var(
        $wpdb->prepare(
            "SELECT COUNT(*) FROM $wpdb->options WHERE option_name LIKE %s",
            $pattern
        )
    );

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cache statistics require direct query
    $cache_size = $wpdb->get_var(
        $wpdb->prepare(
            "SELECT SUM(LENGTH(option_value)) FROM $wpdb->options WHERE option_name LIKE %s",
            $pattern
        )
    );

    return [
        'count' => (int) $total_cached,
        'size' => size_format($cache_size, 2)
    ];
}

/**
 * ═══════════════════════════════════════════════════════════
 * DATABASE CLEANUP - Legacy Options Migration
 * ═══════════════════════════════════════════════════════════
 *
 * Cleanup legacy ssb_* options after 30-day grace period.
 * This function is scheduled via WP-Cron during activation.
 *
 * Note: Pro features (Local Business, Video, Event schemas) are
 * distributed separately via cirvgreen.com, not WordPress.org.
 */
function cirvbo_cleanup_legacy_options() {
    global $wpdb;
    if (!get_option('cirvbo_migration_complete')) {
        return;
    }
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Legacy cleanup requires direct query
    $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'ssb_%'");
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Legacy cleanup requires direct query
    $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_transient_ssb_%'");
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Legacy cleanup requires direct query
    $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_transient_timeout_ssb_%'");
    update_option('cirvbo_legacy_cleanup_complete', time());
}
add_action('cirvbo_cleanup_legacy_options', 'cirvbo_cleanup_legacy_options');


/**
 * Enqueue rating notice script (WordPress.org compliance)
 */
function cirvbo_enqueue_rating_notice_script() {
    if (!is_admin()) {
        return;
    }

    $activation_time = get_option('cirvbo_activation_time');
    $notice_dismissed = get_option('cirvbo_rating_notice_dismissed');

    if (!$activation_time || $notice_dismissed === 'forever') {
        return;
    }

    $days_active = (time() - $activation_time) / DAY_IN_SECONDS;

    if ($days_active < 7 || (is_numeric($notice_dismissed) && time() < $notice_dismissed)) {
        return;
    }

    $script = "
        jQuery(document).ready(function($) {
            $(document).on('click', '.cirvbo-rating-notice .notice-dismiss', function() {
                $.post(ajaxurl, {
                    action: 'cirvbo_dismiss_rating_notice',
                    security: '" . esc_js(wp_create_nonce('cirvbo-dismiss-rating-notice')) . "'
                });
            });
        });
    ";

    wp_add_inline_script('jquery', $script);
}
add_action('admin_enqueue_scripts', 'cirvbo_enqueue_rating_notice_script');

