<?php

if ( ! defined( 'ABSPATH' ) ) exit;
require_once plugin_dir_path(__FILE__) . './Logger.php';
require_once plugin_dir_path(__FILE__) . './helpers.php';
require_once plugin_dir_path(__FILE__) . './ProductDimensionAggregator.php';


class multicourier_CustomShippingService {

    private function apiConfig() {
        $api_url = multicourier_shipping_get_setting('api_url');
        $api_key = multicourier_shipping_get_setting('api_key');

        return [
            'base_url' => $api_url,
            'headers' => [
                'Accept'          => 'application/json',
                'X-RapidAPI-Key'  => $api_key,
            ],
            'timeout' => 10,
        ];
    }

    private function api_get($endpoint) {
        $config = $this->apiConfig();
        $url = rtrim($config['base_url'], '/') . '/' . ltrim($endpoint, '/');
    
        $response = wp_remote_get($url, [
            'headers' => $config['headers'],
            'timeout' => $config['timeout'],
        ]);
    
        if (is_wp_error($response)) {
            multicourier_Logger::log("Error en la conexión con la API: " . $response->get_error_message(), 'error');
            return $response;
        }

        $http_code = wp_remote_retrieve_response_code($response);
        if ($http_code !== 200) {
            multicourier_Logger::log("Error HTTP $http_code : " . wp_remote_retrieve_body($response), 'error');
            return new WP_Error('http_error', "La API respondió con un código no exitoso: $http_code");
        }
    
        $body = wp_remote_retrieve_body($response);
        return json_decode($body, true);
    }

    private function api_post($endpoint, $body = []) {

        $config = $this->apiConfig();
        $url = rtrim($config['base_url'], '/') . '/' . ltrim($endpoint, '/');

        $headers = array_merge($config['headers'], ['Content-Type' => 'application/json']);

        $response = wp_remote_post($url, [
            'headers' => $headers,
            'timeout' => $config['timeout'],
            'body'    => json_encode($body),
        ]);

        if (is_wp_error($response)) {
            multicourier_Logger::log("Error en la conexión con la API: " . $response->get_error_message(), 'error');
            return $response;
        }

        $http_code = wp_remote_retrieve_response_code($response);
        //multicourier_Logger::log("Recibiendo respuesta de la API: ". $http_code . " - " . gettype($http_code));

        if ($http_code !== 200 && $http_code !== 201) {
            multicourier_Logger::log("Error HTTP $http_code : " . wp_remote_retrieve_body($response), 'error');
            return new WP_Error('http_error', "La API respondió con un código no exitoso: $http_code");
        }
    
        $body = wp_remote_retrieve_body($response);
        return json_decode($body, true);
    }

    public function getStates() {
        $cache_key = 'multicourier_shipping_calculator_states';
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            return $cached;
        }

        $data = $this->api_get('/location/state');

        if (is_wp_error($data) || !is_array($data)) {
            return new WP_Error('json_error', 'Error al obtener los estados');
        }

        $results = [];

        foreach ($data as $region) {
            if (isset($region['codeRegion']) && isset($region['name'])) {
                $results[$region['codeRegion']] = $region['name'];
            }
        }

        multicourier_Logger::log("Leyendo estados del API");
        if (!empty($results)) {
            set_transient($cache_key, $results, 15 * DAY_IN_SECONDS);
        }

        return $results;

    }

    public function getDistricts($state) {
        $cache_key = 'multicourier_districts_' . sanitize_key($state);
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            return $cached;
        }


        $data = $this->api_get('/location/district/' . urlencode($state));

        if (is_wp_error($data) || empty($data['locations']) || !is_array($data['locations'])) {
            return new WP_Error('json_error', 'Error al obtener los distritos');
        }

        $results = [];

        foreach ($data["locations"] as $region) {
            if (isset($region['code']) && isset($region['name'])) {
                $results[$region['code']] = $region['name'];
            }
        }

        multicourier_Logger::log("Leyendo distritos del API");
        if (!empty($results)) {
            set_transient($cache_key, $results, 15 * DAY_IN_SECONDS);

            $keys = get_option('multicourier_district_keys', []);
            if (!in_array($state, $keys, true)) {
                $keys[] = $state;

                multicourier_Logger::log("Nueva llave para distritos: " . $state);
                update_option('multicourier_district_keys', $keys);
            }
        }

        return $results;

    }

    public function calculateShippingCost($toState, $toDistrict, $productsInfo) {
        // multicourier_Logger::log("Estado y distrito: " . $toState . ", " . $toDistrict);
        // multicourier_Logger::log("Productos: " . print_r($productsInfo, true));
    
        $aggregator = new multicourier_ProductDimensionAggregator($productsInfo);
        $dimensions = $aggregator->getFinalDimensions();
    
        //multicourier_Logger::log("final dimensiones" . print_r($dimensions, true));

        $body = [
            "from" => [
                "state"    => multicourier_shipping_get_setting('origin_state', 'RM'),
                "district" => multicourier_shipping_get_setting('origin_district', 10)
            ],
            "to" => [
                "state"    => $toState,
                "district" => $toDistrict
            ],
            "datosProducto" => [
                "largo"        => $dimensions['length'],
                "ancho"        => $dimensions['width'],
                "alto"         => $dimensions['height'],
                "pesoFisico"   => $dimensions['weight']
            ]
        ];
    
        return $this->api_post('/pricing', $body);
    }

    
    
    

}