<?php
/**
 * Plugin Name: Chilenvíos
 * Description: Cotiza tus envíos con multiples proveedores en un solo lugar.
 * Version: 1.0.1
 * Author: Chilenvios
 * Author URI: https://chilenvios.cl
 * WC requires at least: 6.0.0
 * License: GPLv2 or later
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once plugin_dir_path(__FILE__) . 'includes/CustomShippingService.php';
require_once plugin_dir_path(__FILE__) . 'includes/AjaxHandlers.php';
require_once plugin_dir_path(__FILE__) . 'includes/Logger.php';

function multicourier_shipping_calculator_activate_plugin() {
    $default_settings = [
        'api_key'          => '',
        'api_url'          => '',
        'origin_state'     => 'RM',
        'origin_district'  => '31',
        'default_length'   => 12,
        'default_width'    => 10,
        'default_height'   => 8,
        'default_weight'   => 1,
        'shipping_extra_fee' => 0,
        'shipping_extra_type' => 'fixed'
    ];

    if (!get_option('multicourier_shipping_calculator_settings')) {
        add_option('multicourier_shipping_calculator_settings', $default_settings);
    }
}
register_activation_hook(__FILE__, 'multicourier_shipping_calculator_activate_plugin');

// Hook que se ejecuta al desactivar el plugin
function multicourier_plugin_desactivar() {
    delete_option('mi_plugin_activado');
    
    delete_transient('multicourier_shipping_calculator_states');

    $keys = get_option('multicourier_district_keys', []);
    foreach ($keys as $estado) {
        delete_transient('multicourier_districts_' . sanitize_key($estado));
    }

    delete_option('multicourier_district_keys');

}

register_deactivation_hook(__FILE__, 'multicourier_plugin_desactivar');

// Hook para inicializar el método de envío en WooCommerce
add_action('woocommerce_shipping_init', function () {
    class multicourier_Custom_Shipping_Method extends WC_Shipping_Method {
        public function __construct() {
            $this->id                 = 'custom_shipping';
            $this->method_title       = __('Custom Shipping - Chile', 'chilenvios');
            $this->method_description = __('Método de envío con tarifas diferenciadas por ciudad en Chile.', 'chilenvios');

            $this->init();
        }

        public function init() {
            $this->init_form_fields();
            $this->init_settings();

            add_action('woocommerce_update_options_shipping_' . $this->id, [$this, 'process_admin_options']);
        }

        public function calculate_shipping($package = []) {
            if (empty($package['destination']['country'])) {
                return;
            }
        
            $country = $package['destination']['country'];
        
            if ($country !== 'CL') {
                return;
            }

            $estado   = $package['destination']['billing_estado'] ?? WC()->session->get('billing_estado', '');
            $distrito = $package['destination']['billing_distrito'] ?? WC()->session->get('billing_distrito', '');
            
            if (empty($estado) || empty($distrito)) {
                return;
            }

            $service = new multicourier_CustomShippingService();

            $products_info = [];

            foreach ($package['contents'] as $item) {
                $product = $item['data'];
                $product_id = $product->get_id();
                $quantity   = $item['quantity'];
                
                $products_info[] = [
                    'product_id' => $product_id,
                    'quantity'   => $quantity,
                    'length'     => $product->get_length(),
                    'width'      => $product->get_width(),
                    'height'     => $product->get_height(),
                    'weight'     => $product->get_weight(),
                ];
            }

            $tarifas_envio = $service->calculateShippingCost($estado, $distrito, $products_info);

            if (empty($tarifas_envio) || !is_array($tarifas_envio)) {
                return;
            }
            
            $settings = get_option('multicourier_shipping_calculator_settings');

            $extra_fee_value = floatval($settings['shipping_extra_fee'] ?? 0);
            $extra_type = $settings['shipping_extra_type'] ?? 'fixed';

            foreach ($tarifas_envio as $tarifa) {
                if (!isset($tarifa['data']['nameService']) || !isset($tarifa['data']['total'])) {
                    continue;
                }
        
                $nameService = $tarifa['data']['nameService'];
                $total = $tarifa['data']['total'];

                $extra_fee = ($extra_type === 'percent')
                    ? ($total * $extra_fee_value) / 100
                    : $extra_fee_value;
        
                $rate = [
                    'id'    => $this->id . ':' . sanitize_title($nameService),
                    'label' => "{$nameService}",
                    'cost'  => $total + $extra_fee,
                    'calc_tax' => 'per_order'
                ];
        
                $this->add_rate($rate);
            }

        }

    }
});

add_filter('woocommerce_shipping_methods', function ($methods) {
    $methods['custom_shipping'] = 'multicourier_Custom_Shipping_Method';
    return $methods;
});

add_filter('woocommerce_checkout_fields', function ($fields) {
    unset($fields['billing']['billing_city']);
    unset($fields['billing']['billing_state']);

    $fields['billing']['billing_estado'] = [
        'type'     => 'select',
        'label'    => __('Región', 'chilenvios'),
        'options'  => [
            '' => __('Cargando regiones...', 'chilenvios'),
        ],
        'required' => true,
        'class'    => ['form-row-wide'],
        'priority' => 41,
    ];

    $fields['billing']['billing_distrito'] = [
        'type'     => 'select',
        'label'    => __('Comuna', 'chilenvios'),
        'options'  => [
            '' => __('Selecciona una región primero', 'chilenvios'),
        ],
        'required' => true,
        'class'    => ['form-row-wide'],
        'priority' => 42,
    ];

    return $fields;
});

add_action('wp_enqueue_scripts', function() {
    wp_enqueue_script('multicourier-custom-shipping', plugins_url('js/multicourier-custom-shipping.js', __FILE__), array('jquery'), '1.0.0', true);
    wp_localize_script('multicourier-custom-shipping', 'customShippingAjax', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('multicourier_nonce')
    ]);
});

add_action('woocommerce_checkout_update_order_meta', function ($order_id) {
    if (!class_exists('multicourier_CustomShippingService')) {
        return;
    }
    
    $estado   = '';
    $distrito = '';
    
    // Intentar obtener de POST primero
    if (isset($_POST['billing_estado'])) {
        $estado = sanitize_text_field($_POST['billing_estado']);
    }
    
    if (isset($_POST['billing_distrito'])) {
        $distrito = sanitize_text_field($_POST['billing_distrito']);
    }
    
    // Si no están en POST, intentar obtenerlos de la sesión
    if (empty($estado) && WC()->session && WC()->session->get('billing_estado')) {
        $estado = WC()->session->get('billing_estado');
    }
    
    if (empty($distrito) && WC()->session && WC()->session->get('billing_distrito')) {
        $distrito = WC()->session->get('billing_distrito');
    }

    // Inicializamos el servicio solo si tenemos al menos un campo con valor
    if (!empty($estado) || !empty($distrito)) {
        $service = new multicourier_CustomShippingService();

        // Procesamos el estado si existe
        if (!empty($estado)) {
            update_post_meta($order_id, '_billing_estado', $estado);

            $states = $service->getStates();
            $estado_nombre = $states[$estado] ?? $estado;
            update_post_meta($order_id, '_billing_estado_nombre', $estado_nombre);
        }

        // Procesamos el distrito si existe
        if (!empty($distrito)) {
            update_post_meta($order_id, '_billing_distrito', $distrito);

            // Solo intentamos obtener el nombre del distrito si tenemos el estado
            if (!empty($estado)) {
                $districts = $service->getDistricts($estado);
                $distrito_nombre = $districts[$distrito] ?? $distrito;
                update_post_meta($order_id, '_billing_distrito_nombre', $distrito_nombre);
            }
        }
    }
});


add_filter('woocommerce_admin_billing_fields', function ($fields) {
    unset($fields['city']);
    unset($fields['state']);
    return $fields;
});

add_filter('woocommerce_cart_shipping_packages', function ($packages) {
    foreach ($packages as &$package) {
        $billing_estado = WC()->session->get('billing_estado');
        $billing_distrito = WC()->session->get('billing_distrito');

        if ($billing_estado) {
            $package['destination']['billing_estado'] = $billing_estado;
        }

        if ($billing_distrito) {
            $package['destination']['billing_distrito'] = $billing_distrito;
        }
    }

    return $packages;
});


add_action('woocommerce_checkout_update_order_review', function ($post_data) {
    parse_str($post_data, $parsed_data);

    if (!empty($parsed_data['billing_estado'])) {
        WC()->session->set('billing_estado', sanitize_text_field($parsed_data['billing_estado']));
    }
    if (!empty($parsed_data['billing_distrito'])) {
        WC()->session->set('billing_distrito', sanitize_text_field($parsed_data['billing_distrito']));
    }
});

add_action('woocommerce_admin_order_data_after_billing_address', function ($order) {
    $estado_nombre   = get_post_meta($order->get_id(), '_billing_estado_nombre', true);
    $distrito_nombre = get_post_meta($order->get_id(), '_billing_distrito_nombre', true);
    
    // Si no tenemos los nombres, intentamos obtener los códigos y convertirlos
    if (empty($estado_nombre) || empty($distrito_nombre)) {
        $estado_code = get_post_meta($order->get_id(), '_billing_estado', true);
        $distrito_code = get_post_meta($order->get_id(), '_billing_distrito', true);
        
        if (!empty($estado_code) || !empty($distrito_code)) {
            $service = new multicourier_CustomShippingService();
            
            if (!empty($estado_code) && empty($estado_nombre)) {
                $states = $service->getStates();
                $estado_nombre = $states[$estado_code] ?? $estado_code;
                update_post_meta($order->get_id(), '_billing_estado_nombre', $estado_nombre);
            }
            
            if (!empty($distrito_code) && empty($distrito_nombre) && !empty($estado_code)) {
                $districts = $service->getDistricts($estado_code);
                $distrito_nombre = $districts[$distrito_code] ?? $distrito_code;
                update_post_meta($order->get_id(), '_billing_distrito_nombre', $distrito_nombre);
            }
        }
    }

    echo '<div class="custom-shipping-details" style="margin-top: 20px;">';
    echo '<h3 style="margin-bottom: 10px;">📦 Detalles de Envío Personalizados</h3>';
    echo '<table cellspacing="0" cellpadding="5">';
    echo '<tr><td><strong>Región:</strong></td><td>' . esc_html($estado_nombre) . '</td></tr>';
    echo '<tr><td><strong>Comuna:</strong></td><td>' . esc_html($distrito_nombre) . '</td></tr>';
    echo '</table>';
    echo '</div>';
});

add_action('admin_enqueue_scripts', function ($hook) {
    if ($hook !== 'toplevel_page_chilenvios-settings') return;

    wp_enqueue_script('multicourier-admin-js', plugin_dir_url(__FILE__) . 'js/admin-settings.js', ['jquery'], null, true);

    wp_localize_script('multicourier-admin-js', 'mcAdminAjax', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('multicourier_nonce'),
        'selected_state' => get_option('multicourier_shipping_calculator_settings')['origin_state'] ?? '',
        'selected_district' => get_option('multicourier_shipping_calculator_settings')['origin_district'] ?? ''
    ]);
});

add_action('admin_menu', function () {
    add_menu_page(
        'Chilenvios Settings',
        'Chilenvios',
        'manage_options',
        'chilenvios-settings',
        'multicourier_render_admin_page',
        'dashicons-admin-generic'
    );
});

function multicourier_render_admin_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    if (isset($_POST['multicourier_save_settings'])) {
        check_admin_referer('multicourier_save_settings_nonce');

        $settings = [
            'api_key' => sanitize_text_field($_POST['api_key'] ?? ''),
            'api_url' => esc_url_raw($_POST['api_url'] ?? ''),
            'origin_state' => sanitize_text_field($_POST['origin_state'] ?? ''),
            'origin_district' => sanitize_text_field($_POST['origin_district'] ?? ''),
            'default_length' => floatval($_POST['default_length'] ?? 0),
            'default_width' => floatval($_POST['default_width'] ?? 0),
            'default_height' => floatval($_POST['default_height'] ?? 0),
            'default_weight' => floatval($_POST['default_weight'] ?? 0),
            'shipping_extra_fee' => floatval($_POST['shipping_extra_fee'] ?? 0),
            'shipping_extra_type' => sanitize_text_field($_POST['shipping_extra_type'] ?? ''),

        ];

        update_option('multicourier_shipping_calculator_settings', $settings);

        echo '<div class="updated"><p>Configuración guardada</p></div>';
    }

    $settings = get_option('multicourier_shipping_calculator_settings', []);

    $estados = function_exists('get_available_states') ? get_available_states() : ['RM' => 'Región Metropolitana'];
    $distritos = function_exists('get_available_districts') ? get_available_districts($settings['origin_state'] ?? '') : ['1' => 'Santiago'];
    ?>

    <div class="wrap">
        <h1>Configuración del cotizador Chilenvios</h1>
        <p>Para más información, visita <a href="https://chilenvios.cl" target="_blank">https://chilenvios.cl</a></p>
        <form method="POST">
            <?php wp_nonce_field('multicourier_save_settings_nonce'); ?>

            <table class="form-table">
            <tr>
                    <th colspan="2" style="padding-top: 0px;">
                        <h2>Configuración inicial</h2>
                        <p style="color: grey;">Configure su clave, paso indispensable para empezar a usar el tarifario.<br>Una vez configurado la clave, guarde y vuelva a la pantalla de configuración para configurar los valores por defecto.</p>
                    </th>
                </tr>
                <tr>
                    <th scope="row"><label for="api_key">API Key</label></th>
                    <td><input name="api_key" type="text" id="api_key" placeholder="Clave obtenida desde su cuenta de RapidAPI" value="<?php echo esc_attr($settings['api_key'] ?? ''); ?>" class="regular-text"><p class="description">Clave obtenida desde su cuenta de RapidAPI.</p></td>
                </tr>
                <tr>
                    <th scope="row"><label for="api_url">API URL</label></th>       
                    <td><input name="api_url" type="text" id="api_url" value="<?php echo esc_attr($settings['api_url'] ?? ''); ?>" class="regular-text" placeholder="https://multicourier.p.rapidapi.com"><p class="description">En caso de tener una API URL, colocarla. De lo contrario, dejarla vacío.</p></td>
                </tr>
                <tr>
                    <th colspan="2">
                        <h2>Valores por defecto</h2>
                        <p style="color: grey;">Configure el tamaño minimo de su paquete, así como también el origen físico desde donde envía sus pedidos.</p>
                        <p style="color: grey;">Nota: Para obtener mejores resultados, recuerde configurar las medidas y peso de cada producto.</p>
                    </th>
                </tr>
                <tr>
                    <th scope="row"><label for="origin_state">Región de Origen</label></th>
                    <td>
                        <select name="origin_state" id="origin_state" class="regular-text">
                            <option value="">Cargando regiones...</option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="origin_district">Comuna de Origen</label></th>
                    <td>
                        <select name="origin_district" id="origin_district" class="regular-text">
                            <option value="">Selecciona una región primero</option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="default_length">Largo por defecto (cm)</label></th>
                    <td><input name="default_length" type="number" step="0.01" id="default_length" value="<?php echo esc_attr($settings['default_length'] ?? ''); ?>" class="small-text"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="default_width">Ancho por defecto (cm)</label></th>
                    <td><input name="default_width" type="number" step="0.01" id="default_width" value="<?php echo esc_attr($settings['default_width'] ?? ''); ?>" class="small-text"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="default_height">Alto por defecto (cm)</label></th>
                    <td><input name="default_height" type="number" step="0.01" id="default_height" value="<?php echo esc_attr($settings['default_height'] ?? ''); ?>" class="small-text"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="default_weight">Peso por defecto (kg)</label></th>
                    <td><input name="default_weight" type="number" step="0.01" id="default_weight" value="<?php echo esc_attr($settings['default_weight'] ?? ''); ?>" class="small-text"></td>
                </tr>
                
                <tr>
                    <th scope="row"><label for="shipping_extra_fee">Incremento extra al envío</label></th>
                    <td>
                        <input name="shipping_extra_fee" type="number" step="0.01" id="shipping_extra_fee"
                            value="<?php echo esc_attr($settings['shipping_extra_fee'] ?? 0); ?>" class="small-text" style="margin-right: 10px;">

                        <select name="shipping_extra_type" id="shipping_extra_type">
                            <option value="fixed" <?php selected($settings['shipping_extra_type'] ?? '', 'fixed'); ?>>Monto fijo</option>
                            <option value="percent" <?php selected($settings['shipping_extra_type'] ?? '', 'percent'); ?>>Porcentaje</option>
                        </select>

                        <p class="description">Aumente el valor de la tarifa entregado al cliente en un monto fijo o porcentaje.</p>
                    </td>
                </tr>


            </table>

            <?php submit_button('Guardar configuración', 'primary', 'multicourier_save_settings'); ?>
        </form>
    </div>
<?php }
