<?php
/**
 * Cheqpay Logger Class
 *
 * Handles all logging for the Cheqpay Payment Gateway plugin.
 * Logs are stored in WooCommerce logs directory (wp-content/wc-logs/).
 *
 * @package Cheqpay_Payment_Gateway
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Cheqpay_Logger
 */
class Cheqpay_Logger {

    /**
     * Log source identifier
     */
    const LOG_SOURCE = 'cheqpay-payment-gateway';

    /**
     * WooCommerce logger instance
     *
     * @var WC_Logger|null
     */
    private static $logger = null;

    /**
     * Get the WooCommerce logger instance
     *
     * @return WC_Logger|null
     */
    private static function get_logger() {
        // Ensure WooCommerce is loaded before getting logger
        if (!function_exists('wc_get_logger')) {
            return null;
        }

        if (is_null(self::$logger)) {
            self::$logger = wc_get_logger();
        }
        return self::$logger;
    }

    /**
     * Check if debug mode is enabled
     *
     * @return bool
     */
    public static function is_debug_enabled() {
        return defined('CHEQPAY_DEBUG_MODE') && CHEQPAY_DEBUG_MODE;
    }

    /**
     * Format log message with order context
     *
     * @param string $message Log message
     * @param int|null $order_id Order ID for context
     * @param array $context Additional context data
     * @return string Formatted message
     */
    private static function format_message($message, $order_id = null, $context = array()) {
        $formatted = '';

        // Add order ID prefix if provided
        if ($order_id) {
            $formatted .= "[Order #{$order_id}] ";
        }

        $formatted .= $message;

        // Add context data if provided
        if (!empty($context)) {
            // Sanitize context to avoid logging sensitive data
            $safe_context = self::sanitize_context($context);
            $json = function_exists('wp_json_encode') ? wp_json_encode($safe_context) : json_encode($safe_context);
            $formatted .= ' | Context: ' . $json;
        }

        return $formatted;
    }

    /**
     * Sanitize context data to remove sensitive information
     *
     * @param array $context Context data
     * @return array Sanitized context
     */
    private static function sanitize_context($context) {
        $sensitive_keys = array(
            'card_number',
            'card_cvc',
            'cvc',
            'cvv',
            'secret_key',
            'api_key',
            'password',
            'token',
            'session_token',
            'x-api-key',
            'x-session-token',
        );

        $sanitized = array();

        foreach ($context as $key => $value) {
            $lower_key = strtolower($key);

            // Check if key contains sensitive data
            $is_sensitive = false;
            foreach ($sensitive_keys as $sensitive) {
                if (strpos($lower_key, $sensitive) !== false) {
                    $is_sensitive = true;
                    break;
                }
            }

            if ($is_sensitive) {
                $sanitized[$key] = '***REDACTED***';
            } elseif (is_array($value)) {
                $sanitized[$key] = self::sanitize_context($value);
            } else {
                $sanitized[$key] = $value;
            }
        }

        return $sanitized;
    }

    /**
     * Log a debug message (only when debug mode is enabled)
     *
     * @param string $message Log message
     * @param int|null $order_id Order ID for context
     * @param array $context Additional context data
     */
    public static function debug($message, $order_id = null, $context = array()) {
        if (!self::is_debug_enabled()) {
            return;
        }

        $logger = self::get_logger();
        if (!$logger) {
            return;
        }

        $logger->debug(
            self::format_message($message, $order_id, $context),
            array('source' => self::LOG_SOURCE)
        );
    }

    /**
     * Log an info message
     *
     * @param string $message Log message
     * @param int|null $order_id Order ID for context
     * @param array $context Additional context data
     */
    public static function info($message, $order_id = null, $context = array()) {
        $logger = self::get_logger();
        if (!$logger) {
            return;
        }

        $logger->info(
            self::format_message($message, $order_id, $context),
            array('source' => self::LOG_SOURCE)
        );
    }

    /**
     * Log a warning message
     *
     * @param string $message Log message
     * @param int|null $order_id Order ID for context
     * @param array $context Additional context data
     */
    public static function warning($message, $order_id = null, $context = array()) {
        $logger = self::get_logger();
        if (!$logger) {
            return;
        }

        $logger->warning(
            self::format_message($message, $order_id, $context),
            array('source' => self::LOG_SOURCE)
        );
    }

    /**
     * Log an error message
     *
     * @param string $message Log message
     * @param int|null $order_id Order ID for context
     * @param array $context Additional context data
     */
    public static function error($message, $order_id = null, $context = array()) {
        $logger = self::get_logger();
        if (!$logger) {
            return;
        }

        $logger->error(
            self::format_message($message, $order_id, $context),
            array('source' => self::LOG_SOURCE)
        );
    }

    /**
     * Log an API request
     *
     * @param string $endpoint API endpoint
     * @param string $method HTTP method
     * @param array $request_data Request data (will be sanitized)
     * @param int|null $order_id Order ID for context
     */
    public static function log_api_request($endpoint, $method, $request_data = array(), $order_id = null) {
        if (!self::is_debug_enabled()) {
            return;
        }

        self::debug(
            "API Request: {$method} {$endpoint}",
            $order_id,
            array('request_body' => $request_data)
        );
    }

    /**
     * Log an API response
     *
     * @param string $endpoint API endpoint
     * @param int $response_code HTTP response code
     * @param array|string $response_body Response body
     * @param int|null $order_id Order ID for context
     */
    public static function log_api_response($endpoint, $response_code, $response_body = array(), $order_id = null) {
        if (!self::is_debug_enabled()) {
            return;
        }

        $level = ($response_code >= 200 && $response_code < 300) ? 'debug' : 'warning';

        $message = "API Response: {$endpoint} - HTTP {$response_code}";
        $context = array('response_body' => $response_body);

        if ($level === 'warning') {
            self::warning($message, $order_id, $context);
        } else {
            self::debug($message, $order_id, $context);
        }
    }

    /**
     * Log a webhook event
     *
     * @param string $event_type Webhook event type
     * @param array $payload Webhook payload (will be sanitized)
     * @param int|null $order_id Order ID for context
     */
    public static function log_webhook($event_type, $payload = array(), $order_id = null) {
        self::info(
            "Webhook received: {$event_type}",
            $order_id,
            array('payload' => $payload)
        );
    }

    /**
     * Log payment status change
     *
     * @param int $order_id Order ID
     * @param string $old_status Old payment status
     * @param string $new_status New payment status
     * @param string $source Source of the change (redirect/webhook/api)
     */
    public static function log_payment_status_change($order_id, $old_status, $new_status, $source = 'unknown') {
        self::info(
            "Payment status changed: {$old_status} -> {$new_status} (via {$source})",
            $order_id
        );
    }

    /**
     * Log an exception
     *
     * @param Exception $exception The exception
     * @param string $context_message Additional context message
     * @param int|null $order_id Order ID for context
     */
    public static function log_exception($exception, $context_message = '', $order_id = null) {
        $message = $context_message ? "{$context_message}: " : '';
        $message .= $exception->getMessage();

        self::error(
            $message,
            $order_id,
            array(
                'exception_class' => get_class($exception),
                'file' => $exception->getFile(),
                'line' => $exception->getLine(),
            )
        );
    }
}
