<?php
if (!defined('ABSPATH')) {
    exit;
}

class WC_Gateway_Cheqpay_Link extends WC_Payment_Gateway {

    public function __construct() {
        $this->id = 'cheqpay_link';
        $this->icon = '';
        $this->method_title = __('LABEL_CHEQPAY', 'cheqpay-payment-gateway');
        $this->method_description = __('DESCRIPTION_PAYMENT_METHOD', 'cheqpay-payment-gateway');

        $this->supports = array('products', 'refunds');

        $this->init_form_fields();
        $this->init_settings();

        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->enabled = $this->get_option('enabled');

        // Set has_fields based on payment method
        $payment_method = $this->get_option('payment_method', 'payment_link');
        $this->has_fields = ($payment_method === 'card_payment');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action('woocommerce_api_cheqpay_device_data', array($this, 'handle_device_data_collection'));
        add_action('woocommerce_api_cheqpay_challenge_return', array($this, 'handle_challenge_return'));
        add_action('woocommerce_api_cheqpay_challenge_page', array($this, 'handle_challenge_page'));
        add_action('wp_enqueue_scripts', array($this, 'payment_scripts'));
        add_action('wp', array($this, 'display_cheqpay_error'));
    }

    public function display_cheqpay_error() {
        if (!is_checkout()) {
            return;
        }

        if (isset($_GET['cheqpay_error']) && !empty($_GET['cheqpay_error'])) {
            $error_message = sanitize_text_field(wp_unslash($_GET['cheqpay_error']));
            wc_add_notice($error_message, 'error');
        }
    }

    public function payment_scripts() {
        if (!is_cart() && !is_checkout() && !isset($_GET['pay_for_order'])) {
            return;
        }

        if ($this->enabled === 'no') {
            return;
        }

        // Only load scripts if payment method is card_payment
        $payment_method = $this->get_option('payment_method', 'payment_link');
        if ($payment_method !== 'card_payment') {
            return;
        }

        $environment = $this->get_option('environment', 'sandbox');

        // VGS configuration
        $vgs_vault_id = ($environment === 'production') ? CHEQPAY_VGS_VAULT_ID_PRODUCTION : CHEQPAY_VGS_VAULT_ID_SANDBOX;
        $vgs_environment = ($environment === 'production') ? 'live' : 'sandbox';

        // Always enqueue VGS Collect.js for both classic and blocks checkout
        wp_enqueue_script(
            'vgs-collect-js',
            'https://js.verygoodvault.com/vgs-collect/2.18.0/vgs-collect.js',
            array(),
            '2.18.0',
            true
        );

        // Always enqueue classic checkout script — it self-guards at runtime
        // by checking for jQuery elements and cheqpayParams before initializing.
        wp_enqueue_script(
            'cheqpay_classic_js',
            CHEQPAY_LINK_PLUGIN_URL . 'assets/js/cheqpay-classic.js',
            array('jquery', 'vgs-collect-js'),
            CHEQPAY_LINK_VERSION,
            true
        );

        // Iframe color settings from admin
        $iframe_options = array(
            'colorBackground' => $this->get_option('color_background', '#ffffff'),
            'colorText' => $this->get_option('color_text', '#333333'),
            'colorBorder' => $this->get_option('color_border', '#e0e0e0'),
            'colorBorderSelected' => $this->get_option('color_border_selected', '#0F172A'),
            'colorPlaceholder' => $this->get_option('color_placeholder', '#64748B'),
            'radiusBorder' => $this->get_option('radius_border', '8px'),
        );

        // Pass configuration to JS
        wp_localize_script('cheqpay_classic_js', 'cheqpayParams', array(
            'environment' => $environment,
            'apiUrl' => $this->get_api_url(),
            'paymentMethod' => $payment_method,
            'vgsVaultId' => $vgs_vault_id,
            'vgsEnvironment' => $vgs_environment,
            'vgsCname' => CHEQPAY_VGS_CNAME,
            'iframe' => $iframe_options,
            'i18n' => array(
                'ERROR_CARD_NUMBER_INVALID' => __('ERROR_CARD_NUMBER_INVALID', 'cheqpay-payment-gateway'),
                'ERROR_CARD_EXPIRY_INVALID' => __('ERROR_CARD_EXPIRY_INVALID', 'cheqpay-payment-gateway'),
                'ERROR_CARD_CVC_INVALID' => __('ERROR_CARD_CVC_INVALID', 'cheqpay-payment-gateway'),
                'PAYMENT_FAILED_FORM_NOT_INITIALIZED' => __('PAYMENT_FAILED_FORM_NOT_INITIALIZED', 'cheqpay-payment-gateway'),
                'PAYMENT_FAILED_FORM_LOADING' => __('PAYMENT_FAILED_FORM_LOADING', 'cheqpay-payment-gateway'),
                'PAYMENT_FAILED_CARD_PROCESSING' => __('PAYMENT_FAILED_CARD_PROCESSING', 'cheqpay-payment-gateway'),
                'PAYMENT_FAILED_CARD_DETAILS' => __('PAYMENT_FAILED_CARD_DETAILS', 'cheqpay-payment-gateway'),
                'PLACEHOLDER_CARD_NUMBER' => __('PLACEHOLDER_CARD_NUMBER', 'cheqpay-payment-gateway'),
                'PLACEHOLDER_EXPIRY_DATE' => __('PLACEHOLDER_EXPIRY_DATE', 'cheqpay-payment-gateway'),
                'PLACEHOLDER_CVC' => __('PLACEHOLDER_CVC', 'cheqpay-payment-gateway'),
            ),
        ));
    }

    public function handle_challenge_return() {
        $order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;

        Cheqpay_Logger::debug('3DS challenge return received', $order_id);

        // Basic validation to ensure we have a valid context
            if (!$order_id || !wc_get_order($order_id)) {
            Cheqpay_Logger::error('3DS challenge return failed - Invalid order', $order_id);
            wp_die(esc_html__('ERROR_INVALID_ORDER', 'cheqpay-payment-gateway'), 'cheqpay-payment-gateway', array('response' => 400));
        }

        ?>
        <!DOCTYPE html>
        <html>
        <body>
            <script type="text/javascript">
                // Notify parent window that challenge is complete
                window.parent.postMessage("challenge_complete", "*");
            </script>
        </body>
        </html>
        <?php
        exit;
    }

    public function handle_challenge_page() {
        $order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;

        Cheqpay_Logger::debug('3DS challenge page loaded', $order_id);

        if (!$order_id) {
            Cheqpay_Logger::error('3DS challenge page failed - Invalid order ID');
            wp_die(esc_html__('ERROR_INVALID_ORDER_ID', 'cheqpay-payment-gateway'));
        }

        $jwt = get_transient('cheqpay_challenge_jwt_' . $order_id);
        $url = get_transient('cheqpay_challenge_url_' . $order_id);

        if (!$jwt || !$url) {
            Cheqpay_Logger::error('3DS challenge page failed - Session expired', $order_id);
            wp_die(esc_html__('ERROR_CHALLENGE_SESSION_EXPIRED', 'cheqpay-payment-gateway'));
        }

        // Cleanup
        delete_transient('cheqpay_challenge_jwt_' . $order_id);
        delete_transient('cheqpay_challenge_url_' . $order_id);

        $ajax_url = admin_url('admin-ajax.php');
        $nonce = wp_create_nonce('cheqpay_validate_' . $order_id);
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title><?php echo esc_html(__('TITLE_3DS_AUTHENTICATION', 'cheqpay-payment-gateway')); ?></title>
            <?php wp_print_scripts('jquery-core'); ?>
            <style>
                html, body {
                    margin: 0;
                    padding: 0;
                    width: 100%;
                    height: 100%;
                    overflow: hidden;
                    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
                }
                #step-up-iframe {
                    width: 100%;
                    height: 100%;
                    border: none;
                    position: absolute;
                    top: 0;
                    left: 0;
                }
                #challenge-container {
                    width: 100%;
                    height: 100%;
                }
                #loader-overlay {
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(255, 255, 255, 0.95);
                    display: flex;
                    flex-direction: column;
                    justify-content: center;
                    align-items: center;
                    z-index: 1000;
                    display: none; /* Hidden by default */
                    text-align: center;
                }
                .loader {
                    border: 4px solid #f3f3f3;
                    border-top: 4px solid #3498db;
                    border-radius: 50%;
                    width: 40px;
                    height: 40px;
                    animation: spin 1s linear infinite;
                    margin-bottom: 20px;
                }
                p {
                    margin: 5px 0;
                    color: #333;
                    font-size: 16px;
                }
                .sub-text {
                    font-size: 14px;
                    color: #666;
                }
                @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
            </style>
        </head>
        <body>
            <div id="loader-overlay">
                <div class="loader"></div>
                <p><strong><?php echo esc_html(__('MESSAGE_FINALIZING_PAYMENT', 'cheqpay-payment-gateway')); ?></strong></p>
                <p class="sub-text"><?php echo esc_html(__('MESSAGE_DO_NOT_CLOSE_WINDOW', 'cheqpay-payment-gateway')); ?></p>
            </div>

            <div id="challenge-container">
                <!-- Visible iframe for the 3DS challenge -->
                <iframe
                    id="step-up-iframe"
                    name="step-up-iframe">
                </iframe>
            </div>

            <!-- Form to submit JWT to Cardinal Commerce (not hidden - following Cybersource docs) -->
            <form
                id="step-up-form"
                method="POST"
                target="step-up-iframe">
                <input type="hidden" name="JWT" value="<?php echo esc_attr($jwt); ?>" />
            </form>

            <script>
            // Set the action URL from the API response
            document.getElementById('step-up-form').action = '<?php echo esc_url_raw($url); ?>';

            // Submit the form on page load to display the challenge
            window.onload = function() {
                var stepUpForm = document.querySelector('#step-up-form');
                if (stepUpForm) {
                    stepUpForm.submit();
                }
            };

            // Listen for completion message from redirection page
            window.addEventListener("message", function(event) {
                if (event.data === "challenge_complete") {
                    console.log("3DS challenge completed successfully");

                    // Hide the challenge iframe and show loader
                    var container = document.getElementById('challenge-container');
                    var loader = document.getElementById('loader-overlay');

                    if (container) container.style.display = 'none';
                    if (loader) loader.style.display = 'flex';

                    // Next step: Call POST /payments/{id}/payer-authentication/validate
                    var payload = {
                        action: 'cheqpay_validate_auth',
                        nonce: '<?php echo esc_js($nonce); ?>',
                        order_id: <?php echo absint($order_id); ?>
                    };

                    jQuery.post('<?php echo esc_url_raw($ajax_url); ?>', payload, function(response) {
                        if (response.success) {
                            window.location.href = response.data.redirect;
                        } else {
                            var errorMsg = '<?php echo esc_js(__('PAYMENT_FAILED_VALIDATION', 'cheqpay-payment-gateway')); ?>';
                            window.location.href = '<?php echo esc_url(wc_get_checkout_url()); ?>?cheqpay_error=' + encodeURIComponent(errorMsg);
                        }
                    }).fail(function() {
                        window.location.href = '<?php echo esc_url(wc_get_checkout_url()); ?>?cheqpay_error=' + encodeURIComponent('<?php echo esc_js(__('PAYMENT_FAILED_COMMUNICATION', 'cheqpay-payment-gateway')); ?>');
                    });
                }
            }, false);
            </script>
        </body>
        </html>
        <?php
        exit;
    }

    public function handle_device_data_collection() {
        $order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;

        Cheqpay_Logger::debug('Device data collection page loaded', $order_id);

        if (!$order_id) {
            Cheqpay_Logger::error('Device data collection failed - Invalid order ID');
            wp_die(esc_html__('ERROR_INVALID_ORDER_ID', 'cheqpay-payment-gateway'));
        }

        $jwt = get_transient('cheqpay_ddc_jwt_' . $order_id);
        $url = get_transient('cheqpay_ddc_url_' . $order_id);
        $payment_id = get_transient('cheqpay_ddc_payment_id_' . $order_id);

        if (!$jwt || !$url || !$payment_id) {
            Cheqpay_Logger::error('Device data collection failed - Session expired', $order_id);
            wp_die(esc_html__('ERROR_SESSION_EXPIRED', 'cheqpay-payment-gateway'));
        }

        // Clean up transients
        delete_transient('cheqpay_ddc_jwt_' . $order_id);
        delete_transient('cheqpay_ddc_url_' . $order_id);
        // Don't delete payment_id yet, needed for next step

        $ajax_url = admin_url('admin-ajax.php');
        $nonce = wp_create_nonce('cheqpay_auth_req_' . $order_id);

        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title><?php echo esc_html(__('MESSAGE_PROCESSING_PAYMENT', 'cheqpay-payment-gateway')); ?></title>
            <?php wp_print_scripts('jquery-core'); ?>
            <style>
                body {
                    margin: 0;
                    padding: 0;
                    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
                }
                #loader-overlay {
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: #fff;
                    display: flex;
                    flex-direction: column;
                    justify-content: center;
                    align-items: center;
                    z-index: 1000;
                    text-align: center;
                }
                .loader {
                    border: 4px solid #f3f3f3;
                    border-top: 4px solid #3498db;
                    border-radius: 50%;
                    width: 40px;
                    height: 40px;
                    animation: spin 1s linear infinite;
                    margin-bottom: 20px;
                }
                p {
                    margin: 5px 0;
                    color: #333;
                    font-size: 16px;
                }
                .sub-text {
                    font-size: 14px;
                    color: #666;
                }
                @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
            </style>
        </head>
        <body>
            <div id="loader-overlay">
                <div class="loader"></div>
                <p><strong><?php echo esc_html(__('MESSAGE_VERIFYING_PAYMENT', 'cheqpay-payment-gateway')); ?></strong></p>
                <p class="sub-text"><?php echo esc_html(__('MESSAGE_DO_NOT_CLOSE_WINDOW', 'cheqpay-payment-gateway')); ?></p>
            </div>

            <iframe id="pa-iframe" name="pa-iframe" width="0" height="0" style="display:none"></iframe>

            <form id="pa-form" method="POST" target="pa-iframe">
                <input type="hidden" name="JWT" value="<?php echo esc_attr($jwt); ?>" />
            </form>

            <script>
                function getBrowserName(userAgent) {
                    if (userAgent.indexOf("Firefox") > -1) return "Firefox";
                    if (userAgent.indexOf("SamsungBrowser") > -1) return "Samsung Browser";
                    if (userAgent.indexOf("Opera") > -1 || userAgent.indexOf("OPR") > -1) return "Opera";
                    if (userAgent.indexOf("Trident") > -1) return "Internet Explorer";
                    if (userAgent.indexOf("Edge") > -1) return "Edge";
                    if (userAgent.indexOf("Chrome") > -1) return "Chrome";
                    if (userAgent.indexOf("Safari") > -1) return "Safari";
                    return "Unknown";
                }

                function getBrowserLanguage() {
                    return (navigator.languages && navigator.languages.length) ? navigator.languages[0] :
                           navigator.language || navigator.userLanguage || navigator.browserLanguage || 'en-US';
                }

                jQuery(document).ready(function($) {
                    // Listen for completion callback from Cardinal Commerce
                    window.addEventListener("message", function(event) {
                        // Verify the message is from Cardinal Commerce
                        if (
                            event.origin === "https://centinelapistag.cardinalcommerce.com" ||
                            event.origin === "https://centinelapi.cardinalcommerce.com"
                        ) {
                            console.log("Device data collection completed.");

                            var sessionId = '';
                            try {
                                var data = JSON.parse(event.data);
                                sessionId = data.SessionId || '';
                            } catch (e) {
                                console.error("Error parsing event data", e);
                            }

                            if (!sessionId) {
                                console.warn("No SessionId found in device data collection response");
                                return;
                            }

                            console.log("Proceeding to API call with SessionId:", sessionId);

                            var userAgent = navigator.userAgent;
                            var screenWidth = window.screen.width;
                            var screenHeight = window.screen.height;

                            var payload = {
                                action: 'cheqpay_authentication_request',
                                nonce: '<?php echo esc_js($nonce); ?>',
                                order_id: <?php echo absint($order_id); ?>,
                                collectionReferenceId: sessionId,
                                userAgent: userAgent,
                                userAgentBrowserValue: getBrowserName(userAgent),
                                httpBrowserLanguage: getBrowserLanguage(),
                                httpBrowserScreenWidth: screenWidth,
                                httpBrowserScreenHeight: screenHeight
                            };

                            $.post('<?php echo esc_url_raw($ajax_url); ?>', payload, function(response) {
                                if (response.success) {
                                    window.location.href = response.data.redirect;
                                } else {
                                    var errorMsg = '<?php echo esc_js(__('PAYMENT_FAILED_GENERIC', 'cheqpay-payment-gateway')); ?>';
                                    window.location.href = '<?php echo esc_url(wc_get_checkout_url()); ?>?cheqpay_error=' + encodeURIComponent(errorMsg);
                                }
                            }).fail(function() {
                                window.location.href = '<?php echo esc_url(wc_get_checkout_url()); ?>?cheqpay_error=' + encodeURIComponent('<?php echo esc_js(__('PAYMENT_FAILED_COMMUNICATION', 'cheqpay-payment-gateway')); ?>');
                            });
                        }
                    }, false);

                    // Submit the form
                    var form = document.getElementById('pa-form');
                    form.action = '<?php echo esc_url_raw($url); ?>';
                    form.submit();
                });
            </script>
        </body>
        </html>
        <?php
        exit;
    }

    public function ajax_handle_authentication_request() {
        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;

        Cheqpay_Logger::info('3DS authentication request started', $order_id, array(
            'has_nonce' => isset($_POST['nonce']),
        ));

        // Verify nonce - log before checking
        $nonce_valid = isset($_POST['nonce']) && wp_verify_nonce($_POST['nonce'], 'cheqpay_auth_req_' . $order_id);
        if (!$nonce_valid) {
            Cheqpay_Logger::error('3DS authentication failed - Nonce verification failed', $order_id);
            wp_send_json_error(array('message' => 'Security check failed'), 403);
            return;
        }

        $order = wc_get_order($order_id);

        Cheqpay_Logger::debug('3DS authentication nonce verified', $order_id);

        if (!$order) {
            Cheqpay_Logger::error('3DS authentication failed - Invalid order', $order_id);
            wp_send_json_error(array('message' => 'Invalid Order'));
        }

        $session_token = get_transient('cheqpay_ddc_session_token_' . $order_id);
        $payment_id = get_transient('cheqpay_ddc_payment_id_' . $order_id);

        if (!$payment_id) {
            Cheqpay_Logger::error('3DS authentication failed - Payment session expired', $order_id);
            wp_send_json_error(array('message' => 'Payment session expired'));
        }

        // Clean up
        delete_transient('cheqpay_ddc_payment_id_' . $order_id);
        // delete_transient('cheqpay_ddc_session_token_' . $order_id);

        // Prepare Body
        $ip_address = WC_Geolocation::get_ip_address();
        // Fallback for localhost
        if ($ip_address === '::1' || empty($ip_address)) {
            $ip_address = '127.0.0.1';
        }

        $device_information = array(
            'ipAddress' => $ip_address,
            'userAgent' => sanitize_text_field($_POST['userAgent']),
            'userAgentBrowserValue' => sanitize_text_field($_POST['userAgentBrowserValue']),
            'httpAcceptBrowserValue' => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.7',
            'httpBrowserLanguage' => sanitize_text_field($_POST['httpBrowserLanguage']),
            'httpBrowserJavaScriptEnabled' => true,
            'httpBrowserScreenWidth' => strval($_POST['httpBrowserScreenWidth']),
            'httpBrowserScreenHeight' => strval($_POST['httpBrowserScreenHeight'])
        );

        $return_url = add_query_arg(
            'order_id',
            $order_id,
            WC()->api_request_url('cheqpay_challenge_return')
        );

        $body = array(
            'collectionReferenceId' => sanitize_text_field($_POST['collectionReferenceId']),
            'returnUrl' => "$return_url",
            // 'returnUrl' => "https://637d5a58ff65.ngrok-free.app/wc-api/cheqpay_challenge_return/?order_id=$order_id", // @todo use me on dev
            'deviceInformation' => $device_information
        );

         // Make API Call to submit device data

        try {
            $api_url = $this->get_api_url() . '/lps/payments/' . $payment_id . '/payer-authentication';

            Cheqpay_Logger::log_api_request($api_url, 'POST', array(
                'collectionReferenceId' => $body['collectionReferenceId'] ?? '',
                'returnUrl' => $body['returnUrl'] ?? '',
            ), $order_id);

            $response = wp_remote_post($api_url, array(
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'x-session-token' => $session_token,
                    'User-Agent' => 'Cheqpay/' . CHEQPAY_LINK_VERSION
                ),
                'body' => json_encode($body),
                'timeout' => 30
            ));

            if (is_wp_error($response)) {
                Cheqpay_Logger::error('3DS payer authentication API failed - WP Error', $order_id, array(
                    'error' => $response->get_error_message(),
                ));
                throw new Exception($response->get_error_message());
            }

            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            Cheqpay_Logger::log_api_response($api_url, $response_code, $response_body, $order_id);

            // Assuming if we get here and status is COMPLETED or similar, it's good.
            // If the API returns another status (e.g. Challenge), logic needs to be added.
            // For now, if 200/201, assume success or handle generic success.

            if ($response_code === 200 || $response_code === 201) {
                 // Check status in response if available
                 $status = $data['status'] ?? '';

                 Cheqpay_Logger::debug('3DS authentication response status', $order_id, array(
                     'status' => $status,
                     'payment_id' => $payment_id,
                 ));

                 if ($status === 'COMPLETED') {
                     Cheqpay_Logger::info('3DS authentication completed - payment successful', $order_id, array(
                         'payment_id' => $payment_id,
                     ));
                     $order->payment_complete($payment_id); // Use payment ID as transaction ID
                     $order->add_order_note(__('NOTE_PAYMENT_COMPLETED_PAYER_AUTH', 'cheqpay-payment-gateway'));
                     wp_send_json_success(array(
                         'redirect' => $this->get_return_url($order)
                     ));
                 } elseif ($status === 'PAYER_AUTHENTICATION_CHALLENGE_REQUIRED') {
                     Cheqpay_Logger::debug('3DS challenge required', $order_id, array(
                         'payment_id' => $payment_id,
                     ));

                     $auth_data = $data['payerAuthentication'] ?? [];
                     $jwt = $auth_data['jwt'] ?? '';
                     $url = $auth_data['url'] ?? '';
                     $authentication_transaction_id = $auth_data['authenticationTransactionId'] ?? '';

                     if ($jwt && $url) {
                         // also save authenticationTransactionId
                         set_transient('cheqpay_authentication_transaction_id_' . $order->get_id(), $authentication_transaction_id, 10 * MINUTE_IN_SECONDS);
                         set_transient('cheqpay_challenge_jwt_' . $order->get_id(), $jwt, 10 * MINUTE_IN_SECONDS);
                         set_transient('cheqpay_challenge_url_' . $order->get_id(), $url, 10 * MINUTE_IN_SECONDS);
                         // Re-save payment ID for the next step (validate)
                         set_transient('cheqpay_ddc_payment_id_' . $order->get_id(), $payment_id, 10 * MINUTE_IN_SECONDS);

                         wp_send_json_success(array(
                             'redirect' => home_url('/?wc-api=cheqpay_challenge_page&order_id=' . $order->get_id())
                         ));
                     } else {
                         Cheqpay_Logger::error('3DS challenge failed - Missing JWT or URL', $order_id);
                         $order->update_status('failed', __('ERROR_AUTHENTICATION_CHALLENGE_FAILED', 'cheqpay-payment-gateway'));
                         wp_send_json_error(array('message' => __('ERROR_AUTHENTICATION_CHALLENGE_FAILED', 'cheqpay-payment-gateway')));
                     }
                 } else {
                     // If status is not COMPLETED (e.g. FAILED or ACTION_REQUIRED), fail for now
                     Cheqpay_Logger::error('3DS authentication failed', $order_id, array(
                         'status' => $status,
                         'payment_id' => $payment_id,
                     ));
                     $order->update_status('failed', __('ERROR_AUTHENTICATION_FAILED', 'cheqpay-payment-gateway') . ': ' . $status);
                     wp_send_json_error(array('message' => __('ERROR_AUTHENTICATION_FAILED', 'cheqpay-payment-gateway') . ': ' . $status));
                 }
             } else {
                 Cheqpay_Logger::error('3DS authentication API returned error', $order_id, array(
                     'response_code' => $response_code,
                 ));
                 $order->update_status('failed', __('ERROR_PAYMENT_PROCESSING_ERROR', 'cheqpay-payment-gateway') . ': ' . $response_code);
                 wp_send_json_error(array('message' => __('ERROR_PAYMENT_PROCESSING_ERROR', 'cheqpay-payment-gateway')));
             }

        } catch (Exception $e) {
            Cheqpay_Logger::log_exception($e, '3DS authentication failed', $order_id);
            $order->update_status('failed', 'Error: ' . $e->getMessage());
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }

    public function ajax_handle_validate_authentication() {
        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;

        Cheqpay_Logger::info('3DS validation request started', $order_id, array(
            'has_nonce' => isset($_POST['nonce']),
        ));

        // Verify nonce - log before checking
        $nonce_valid = isset($_POST['nonce']) && wp_verify_nonce($_POST['nonce'], 'cheqpay_validate_' . $order_id);
        if (!$nonce_valid) {
            Cheqpay_Logger::error('3DS validation failed - Nonce verification failed', $order_id);
            wp_send_json_error(array('message' => 'Security check failed'), 403);
            return;
        }

        $order = wc_get_order($order_id);

        Cheqpay_Logger::debug('3DS validation nonce verified', $order_id);

        if (!$order) {
            Cheqpay_Logger::error('3DS validation failed - Invalid order', $order_id);
            wp_send_json_error(array('message' => __('ERROR_INVALID_ORDER', 'cheqpay-payment-gateway')));
        }

        $session_token = get_transient('cheqpay_ddc_session_token_' . $order_id);
        $payment_id = get_transient('cheqpay_ddc_payment_id_' . $order_id);
        $authentication_transaction_id = get_transient('cheqpay_authentication_transaction_id_' . $order_id);

        if (!$payment_id) {
            Cheqpay_Logger::error('3DS validation failed - Payment session expired', $order_id);
            wp_send_json_error(array('message' => __('ERROR_PAYMENT_SESSION_EXPIRED', 'cheqpay-payment-gateway')));
        }

        // Clean up
        delete_transient('cheqpay_ddc_payment_id_' . $order_id);
        delete_transient('cheqpay_ddc_session_token_' . $order_id);

        try {
            $api_url = $this->get_api_url() . '/lps/payments/' . $payment_id . '/payer-authentication/validate';

            Cheqpay_Logger::log_api_request($api_url, 'POST', array(
                'authenticationTransactionId' => $authentication_transaction_id,
            ), $order_id);

            $response = wp_remote_post($api_url, array(
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'x-session-token' => $session_token,
                    'User-Agent' => 'Cheqpay/' . CHEQPAY_LINK_VERSION
                ),
                'body' => json_encode(array(
                    'authenticationTransactionId' => $authentication_transaction_id
                )),
                'timeout' => 30
            ));

            if (is_wp_error($response)) {
                Cheqpay_Logger::error('3DS validation API failed - WP Error', $order_id, array(
                    'error' => $response->get_error_message(),
                ));
                throw new Exception($response->get_error_message());
            }

            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            Cheqpay_Logger::log_api_response($api_url, $response_code, $response_body, $order_id);

            $status = $data['status'] ?? '';

            if (($response_code === 200 || $response_code === 201) && $status === 'COMPLETED') {
                 Cheqpay_Logger::info('3DS validation completed - payment successful', $order_id, array(
                     'payment_id' => $payment_id,
                 ));
                 $order->payment_complete($payment_id);
                 $order->add_order_note(__('NOTE_PAYMENT_COMPLETED_3DS', 'cheqpay-payment-gateway'));
                 wp_send_json_success(array(
                     'redirect' => $this->get_return_url($order)
                 ));
            } else {
                Cheqpay_Logger::error('3DS validation failed', $order_id, array(
                    'status' => $status,
                    'response_code' => $response_code,
                ));
                $order->update_status('failed', __('ERROR_AUTHENTICATION_VALIDATION_FAILED', 'cheqpay-payment-gateway') . ': ' . $status);
                wp_send_json_error(array('message' => __('ERROR_AUTHENTICATION_VALIDATION_FAILED', 'cheqpay-payment-gateway') . ': ' . $status));
            }

        } catch (Exception $e) {
            Cheqpay_Logger::log_exception($e, '3DS validation failed', $order_id);
            $order->update_status('failed', 'Error: ' . $e->getMessage());
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }

    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('LABEL_ENABLED', 'cheqpay-payment-gateway'),
                'type' => 'select',
                'label' => __('LABEL_ENABLE_GATEWAY', 'cheqpay-payment-gateway'),
                'default' => 'no',
                'options' => array(
                    'yes' => __('LABEL_YES', 'cheqpay-payment-gateway'),
                    'no' => __('LABEL_NO', 'cheqpay-payment-gateway')
                )
            ),
            'title' => array(
                'title' => __('LABEL_TITLE', 'cheqpay-payment-gateway'),
                'type' => 'text',
                'description' => __('DESCRIPTION_PAYMENT_TITLE', 'cheqpay-payment-gateway'),
                'default' => __('DEFAULT_PAYMENT_TITLE', 'cheqpay-payment-gateway'),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __('LABEL_DESCRIPTION', 'cheqpay-payment-gateway'),
                'type' => 'textarea',
                'description' => __('DESCRIPTION_PAYMENT_DESCRIPTION', 'cheqpay-payment-gateway'),
                'default' => __('DEFAULT_PAYMENT_DESCRIPTION', 'cheqpay-payment-gateway'),
                'desc_tip' => true,
            ),
            'payment_method' => array(
                'title' => __('LABEL_PAYMENT_METHOD', 'cheqpay-payment-gateway'),
                'type' => 'select',
                'description' => __('DESCRIPTION_SELECT_PAYMENT_METHOD', 'cheqpay-payment-gateway'),
                'default' => 'payment_link',
                'options' => array(
                    'payment_link' => __('OPTION_PAYMENT_LINK', 'cheqpay-payment-gateway'),
                    'card_payment' => __('OPTION_CARD_PAYMENT', 'cheqpay-payment-gateway')
                ),
                'desc_tip' => true,
            ),
            'environment' => array(
                'title' => __('LABEL_ENVIRONMENT', 'cheqpay-payment-gateway'),
                'type' => 'select',
                'description' => __('DESCRIPTION_SELECT_ENVIRONMENT', 'cheqpay-payment-gateway'),
                'default' => 'sandbox',
                'options' => array(
                    'sandbox' => __('OPTION_SANDBOX', 'cheqpay-payment-gateway'),
                    'production' => __('OPTION_PRODUCTION', 'cheqpay-payment-gateway')
                ),
                'desc_tip' => true,
            ),
            'public_key' => array(
                'title' => __('LABEL_PUBLIC_KEY', 'cheqpay-payment-gateway'),
                'type' => 'text',
                'description' => __('DESCRIPTION_PUBLIC_KEY', 'cheqpay-payment-gateway'),
                'default' => '',
                'desc_tip' => true,
            ),
            'secret_key' => array(
                'title' => __('LABEL_SECRET_KEY', 'cheqpay-payment-gateway'),
                'type' => 'password',
                'description' => __('DESCRIPTION_SECRET_KEY', 'cheqpay-payment-gateway'),
                'default' => '',
                'desc_tip' => true,
            ),
            'webhook_secret' => array(
                'title' => __('LABEL_WEBHOOK_SECRET', 'cheqpay-payment-gateway'),
                'type' => 'password',
                'description' => __('DESCRIPTION_WEBHOOK_SECRET', 'cheqpay-payment-gateway'),
                'default' => '',
                'desc_tip' => true,
            ),
            'webhook_url' => array(
                'title' => __('LABEL_WEBHOOK_URL', 'cheqpay-payment-gateway'),
                'type' => 'text',
                'description' => sprintf(
                    /* translators: %s: webhook URL wrapped in HTML code tags */
                    __('DESCRIPTION_WEBHOOK_URL', 'cheqpay-payment-gateway'),
                    '<br><code>' . home_url('/?wc-api=cheqpay_link') . '</code>'
                ),
                'default' => home_url('/?wc-api=cheqpay_link'),
                'custom_attributes' => array('readonly' => 'readonly'),
            ),
            'iframe_style_heading' => array(
                'title' => __('LABEL_IFRAME_STYLE_HEADING', 'cheqpay-payment-gateway'),
                'type' => 'title',
                'description' => __('DESCRIPTION_IFRAME_STYLE_HEADING', 'cheqpay-payment-gateway'),
            ),
            'color_background' => array(
                'title' => __('LABEL_COLOR_BACKGROUND', 'cheqpay-payment-gateway'),
                'type' => 'color',
                'description' => __('DESCRIPTION_COLOR_BACKGROUND', 'cheqpay-payment-gateway'),
                'default' => '#ffffff',
                'desc_tip' => true,
            ),
            'color_text' => array(
                'title' => __('LABEL_COLOR_TEXT', 'cheqpay-payment-gateway'),
                'type' => 'color',
                'description' => __('DESCRIPTION_COLOR_TEXT', 'cheqpay-payment-gateway'),
                'default' => '#333333',
                'desc_tip' => true,
            ),
            'color_border' => array(
                'title' => __('LABEL_COLOR_BORDER', 'cheqpay-payment-gateway'),
                'type' => 'color',
                'description' => __('DESCRIPTION_COLOR_BORDER', 'cheqpay-payment-gateway'),
                'default' => '#e0e0e0',
                'desc_tip' => true,
            ),
            'color_border_selected' => array(
                'title' => __('LABEL_COLOR_BORDER_SELECTED', 'cheqpay-payment-gateway'),
                'type' => 'color',
                'description' => __('DESCRIPTION_COLOR_BORDER_SELECTED', 'cheqpay-payment-gateway'),
                'default' => '#0F172A',
                'desc_tip' => true,
            ),
            'color_placeholder' => array(
                'title' => __('LABEL_COLOR_PLACEHOLDER', 'cheqpay-payment-gateway'),
                'type' => 'color',
                'description' => __('DESCRIPTION_COLOR_PLACEHOLDER', 'cheqpay-payment-gateway'),
                'default' => '#64748B',
                'desc_tip' => true,
            ),
            'radius_border' => array(
                'title' => __('LABEL_RADIUS_BORDER', 'cheqpay-payment-gateway'),
                'type' => 'text',
                'description' => __('DESCRIPTION_RADIUS_BORDER', 'cheqpay-payment-gateway'),
                'default' => '8px',
                'desc_tip' => true,
            ),
        );
    }

    public function payment_fields() {
        $payment_method = $this->get_option('payment_method', 'payment_link');
        $description = $this->get_description();

        if ($payment_method !== 'card_payment') {
            if ($description) {
                echo wp_kses_post(wpautop(wptexturize($description)));
            }
            return;
        }

        $color_bg = esc_attr($this->get_option('color_background', '#ffffff'));
        $color_text = esc_attr($this->get_option('color_text', '#333333'));
        $color_border = esc_attr($this->get_option('color_border', '#e0e0e0'));
        $radius_border = esc_attr($this->get_option('radius_border', '8px'));

        ?>
        <style>
            #cheqpay-payment-form {
                display: flex;
                flex-direction: column;
                background: <?php echo $color_bg; ?>;
                color: <?php echo $color_text; ?>;
            }
            #cheqpay-payment-form .cheqpay-field-group {
                display: flex;
                flex-direction: column;
            }
            #cheqpay-payment-form #cheqpay-card-number-container {
                margin-bottom: 15px;
            }
            #cheqpay-payment-form label {
                font-size: 14px;
                font-weight: 500;
                color: <?php echo $color_text; ?>;
                margin-bottom: 8px;
            }
            #cheqpay-payment-form .cheqpay-element {
                position: relative;
                width: 100%;
                height: 40px;
                background: <?php echo $color_bg; ?>;
                border: 1px solid <?php echo $color_border; ?>;
                border-radius: <?php echo $radius_border; ?>;
                box-sizing: border-box;
            }
            #cheqpay-payment-form .cheqpay-element iframe {
                width: 100%;
                height: 100%;
                border: none;
                overflow: hidden;
            }
            #cheqpay-payment-form .cheqpay-element.error {
                border-color: #ef4444;
            }
            #cheqpay-payment-form .cheqpay-element.valid {
                border-color: #10b981;
            }
            #cheqpay-payment-form .error-message {
                color: #ef4444;
                font-size: 12px;
                margin-top: 4px;
                min-height: 16px;
                line-height: 16px;
                font-weight: 400;
                display: none;
            }
            #cheqpay-payment-form .error-message.show {
                display: block;
            }
            #cheqpay-payment-form .cheqpay-row {
                display: grid;
                grid-template-columns: 1fr 1fr;
                gap: 16px;
                margin-left: 0 !important;
                margin-right: 0 !important;
            }
        </style>
        <fieldset id="cheqpay-payment-form" class="wc-payment-form">
            <!-- VGS Iframe Containers -->
            <div class="cheqpay-field-group" id="cheqpay-card-number-container">
                <label for="cheqpay-card-number"><?php echo esc_html(__('LABEL_CARD_NUMBER', 'cheqpay-payment-gateway')); ?> <span class="required"><?php echo esc_html(__('LABEL_REQUIRED', 'cheqpay-payment-gateway')); ?></span></label>
                <div id="cheqpay-card-number" class="cheqpay-element">
                    <!-- VGS iframe will be injected here -->
                </div>
                <div id="cheqpay-card-number-error" class="error-message"></div>
            </div>

            <div class="cheqpay-row">
                <div class="cheqpay-field-group">
                    <label for="cheqpay-card-expiry"><?php echo esc_html(__('LABEL_EXPIRY_DATE', 'cheqpay-payment-gateway')); ?> <span class="required"><?php echo esc_html(__('LABEL_REQUIRED', 'cheqpay-payment-gateway')); ?></span></label>
                    <div id="cheqpay-card-expiry" class="cheqpay-element">
                        <!-- VGS iframe will be injected here -->
                    </div>
                    <div id="cheqpay-card-expiry-error" class="error-message"></div>
                </div>
                <div class="cheqpay-field-group">
                    <label for="cheqpay-card-cvc"><?php echo esc_html(__('LABEL_CVC', 'cheqpay-payment-gateway')); ?> <span class="required"><?php echo esc_html(__('LABEL_REQUIRED', 'cheqpay-payment-gateway')); ?></span></label>
                    <div id="cheqpay-card-cvc" class="cheqpay-element">
                        <!-- VGS iframe will be injected here -->
                    </div>
                    <div id="cheqpay-card-cvc-error" class="error-message"></div>
                </div>
            </div>

            <!-- Hidden fields to store encrypted values -->
            <input type="hidden" id="cheqpay_card_number" name="cheqpay_card_number" />
            <input type="hidden" id="cheqpay_card_expiry" name="cheqpay_card_expiry" />
            <input type="hidden" id="cheqpay_card_cvc" name="cheqpay_card_cvc" />

            <div class="clear"></div>
        </fieldset>
        <?php
    }

    public function process_payment($order_id) {
        $order = wc_get_order($order_id);

        Cheqpay_Logger::debug('Processing payment', $order_id);

        if (!$order) {
            Cheqpay_Logger::error('Order not found during payment processing', $order_id);
            wc_add_notice(__('ERROR_ORDER_NOT_FOUND', 'cheqpay-payment-gateway'), 'error');
            return array('result' => 'fail');
        }

        $payment_method = $this->get_option('payment_method', 'payment_link');

        Cheqpay_Logger::debug('Payment method selected', $order_id, array(
            'payment_method' => $payment_method,
            'order_total' => $order->get_total(),
            'currency' => $order->get_currency(),
        ));

        if ($payment_method === 'card_payment') {
            return $this->process_card_payment($order);
        } else {
            return $this->process_payment_link($order);
        }
    }

    private function process_payment_link($order) {
        $order_id = $order->get_id();
        $secret_key = $this->get_option('secret_key');

        Cheqpay_Logger::debug('Processing payment link', $order_id);

        if (empty($secret_key)) {
            Cheqpay_Logger::error('Payment link not configured - missing secret key', $order_id);
            wc_add_notice(__('ERROR_PAYMENT_LINK_NOT_CONFIGURED', 'cheqpay-payment-gateway'), 'error');
            return array('result' => 'fail');
        }

        try {
            // Create new payment link if none exists
            $payment_link = $this->create_payment_link($order);

            if ($payment_link && isset($payment_link['id'])) {
                Cheqpay_Logger::info('Payment link created successfully', $order_id, array(
                    'payment_link_id' => $payment_link['id'],
                ));

                $order->update_status('pending', __('NOTE_AWAITING_PAYMENT_LINK', 'cheqpay-payment-gateway'));
                $order->update_meta_data('_cheqpay_payment_link_id', $payment_link['id']);
                $order->set_payment_method_title(__('LABEL_CHEQPAY_LINK', 'cheqpay-payment-gateway'));
                $order->update_meta_data('_cheqpay_method_type', 'payment_link');
                $order->save();

                return array(
                    'result' => 'success',
                    'redirect' => $this->get_hpp_url() . '/' . $payment_link['id']
                );
            } else {
                throw new Exception('Failed to create payment link');
            }

        } catch (Exception $e) {
            Cheqpay_Logger::log_exception($e, 'Payment link creation failed', $order_id);
            wc_add_notice(
                sprintf(
                    /* translators: %s: error message */
                    __('Payment error: %s', 'cheqpay-payment-gateway'),
                    $e->getMessage()
                ),
                'error'
            );
            return array('result' => 'fail');
        }
    }

    private function process_card_payment($order) {
        $order_id = $order->get_id();
        $secret_key = $this->get_option('secret_key');
        $public_key = $this->get_option('public_key');

        Cheqpay_Logger::debug('Processing card payment', $order_id);

        if (empty($secret_key) || empty($public_key)) {
            Cheqpay_Logger::error('Card payment not configured - missing keys', $order_id);
            wc_add_notice(__('ERROR_CARD_PAYMENT_NOT_CONFIGURED', 'cheqpay-payment-gateway'), 'error');
            return array('result' => 'fail');
        }

        // Validate card fields
        $card_number = isset($_POST['cheqpay_card_number']) ? sanitize_text_field(wp_unslash($_POST['cheqpay_card_number'])) : '';
        $card_expiry = isset($_POST['cheqpay_card_expiry']) ? sanitize_text_field(wp_unslash($_POST['cheqpay_card_expiry'])) : '';
        $card_cvc = isset($_POST['cheqpay_card_cvc']) ? sanitize_text_field(wp_unslash($_POST['cheqpay_card_cvc'])) : '';

        if (empty($card_number) || empty($card_expiry) || empty($card_cvc)) {
            Cheqpay_Logger::warning('Card payment failed - empty card fields', $order_id);
            wc_add_notice(__('ERROR_CARD_FIELDS_EMPTY', 'cheqpay-payment-gateway'), 'error');
            return array('result' => 'fail');
        }

        $card_number = str_replace(' ', '', $card_number); // Remove spaces

        // Set payment method title and type
        $order->set_payment_method_title($this->get_option('title'));
        $order->update_meta_data('_cheqpay_method_type', 'card_payment');
        $order->save();

        // Process card payment via API
        try {
            // Step 1: Create Payment Session
            Cheqpay_Logger::debug('Creating payment session', $order_id);
            $session_data = $this->create_payment_session($order_id);
            $session_token = $session_data['token'] ?? '';

            if (empty($session_token)) {
                throw new Exception('Failed to create payment session');
            }

            Cheqpay_Logger::debug('Payment session created successfully', $order_id);

            // Step 2: Create Payment Charge
            Cheqpay_Logger::debug('Creating payment charge', $order_id);
            $payment_result = $this->create_payment_charge($order, $card_number, $card_expiry, $card_cvc, $session_token);
            $status = $payment_result['status'] ?? '';

            Cheqpay_Logger::debug('Payment charge response received', $order_id, array(
                'status' => $status,
                'payment_id' => $payment_result['id'] ?? 'N/A',
            ));

            if ($status === 'COMPLETED' && isset($payment_result['id'])) {
                Cheqpay_Logger::info('Card payment completed successfully', $order_id, array(
                    'payment_id' => $payment_result['id'],
                ));

                $order->payment_complete($payment_result['id']);
                $order->add_order_note(
                    sprintf(
                        /* translators: %s: payment ID */
                        __('NOTE_PAYMENT_COMPLETED_CARD', 'cheqpay-payment-gateway'),
                        $payment_result['id']
                    )
                );

                // Remove card data from order meta (security)
                $order->delete_meta_data('_cheqpay_card_number');
                $order->delete_meta_data('_cheqpay_card_expiry');
                $order->delete_meta_data('_cheqpay_card_cvc');
                $order->save();

                return array(
                    'result' => 'success',
                    'redirect' => $this->get_return_url($order)
                );
            } elseif ($status === 'PAYER_AUTHENTICATION_DEVICE_DATA_REQUIRED') {
                Cheqpay_Logger::debug('3DS device data collection required', $order_id, array(
                    'payment_id' => $payment_result['id'] ?? 'N/A',
                ));

                $auth_data = $payment_result['payerAuthentication'] ?? [];
                $jwt = $auth_data['jwt'] ?? '';
                $url = $auth_data['url'] ?? '';

                if ($jwt && $url) {
                    set_transient('cheqpay_ddc_session_token_' . $order->get_id(), $session_token, 10 * MINUTE_IN_SECONDS);
                    set_transient('cheqpay_ddc_jwt_' . $order->get_id(), $jwt, 10 * MINUTE_IN_SECONDS);
                    set_transient('cheqpay_ddc_url_' . $order->get_id(), $url, 10 * MINUTE_IN_SECONDS);
                    set_transient('cheqpay_ddc_payment_id_' . $order->get_id(), $payment_result['id'], 10 * MINUTE_IN_SECONDS);

                    return array(
                        'result' => 'success',
                        'redirect' => home_url('/?wc-api=cheqpay_device_data&order_id=' . $order->get_id())
                    );
                } else {
                    throw new Exception('Missing payer authentication data.');
                }
            } else {
                throw new Exception('Payment status not completed: ' . $status);
            }

        } catch (Exception $e) {
            Cheqpay_Logger::log_exception($e, 'Card payment failed', $order_id);
            wc_add_notice(
                sprintf(
                    /* translators: %s: error message */
                    __('Payment error: %s', 'cheqpay-payment-gateway'),
                    $e->getMessage()
                ),
                'error'
            );
            return array('result' => 'fail');
        }
    }

    private function create_payment_session($order_id = null) {
        $api_url = $this->get_api_url() . '/lps/payments/session';

        Cheqpay_Logger::log_api_request($api_url, 'POST', array(), $order_id);

        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $this->get_option('secret_key'),
                'User-Agent' => 'Cheqpay/' . CHEQPAY_LINK_VERSION . ' WordPress/' . get_bloginfo('version') . ' WooCommerce/' . WC()->version
            ),
            'body' => '{}', // Empty body
            'timeout' => 30
        ));

        if (is_wp_error($response)) {
            Cheqpay_Logger::error('Payment session API request failed - WP Error', $order_id, array(
                'error' => $response->get_error_message(),
            ));
            throw new Exception(esc_html($response->get_error_message()));
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        Cheqpay_Logger::log_api_response($api_url, $response_code, $response_body, $order_id);

        if ($response_code !== 200 && $response_code !== 201) {
            $error_data = json_decode($response_body, true);
            $error_message = isset($error_data['message']) ? $error_data['message'] : 'Session API request failed with code: ' . $response_code;
            Cheqpay_Logger::error('Payment session API request failed', $order_id, array(
                'response_code' => $response_code,
                'error_message' => $error_message,
            ));
            throw new Exception(esc_html($error_message));
        }

        $data = json_decode($response_body, true);

        if (!$data || !isset($data['token'])) {
            Cheqpay_Logger::error('Invalid response from Payment Session API', $order_id);
            throw new Exception(esc_html('Invalid response from Cheqpay Session API'));
        }

        return $data;
    }

    private function create_payment_charge($order, $card_number, $card_expiry, $card_cvc, $session_token) {
        $order_id = $order->get_id();
        $api_url = $this->get_api_url() . '/lps/payments/charge';

        // Parse expiry date
        list($exp_month, $exp_year) = explode('/', $card_expiry);

        // Prepare customer data
        $customer = array(
            'firstName' => $order->get_billing_first_name(),
            'lastName' => $order->get_billing_last_name(),
            'email' => $order->get_billing_email(),
            // 'phone' => $order->get_billing_phone()
        );

        // Prepare billing address
        $billing_address = array(
            'firstName' => $order->get_billing_first_name(),
            'lastName' => $order->get_billing_last_name(),
            'organizationName' => $order->get_billing_company(),
            'email' => $order->get_billing_email(),
            // 'phone' => $order->get_billing_phone(),
            'addressLine1' => $order->get_billing_address_1(),
            'addressLine2' => $order->get_billing_address_2(),
            'city' => $order->get_billing_city(),
            'state' => !empty($order->get_billing_state()) ? $order->get_billing_state() : $order->get_billing_city(), // @todo handle this in backend
            'postalCode' => $order->get_billing_postcode(),
            'country' => $order->get_billing_country()
        );

        $request_data = array(
            'amount' => (float) $order->get_total(),
            'currency' => $order->get_currency(),
            /* translators: %s: order number */
            'description' => sprintf(__('DESCRIPTION_ORDER_NUMBER', 'cheqpay-payment-gateway'), $order->get_order_number()),
            'merchantReference' => (string) $order->get_id(),
            'paymentMethod' => array(
                'type' => 'card',
                'card' => array(
                    'number' => $card_number,
                    'expiryMonth' => intval($exp_month),
                    'expiryYear' => intval($exp_year),
                    'cvc' => $card_cvc
                )
            ),
            'customer' => $customer,
            'billingAddress' => $billing_address
        );

        // Log request without sensitive card data
        Cheqpay_Logger::log_api_request($api_url, 'POST', array(
            'amount' => $request_data['amount'],
            'currency' => $request_data['currency'],
            'merchantReference' => $request_data['merchantReference'],
            'customer' => $customer,
            'billingAddress' => $billing_address,
        ), $order_id);

        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-session-token' => $session_token,
                'User-Agent' => 'Cheqpay/' . CHEQPAY_LINK_VERSION . ' WordPress/' . get_bloginfo('version') . ' WooCommerce/' . WC()->version
            ),
            'body' => json_encode($request_data),
            'timeout' => 30
        ));

        if (is_wp_error($response)) {
            Cheqpay_Logger::error('Payment charge API request failed - WP Error', $order_id, array(
                'error' => $response->get_error_message(),
            ));
            throw new Exception(esc_html($response->get_error_message()));
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        Cheqpay_Logger::log_api_response($api_url, $response_code, $response_body, $order_id);

        if ($response_code !== 200 && $response_code !== 201) {
            $error_data = json_decode($response_body, true);
            $error_message = isset($error_data['message']) ? $error_data['message'] : 'Charge API request failed with code: ' . $response_code;
            Cheqpay_Logger::error('Payment charge API request failed', $order_id, array(
                'response_code' => $response_code,
                'error_message' => $error_message,
            ));
            throw new Exception(esc_html($error_message));
        }

        $data = json_decode($response_body, true);

        if (!$data || !isset($data['id'])) {
            Cheqpay_Logger::error('Invalid response from Payment Charge API', $order_id);
            throw new Exception(esc_html('Invalid response from Cheqpay Charge API'));
        }

        return $data;
    }

    private function get_api_url() {
        $environment = $this->get_option('environment', 'sandbox');
        return ($environment === 'production') ? CHEQPAY_LINK_API_URL_PRODUCTION : CHEQPAY_LINK_API_URL_SANDBOX;
    }

    private function get_hpp_url() {
        $environment = $this->get_option('environment', 'sandbox');
        return ($environment === 'production') ? CHEQPAY_LINK_HPP_URL_PRODUCTION : CHEQPAY_LINK_HPP_URL_SANDBOX;
    }

    /**
     * Create payment link via API
     * @param $order WC_Order
     * @return mixed
     * @throws Exception
     */
    private function create_payment_link($order) {
        $order_id = $order->get_id();
        $api_url = $this->get_api_url() . '/lps/payment-links';

        $products = array();
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();

            // Get product image URL
            $image_id = $product->get_image_id();
            $image_url = '';
            if ($image_id) {
                $image_url = wp_get_attachment_image_url($image_id, 'full');
            }

            $item_total = (float) $item->get_total();
            $item_quantity = (float) $item->get_quantity();
            $unit_price = $item_quantity > 0 ? $item_total / $item_quantity : 0;

            $products[] = array(
                'id' => (string) $product->get_id(),
                'name' => $item->get_name(),
                'description' => $product->get_short_description() ?: '',
                'qty' => $item->get_quantity(),
                'price' => $unit_price,
                'total' => $item_total,
                'imageUrl' => $image_url ?: '',
                'isQtyAdjustable' => false
            );
        }

        $redirect_url = add_query_arg(array(
            'hpp_return' => '1',
            'order_id' => $order->get_id(),
            '_wpnonce' => wp_create_nonce('cheqpay_return_' . $order->get_id())
        ), home_url('/'));

        $billing_address = array(
            'firstName' => $order->get_billing_first_name(),
            'lastName' => $order->get_billing_last_name(),
            'organizationName' => $order->get_billing_company(),
            'email' => $order->get_billing_email(),
            // 'phone' => $order->get_billing_phone(),  // TODO: Add phone to billing address
            'addressLine1' => $order->get_billing_address_1(),
            'addressLine2' => $order->get_billing_address_2(),
            'city' => $order->get_billing_city(),
            'state' => $order->get_billing_state(),
            'postalCode' => $order->get_billing_postcode(),
            'country' => $order->get_billing_country()
        );

        $shipping_address = array(
            'firstName' => $order->get_shipping_first_name(),
            'lastName' => $order->get_shipping_last_name(),
            'organizationName' => $order->get_shipping_company(),
            'email' => $order->get_billing_email(), // Shipping doesn't have email, use billing
            // 'phone' => $order->get_billing_phone(), // Shipping doesn't have phone, use billing // TODO: Add phone to shipping address
            'addressLine1' => $order->get_shipping_address_1(),
            'addressLine2' => $order->get_shipping_address_2(),
            'city' => $order->get_shipping_city(),
            'state' => $order->get_shipping_state(),
            'postalCode' => $order->get_shipping_postcode(),
            'country' => $order->get_shipping_country()
        );

        // Calculate fees total
        $fees_total = 0;
        foreach ($order->get_fees() as $fee) {
            $fees_total += (float) $fee->get_total();
        }

        $request_data = array(
            'type' => 'products_subscriptions',
            'products' => $products,
            'currency' => $order->get_currency(),
            'totals' => array(
                'subtotal' => (float) $order->get_subtotal(),
                'total' => (float) $order->get_total(),
                'shipping' => (float) $order->get_shipping_total(),
                'taxes' => (float) $order->get_total_tax(),
                'fees' => $fees_total,
                'discounts' => (float) $order->get_total_discount()
            ),
            'paymentMethods' => array('card'),
            'redirectUrl' => $redirect_url,
            'isEnabled' => true,
            'numberOfPayments' => 1,
            'reference' => (string)$order->get_id(),
            'defaults' => array(
                'billingAddress' => $billing_address,
                'shippingAddress' => $shipping_address
            ),
            'metadata' => array(
                'source' => 'woocommerce_cheqpay_link',
                'siteUrl' => home_url(),
                'pluginVersion' => CHEQPAY_LINK_VERSION,
                'woocommerceVersion' => WC()->version,
                'wordpressVersion' => get_bloginfo('version'),
                'phpVersion' => phpversion(),
            )
        );

        Cheqpay_Logger::log_api_request($api_url, 'POST', array(
            'currency' => $request_data['currency'],
            'totals' => $request_data['totals'],
            'reference' => $request_data['reference'],
            'product_count' => count($products),
        ), $order_id);

        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $this->get_option('secret_key'),
                'User-Agent' => 'Cheqpay/' . CHEQPAY_LINK_VERSION . ' WordPress/' . get_bloginfo('version') . ' WooCommerce/' . WC()->version
            ),
            'body' => json_encode($request_data),
            'timeout' => 30
        ));

        if (is_wp_error($response)) {
            Cheqpay_Logger::error('Payment link API request failed - WP Error', $order_id, array(
                'error' => $response->get_error_message(),
            ));
            throw new Exception(esc_html($response->get_error_message()));
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        Cheqpay_Logger::log_api_response($api_url, $response_code, $response_body, $order_id);

        if ($response_code !== 200 && $response_code !== 201) {
            Cheqpay_Logger::error('Payment link API request failed', $order_id, array(
                'response_code' => $response_code,
            ));
            throw new Exception(esc_html('API request failed with code: ' . $response_code));
        }

        $data = json_decode($response_body, true);

        if (!$data || !isset($data['id'])) {
            Cheqpay_Logger::error('Invalid response from Payment Link API', $order_id);
            throw new Exception(esc_html('Invalid response from Cheqpay API'));
        }

        return $data;
    }

    public function is_available() {
        if ($this->enabled === 'no') {
            return false;
        }

        $payment_method = $this->get_option('payment_method', 'payment_link');
        $secret_key = $this->get_option('secret_key');

        // For card payments, also require public key
        if ($payment_method === 'card_payment') {
            $public_key = $this->get_option('public_key');
            if (empty($secret_key) || empty($public_key)) {
                return false;
            }
        } else {
            if (empty($secret_key)) {
                return false;
            }
        }

        return parent::is_available();
    }

    /**
     * Process refund
     *
     * @param int $order_id Order ID
     * @param float|null $amount Refund amount
     * @param string $reason Refund reason
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public function process_refund($order_id, $amount = null, $reason = '') {
        Cheqpay_Logger::debug('Processing refund', $order_id, array(
            'amount' => $amount,
            'reason' => $reason,
        ));

        $order = wc_get_order($order_id);

        if (!$order) {
            Cheqpay_Logger::error('Refund failed - Order not found', $order_id);
            return new WP_Error('invalid_order', __('ERROR_ORDER_NOT_FOUND', 'cheqpay-payment-gateway'));
        }

        // Get the invoice ID from transaction ID
        $transaction_id = $order->get_transaction_id();
        // as we are currently live with payment link only, so for now we can assume that transaction id is mongoId
        // after sometime we can remove this defaulting, as this value will be always present
        $is_mongoId = preg_match('/^[0-9a-f]{24}$/i', $transaction_id) === 1;

        if (empty($transaction_id)) {
            Cheqpay_Logger::error('Refund failed - No transaction ID', $order_id);
            return new WP_Error('no_invoice_id', __('ERROR_NO_INVOICE_ID', 'cheqpay-payment-gateway'));
        }

        if (is_null($amount) || $amount <= 0) {
            Cheqpay_Logger::error('Refund failed - Invalid amount', $order_id, array('amount' => $amount));
            return new WP_Error('invalid_amount', __('ERROR_INVALID_REFUND_AMOUNT', 'cheqpay-payment-gateway'));
        }

        Cheqpay_Logger::debug('Refund details', $order_id, array(
            'transaction_id' => $transaction_id,
            'is_mongoId' => $is_mongoId,
            'amount' => $amount,
        ));

        try {
            $refund_response = $is_mongoId
                ? $this->create_refund_invoice($transaction_id, $amount, $order_id)
                : $this->create_refund_payment_order($transaction_id, $amount, $order_id);

            if ($refund_response && isset($refund_response['id'])) {
                $refund_id = $refund_response['id'];
                $refund_status = $refund_response['status'] ?? 'unknown';

                Cheqpay_Logger::info('Refund processed successfully', $order_id, array(
                    'refund_id' => $refund_id,
                    'refund_status' => $refund_status,
                    'amount' => $amount,
                ));

                // Add order note
                $order->add_order_note(
                    sprintf(
                        /* translators: 1: refund amount with currency symbol, 2: refund ID, 3: refund status */
                        __('NOTE_REFUND_PROCESSED', 'cheqpay-payment-gateway'),
                        wc_price($amount),
                        $refund_id,
                        $refund_status
                    )
                );

                return true;
            } else {
                Cheqpay_Logger::error('Refund failed - Invalid response', $order_id);
                return new WP_Error('refund_failed', __('ERROR_REFUND_FAILED', 'cheqpay-payment-gateway'));
            }

        } catch (Exception $e) {
            Cheqpay_Logger::log_exception($e, 'Refund failed', $order_id);
            return new WP_Error('refund_error', $e->getMessage());
        }
    }

    /**
     * Create refund via API
     *
     * @param string $id Payment Order ID
     * @param float $amount Refund amount
     * @param int|null $order_id Order ID for logging
     * @return array|false Refund response or false on failure
     */
    private function create_refund_payment_order($id, $amount, $order_id = null) {
        $api_url = $this->get_api_url() . '/pos/payment-orders/' . $id . '/refund';

        $body = array(
            'amount' => floatval($amount)
        );

        Cheqpay_Logger::log_api_request($api_url, 'POST', $body, $order_id);

        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $this->get_option('secret_key'),
                'User-Agent' => 'Cheqpay/' . CHEQPAY_LINK_VERSION . ' WordPress/' . get_bloginfo('version') . ' WooCommerce/' . WC()->version
            ),
            'body' => wp_json_encode($body),
            'timeout' => 30
        ));

        if (is_wp_error($response)) {
            Cheqpay_Logger::error('Refund payment order API failed - WP Error', $order_id, array(
                'error' => $response->get_error_message(),
            ));
            throw new Exception(esc_html($response->get_error_message()));
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        Cheqpay_Logger::log_api_response($api_url, $response_code, $response_body, $order_id);

        if ($response_code !== 200) {
            Cheqpay_Logger::error('Refund payment order API failed', $order_id, array(
                'response_code' => $response_code,
            ));
            throw new Exception(
                esc_html(
                    sprintf(
                        /* translators: %s: HTTP response code */
                        __('Refund API returned error code: %s', 'cheqpay-payment-gateway'),
                        $response_code
                    )
                )
            );
        }

        $data = json_decode($response_body, true);

        if (!$data) {
            Cheqpay_Logger::error('Invalid response from refund payment order API', $order_id);
            throw new Exception(esc_html__('Invalid response from refund API.', 'cheqpay-payment-gateway'));
        }

        return $data;
    }

    /**
     * Create refund via API
     *
     * @param string $invoice_id Invoice ID
     * @param float $amount Refund amount
     * @param int|null $order_id Order ID for logging
     * @return array|false Refund response or false on failure
     */
    private function create_refund_invoice($invoice_id, $amount, $order_id = null) {
        $api_url = $this->get_api_url() . '/lps/invoices/' . $invoice_id . '/refund';

        $body = array(
            'amount' => floatval($amount)
        );

        Cheqpay_Logger::log_api_request($api_url, 'POST', $body, $order_id);

        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $this->get_option('secret_key'),
                'User-Agent' => 'Cheqpay/' . CHEQPAY_LINK_VERSION . ' WordPress/' . get_bloginfo('version') . ' WooCommerce/' . WC()->version
            ),
            'body' => wp_json_encode($body),
            'timeout' => 30
        ));

        if (is_wp_error($response)) {
            Cheqpay_Logger::error('Refund invoice API failed - WP Error', $order_id, array(
                'error' => $response->get_error_message(),
            ));
            throw new Exception(esc_html($response->get_error_message()));
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        Cheqpay_Logger::log_api_response($api_url, $response_code, $response_body, $order_id);

        if ($response_code !== 200) {
            Cheqpay_Logger::error('Refund invoice API failed', $order_id, array(
                'response_code' => $response_code,
            ));
            throw new Exception(
                esc_html(
                    sprintf(
                        /* translators: %s: HTTP response code */
                        __('Refund API returned error code: %s', 'cheqpay-payment-gateway'),
                        $response_code
                    )
                )
            );
        }

        $data = json_decode($response_body, true);

        if (!$data) {
            Cheqpay_Logger::error('Invalid response from refund invoice API', $order_id);
            throw new Exception(esc_html__('Invalid response from refund API.', 'cheqpay-payment-gateway'));
        }

        return $data;
    }

    /**
     * Get payment gateway icon
     *
     * @return string
     */
    public function get_icon() {
        $icon_html = '<span style="float: right; display: inline-flex; align-items: center; gap: 3px;">';

        $visa_svg = '<svg xmlns="http://www.w3.org/2000/svg" width="42" height="42" viewBox="0 0 32 32"> <g class="nc-icon-wrapper"> <rect x="2" y="7" width="28" height="18" rx="3" ry="3" fill="#1434cb" stroke-width="0"></rect> <path d="m27,7H5c-1.657,0-3,1.343-3,3v12c0,1.657,1.343,3,3,3h22c1.657,0,3-1.343,3-3v-12c0-1.657-1.343-3-3-3Zm2,15c0,1.103-.897,2-2,2H5c-1.103,0-2-.897-2-2v-12c0-1.103.897-2,2-2h22c1.103,0,2,.897,2,2v12Z" stroke-width="0" opacity=".15"></path> <path d="m27,8H5c-1.105,0-2,.895-2,2v1c0-1.105.895-2,2-2h22c1.105,0,2,.895,2,2v-1c0-1.105-.895-2-2-2Z" fill="#fff" opacity=".2" stroke-width="0"></path> <path d="m13.392,12.624l-2.838,6.77h-1.851l-1.397-5.403c-.085-.332-.158-.454-.416-.595-.421-.229-1.117-.443-1.728-.576l.041-.196h2.98c.38,0,.721.253.808.69l.738,3.918,1.822-4.608h1.84Z" fill="#fff" stroke-width="0"></path> <path d="m20.646,17.183c.008-1.787-2.47-1.886-2.453-2.684.005-.243.237-.501.743-.567.251-.032.943-.058,1.727.303l.307-1.436c-.421-.152-.964-.299-1.638-.299-1.732,0-2.95.92-2.959,2.238-.011.975.87,1.518,1.533,1.843.683.332.912.545.909.841-.005.454-.545.655-1.047.663-.881.014-1.392-.238-1.799-.428l-.318,1.484c.41.188,1.165.351,1.947.359,1.841,0,3.044-.909,3.05-2.317" fill="#fff" stroke-width="0"></path> <path d="m25.423,12.624h-1.494c-.337,0-.62.195-.746.496l-2.628,6.274h1.839l.365-1.011h2.247l.212,1.011h1.62l-1.415-6.77Zm-2.16,4.372l.922-2.542.53,2.542h-1.452Z" fill="#fff" stroke-width="0"></path> <path fill="#fff" stroke-width="0" d="M15.894 12.624L14.446 19.394 12.695 19.394 14.143 12.624 15.894 12.624z"></path> </g> </svg>';
        $mastercard_svg = '<svg xmlns="http://www.w3.org/2000/svg" width="42" height="42" viewBox="0 0 32 32"> <g class="nc-icon-wrapper"> <rect x="2" y="7" width="28" height="18" rx="3" ry="3" fill="#fff" stroke-width="0"></rect> <path d="m27,7H5c-1.657,0-3,1.343-3,3v12c0,1.657,1.343,3,3,3h22c1.657,0,3-1.343,3-3v-12c0-1.657-1.343-3-3-3Zm2,15c0,1.103-.897,2-2,2H5c-1.103,0-2-.897-2-2v-12c0-1.103.897-2,2-2h22c1.103,0,2,.897,2,2v12Z" stroke-width="0" opacity=".15"></path> <path d="m27,8H5c-1.105,0-2,.895-2,2v1c0-1.105.895-2,2-2h22c1.105,0,2,.895,2,2v-1c0-1.105-.895-2-2-2Z" fill="#fff" opacity=".2" stroke-width="0"></path> <path fill="#ff5f00" stroke-width="0" d="M13.597 11.677H18.407V20.32H13.597z"></path> <path d="m13.902,15.999c0-1.68.779-3.283,2.092-4.322-2.382-1.878-5.849-1.466-7.727.932-1.863,2.382-1.451,5.833.947,7.712,2,1.573,4.795,1.573,6.795,0-1.329-1.038-2.107-2.642-2.107-4.322Z" fill="#eb001b" stroke-width="0"></path> <path d="m24.897,15.999c0,3.039-2.459,5.497-5.497,5.497-1.237,0-2.428-.412-3.39-1.176,2.382-1.878,2.795-5.329.916-7.727-.275-.336-.58-.657-.916-.916,2.382-1.878,5.849-1.466,7.712.932.764.962,1.176,2.153,1.176,3.39Z" fill="#f79e1b" stroke-width="0"></path> </g> </svg>';

        // Only show Cheqpay icon for Mexico
        $base_country = WC()->countries->get_base_country();
        if ($base_country === 'MX') {
            $cheqpay_svg = '<svg width="75" height="100%" viewBox="0 0 1422 375" fill="none" xmlns="http://www.w3.org/2000/svg"> <path d="M625.468 83.0351C638.966 80.0776 654.529 78.5679 669.158 81.4169C677.949 83.1293 687.011 85.9515 693.375 92.2861C693.375 92.2861 693.375 86.8204 693.375 84.3447H755.767V374.111C738.839 367.878 722.836 357.869 706.547 350.147C696.599 345.432 694.497 339.729 694.497 329.435V284.88C678.226 296.315 661.4 296.031 644.775 295.403C632.352 294.93 619.919 292.636 609.428 285.372C597.855 277.358 590.61 266.03 590.469 251.771C590.059 210.177 590.29 168.577 590.225 126.977C590.195 104.664 604.262 87.6793 625.468 83.0351ZM672.104 133.924C652.919 133.924 652.126 148.299 652.126 151.283C652.126 154.266 652.111 216.294 652.111 219.812C652.111 223.331 651 240.727 672.014 240.727C693.028 240.726 693.367 224.294 693.367 219.814V187.643C693.367 175.525 693.428 163.407 693.428 151.29C693.428 145.705 691.289 133.924 672.104 133.924Z" fill="#01467F"/> <path d="M874.717 81.4169C889.345 78.5678 904.909 80.0775 918.407 83.0351C939.613 87.6794 953.679 104.664 953.649 126.977C953.584 168.577 953.816 210.177 953.406 251.771C953.265 266.03 946.019 277.358 934.446 285.372C923.955 292.636 911.522 294.93 899.1 295.403C882.474 296.031 865.648 296.315 849.377 284.88V329.435C849.377 339.729 847.276 345.432 837.327 350.147C821.038 357.869 805.035 367.878 788.108 374.111V84.3447H850.499V92.2861C856.863 85.9514 865.925 83.1293 874.717 81.4169ZM871.771 133.924C852.586 133.924 850.446 145.705 850.446 151.29C850.446 163.407 850.508 175.525 850.508 187.643V219.814C850.508 224.294 850.846 240.727 871.861 240.727C892.875 240.726 891.763 223.331 891.763 219.812C891.763 216.294 891.748 154.266 891.748 151.283C891.748 148.299 890.956 133.924 871.771 133.924Z" fill="#01467F"/> <path d="M362.478 290.801L299.876 290.837V283.899C299.876 241.161 300.098 156.255 300.098 156.255C300.098 156.255 301.303 136.355 279.488 136.355C257.673 136.355 258.466 155.709 258.466 155.709V290.866L196.975 291.017V0.882629C204.604 4.25424 211.868 7.33908 219.019 10.6649C225.101 13.4918 228.321 14.8826 237.097 19.4884C252.252 27.4416 258.087 38.4522 258.415 54.0393C258.66 65.6999 258.466 77.3674 258.466 90.0495C269.469 80.7135 281.811 80.5778 293.967 80.5778C309.435 80.5778 319.199 80.5778 331.435 84.8945C349.021 91.0981 358.903 103.315 361.387 121.555C362.158 127.247 362.423 133.046 362.434 138.794C362.512 187.129 362.478 235.466 362.478 283.804V290.801Z" fill="#01467F"/> <path d="M1038.03 83.0687C1058.9 79.2339 1079.76 78.7804 1100.43 84.8821C1125.74 92.3567 1140.85 109.504 1146.09 134.768C1148.64 147.022 1149.33 159.82 1149.47 172.385C1149.94 212.911 1149.62 253.442 1149.62 293.969C1129.19 293.969 1086.8 293.969 1086.8 293.969V284.899C1086.8 284.899 1075.19 292.028 1063.23 293.118C1046.04 294.686 1028.81 294.773 1014.57 289.174C998.576 282.889 984.517 268.408 984.283 251.342C984.122 239.589 984.389 227.727 984.389 215.973C984.389 193.363 996.001 167.223 1021.09 167.223C1046.17 167.223 1085.34 167.223 1085.34 167.223V146.578C1085.34 142.789 1083.56 134.073 1065.15 134.073C1046.81 134.073 1046.18 146.544 1046.17 146.628C1046.17 146.628 1004.9 146.628 984.263 146.628C986.048 129.735 991.313 114.136 1002.7 101.775C1012.11 91.553 1024.41 85.5698 1038.03 83.0687ZM1085.49 208.252C1085.49 208.252 1067.11 208.252 1057.61 208.252C1048.1 208.252 1045.2 215.734 1044.93 223.877C1044.68 231.123 1047.85 238.354 1056.67 240.177C1065.5 242 1085.42 242.904 1085.42 230.683C1085.42 225.009 1085.49 208.252 1085.49 208.252Z" fill="#01467F"/> <path d="M450.492 82.7717C472.976 78.9538 495.268 78.7337 516.972 87.1008C538.835 95.5307 552.712 111.806 556.414 134.243C559.84 155.014 558.886 208.41 558.886 208.41H455.79C455.79 215.219 455.904 222.527 457.424 229.181C459.264 237.227 466.082 240.88 473.672 241.626C481.644 242.408 489.227 240.835 494.386 233.403C495.639 231.595 496.292 229.548 496.296 229.534L558.886 229.569C546.92 263.672 527.021 292.594 488.891 295.171C488.782 295.18 451.489 298.365 432.151 287.186C413.621 276.473 404.32 264.911 397.734 244.567C394.344 234.096 394.346 225.969 394.147 216.616C393.673 194.232 392.997 171.766 394.339 149.447C396.351 115.983 413.945 88.9821 450.492 82.7717ZM475.903 134.26C454.749 134.26 455.861 150.243 455.861 151.321C455.861 157.169 455.746 161.261 455.746 166.541H497.103C497.103 160.485 497.063 157.245 497.063 151.288C497.063 147.777 497.057 134.26 475.903 134.26Z" fill="#01467F"/> <path d="M104.307 208.883H166.621C166.621 237.041 162.022 262.332 138.069 280.59C127.603 288.567 115.289 292.324 102.544 294.275C85.3204 296.912 68.1663 296.126 51.1975 291.686C26.2912 285.163 10.5507 269.337 4.18821 244.707C1.80972 235.5 0.780169 225.749 0.574599 216.212C0.0955023 194.025 -0.522906 171.755 0.766578 149.629C2.73393 115.897 20.4537 89.0636 56.7037 82.7989C79.3571 78.8843 101.871 78.6587 123.694 87.1937C145.449 95.705 157.936 112.298 162.783 134.546C165.008 144.771 166.234 155.383 166.616 166.405H104.297C104.297 162.825 104.142 155.498 104.142 155.498C104.142 155.498 104.278 134.778 83.0121 134.934C61.7465 135.09 62.3288 154.327 62.3288 154.327L62.39 221.518C62.39 221.518 62.0074 243.107 82.0948 242.798C102.182 242.489 103.489 221.518 103.489 221.518C103.489 221.518 104.256 213.653 104.307 208.883Z" fill="#01467F"/> <path d="M1330.33 278.654C1331.01 277.323 1330.07 275.737 1328.57 275.737C1318.61 275.737 1288.36 275.737 1272.49 275.737C1249.18 275.534 1244.55 285.196 1243.95 286.751C1243.9 286.894 1243.84 287.029 1243.77 287.163L1206.06 355.006C1205.32 356.339 1206.28 357.977 1207.81 357.977H1237.25C1237.25 357.977 1258.25 358.057 1267.8 357.875C1274.54 357.746 1281.64 357.584 1287.77 353.229C1295.83 347.503 1303.19 331.132 1303.19 331.132L1330.33 278.654Z" fill="#01467F"/> <path d="M1167.89 86.4507C1167.89 86.4507 1204.05 86.508 1222.1 86.508C1245.97 86.2874 1255.07 102.448 1261 121.443C1275.33 167.307 1302.03 260.99 1302.03 260.99H1277.52C1277.52 260.99 1257.75 261.071 1248.75 260.888C1242.41 260.759 1235.73 260.598 1229.96 256.242C1222.36 250.517 1218.36 242.542 1215.43 234.146C1198.4 185.322 1183.23 135.821 1167.89 86.4507Z" fill="#219D73"/> <path d="M1421.94 6C1413.38 33.5429 1404.88 61.1264 1396.07 88.6104C1380.96 137.265 1365.99 186.034 1349.2 234.146C1346.27 242.542 1342.27 250.517 1334.68 256.242C1329.52 260.134 1323.64 260.675 1317.92 260.839L1294.86 183.117C1288.47 161.566 1298.12 142.989 1304.75 121.743C1304.76 121.72 1304.77 121.697 1304.78 121.674C1312.85 93.9166 1322.17 62.2948 1328.83 40.9912C1334.76 21.997 1343.87 5.83606 1367.73 6.05664C1385.75 6.05664 1421.82 6.00019 1421.94 6Z" fill="#219D73"/> </svg>';
            $icon_html .= $cheqpay_svg;
        }

        $icon_html .= $visa_svg;
        $icon_html .= $mastercard_svg;
        $icon_html .= '</span>';

        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
        return apply_filters('woocommerce_gateway_icon', $icon_html, $this->id);
    }
}