<?php
/**
 * Plugin Name: Cheqpay Payment Gateway
 * Plugin URI: https://cheqpay.mx
 * Description: Accept payments via Cheqpay Link - redirect customers to Cheqpay HPP for secure payment processing.
 * Version: 2.2.22
 * Author: Cheqpay
 * License: GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: cheqpay-payment-gateway
 * WC requires at least: 5.0
 * WC tested up to: 9.0
 */

if (!defined('ABSPATH')) {
    exit;
}

define('CHEQPAY_LINK_VERSION', '2.2.22');
define('CHEQPAY_LINK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CHEQPAY_LINK_PLUGIN_PATH', plugin_dir_path(__FILE__));

// Enable/disable debug logging - set to false to disable all debug logs
define('CHEQPAY_DEBUG_MODE', true);

// API URL constants
define('CHEQPAY_LINK_API_URL_SANDBOX', 'https://api.cheqpay.dev');
define('CHEQPAY_LINK_API_URL_PRODUCTION', 'https://prod.cheqpay.mx');
define('CHEQPAY_LINK_HPP_URL_SANDBOX', 'https://pay.cheqpay.dev');
define('CHEQPAY_LINK_HPP_URL_PRODUCTION', 'https://pay.cheqpay.mx');

// VGS constants
define('CHEQPAY_VGS_VAULT_ID_SANDBOX', 'tntz1e5lzeb');
define('CHEQPAY_VGS_VAULT_ID_PRODUCTION', 'tntlmeijl9w');
define('CHEQPAY_VGS_CNAME', 'vault.cheqpay.mx');

/**
 * Main Plugin Class
 */
class Cheqpay_Link_Plugin {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        add_action('plugins_loaded', array($this, 'load_textdomain'));
        add_action('before_woocommerce_init', array($this, 'declare_compatibility'));
    }

    public function load_textdomain() {
        load_plugin_textdomain('cheqpay-payment-gateway', false, dirname(plugin_basename(__FILE__)) . '/languages/');
    }

    public function init() {
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }

        require_once CHEQPAY_LINK_PLUGIN_PATH . 'includes/class-cheqpay-logger.php';
        require_once CHEQPAY_LINK_PLUGIN_PATH . 'includes/class-cheqpay-link-gateway.php';

        add_filter('woocommerce_payment_gateways', array($this, 'add_gateway'));
        add_filter('query_vars', array($this, 'add_query_vars'));
        add_action('parse_request', array($this, 'handle_redirect'));
        add_action('woocommerce_blocks_loaded', array($this, 'register_block_support'));
        add_action('woocommerce_api_cheqpay_link', array($this, 'handle_webhook'));

        // Register AJAX handlers early to ensure they're available during AJAX requests
        // These must be registered here because the gateway may not be
        // instantiated when admin-ajax.php processes the request
        add_action('wp_ajax_cheqpay_authentication_request', array($this, 'ajax_handle_authentication_request'));
        add_action('wp_ajax_nopriv_cheqpay_authentication_request', array($this, 'ajax_handle_authentication_request'));
        add_action('wp_ajax_cheqpay_validate_auth', array($this, 'ajax_handle_validate_authentication'));
        add_action('wp_ajax_nopriv_cheqpay_validate_auth', array($this, 'ajax_handle_validate_authentication'));
    }

    /**
     * Get or create gateway instance for AJAX handlers
     */
    private function get_gateway_instance() {
        $gateways = WC()->payment_gateways()->payment_gateways();
        return $gateways['cheqpay_link'] ?? null;
    }

    /**
     * AJAX handler for 3DS authentication request
     */
    public function ajax_handle_authentication_request() {
        $gateway = $this->get_gateway_instance();
        if ($gateway) {
            $gateway->ajax_handle_authentication_request();
        } else {
            wp_send_json_error(array('message' => 'Payment gateway not available'), 500);
        }
    }

    /**
     * AJAX handler for 3DS validation
     */
    public function ajax_handle_validate_authentication() {
        $gateway = $this->get_gateway_instance();
        if ($gateway) {
            $gateway->ajax_handle_validate_authentication();
        } else {
            wp_send_json_error(array('message' => 'Payment gateway not available'), 500);
        }
    }

    public function register_block_support() {
        if (!class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
            return;
        }

        require_once CHEQPAY_LINK_PLUGIN_PATH . 'includes/class-cheqpay-link-blocks.php';

        add_action('woocommerce_blocks_payment_method_type_registration', function($payment_method_registry) {
            $payment_method_registry->register(new Cheqpay_Link_Blocks());
        });
    }

    public function declare_compatibility() {
        if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', __FILE__, true);
        }
    }

    public function add_gateway($gateways) {
        $gateways[] = 'WC_Gateway_Cheqpay_Link';
        return $gateways;
    }

    public function add_query_vars($vars) {
        $vars[] = 'hpp_return';
        $vars[] = 'order_id';
        $vars[] = 'cp_status';
        $vars[] = 'cp_invoice_id';
        return $vars;
    }

    public function handle_redirect() {
        if (isset($_GET['hpp_return']) && $_GET['hpp_return'] === '1') {
            $order_id = isset($_GET['order_id']) ? absint(wp_unslash($_GET['order_id'])) : 0;
            $invoice_id = isset($_GET['cp_invoice_id']) ? sanitize_text_field(wp_unslash($_GET['cp_invoice_id'])) : '';
            $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';

            Cheqpay_Logger::debug('HPP redirect received', $order_id, array(
                'invoice_id' => $invoice_id,
                'has_nonce' => !empty($nonce),
            ));

            if (!$order_id) {
                Cheqpay_Logger::warning('HPP redirect: Missing order ID');
                wp_safe_redirect(wc_get_page_permalink('shop'));
                exit;
            }

            // Verify nonce
            if (empty($nonce) || !wp_verify_nonce($nonce, 'cheqpay_return_' . $order_id)) {
                Cheqpay_Logger::error('HPP redirect: Security check failed - invalid nonce', $order_id);
                wp_die(esc_html__('ERROR_SECURITY_CHECK_FAILED', 'cheqpay-payment-gateway'));
            }

            $order = wc_get_order($order_id);
            if (!$order) {
                Cheqpay_Logger::error('HPP redirect: Order not found', $order_id);
                wp_safe_redirect(wc_get_page_permalink('shop'));
                exit;
            }

            // Get gateway instance to access settings
            $gateway = WC()->payment_gateways()->payment_gateways()['cheqpay_link'] ?? null;
            if (!$gateway) {
                Cheqpay_Logger::error('HPP redirect: Gateway instance not found', $order_id);
                wp_safe_redirect(wc_get_page_permalink('shop'));
                exit;
            }

            // Verify payment status from Cheqpay API
            Cheqpay_Logger::debug('Verifying payment invoice status', $order_id, array('invoice_id' => $invoice_id));
            ['status' => $payment_status, 'payment_id' => $payment_id] = $this->verify_payment_invoice_status($invoice_id, $gateway);

            Cheqpay_Logger::log_payment_status_change($order_id, $order->get_status(), $payment_status, 'redirect');

            // Update order based on verified status
            $this->update_order_status($order, $payment_id, $payment_status, 'redirect');
            $order->save();

            // Redirect user based on status
            if ($payment_status === 'success') {
                Cheqpay_Logger::info('Payment completed successfully via HPP redirect', $order_id, array('payment_id' => $payment_id));
                wc_add_notice(__('PAYMENT_COMPLETED_SUCCESS', 'cheqpay-payment-gateway'), 'success');
                wp_safe_redirect($order->get_checkout_order_received_url());
            } else if ($payment_status === 'pending') {
                Cheqpay_Logger::info('Payment pending via HPP redirect', $order_id, array('payment_id' => $payment_id));
                wc_add_notice(__('PAYMENT_PENDING_PROCESSING', 'cheqpay-payment-gateway'), 'notice');
                wp_safe_redirect($order->get_checkout_order_received_url());
            } else if ($payment_status === 'refunded') {
                Cheqpay_Logger::info('Payment refunded via HPP redirect', $order_id, array('payment_id' => $payment_id));
                wc_add_notice(__('PAYMENT_REFUNDED', 'cheqpay-payment-gateway'), 'notice');
                wp_safe_redirect($order->get_checkout_order_received_url());
            } else {
                Cheqpay_Logger::error('Payment verification failed via HPP redirect', $order_id, array(
                    'payment_id' => $payment_id,
                    'status' => $payment_status,
                ));
                wc_add_notice(__('PAYMENT_FAILED_VERIFICATION', 'cheqpay-payment-gateway'), 'error');
                wp_safe_redirect(wc_get_checkout_url());
            }
            exit;
        }
    }

    /**
     * Update order status based on payment verification
     *
     * @param WC_Order $order Order object
     * @param string $payment_id Payment ID
     * @param string $payment_status Payment status (success/pending/failed/refunded)
     * @param string $source Source of the update (redirect/webhook)
     */
    private function update_order_status($order, $payment_id, $payment_status, $source = 'unknown') {
        $source_label = ($source === 'webhook') ? 'webhook' : 'redirect';

        switch ($payment_status) {
            case 'success':
                if ($order->get_status() !== 'completed' && $order->get_status() !== 'processing') {
                    $order->payment_complete($payment_id);
                    $order->add_order_note(
                        sprintf(
                            /* translators: 1: source label (redirect or webhook), 2: payment ID */
                            __('NOTE_PAYMENT_COMPLETED', 'cheqpay-payment-gateway'),
                            $source_label,
                            $payment_id
                        )
                    );
                }
                break;

            case 'pending':
                if ($order->get_status() === 'pending') {
                    $order->update_status('on-hold',
                        sprintf(
                            /* translators: 1: source label (redirect or webhook), 2: payment ID */
                            __('NOTE_PAYMENT_PENDING', 'cheqpay-payment-gateway'),
                            $source_label,
                            $payment_id
                        )
                    );
                }
                break;

            case 'refunded':
                if ($order->get_status() !== 'refunded') {
                    $order->update_status('refunded',
                        sprintf(
                            /* translators: 1: source label (redirect or webhook), 2: payment ID */
                            __('NOTE_PAYMENT_REFUNDED', 'cheqpay-payment-gateway'),
                            $source_label,
                            $payment_id
                        )
                    );
                }
                break;

            case 'failed':
            default:
                if ($order->get_status() === 'pending' || $order->get_status() === 'on-hold') {
                    $order->update_status('failed',
                        sprintf(
                            /* translators: 1: source label (redirect or webhook), 2: payment ID */
                            __('NOTE_PAYMENT_VERIFICATION_FAILED', 'cheqpay-payment-gateway'),
                            $source_label,
                            $payment_id
                        )
                    );
                }
                break;
        }
    }

    private function verify_payment_order_status($payment_id, $gateway) {
        $environment = $gateway->get_option('environment', 'sandbox');
        $base_url = ($environment === 'production') ? CHEQPAY_LINK_API_URL_PRODUCTION : CHEQPAY_LINK_API_URL_SANDBOX;
        $api_url = $base_url . '/pos/v2/payment-orders/' . $payment_id;

        Cheqpay_Logger::log_api_request($api_url, 'GET', array(), null);

        $response = wp_remote_get($api_url, array(
            'headers' => array(
                'x-api-key' => $gateway->get_option('secret_key'),
                'User-Agent' => 'Cheqpay/' . CHEQPAY_LINK_VERSION . ' WordPress/' . get_bloginfo('version') . ' WooCommerce/' . WC()->version
            ),
            'timeout' => 15
        ));

        if (is_wp_error($response)) {
            Cheqpay_Logger::error('Payment order verification failed - WP Error', null, array(
                'payment_id' => $payment_id,
                'error' => $response->get_error_message(),
            ));
            return array(
                'status' => 'failed',
                'order_id' => ''
            );
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        Cheqpay_Logger::log_api_response($api_url, $response_code, $body, null);

        if ($response_code !== 200) {
            Cheqpay_Logger::error('Payment order verification failed - HTTP error', null, array(
                'payment_id' => $payment_id,
                'response_code' => $response_code,
            ));
            return array(
                'status' => 'failed',
                'order_id' => ''
            );
        }

        $data = json_decode($body, true);

        if (!$data || !isset($data['status'])) {
            Cheqpay_Logger::error('Payment order verification failed - Invalid response', null, array(
                'payment_id' => $payment_id,
            ));
            return array(
                'status' => 'failed',
                'order_id' => ''
            );
        }

        $order_id = $data['merchantReference'];
        $status = strtolower($data['status']);

        Cheqpay_Logger::debug('Payment order status retrieved', $order_id, array(
            'payment_id' => $payment_id,
            'api_status' => $status,
        ));

        if (in_array($status, array('completed', 'partially_paid'))) {
            return array(
                'status' => 'success',
                'order_id' => $order_id,
                'payment_id' => $payment_id,
            );
        } else if (in_array($status, array(
            'pending',
            'processing',
            'authorized',
            'partially_authorized',
            'action_required',
            'cancellation_requested',
            'payer_authentication_device_data_required',
            'payer_authentication_challenge_required'
        ))) {
            return array(
                'status' => 'pending',
                'order_id' => $order_id,
                'payment_id' => $payment_id,
            );
        } else if (in_array($status, array('refund_processing', 'partially_refunded', 'refunded'))) {
            return array(
                'status' => 'refunded',
                'order_id' => $order_id,
                'payment_id' => $payment_id,
            );
         } else {
            return array(
                'status' => 'failed',
                'order_id' => $order_id,
                'payment_id' => $payment_id,
            );
        }
    }

    private function verify_payment_invoice_status($invoice_id, $gateway) {
        $environment = $gateway->get_option('environment', 'sandbox');
        $base_url = ($environment === 'production') ? CHEQPAY_LINK_API_URL_PRODUCTION : CHEQPAY_LINK_API_URL_SANDBOX;
        $api_url = $base_url . '/lps/invoices/' . $invoice_id;

        Cheqpay_Logger::log_api_request($api_url, 'GET', array(), null);

        $response = wp_remote_get($api_url, array(
            'headers' => array(
                'x-api-key' => $gateway->get_option('secret_key'),
                'User-Agent' => 'Cheqpay/' . CHEQPAY_LINK_VERSION . ' WordPress/' . get_bloginfo('version') . ' WooCommerce/' . WC()->version
            ),
            'timeout' => 15
        ));

        if (is_wp_error($response)) {
            Cheqpay_Logger::error('Invoice verification failed - WP Error', null, array(
                'invoice_id' => $invoice_id,
                'error' => $response->get_error_message(),
            ));
            return array(
                'status' => 'failed',
                'order_id' => ''
            );
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        Cheqpay_Logger::log_api_response($api_url, $response_code, $body, null);

        if ($response_code !== 200) {
            Cheqpay_Logger::error('Invoice verification failed - HTTP error', null, array(
                'invoice_id' => $invoice_id,
                'response_code' => $response_code,
            ));
            return array(
                'status' => 'failed',
                'order_id' => ''
            );
        }

        $data = json_decode($body, true);

        if (!$data || !isset($data['status'])) {
            Cheqpay_Logger::error('Invoice verification failed - Invalid response', null, array(
                'invoice_id' => $invoice_id,
            ));
            return array(
                'status' => 'failed',
                'order_id' => ''
            );
        }

        $order_id = $data['reference'];
        $payment_id = $data['paymentOrderId'];
        $status = strtolower($data['status']);

        Cheqpay_Logger::debug('Invoice status retrieved', $order_id, array(
            'invoice_id' => $invoice_id,
            'payment_id' => $payment_id,
            'api_status' => $status,
        ));

        if (in_array($status, array('paid'))) {
            return array(
                'status' => 'success',
                'order_id' => $order_id,
                'payment_id' => $payment_id,
            );
        } else if (in_array($status, array('pending'))) {
            return array(
                'status' => 'pending',
                'order_id' => $order_id,
                'payment_id' => $payment_id,
            );
        } else if (in_array($status, array('refunded'))) {
            return array(
                'status' => 'refunded',
                'order_id' => $order_id,
                'payment_id' => $payment_id,
            );
         } else {
            return array(
                'status' => 'failed',
                'order_id' => $order_id,
                'payment_id' => $payment_id,
            );
        }
    }

    /**
     * Handle incoming webhook from Cheqpay
     */
    public function handle_webhook() {
        $payload = file_get_contents('php://input');
        $data = json_decode($payload, true);

        Cheqpay_Logger::debug('Webhook received', null, array(
            'payload_length' => strlen($payload),
        ));

        // Validate JSON structure
        if (!is_array($data)) {
            Cheqpay_Logger::error('Webhook rejected - Invalid JSON payload');
            http_response_code(400);
            echo wp_json_encode(array('error' => 'Invalid JSON payload'));
            exit;
        }

        // @todo sanitize webhook data
        // $data = $this->sanitize_webhook_data($data);

        // Get gateway instance
        $gateway = WC()->payment_gateways()->payment_gateways()['cheqpay_link'] ?? null;
        if (!$gateway) {
            Cheqpay_Logger::error('Webhook rejected - Gateway instance not found');
            http_response_code(500);
            exit;
        }

        // Verify webhook signature
        if (!$this->verify_webhook_signature($data, $gateway)) {
            Cheqpay_Logger::error('Webhook rejected - Invalid signature', null, array(
                'webhook_id' => $data['id'] ?? 'unknown',
            ));
            http_response_code(401);
            echo wp_json_encode(array('error' => 'Invalid signature'));
            exit;
        }

        $webhook_id = $data['id'] ?? '';
        $event_type = $data['event'] ?? 'unknown';
        $merchant_reference = $data['data']['paymentOrder']['merchantReference'] ?? '';
        $payment_id = sanitize_text_field($data['data']['paymentOrder']['id'] ?? '');
        $invoice_id = str_replace('plnk_', '', sanitize_text_field($merchant_reference));

        Cheqpay_Logger::log_webhook($event_type, array(
            'webhook_id' => $webhook_id,
            'merchant_reference' => $merchant_reference,
            'payment_id' => $payment_id,
        ), null);

        // Verify payment status from API (single source of truth)
        ['status' => $payment_status, 'order_id' => $order_id] = strpos($merchant_reference, 'plnk_') === 0
            ? $this->verify_payment_invoice_status($invoice_id, $gateway)
            : $this->verify_payment_order_status($payment_id, $gateway);

        $order = wc_get_order($order_id);
        if (!$order) {
            Cheqpay_Logger::error('Webhook processing failed - Order not found', null, array(
                'order_id' => $order_id,
                'webhook_id' => $webhook_id,
            ));
            http_response_code(404);
            echo wp_json_encode(array('error' => 'Order not found'));
            exit;
        }

        // Check for duplicate webhook (idempotency)
        $processed_events = $order->get_meta('_cheqpay_processed_webhook_events', true);
        if (!is_array($processed_events)) {
            $processed_events = array();
        }

        // Validate and sanitize webhook ID before storing
        $event_key = sanitize_text_field($webhook_id);
        if (empty($event_key)) {
            Cheqpay_Logger::error('Webhook rejected - Invalid webhook ID', $order_id);
            http_response_code(400);
            echo wp_json_encode(array('error' => 'Invalid webhook ID'));
            exit;
        }

        if (in_array($event_key, $processed_events, true)) {
            Cheqpay_Logger::debug('Webhook skipped - Already processed', $order_id, array(
                'webhook_id' => $webhook_id,
            ));
            http_response_code(200);
            echo wp_json_encode(array('success' => true, 'message' => 'Already processed'));
            exit;
        }

        Cheqpay_Logger::log_payment_status_change($order_id, $order->get_status(), $payment_status, 'webhook');

        // Update order based on verified status
        $this->update_order_status($order, $payment_id, $payment_status, 'webhook');

        // Mark event as processed (limit array size to prevent memory issues)
        $processed_events[] = $event_key;
        $processed_events = array_slice($processed_events, -100); // Keep only last 100 events
        $order->update_meta_data('_cheqpay_processed_webhook_events', $processed_events);
        $order->save();

        Cheqpay_Logger::info('Webhook processed successfully', $order_id, array(
            'webhook_id' => $webhook_id,
            'event_type' => $event_type,
            'payment_status' => $payment_status,
        ));

        http_response_code(200);
        echo wp_json_encode(array('success' => true));
        exit;
    }

    /**
     * Sanitize webhook data recursively
     */
    private function sanitize_webhook_data($data) {
        if (!is_array($data)) {
            return sanitize_text_field($data);
        }

        $sanitized = array();
        foreach ($data as $key => $value) {
            $sanitized_key = sanitize_key($key);

            if (is_array($value)) {
                $sanitized[$sanitized_key] = $this->sanitize_webhook_data($value);
            } else if (is_numeric($value)) {
                // Preserve numeric values
                $sanitized[$sanitized_key] = $value;
            } else {
                $sanitized[$sanitized_key] = sanitize_text_field($value);
            }
        }

        return $sanitized;
    }

    /**
     * Verify webhook signature
     */
    private function verify_webhook_signature($data, $gateway) {
        $signature_header = isset($_SERVER['HTTP_X_WEBHOOK_SIGNATURE']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_X_WEBHOOK_SIGNATURE'])) : '';

        if (empty($signature_header)) {
            return false;
        }

        $webhook_secret = $gateway->get_option('webhook_secret');

        if (empty($webhook_secret)) {
            return false;
        }

        $payment_method_id = $data['data']['paymentMethod']['options']['card']['id'] ?? '';
        $amount = $data['data']['amount'] ?? '';
        $currency = $data['data']['currency'] ?? '';
        $event = $data['event'] ?? '';

        // Build payload string: paymentMethodId|amount|currency|event
        $payload_parts = array($payment_method_id, $amount, $currency, $event);
        $payload_string = implode('|', $payload_parts);

        $calculated_signature = hash_hmac('sha256', $payload_string, $webhook_secret);

        return hash_equals($calculated_signature, $signature_header);
    }

    public function woocommerce_missing_notice() {
        echo '<div class="error"><p><strong>' . esc_html__('LABEL_CHEQPAY_LINK', 'cheqpay-payment-gateway') . '</strong> ' . esc_html__('DESCRIPTION_REQUIRES_WOOCOMMERCE', 'cheqpay-payment-gateway') . '</p></div>';
    }
}

Cheqpay_Link_Plugin::get_instance();