(function() {
    const { registerPaymentMethod } = window.wc.wcBlocksRegistry;
    const { getSetting } = window.wc.wcSettings;
    const { createElement, useState, useEffect, useRef, useCallback } = window.wp.element;

    const getCheqpaySettings = () => {
        const settings = getSetting('cheqpay_link_data', {});
        return {
            title: settings.title || 'Cheqpay Link',
            description: settings.description || '',
            icons: settings.icons || [],
            paymentMethod: settings.paymentMethod || 'payment_link',
            supports: settings.supports || ['products'],
            vgsVaultId: settings.vgsVaultId || '',
            vgsEnvironment: settings.vgsEnvironment || 'sandbox',
            vgsCname: settings.vgsCname || '',
            iframe: settings.iframe || {},
            i18n: settings.i18n
        };
    };

    const Label = () => {
        const { title, icons } = getCheqpaySettings();

        return createElement(
            'div',
            {
                style: {
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'space-between',
                    width: '100%'
                }
            },
            createElement('span', null, title),
            icons.length > 0 && createElement(
                'div',
                {
                    style: {
                        display: 'flex',
                        alignItems: 'center',
                        gap: '3px',
                        marginLeft: 'auto'
                    }
                },
                icons.map((icon, index) =>
                    createElement('img', {
                        key: icon.id || index,
                        src: icon.src,
                        alt: icon.alt || '',
                        style: icon.id !== 'cheqpay' ? {
                            height: '52px',
                            width: 'auto',
                            maxHeight: '52px'
                        }: {
                            height: '30px',
                            width: 'auto',
                            maxHeight: '30px'
                        }
                    })
                )
            )
        );
    };

// Helper functions for error display
    const getFieldNameFromKey = (fieldKey) => {
        if (fieldKey.includes('card_number') || fieldKey.includes('cardNumber')) {
            return 'card-number';
        }
        if (fieldKey.includes('card_exp') || fieldKey.includes('cardExp')) {
            return 'card-expiry';
        }
        if (fieldKey.includes('card_cvc') || fieldKey.includes('cardCvc')) {
            return 'card-cvc';
        }
        return null;
    };

    const updateErrorDisplay = (fieldName, field) => {
        const errorElement = document.getElementById('cheqpay-' + fieldName + '-error');
        const fieldElement = document.getElementById('cheqpay-' + fieldName);

        if (!errorElement || !fieldElement) {
            return;
        }

        const { i18n } = getCheqpaySettings();

        const isValid = field.isValid === true;
        const isDirty = field.isDirty === true;
        const hasValue = !field.isEmpty;

        // Show error if invalid and (dirty or has value)
        const shouldShowError = !isValid && (isDirty || hasValue);

        // Custom error messages based on field type
        let customErrorMessage = '';
        if (shouldShowError) {
            if (fieldName === 'card-number') {
                customErrorMessage = i18n.ERROR_CARD_NUMBER_INVALID;
            } else if (fieldName === 'card-expiry') {
                customErrorMessage = i18n.ERROR_CARD_EXPIRY_INVALID;
            } else if (fieldName === 'card-cvc') {
                customErrorMessage = i18n.ERROR_CARD_CVC_INVALID;
            }
        }

        if (shouldShowError && customErrorMessage) {
            errorElement.textContent = customErrorMessage;
            errorElement.classList.add('show');
            fieldElement.classList.add('error');
            fieldElement.classList.remove('valid');
        } else if (isValid && hasValue) {
            errorElement.textContent = '';
            errorElement.classList.remove('show');
            fieldElement.classList.remove('error');
            fieldElement.classList.add('valid');
        } else {
            errorElement.textContent = '';
            errorElement.classList.remove('show');
            fieldElement.classList.remove('error', 'valid');
        }
    };

// VGS Card Fields Component
    const VGSCardFields = ({ onVGSFormReady, vgsFormRef }) => {
        const settings = getCheqpaySettings();
        const { vgsVaultId, vgsEnvironment, vgsCname, iframe: iframeSettings, i18n } = settings;
        const [fieldsLoaded, setFieldsLoaded] = useState(false);
        const [error, setError] = useState(null);

        useEffect(() => {
            // Check if VGSCollect is available
            if (typeof VGSCollect === 'undefined') {
                setError(i18n.PAYMENT_FAILED_FORM_NOT_INITIALIZED);
                return;
            }

            // Check if container exists and has no iframe (needs initialization)
            const container = document.getElementById('cheqpay-card-number');
            const hasExistingIframe = container && container.querySelector('iframe');

            // Don't reinitialize if already done and iframes exist
            if (vgsFormRef.current && hasExistingIframe) {
                return;
            }

            // Reset form ref if we need to reinitialize
            vgsFormRef.current = null;
            setFieldsLoaded(false);

            // Small delay to ensure DOM is ready
            const initTimeout = setTimeout(() => {
                try {
                    const form = VGSCollect.create(vgsVaultId, vgsEnvironment, function(state) {
                        // Form state callback - update error displays
                        Object.entries(state).forEach(function([fieldKey, field]) {
                            const fieldName = getFieldNameFromKey(fieldKey);
                            if (fieldName) {
                                updateErrorDisplay(fieldName, field);
                            }
                        });
                    });

                    if (vgsEnvironment === 'live' && vgsCname) {
                        form.useCname(vgsCname);
                    }

                    // Common CSS for VGS fields - colors from admin settings
                    const iframeColors = iframeSettings || {};
                    const fieldCss = {
                        position: "absolute",
                        top: "0",
                        left: "0",
                        width: "100%",
                        height: "100%",
                        maxWidth: "100%",
                        maxHeight: "100%",
                        background: iframeColors.colorBackground || "white",
                        color: iframeColors.colorText || "#333333",
                        padding: "8px 12px",
                        border: "none",
                        margin: "0",
                        boxSizing: "border-box",
                        outline: "none",
                        fontSize: "14px",
                        lineHeight: "20px",
                        overflow: "hidden",
                        borderRadius: iframeColors.radiusBorder || "8px",
                        "&.invalid.touched": {
                            border: '1px solid #DC2626',
                        },
                        "&:focus": {
                            border: '1px solid ' + (iframeColors.colorBorderSelected || '#0F172A'),
                            boxShadow: '0 0 0 3px #7DD3FC',
                            margin: "2px",
                            // Responsible for compensating for the extra margin size
                            padding: "8px 9px",
                            maxWidth: "calc(100% - 4px)",
                            maxHeight: "calc(100% - 4px)",
                        },
                        "&::placeholder": { color: iframeColors.colorPlaceholder || '#64748B' }
                    };

                    // Card Number Field - no validations to avoid VGS red border
                    const cardNumberField = form.field('#cheqpay-card-number', {
                        type: 'card-number',
                        name: 'card_number',
                        placeholder: i18n.PLACEHOLDER_CARD_NUMBER,
                        showCardIcon: true,
                        validations: ['required', 'validCardNumber'],
                        css: fieldCss
                    });

                    // Expiry Field
                    const expiryField = form.field('#cheqpay-card-expiry', {
                        type: 'card-expiration-date',
                        name: 'card_exp',
                        placeholder: i18n.PLACEHOLDER_EXPIRY_DATE,
                        yearLength: 2,
                        validations: ['required', 'validCardExpirationDate'],
                        css: fieldCss
                    });

                    // CVC Field
                    const cvcField = form.field('#cheqpay-card-cvc', {
                        type: 'card-security-code',
                        name: 'card_cvc',
                        placeholder: i18n.PLACEHOLDER_CVC,
                        maxLength: 4,
                        validations: ['required', 'validCardSecurityCode'],
                        css: fieldCss
                    });

                    // Store form reference
                    vgsFormRef.current = form;

                    // Wait for all fields to load
                    Promise.all([cardNumberField.promise, expiryField.promise, cvcField.promise])
                        .then(function() {
                            setFieldsLoaded(true);
                            if (onVGSFormReady) {
                                onVGSFormReady(form);
                            }
                        })
                        .catch(function(err) {
                            console.error('Error loading VGS fields:', err);
                            setError(i18n.PAYMENT_FAILED_FORM_NOT_INITIALIZED);
                        });

                } catch (err) {
                    console.error('Error initializing VGS:', err);
                    setError(i18n.PAYMENT_FAILED_FORM_NOT_INITIALIZED);
                }
            }, 100);

            return () => {
                clearTimeout(initTimeout);
                // Reset form ref on cleanup so it reinitializes on remount
                vgsFormRef.current = null;
            };
        }, [vgsVaultId, vgsEnvironment, onVGSFormReady, vgsFormRef]);

        const iColors = iframeSettings || {};
        const containerStyle = {
            position: 'relative',
            width: '100%',
            height: '40px',
            background: iColors.colorBackground || 'white',
            border: '1px solid ' + (iColors.colorBorder || '#e0e0e0'),
            borderRadius: iColors.radiusBorder || '8px',
            boxSizing: 'border-box'
        };

        const labelStyle = {
            display: 'block',
            marginBottom: '8px',
            fontWeight: '500',
            fontSize: '14px',
            color: iColors.colorText || '#333'
        };

        const rowStyle = {
            marginBottom: '15px'
        };

        const halfRowStyle = {
            display: 'grid',
            gridTemplateColumns: '1fr 1fr',
            gap: '16px'
        };

        const errorMessageStyle = {
            color: '#ef4444',
            fontSize: '12px',
            marginTop: '4px',
            minHeight: '16px',
            lineHeight: '16px',
            fontWeight: '400'
        };

        if (error) {
            return createElement(
                'div',
                {
                    style: {
                        padding: '12px',
                        background: '#fef1f1',
                        border: '1px solid #e2401c',
                        borderRadius: '4px',
                        color: '#e2401c'
                    }
                },
                error
            );
        }

        return createElement(
            'div',
            {
                className: 'cheqpay-vgs-fields',
                style: { padding: '12px 0' }
            },
            createElement('style', null, `
            .cheqpay-vgs-fields .cheqpay-element {
                overflow: hidden;
            }
            .cheqpay-vgs-fields .cheqpay-element iframe {
                width: 100% !important;
                height: 100% !important;
                border: none !important;
            }
            .cheqpay-vgs-fields .cheqpay-element.error {
                border-color: #ef4444 !important;
            }
            .cheqpay-vgs-fields .cheqpay-element.valid {
                border-color: #10b981 !important;
            }
            .cheqpay-vgs-fields .error-message {
                display: none;
            }
            .cheqpay-vgs-fields .error-message.show {
                display: block;
            }
        `),
            // Card Number
            createElement(
                'div',
                { style: rowStyle },
                createElement('label', { style: labelStyle }, i18n.LABEL_CARD_NUMBER, createElement('span', { style: { color: 'red' } }, ' *')),
                createElement('div', {
                    id: 'cheqpay-card-number',
                    className: 'cheqpay-element',
                    style: containerStyle
                }),
                createElement('div', {
                    id: 'cheqpay-card-number-error',
                    className: 'error-message',
                    style: errorMessageStyle
                })
            ),
            // Expiry & CVC
            createElement(
                'div',
                { style: halfRowStyle },
                createElement(
                    'div',
                    null,
                    createElement('label', { style: labelStyle }, i18n.LABEL_EXPIRY_DATE, createElement('span', { style: { color: 'red' } }, ' *')),
                    createElement('div', {
                        id: 'cheqpay-card-expiry',
                        className: 'cheqpay-element',
                        style: containerStyle
                    }),
                    createElement('div', {
                        id: 'cheqpay-card-expiry-error',
                        className: 'error-message',
                        style: errorMessageStyle
                    })
                ),
                createElement(
                    'div',
                    null,
                    createElement('label', { style: labelStyle }, i18n.LABEL_CVC, createElement('span', { style: { color: 'red' } }, ' *')),
                    createElement('div', {
                        id: 'cheqpay-card-cvc',
                        className: 'cheqpay-element',
                        style: containerStyle
                    }),
                    createElement('div', {
                        id: 'cheqpay-card-cvc-error',
                        className: 'error-message',
                        style: errorMessageStyle
                    })
                )
            ),
            !fieldsLoaded && createElement(
                'div',
                {
                    style: {
                        textAlign: 'center',
                        padding: '10px',
                        color: '#666',
                        fontSize: '14px'
                    }
                },
                i18n.MESSAGE_LOADING_PAYMENT_FORM
            )
        );
    };

    const Content = (props) => {
        const { eventRegistration, emitResponse } = props;
        const { onPaymentSetup } = eventRegistration;
        const { paymentMethod, description, i18n } = getCheqpaySettings();

        const vgsFormRef = useRef(null);
        const [isVGSReady, setIsVGSReady] = useState(false);

        const handleVGSFormReady = useCallback((form) => {
            setIsVGSReady(true);
        }, []);

        // Register payment processing handler
        useEffect(() => {
            if (paymentMethod !== 'card_payment') return;

            const unsubscribe = onPaymentSetup(async () => {
                // Check if VGS form is ready
                if (!vgsFormRef.current) {
                    return {
                        type: emitResponse.responseTypes.ERROR,
                        message: i18n.PAYMENT_FAILED_FORM_LOADING,
                    };
                }

                // Tokenize via VGS
                return new Promise((resolve) => {
                    vgsFormRef.current.submit(
                        '/lps/card-tokens/hash',
                        {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json'
                            }
                        },
                        function(status, response) {
                            if (status >= 200 && status < 300 && response) {
                                resolve({
                                    type: emitResponse.responseTypes.SUCCESS,
                                    meta: {
                                        paymentMethodData: {
                                            cheqpay_card_number: response.card_number || '',
                                            cheqpay_card_expiry: response.card_exp || '',
                                            cheqpay_card_cvc: response.card_cvc || ''
                                        }
                                    }
                                });
                            } else {
                                var errorMessage = i18n.PAYMENT_FAILED_CARD_PROCESSING;
                                if (response && response.message) {
                                    errorMessage = response.message;
                                }
                                resolve({
                                    type: emitResponse.responseTypes.ERROR,
                                    message: errorMessage,
                                });
                            }
                        },
                        function(errors) {
                            console.error('VGS Validation Errors:', errors);
                            resolve({
                                type: emitResponse.responseTypes.ERROR,
                                message: i18n.PAYMENT_FAILED_CARD_DETAILS,
                            });
                        }
                    );
                });
            });

            return () => unsubscribe();
        }, [onPaymentSetup, emitResponse.responseTypes, paymentMethod, i18n]);

        if (paymentMethod === 'card_payment') {
            return createElement(VGSCardFields, {
                onVGSFormReady: handleVGSFormReady,
                vgsFormRef: vgsFormRef
            });
        }

        return createElement(
            'div',
            {
                className: 'cheqpay-payment-method-description',
                style: {
                    padding: '12px 0',
                    color: '#666',
                    fontSize: '14px',
                    lineHeight: '1.5'
                },
                dangerouslySetInnerHTML: { __html: description }
            }
        );
    };

    const initialSettings = getCheqpaySettings();

    const CheqpayLinkPayment = {
        name: 'cheqpay_link',
        label: createElement(Label),
        content: createElement(Content),
        edit: createElement(Content),
        canMakePayment: () => true,
        ariaLabel: initialSettings.title,
        supports: {
            features: initialSettings.supports,
        },
    };

    registerPaymentMethod(CheqpayLinkPayment);
})();
