jQuery(document).ready(function($) {
    // Only run if cheqpayParams is available
    if (typeof cheqpayParams === 'undefined') {
        return;
    }

    // VGS Configuration from PHP
    var vaultId = cheqpayParams.vgsVaultId;
    var vgsEnv = cheqpayParams.vgsEnvironment;
    var vgsCname = cheqpayParams.vgsCname;

    var vgsForm = null;
    var isTokenizing = false;
    var hasTokenized = false;
    var fieldsLoaded = 0;
    var totalFields = 3;

    // Common CSS for VGS fields - applied to the input element inside the iframe
    // Colors are configurable via admin settings, passed through cheqpayParams.iframe
    var iframeColors = cheqpayParams.iframe || {};
    var fieldCss = {
        position: "absolute",
        top: "0",
        left: "0",
        width: "100%",
        height: "100%",
        maxWidth: "100%",
        maxHeight: "100%",
        background: iframeColors.colorBackground || "white",
        color: iframeColors.colorText || "#333333",
        padding: "8px 12px",
        border: "none",
        margin: "0",
        boxSizing: "border-box",
        outline: "none",
        fontSize: "14px",
        lineHeight: "20px",
        overflow: "hidden",
        borderRadius: iframeColors.radiusBorder || "8px",
        "&.invalid.touched": {
            border: '1px solid #DC2626',
        },
        "&:focus": {
            border: '1px solid ' + (iframeColors.colorBorderSelected || '#0F172A'),
            boxShadow: '0 0 0 3px #7DD3FC',
            margin: "2px",
            // Responsible for compensating for the extra margin size
            padding: "8px 9px",
            maxWidth: "calc(100% - 4px)",
            maxHeight: "calc(100% - 4px)",
        },
        "&::placeholder": { color: iframeColors.colorPlaceholder || '#64748B' }
    };

    function getFieldNameFromKey(fieldKey) {
        if (fieldKey.includes('card_number') || fieldKey.includes('cardNumber')) {
            return 'card-number';
        }
        if (fieldKey.includes('card_exp') || fieldKey.includes('cardExp')) {
            return 'card-expiry';
        }
        if (fieldKey.includes('card_cvc') || fieldKey.includes('cardCvc')) {
            return 'card-cvc';
        }
        return null;
    }

    function updateErrorDisplay(fieldName, field) {
        const errorElement = document.getElementById('cheqpay-' + fieldName + '-error');
        const fieldElement = document.getElementById('cheqpay-' + fieldName);

        if (!errorElement || !fieldElement) {
            return;
        }

        const isValid = field.isValid === true;
        const isDirty = field.isDirty === true;
        const hasValue = !field.isEmpty;

        // Show error if invalid and (dirty or has value)
        const shouldShowError = !isValid && (isDirty || hasValue);

        // Custom error messages based on field type
        let customErrorMessage = '';
        if (shouldShowError) {
            if (fieldName === 'card-number') {
                customErrorMessage = cheqpayParams.i18n.ERROR_CARD_NUMBER_INVALID;
            } else if (fieldName === 'card-expiry') {
                customErrorMessage = cheqpayParams.i18n.ERROR_CARD_EXPIRY_INVALID;
            } else if (fieldName === 'card-cvc') {
                customErrorMessage = cheqpayParams.i18n.ERROR_CARD_CVC_INVALID;
            }
        }

        if (shouldShowError && customErrorMessage) {
            errorElement.textContent = customErrorMessage;
            errorElement.classList.add('show');
            fieldElement.classList.add('error');
            fieldElement.classList.remove('valid');
        } else if (isValid && hasValue) {
            errorElement.textContent = '';
            errorElement.classList.remove('show');
            fieldElement.classList.remove('error');
            fieldElement.classList.add('valid');
        } else {
            errorElement.textContent = '';
            errorElement.classList.remove('show');
            fieldElement.classList.remove('error', 'valid');
        }
    }

    function initVGSForm() {
        // Check if VGSCollect is loaded
        if (typeof VGSCollect === 'undefined') {
            console.error('VGS Collect.js not loaded');
            return;
        }

        // Check if containers exist
        var $cardContainer = $('#cheqpay-card-number');
        if ($cardContainer.length === 0) {
            return;
        }

        // Check if VGS iframes are already in the containers
        // If container exists but has no iframe, we need to reinitialize
        var hasExistingIframe = $cardContainer.find('iframe').length > 0;

        // Don't reinitialize if already done and iframes exist
        if (vgsForm !== null && hasExistingIframe) {
            return;
        }

        // Reset if we need to reinitialize
        vgsForm = null;
        fieldsLoaded = 0;

        vgsForm = VGSCollect.create(vaultId, vgsEnv, function(state) {
            Object.entries(state).forEach(function([fieldKey, field]) {
                const fieldName = getFieldNameFromKey(fieldKey);
                if (fieldName) {
                    updateErrorDisplay(fieldName, field);
                }
            });
        });

        if (vgsEnv === 'live' && vgsCname) {
            vgsForm.useCname(vgsCname);
        }

        // Card Number Field - no validations to avoid VGS red border
        var cardNumberField = vgsForm.field('#cheqpay-card-number', {
            type: 'card-number',
            name: 'card_number',
            placeholder: cheqpayParams.i18n.PLACEHOLDER_CARD_NUMBER,
            showCardIcon: true,
            validations: ['required', 'validCardNumber'],
            css: fieldCss
        });

        // Expiry Field
        var expiryField = vgsForm.field('#cheqpay-card-expiry', {
            type: 'card-expiration-date',
            name: 'card_exp',
            placeholder: cheqpayParams.i18n.PLACEHOLDER_EXPIRY_DATE,
            yearLength: 2,
            validations: ['required', 'validCardExpirationDate'],
            css: fieldCss
        });

        // CVC Field
        var cvcField = vgsForm.field('#cheqpay-card-cvc', {
            type: 'card-security-code',
            name: 'card_cvc',
            placeholder: cheqpayParams.i18n.PLACEHOLDER_CVC,
            maxLength: 4,
            validations: ['required', 'validCardSecurityCode'],
            css: fieldCss
        });

        // Track when fields are loaded
        Promise.all([cardNumberField.promise, expiryField.promise, cvcField.promise])
            .then(function() {
                fieldsLoaded = totalFields;
                console.log('VGS fields loaded successfully');
            })
            .catch(function(error) {
                console.error('Error loading VGS fields:', error);
            });
    }

    function showNotice(message, type) {
        var noticeClass = type === 'error' ? 'woocommerce-error' : 'woocommerce-message';
        var $notices = $('.woocommerce-notices-wrapper').first();

        if ($notices.length === 0) {
            $notices = $('<div class="woocommerce-notices-wrapper"></div>');
            $('form.checkout').prepend($notices);
        }

        $notices.html('<div class="' + noticeClass + '">' + message + '</div>');

        $('html, body').animate({
            scrollTop: $notices.offset().top - 100
        }, 500);
    }

    function clearNotices() {
        $('.woocommerce-notices-wrapper').html('');
    }

    // Initialize VGS form when checkout updates or payment method selected
    $(document.body).on('updated_checkout', function() {
        if ($('#payment_method_cheqpay_link').is(':checked')) {
            // Small delay to ensure DOM is ready
            setTimeout(initVGSForm, 200);
        }
    });

    $(document.body).on('payment_method_selected', function() {
        hasTokenized = false;

        if ($('#payment_method_cheqpay_link').is(':checked')) {
            setTimeout(initVGSForm, 200);
        }
    });

    // Initialize on page load if payment method is already selected
    if ($('#payment_method_cheqpay_link').is(':checked')) {
        setTimeout(initVGSForm, 500);
    }

    // Intercept checkout submission for our payment method
    $('form.checkout').on('checkout_place_order_cheqpay_link', function(e) {
        // If payment method is payment_link, skip VGS
        if (cheqpayParams.paymentMethod !== 'card_payment') {
            return true;
        }

        // If already tokenized, allow submission
        if (hasTokenized) {
            return true;
        }

        // Prevent double submissions
        if (isTokenizing) {
            return false;
        }

        // Check if VGS form is initialized
        if (!vgsForm) {
            showNotice(cheqpayParams.i18n.PAYMENT_FAILED_FORM_NOT_INITIALIZED, 'error');
            return false;
        }

        // Check if fields are loaded
        if (fieldsLoaded < totalFields) {
            showNotice(cheqpayParams.i18n.PAYMENT_FAILED_FORM_LOADING, 'error');
            return false;
        }

        isTokenizing = true;
        clearNotices();

        // Block UI
        $('form.checkout').block({
            message: null,
            overlayCSS: {
                background: '#fff',
                opacity: 0.6
            }
        });

        // Submit to VGS/Cheqpay to get encrypted card data
        vgsForm.submit(
            '/lps/card-tokens/hash',
            {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                }
            },
            function(status, response) {
                isTokenizing = false;

                if (status >= 200 && status < 300 && response) {
                    // Success - inject encrypted values into hidden fields
                    $('#cheqpay_card_number').val(response.card_number || '');
                    $('#cheqpay_card_expiry').val(response.card_exp || '');
                    $('#cheqpay_card_cvc').val(response.card_cvc || '');

                    hasTokenized = true;

                    // Re-trigger form submission
                    $('form.checkout').submit();
                } else {
                    // Error from API
                    $('form.checkout').unblock();

                    var errorMessage = cheqpayParams.i18n.PAYMENT_FAILED_CARD_PROCESSING;
                    if (response && response.message) {
                        errorMessage = response.message;
                    }

                    showNotice(errorMessage, 'error');
                }
            },
            function(errors) {
                // VGS validation errors
                isTokenizing = false;
                $('form.checkout').unblock();

                console.error('VGS Validation Errors:', errors);

                showNotice(cheqpayParams.i18n.PAYMENT_FAILED_CARD_DETAILS, 'error');
            }
        );

        return false; // Prevent form submission until tokenization completes
    });

    // Reset tokenization flag when checkout form is reset/updated
    $(document.body).on('checkout_error', function() {
        hasTokenized = false;
        isTokenizing = false;
    });
});
