<?php
/**
 * Plugin Name: CHEQ Enforce
 * Version: 1.0.0
 * Plugin URI: https://cheq.ai/enforce/
 * Description: Auto-block unauthorized tracking, protect data, give legal, compliance & martech teams control to meet global privacy laws without delay.
 * Author: CHEQ
 * Author URI: https://cheq.ai/
 * License: Apache-2.0
 * License URI: https://www.apache.org/licenses/LICENSE-2.0.txt
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class CHEQ_Enforce {
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Plugin settings array
     */
    private $settings;

    /**
     * Constructor: initialize settings and hooks
     */
    private function __construct() {
        $this->settings = get_option( 'cheq_enforce_settings', $this->get_defaults() );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'admin_menu', [ $this, 'add_admin_page' ] );
        add_action( 'wp_head', [ $this, 'enqueue_script' ], 0 );
    }

    /**
     * Initialize singleton instance
     */
    public static function init() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Default settings
     */
    private function get_defaults() {
        return [
            'domain'  => 'nexus.ensighten.com',
            'account' => '',
            'path'    => '',
        ];
    }

    /**
     * Register plugin settings and settings fields
     */
    public function register_settings() {
        register_setting(
            'cheq_enforce_settings_group',
            'cheq_enforce_settings',
            [ $this, 'sanitize_settings' ]
        );

        add_settings_section(
            'cheq_enforce_main_section',
            'CHEQ Enforce Settings',
            [ $this, 'section_callback' ],
            'cheq-enforce-settings'
        );

        add_settings_field(
            'cheq_enforce_domain',
            'Domain',
            [ $this, 'field_domain_cb' ],
            'cheq-enforce-settings',
            'cheq_enforce_main_section',
            [ 'label_for' => 'cheq_enforce_domain' ]
        );

        add_settings_field(
            'cheq_enforce_account',
            'Account',
            [ $this, 'field_account_cb' ],
            'cheq-enforce-settings',
            'cheq_enforce_main_section',
            [ 'label_for' => 'cheq_enforce_account' ]
        );

        add_settings_field(
            'cheq_enforce_path',
            'Publish Path',
            [ $this, 'field_path_cb' ],
            'cheq-enforce-settings',
            'cheq_enforce_main_section',
            [ 'label_for' => 'cheq_enforce_path' ]
        );
    }

    /**
     * Add options page under Settings
     */
    public function add_admin_page() {
        add_options_page(
            'CHEQ Enforce',
            'CHEQ Enforce',
            'manage_options',
            'cheq-enforce-settings',
            [ $this, 'options_page_html' ]
        );
    }

    /**
     * Section description callback
     */
    public function section_callback() {
        echo '<p>Configure the CHEQ Enforce script URL components.</p>';
    }

    /**
     * Domain field callback
     */
    public function field_domain_cb( $args ) {
        $value = isset( $this->settings['domain'] ) ? esc_attr( $this->settings['domain'] ) : '';
        printf("<input type='text' id='%1\$s' name='cheq_enforce_settings[domain]' value='%2\$s' class='regular-text' />",
            esc_attr( $args['label_for'] ), esc_html($value));
    }

    /**
     * Account field callback
     */
    public function field_account_cb( $args ) {
        $value = isset( $this->settings['account'] ) ? esc_attr( $this->settings['account'] ) : '';
        printf("<input type='text' id='%1\$s' name='cheq_enforce_settings[account]' value='%2\$s' class='regular-text' />",
            esc_attr( $args['label_for'] ), esc_html($value));
    }

    /**
     * Path field callback
     */
    public function field_path_cb( $args ) {
        $value = isset( $this->settings['path'] ) ? esc_attr( $this->settings['path'] ) : '';
        printf("<input type='text' id='%1\$s' name='cheq_enforce_settings[path]' value='%2\$s' class='regular-text' />",
            esc_attr( $args['label_for'] ), esc_html($value));
    }

    /**
     * Sanitize settings input
     */
    public function sanitize_settings( $input ) {
        $sanitized = [];
        $sanitized['domain']  = isset( $input['domain'] )  ? sanitize_text_field( wp_strip_all_tags( $input['domain'] ) ) : '';
        $sanitized['account'] = isset( $input['account'] ) ? sanitize_text_field( wp_strip_all_tags( $input['account'] ) ) : '';
        $sanitized['path']    = isset( $input['path'] )    ? sanitize_text_field( wp_strip_all_tags( $input['path'] ) ) : '';
        return $sanitized;
    }

    /**
     * Options page HTML
     */
    public function options_page_html() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        ?>
        <div class="wrap">
            <h1>CHEQ Enforce Settings</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'cheq_enforce_settings_group' );
                do_settings_sections( 'cheq-enforce-settings' );
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Enqueue the Bootstrap.js script in the head at priority 1
     */
    public function enqueue_script() {
        $domain  = trim( $this->settings['domain'], "\/ " );
        $account = trim( $this->settings['account'], "\/ " );
        $path    = trim( $this->settings['path'], "\/ " );

        $url = esc_url_raw( "https://{$domain}/{$account}/{$path}/Bootstrap.js" );
        wp_enqueue_script(
            'cheq-enforce-bootstrap',
            $url,
            [],
            '1.0.0',
            false
        );
    }
}

// Initialize the plugin
CHEQ_Enforce::init();