<?php

namespace CheerlinkAIChat\RestApi;

if (!defined('WPINC')) {
    die('File loaded directly. Exiting.');
}

require_once __DIR__ . '/../config/CheerlinkConfig.php';

use WP_REST_Response;
use WP_Error;

class CheerlinkRestRoute
{
    public function __construct()
    {
        add_action('rest_api_init', function () {

            /**
             * Register REST route for fetching and updating AppID.
             * @api {get} /cheerlink-ai-chat/v1/notify-update-appid, Fetch and update AppID from Cheerlink API.
             * Only Notify the Cheerlink-ai-chat to update the AppID, no arguments needed.
             */
            register_rest_route('cheerlink-ai-chat/v1', '/notify-update-appid', array(
                'methods' => 'GET',
                'callback' => array($this, 'fetch_and_update_appid'),
                'permission_callback' => '__return_true',
            ));
        });
    }

    /**
     * Fetch and update AppID from Cheerlink API.
     *
     * @param string $site_id Site ID
     * @return WP_REST_Response Returns array on success, WP_Error on failure
     */
    public function fetch_and_update_appid()
    {
        $site_id = clac_getOrInitSiteId();
        $body_json = wp_json_encode(array(
            'siteId' => $site_id,
        ));

        /**
         * Rest API request to fetch and update AppID from Cheerlink API, via https.
         *
         */
        $response = wp_remote_post(CHEERLINK_API_PREFIX . '/clapps/wordpress/webhook/site/appid', array(
            'headers' => array(
                'Content-Type' => 'application/json', // Set content type to JSON
                'cl-plugin-version' => CHEERLINK_AI_CHAT_VERSION // Set plugin version
            ),
            'body' => $body_json,
            'timeout' => 20,
        ));

        // Check if request was fail.
        if (is_wp_error($response)) {
            error_log('Cheerlink API request failed: ' . $response->get_error_message());
            return new WP_REST_Response(array(
                'code' => 1,
                'message' => 'fail'
            ));
        }

        // Get response status code
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return new WP_REST_Response(array(
                'code' => 1001,
                'message' => 'server error, status code: ' . $status_code
            ));
        }

        // Get and parse response body
        $body = wp_remote_retrieve_body($response);
        error_log('cheerlink access /clapps/wordpress/webhook/site/appid Response body: ' . $body);

        $data = json_decode($body, true);

        // Validate JSON parsing
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('Failed to parse API response: ' . json_last_error_msg());
            return new WP_REST_Response(array(
                'code' => 1002,
                'message' => 'json parse error'
            ));
        }

        // Check if API returned error code
        $res_code = $data['code'];
        if ($res_code !== 0) {
            error_log('API returned error code: ' . $res_code);
            return new WP_REST_Response(array(
                'code' => 1003,
                'message' => 'api error, code: ' . $res_code
            ));
        }

        // Validate required fields. JSON format: {"appId":"xxxx","siteId":"xxxx","status":"attached"}
        $appid = $data['data']['appId'];
        $woo_was_installed = $data['data']['wooCommerceWasInstall'];
        error_log('cheerlink access /clapps/wordpress/webhook/site/appid Response appId: ' . $appid . ', wooAckey: ' . $woo_was_installed);

        //
        if (empty($appid)) {
            return new WP_REST_Response(array(
                'code' => 1004,
                'message' => 'no attached, appId was empty: ' . $res_code
            ));
        }

        // Update database
        update_option(CHEERLINK_APPID_OPTION_NAME, $appid);

        if ($woo_was_installed) { // check if the woocommerce was installed.
            // Update WooCommerce access key was getted.
            update_option(CHEERLINK_WOO_INSTALLED_OPTION_NAME, true);
        }

        // Return response
        return new WP_REST_Response(array(
            'code' => 0,
            'message' => 'success',
            'data' => array(
                'appId' => $appid,
                'siteId' => $site_id
            )
        ));
    }
}
