<?php

if (!defined('WPINC')) {
    die('File loaded directly. Exiting.');
}

/**
 * Plugin Name: Cheerlink AI Chat
 * Plugin URI: https://cheerlink.app
 * Description: Cheerlink Livechat, an AI Chatbot, effortlessly integrates with your site to enhance customer service and sales.
 * Version: 1.0.1
 * Text Domain: cheerlink-ai-chat
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.2
 * Author: Cheerlink Labs
 * Author URI: https://www.cheerlink.app
 * License: GPLv3
 */

require_once __DIR__ . '/src/utils/UUIDHelper.php';
require_once __DIR__ . '/src/restapi/CheerlinkRestRoute.php';
require_once __DIR__ . '/src/config/CheerlinkConfig.php';

use CheerlinkAIChat\Utils\UUIDHelper;
use CheerlinkAIChat\RestApi\CheerlinkRestRoute;

/**
 * the function of inject javascript for widget.
 */
function clac_enqueue_js()
{
    $cheerlink_appid = esc_html(get_option(CHEERLINK_APPID_OPTION_NAME, ''));

    $settings = array(
        'appId' => $cheerlink_appid,
        'source' => 'WORDPRESS'
    );

    // add user info if user is logged in
    $current_user = wp_get_current_user();
    if ($current_user->ID) {
        $settings = array(
            'appId' => $cheerlink_appid,
            'source' => 'WORDPRESS',
            'user' => array(
                'bizUserId' => strval($current_user->ID),
                'name' => $current_user->user_login,
                'email' => $current_user->user_email
            )
        );
    }

    $init_script = 'window.cheerlinkSettings = ' . wp_json_encode($settings) . ';';

    wp_register_script('cheerlink-init-script', false);
    wp_add_inline_script('cheerlink-init-script', $init_script);
    wp_enqueue_script('cheerlink-init-script');

    if (empty($cheerlink_appid)) {
        return;
    }

    $cheerlink_launch_js = esc_url(CHEERLINK_LAUNCH_JS_URL . '?appId=' . $cheerlink_appid . '&source=WORDPRESS');
    wp_enqueue_script(
        'cheerlink-seed-script', // name of the script
        $cheerlink_launch_js, // widget js url, use the CDN url
        array(), // no depends
        null,    // no version
        array(
            'strategy'  => 'async'
        )
    );
}

/**
 * Define the left-hand menu button
 */
function clac_dashboard_menu()
{
    add_menu_page(
        'Cheerlink AI Chat',
        'Cheerlink AI Chat',
        'manage_options',
        "Cheerlink-AI-Chat",
        "clac_dashboard_options"
    );
}

/**
 * Define the dashboard page for cheerlink ai chat.
 */
function clac_dashboard_options()
{
    if (!current_user_can('manage_options')) {
        wp_die(
            esc_html__('You do not have sufficient permissions to access this page.', 'cheerlink-ai-chat')
        );
    }

    $cheerlink_appid = esc_html(get_option(CHEERLINK_APPID_OPTION_NAME, ''));
    $current_locale = get_locale();

    $site_id = clac_getOrInitSiteId();
    $site_url = get_site_url();
    $current_user = wp_get_current_user();
    //$admin_email = get_option('admin_email');
    $admin_email = rawurlencode($current_user->user_email);

    // url encode
    $url_base64 = strtr(base64_encode($site_url), '+/', '-_');
    $url_base64 = str_replace("=", "", $url_base64);
    $version = CHEERLINK_AI_CHAT_VERSION;

    $url_parm = 'version=' . esc_attr($version)
        . '&appId=' . esc_attr($cheerlink_appid)
        . '&siteId=' . esc_attr($site_id)
        . '&platform=wordpress'
        . '&locale=' . esc_attr($current_locale)
        . '&siteUrlBase64=' . esc_attr($url_base64)
        . '&adminEmail=' . urlencode($admin_email);

    $url = CHEERLINK_EMBED_HTML_URL . '?' . $url_parm;

    $woo_cheerlink_was_get_ackey = get_option(CHEERLINK_WOO_INSTALLED_OPTION_NAME, false);

    // if woocommerce is active and the access key was not get, then go to the woocommerce authorize page.
    if (clac_woocommerce_active() && !$woo_cheerlink_was_get_ackey) {

        $url_parm = urlencode('?' . $url_parm . '&woocommerce=true');

        $callbackUrl = urlencode(CHEERLINK_API_PREFIX . "/clapps/woocommerce/auth-callback");
        $callbackUrl = CHEERLINK_API_PREFIX . "/clapps/woocommerce/auth-callback";
        $woc_return_url = CHEERLINK_EMBED_HTML_URL . $url_parm;

        /**
         * go to the woocommerce authorize page.
         *
         * https://{currentHost}/wc-auth/v1/authorize?
        app_name=Cheerlink-ai-chat
        &scope=read_write
        &user_id={siteId}
        &return_url={$src}
        &callback_url={callbackUrl}
         */
        $url = get_site_url() . '/wc-auth/v1/authorize?app_name=Cheerlink-ai-chat&scope=read_write'
            . '&user_id=' . $site_id
            . '&callback_url=' . urlencode($callbackUrl)
            . '&return_url=' . urlencode($woc_return_url);
    }

    echo '<div style="min-width: 900px; height:calc(100vh - 120px)">';
    echo '<iframe src="' . $url . '" width="100%" height="100%"></iframe>';
    echo '</div>';
}

/**
 * Define the settings link.
 * @param array $links Plugin action links
 * @return array Modified plugin action links
 */
function clac_add_settings_link($links)
{
    $settings_link = sprintf(
        '<a href="%s">%s</a>',
        esc_url(admin_url('admin.php?page=Cheerlink-AI-Chat')),
        esc_html__('Settings', 'cheerlink-ai-chat')
    );
    array_unshift($links, $settings_link);
    return $links;
}

/**
 * Check if WooCommerce is active.
 * @return bool
 */
function clac_woocommerce_active()
{
    return class_exists('WooCommerce');
    //return in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')));
}

/**
 * init the unique site id.
 */
function clac_getOrInitSiteId()
{
    $cheerlink_site_id = get_option(CHEERLINK_SITEID_OPTION_NAME, '');
    if (empty($cheerlink_site_id)) {
        $cheerlink_site_id = UUIDHelper::generateUUID();
        update_option(CHEERLINK_SITEID_OPTION_NAME, $cheerlink_site_id);
    }
    return $cheerlink_site_id;
}

/**
 * process event for unInstall cheerlink plugin.
 */
function clac_un_install()
{

    $site_id = clac_getOrInitSiteId();
    $cheerlink_appid = get_option(CHEERLINK_APPID_OPTION_NAME, '');

    //clear appId & woo_installed flag
    update_option(CHEERLINK_APPID_OPTION_NAME, '');
    update_option(CHEERLINK_WOO_INSTALLED_OPTION_NAME, false);

    // send uninstall event to cheerlink api.
    $body_json = wp_json_encode(array(
        'siteId' => $site_id,
        'appId' => $cheerlink_appid
    ));

    $response = wp_remote_post(CHEERLINK_API_PREFIX . '/clapps/wordpress/webhook/app/unInstall', array(
        'body' => $body_json,
        'headers' => array(
            'Content-Type' => 'application/json', // Set content type to JSON
            'cl-plugin-version' => CHEERLINK_AI_CHAT_VERSION // Set plugin version
        ),
        'timeout' => 10,
    ));

    if (is_wp_error($response)) {
        $error_message = $response->get_error_message();
        // translators: %s: Error message from API response
        error_log(sprintf(
            esc_html__('Something went wrong during plugin uninstall: %s', 'cheerlink-ai-chat'),
            $error_message
        ));
    } else {
        error_log(
            esc_html__('Callback to Cheerlink API for uninstall plugin succeeded.', 'cheerlink-ai-chat')
        );
    }
}

/**
 * process event for activatiton plugin
 */
function clac_activate_plugin()
{
    $site_id = clac_getOrInitSiteId();
    $cheerlink_appid = get_option(CHEERLINK_APPID_OPTION_NAME, '');
    $site_name = get_bloginfo('name');
    $language = get_bloginfo('language');
    $wordpress_version = get_bloginfo('version');
    $admin_email = get_bloginfo('admin_email');
    $site_url = get_site_url();

    // send basic info to cheerlink api.
    $body_json = wp_json_encode(array(
        'siteId' => $site_id,
        'appId' => $cheerlink_appid,
        'siteUrl' => $site_url,
        'language' => $language,
        'wordpressVersion' => $wordpress_version,
        'siteName' => $site_name,
        'adminEmail' => $admin_email
    ));

    // update basic information for normal operation.
    $response = wp_remote_post(CHEERLINK_API_PREFIX . '/clapps/wordpress/webhook/site/info', array(
        'body' => $body_json,
        'headers' => array(
            'Content-Type' => 'application/json', // Set content type to JSON
            'cl-plugin-version' => CHEERLINK_AI_CHAT_VERSION // Set plugin version
        ),
        'timeout' => 20,
    ));

    if (is_wp_error($response)) {
        $error_message = $response->get_error_message();
        // translators: %s: Error message from API response
        error_log(sprintf(
            esc_html__('Something went wrong during plugin activation: %s', 'cheerlink-ai-chat'),
            $error_message
        ));
    } else {
        error_log(
            esc_html__('Callback to Cheerlink API for activate plugin succeeded.', 'cheerlink-ai-chat')
        );
    }
}

//hook for update site info
function clac_hook_update_site_info()
{
    add_action('update_option_siteurl', 'clac_activate_plugin', 10, 0);
    add_action('update_option_blogname', 'clac_activate_plugin', 10, 0);
    add_action('update_option_url', 'clac_activate_plugin', 10, 0);
}

// load textdomain
function clac_load_textdomain()
{
    load_plugin_textdomain(
        'cheerlink-ai-chat',
        false,
        dirname(plugin_basename(__FILE__)) . '/languages/'
    );
}

/**
 * init plugin settings when installed.
 */
function clac_init()
{

    // enqueue js url
    add_action('wp_enqueue_scripts', 'clac_enqueue_js');

    // hook for add menu
    add_action('admin_menu', 'clac_dashboard_menu');

    // hook for add settings link
    add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'clac_add_settings_link');

    // init rest api and route.
    $cheerlinkRestRoute = new CheerlinkRestRoute();

    // register unInstall hook
    register_uninstall_hook(__FILE__, 'clac_un_install');

    // register activate plugin
    register_activation_hook(__FILE__, 'clac_activate_plugin');

    // hook for update site info
    clac_hook_update_site_info();
}

// init plugin
clac_init();

add_action('plugins_loaded', 'clac_load_textdomain');
