<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://chatolia.com
 * @since      1.0.0
 *
 * @package    Chatolia
 * @subpackage Chatolia/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Chatolia
 * @subpackage Chatolia/public
 * @author     Senol Sahin <senols@gmail.com>
 */
class Chatolia_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
    private $version;

    /**
     * Track whether widget script has been printed.
     * Ensures only one widget per page even if shortcode and site-wide are both used.
     *
     * @var bool
     */
    private static $printed_widget = false;

    /**
     * Handle for the external widget script.
     *
     * @var string
     */
    private $widget_handle = 'chatolia-widget';

    /**
     * Attributes to add to the widget script tag.
     * Captured from the first invocation and reused to avoid duplicates.
     *
     * @var array{agent:string,position:string,theme:string}
     */
    private static $widget_attrs = array(
        'agent'    => '',
        'position' => '',
        'theme'    => '',
    );

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Chatolia_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Chatolia_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/chatolia-public.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Chatolia_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Chatolia_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/chatolia-public.js', array( 'jquery' ), $this->version, false );

	}

	/**
	 * Register shortcode(s).
	 */
    public function register_shortcodes() {
        add_shortcode( 'chatolia', array( $this, 'shortcode_chatolia' ) );
    }

	/**
	 * Shortcode callback for [chatolia].
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string HTML output
	 */
    public function shortcode_chatolia( $atts = array() ) {
        $defaults = array(
            'theme'  => '',
        );
        $atts = shortcode_atts( $defaults, $atts, 'chatolia' );

        $settings = get_option( 'chatolia_settings', array() );
        $settings = wp_parse_args( is_array( $settings ) ? $settings : array(), array(
            'agent_id'    => '',
            'theme'       => 'light',
            'position'    => 'bottom-right',
            'icon_size'   => '48',
            'icon_url'    => '',
            'hint_enabled'=> '0',
        ) );

		$agent_id = isset( $settings['agent_id'] ) ? $settings['agent_id'] : '';
		if ( empty( $agent_id ) ) {
			return '<!-- Chatolia: missing agent_id. Configure the plugin settings. -->';
		}

        $theme  = $atts['theme'] ? $atts['theme'] : $settings['theme'];

        // Enqueue the widget script (printed in footer) with attributes.
        if ( self::$printed_widget ) {
            return '<!-- Chatolia widget already loaded -->';
        }
        $this->enqueue_widget_script( $agent_id, $settings['position'], $theme );
        return '';
    }

    /**
     * Inject the widget site-wide if enabled in settings.
     * Prints in footer. Does nothing if not enabled or already printed.
     */
    public function render_sitewide_widget() {
        if ( is_admin() ) {
            return;
        }
        $settings = get_option( 'chatolia_settings', array() );
        $settings = wp_parse_args( is_array( $settings ) ? $settings : array(), array(
            'agent_id'        => '',
            'theme'           => 'light',
            'position'        => 'bottom-right',
            'sitewide_enabled'=> '0',
        ) );

        if ( '1' !== (string) $settings['sitewide_enabled'] ) {
            return;
        }
        $agent_id = isset( $settings['agent_id'] ) ? $settings['agent_id'] : '';
        if ( empty( $agent_id ) ) {
            return;
        }
        if ( self::$printed_widget ) {
            return;
        }

        $theme = $settings['theme'];
        // Enqueue once; will print in footer via wp_print_footer_scripts.
        $this->enqueue_widget_script( $agent_id, $settings['position'], $theme );
    }

    /**
     * Get widget script URL with strict host/path validation.
     * Allows filtering but restricts to chatolia.com hosts over HTTPS and /widget.js path.
     *
     * @return string
     */
    private function widget_script_url() {
        $default = 'https://www.chatolia.com/widget.js';
        $url     = apply_filters( 'chatolia_widget_script_url', $default );
        if ( ! is_string( $url ) || '' === $url ) {
            return $default;
        }
        $parts = wp_parse_url( $url );
        if ( ! is_array( $parts ) ) {
            return $default;
        }
        $scheme = isset( $parts['scheme'] ) ? strtolower( $parts['scheme'] ) : '';
        $host   = isset( $parts['host'] ) ? strtolower( $parts['host'] ) : '';
        $path   = isset( $parts['path'] ) ? $parts['path'] : '';
        $allowed_hosts = array( 'www.chatolia.com', 'chatolia.com' );
        if ( 'https' !== $scheme || ! in_array( $host, $allowed_hosts, true ) ) {
            return $default;
        }
        if ( ! is_string( $path ) || ! preg_match( '#/widget\.js$#', $path ) ) {
            return $default;
        }
        return $url;
    }

    /**
     * Enqueue the widget script with attributes using the script_loader_tag filter.
     *
     * @param string $agent_id
     * @param string $position
     * @param string $theme
     * @return void
     */
    public function enqueue_widget_script( $agent_id, $position, $theme ) {
        self::$widget_attrs = array(
            'agent'    => sanitize_text_field( (string) $agent_id ),
            'position' => sanitize_text_field( (string) $position ),
            'theme'    => sanitize_text_field( (string) $theme ),
        );

        $handle = $this->widget_handle;
        $src    = $this->widget_script_url();

        if ( ! wp_script_is( $handle, 'registered' ) ) {
            wp_register_script( $handle, $src, array(), $this->version, true );
            // Ask WP to add async/defer when rendering.
            if ( function_exists( 'wp_script_add_data' ) ) {
                wp_script_add_data( $handle, 'async', true );
                wp_script_add_data( $handle, 'defer', true );
            }
        }

        wp_enqueue_script( $handle );
        self::$printed_widget = true;
    }

    /**
     * Filter the script tag to inject data-* attributes for the widget handle.
     *
     * @param string $tag
     * @param string $handle
     * @param string $src
     * @return string
     */
    public function filter_script_tag( $tag, $handle, $src ) {
        if ( $handle !== $this->widget_handle ) {
            return $tag;
        }

        $attrs = self::$widget_attrs;
        $agent = isset( $attrs['agent'] ) ? $attrs['agent'] : '';
        if ( '' === $agent ) {
            return '';
        }
        $position = isset( $attrs['position'] ) ? $attrs['position'] : 'bottom-right';
        $theme    = isset( $attrs['theme'] ) ? $attrs['theme'] : 'light';

        // Inject data-* attributes right after the opening <script ...> tag.
        $injection = sprintf(
            ' data-agent="%1$s" data-position="%2$s" data-theme="%3$s"',
            esc_attr( $agent ),
            esc_attr( $position ),
            esc_attr( $theme )
        );
        $tag = preg_replace( '/^<script\b/i', '<script' . $injection, (string) $tag, 1 );
        return $tag;
    }

}
