<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://chatolia.com
 * @since      1.0.0
 *
 * @package    Chatolia
 * @subpackage Chatolia/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Chatolia
 * @subpackage Chatolia/admin
 * @author     Senol Sahin <senols@gmail.com>
 */
class Chatolia_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Settings option name used to persist plugin settings.
	 *
	 * @var string
	 */
	private $option_name = 'chatolia_settings';

    /**
     * Cached agents for the current request.
     *
     * @var array<int, array<string,string>>|null
     */
    private $agents_cache = null;

    /**
     * Cached token used to fetch agents.
     *
     * @var string
     */
    private $agents_cache_token = '';

    /**
     * Last error encountered when fetching agents.
     *
     * @var WP_Error|null
     */
    private $agents_cache_error = null;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

		// Menu & settings.
		add_action( 'admin_menu', array( $this, 'add_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_post_chatolia_create_agent', array( $this, 'handle_create_agent' ) );
        add_action( 'admin_post_chatolia_widget_preview', array( $this, 'render_widget_preview' ) );
        add_action( 'admin_notices', array( $this, 'maybe_render_admin_notices' ) );

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Chatolia_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Chatolia_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/chatolia-admin.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Chatolia_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Chatolia_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/chatolia-admin.js', array( 'jquery' ), $this->version, false );

	}

	/**
	 * Capability required to manage the plugin settings.
	 *
	 * @return string
	 */
	private function capability() {
		/**
		 * Filter the capability required for managing Chatolia settings.
		 *
		 * @param string $cap Capability slug.
		 */
		return apply_filters( 'chatolia_manage_capability', 'manage_options' );
	}

	/**
	 * Add the top-level Chatolia menu.
	 */
	public function add_menu() {
		add_menu_page(
			esc_html__( 'Chatolia', 'chatolia' ),
			esc_html__( 'Chatolia', 'chatolia' ),
			$this->capability(),
			'chatolia',
			array( $this, 'render_settings_page' ),
			'dashicons-format-chat',
			56
		);
	}

	/**
	 * Register settings, sections and fields for the settings page.
	 */
	public function register_settings() {
		register_setting( 'chatolia_settings_group', $this->option_name, array( $this, 'sanitize_settings' ) );

		add_settings_section(
			'chatolia_main',
			'',
			'__return_false',
			'chatolia'
		);

		add_settings_field( 'chatolia_access_token', esc_html__( 'Access Token', 'chatolia' ), array( $this, 'field_access_token' ), 'chatolia', 'chatolia_main' );
		add_settings_field( 'chatolia_agent_id', esc_html__( 'Agent', 'chatolia' ), array( $this, 'field_agent_id' ), 'chatolia', 'chatolia_main' );
        add_settings_field( 'chatolia_theme', esc_html__( 'Theme', 'chatolia' ), array( $this, 'field_theme' ), 'chatolia', 'chatolia_main' );
        add_settings_field( 'chatolia_position', esc_html__( 'Position', 'chatolia' ), array( $this, 'field_position' ), 'chatolia', 'chatolia_main' );
        add_settings_field( 'chatolia_sitewide_enabled', esc_html__( 'Site-wide Widget', 'chatolia' ), array( $this, 'field_sitewide_enabled' ), 'chatolia', 'chatolia_main' );
        // Removed Primary Color setting; color is managed on chatolia.com.
    }

	/**
	 * Default settings.
	 *
	 * @return array
	 */
	private function defaults() {
        return array(
            'access_token' => '',
            'agent_id'     => '',
            'position'     => 'bottom-right',
            'theme'        => 'light',
            'icon_size'    => '48',
            'icon_url'     => '',
            'icon_style'   => '',
            'hint_enabled' => '0',
            'sitewide_enabled' => '0',
        );
	}

	/**
	 * Get settings merged with defaults.
	 *
	 * @return array
	 */
	private function get_settings() {
		$settings = get_option( $this->option_name, array() );
		if ( ! is_array( $settings ) ) {
			$settings = array();
		}
		return wp_parse_args( $settings, $this->defaults() );
	}

	/**
	 * Sanitize settings callback.
	 *
	 * @param array $input Raw input.
	 * @return array
	 */
	public function sanitize_settings( $input ) {
		$input = is_array( $input ) ? $input : array();
		$clean = $this->defaults();

        // Preserve existing agent_id when the field is not shown in the UI.
        $existing_settings     = $this->get_settings();

        $clean['access_token'] = isset( $input['access_token'] ) ? sanitize_text_field( wp_unslash( $input['access_token'] ) ) : '';
        $clean['agent_id']     = isset( $existing_settings['agent_id'] ) ? sanitize_text_field( (string) $existing_settings['agent_id'] ) : '';
        $clean['position']     = isset( $input['position'] ) ? sanitize_text_field( wp_unslash( $input['position'] ) ) : 'bottom-right';
        $clean['theme']        = in_array( isset( $input['theme'] ) ? $input['theme'] : '', array( 'light', 'dark', 'auto' ), true ) ? $input['theme'] : 'light';
        $clean['icon_size']    = isset( $input['icon_size'] ) ? absint( $input['icon_size'] ) : 48;
        $clean['icon_url']     = isset( $input['icon_url'] ) ? esc_url_raw( wp_unslash( $input['icon_url'] ) ) : '';
        $clean['icon_style']   = isset( $input['icon_style'] ) ? sanitize_text_field( wp_unslash( $input['icon_style'] ) ) : '';
        $clean['hint_enabled'] = isset( $input['hint_enabled'] ) && '1' === (string) $input['hint_enabled'] ? '1' : '0';
        $clean['sitewide_enabled'] = isset( $input['sitewide_enabled'] ) && '1' === (string) $input['sitewide_enabled'] ? '1' : '0';
        // Primary color removed; controlled in Chatolia dashboard.

        // Warn when no token is provided; without it only the demo preview is shown and nothing is published site-wide.
        if ( '' === $clean['access_token'] ) {
            add_settings_error( 'chatolia_settings', 'chatolia_token_missing', __( 'Add your Chatolia token to publish your bot. Without a token, only the demo preview is shown and nothing loads on your site.', 'chatolia' ) );
        }

        $available_agents = array();
        $selected_agent   = isset( $input['agent_id'] ) ? sanitize_text_field( wp_unslash( $input['agent_id'] ) ) : '';

        if ( '' !== $clean['access_token'] ) {
            $available_agents = $this->get_agents_for_token( $clean['access_token'] );
            $agents_error     = $this->get_agents_fetch_error();
            if ( is_wp_error( $agents_error ) ) {
                /* translators: %s: error message returned when listing agents */
                add_settings_error( 'chatolia_settings', 'chatolia_agents_fetch', sprintf( __( 'Could not fetch agents: %s', 'chatolia' ), $agents_error->get_error_message() ) );
            }
        }

        $available_ids = array();
        foreach ( $available_agents as $agent ) {
            if ( isset( $agent['publicId'] ) ) {
                $available_ids[] = (string) $agent['publicId'];
            }
        }

        // Respect user selection when it matches the fetched list.
        if ( '' !== $selected_agent && in_array( $selected_agent, $available_ids, true ) ) {
            $clean['agent_id'] = $selected_agent;
        }

        // If no stored agent and agents exist, default to the first one.
        if ( '' === $clean['agent_id'] && ! empty( $available_ids ) ) {
            $clean['agent_id'] = $available_ids[0];
        }

        // Auto-create an agent if none exist for this token.
        if ( '' !== $clean['access_token'] && '' === $clean['agent_id'] && empty( $available_ids ) ) {
            $agent = $this->maybe_create_agent_with_token( $clean['access_token'] );
            if ( $agent ) {
                $clean['agent_id'] = $agent;
                add_settings_error( 'chatolia_settings', 'chatolia_agent_created', __( 'Agent created and linked automatically.', 'chatolia' ), 'updated' );
            }
        }

        return $clean;
    }

    /**
     * Create an agent using the provided token.
     * Returns agent publicId on success, false on failure.
     *
     * @param string $token Access token.
     * @return string|false
     */
    private function maybe_create_agent_with_token( $token ) {
        $token = sanitize_text_field( (string) $token );
        if ( '' === $token ) {
            return false;
        }

        $site_title = wp_strip_all_tags( (string) get_bloginfo( 'name' ) );
        $site_title = trim( $site_title );
        $suffix     = ' Assistant';
        $max_total  = 40;
        $max_base   = max( 1, $max_total - strlen( $suffix ) );
        $base_name  = '' !== $site_title ? $site_title : 'Chatolia';
        $base_name  = $this->truncate_utf8( $base_name, $max_base );
        $agent_name = sanitize_text_field( trim( $base_name . $suffix ) );
        if ( '' === $agent_name ) {
            $agent_name = 'Chatolia Assistant';
        }

        $api_base = $this->api_base();
        $url      = trailingslashit( $api_base ) . 'api/agents';
        $args     = array(
            'timeout' => 15,
            'headers' => array(
                'Authorization' => 'Bearer ' . $token,
                'Content-Type'  => 'application/json',
                'Accept'        => 'application/json',
                'User-Agent'    => 'Chatolia-WP/' . $this->version . '; ' . home_url( '/' ),
            ),
            'body'    => wp_json_encode( array(
                'name'            => $agent_name,
                'defaultModelId'  => 'openai/gpt-4o',
                'visibility'      => 'public',
            ) ),
        );

        $response = wp_remote_post( $url, $args );
        if ( is_wp_error( $response ) ) {
            /* translators: %s: error message returned when creating an agent */
            add_settings_error( 'chatolia_settings', 'chatolia_agent_error', sprintf( __( 'Could not create agent: %s', 'chatolia' ), $response->get_error_message() ) );
            return false;
        }

        $status = (int) wp_remote_retrieve_response_code( $response );
        $body   = wp_remote_retrieve_body( $response );
        $data   = json_decode( $body, true );

        if ( $status >= 200 && $status < 300 && is_array( $data ) && isset( $data['agent']['publicId'] ) ) {
            return sanitize_text_field( (string) $data['agent']['publicId'] );
        }

        $status_text = wp_remote_retrieve_response_message( $response );
        $detail      = '';
        if ( is_array( $data ) ) {
            if ( isset( $data['error'] ) && is_string( $data['error'] ) ) {
                $detail = $data['error'];
            } elseif ( isset( $data['message'] ) && is_string( $data['message'] ) ) {
                $detail = $data['message'];
            }
        }
        if ( '' === $detail && is_string( $body ) ) {
            $detail = wp_strip_all_tags( ( function_exists( 'mb_substr' ) ? mb_substr( $body, 0, 200 ) : substr( $body, 0, 200 ) ) );
        }

        $base_msg = trim( sprintf( '%d%s%s', $status, $status_text ? ' ' . $status_text : '', $detail ? ': ' . $detail : '' ) );
        /* translators: %s: error message returned when creating an agent */
        add_settings_error( 'chatolia_settings', 'chatolia_agent_error', sprintf( __( 'Could not create agent: %s', 'chatolia' ), $base_msg ) );

        return false;
    }

    /**
     * Truncate a UTF-8 string to a maximum length without breaking multibyte chars.
     *
     * @param string $string     Input string.
     * @param int    $max_length Maximum characters.
     * @return string
     */
    private function truncate_utf8( $string, $max_length ) {
        $string     = (string) $string;
        $max_length = (int) $max_length;
        if ( $max_length <= 0 ) {
            return '';
        }

        if ( function_exists( 'mb_strlen' ) && function_exists( 'mb_substr' ) ) {
            if ( mb_strlen( $string, 'UTF-8' ) > $max_length ) {
                return mb_substr( $string, 0, $max_length, 'UTF-8' );
            }
            return $string;
        }

        if ( strlen( $string ) > $max_length ) { // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obsolete_functions_strlen
            return substr( $string, 0, $max_length ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obsolete_functions_substr
        }

        return $string;
    }

    /**
     * Fetch agents for the given token with caching per request.
     *
     * @param string $token Access token.
     * @return array<int, array<string,string>> Array of agents.
     */
    private function get_agents_for_token( $token ) {
        $token = trim( (string) $token );
        if ( '' === $token ) {
            $this->agents_cache       = array();
            $this->agents_cache_token = '';
            $this->agents_cache_error = null;
            return array();
        }

        if ( null !== $this->agents_cache && $this->agents_cache_token === $token ) {
            return $this->agents_cache;
        }

        $fetched = $this->request_agents( $token );
        if ( is_wp_error( $fetched ) ) {
            $this->agents_cache       = array();
            $this->agents_cache_token = $token;
            $this->agents_cache_error = $fetched;
            return array();
        }

        $this->agents_cache       = $fetched;
        $this->agents_cache_token = $token;
        $this->agents_cache_error = null;
        return $fetched;
    }

    /**
     * Get the last agents fetch error.
     *
     * @return WP_Error|null
     */
    private function get_agents_fetch_error() {
        return $this->agents_cache_error instanceof WP_Error ? $this->agents_cache_error : null;
    }

    /**
     * Perform the HTTP request to list agents.
     *
     * @param string $token Access token.
     * @return array<int, array<string,string>>|WP_Error
     */
    private function request_agents( $token ) {
        $api_base = $this->api_base();
        $url      = trailingslashit( $api_base ) . 'api/agents';
        $args     = array(
            'timeout' => 15,
            'headers' => array(
                'Authorization' => 'Bearer ' . $token,
                'Accept'        => 'application/json',
                'User-Agent'    => 'Chatolia-WP/' . $this->version . '; ' . home_url( '/' ),
            ),
        );

        $response = wp_remote_get( $url, $args );
        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $status = (int) wp_remote_retrieve_response_code( $response );
        $body   = wp_remote_retrieve_body( $response );

        if ( $status < 200 || $status >= 300 ) {
            $status_text = wp_remote_retrieve_response_message( $response );
            $detail      = '';
            $data        = json_decode( $body, true );
            if ( is_array( $data ) && isset( $data['error'] ) && is_string( $data['error'] ) ) {
                $detail = $data['error'];
            } elseif ( '' === $detail && is_string( $body ) ) {
                $detail = wp_strip_all_tags( ( function_exists( 'mb_substr' ) ? mb_substr( $body, 0, 200 ) : substr( $body, 0, 200 ) ) );
            }
            $base_msg = trim( sprintf( '%d%s%s', $status, $status_text ? ' ' . $status_text : '', $detail ? ': ' . $detail : '' ) );
            /* translators: %s: error message returned when listing agents */
            return new WP_Error( 'chatolia_agents_fetch', sprintf( __( 'Could not fetch agents: %s', 'chatolia' ), $base_msg ) );
        }

        $data   = json_decode( $body, true );
        $agents = array();
        if ( is_array( $data ) && isset( $data['agents'] ) && is_array( $data['agents'] ) ) {
            foreach ( $data['agents'] as $agent ) {
                if ( isset( $agent['publicId'] ) ) {
                    $agents[] = array(
                        'publicId' => sanitize_text_field( (string) $agent['publicId'] ),
                        'name'     => isset( $agent['name'] ) ? sanitize_text_field( (string) $agent['name'] ) : '',
                    );
                }
            }
        }

        return $agents;
    }

	/**
	 * Render settings page content.
	 */
    public function render_settings_page() {
        if ( ! current_user_can( $this->capability() ) ) {
            return;
        }
        $settings  = $this->get_settings();
        $agent_id  = $settings['agent_id'];
        $theme     = $settings['theme'];
        $position  = $settings['position'];

        // Prefetch agents for the token to render the dropdown without extra requests.
        if ( ! empty( $settings['access_token'] ) ) {
            $this->get_agents_for_token( $settings['access_token'] );
        }

        $preview_mode          = ! empty( $agent_id ) ? 'agent' : 'demo';
        $showing_demo_preview  = ( 'demo' === $preview_mode );
        $preview_url = wp_nonce_url( admin_url( 'admin-post.php?action=chatolia_widget_preview' ), 'chatolia_widget_preview' );
        $preview_url = add_query_arg( 'preview_mode', $preview_mode, $preview_url );

        include plugin_dir_path( __FILE__ ) . 'partials/chatolia-admin-display.php';
    }

	/**
	 * Field: Access Token.
	 */
	public function field_access_token() {
        $settings  = $this->get_settings();
        $token_url = 'https://www.chatolia.com/dashboard/settings/tokens';
        echo '<div class="chatolia-token-row">';
        echo '<input type="password" autocomplete="off" class="regular-text" name="' . esc_attr( $this->option_name ) . '[access_token]" value="' . esc_attr( $settings['access_token'] ) . '" placeholder="' . esc_attr__( 'Paste your Chatolia token', 'chatolia' ) . '" />';
        echo '<a class="chatolia-token-link" href="' . esc_url( $token_url ) . '" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Get your free token', 'chatolia' ) . '</a>';
        echo '</div>';
    }

    /**
     * Field: Agent selector.
     */
    public function field_agent_id() {
        $settings   = $this->get_settings();
        $token      = isset( $settings['access_token'] ) ? $settings['access_token'] : '';
        $current_id = isset( $settings['agent_id'] ) ? $settings['agent_id'] : '';

        if ( '' === $token ) {
            echo '<p class="description">' . esc_html__( 'Enter your token to load your agents. If you have none, we will create one automatically after saving.', 'chatolia' ) . '</p>';
            return;
        }

        $agents = $this->get_agents_for_token( $token );
        $error  = $this->get_agents_fetch_error();

        if ( is_wp_error( $error ) ) {
            echo '<p class="description">' . esc_html( $error->get_error_message() ) . '</p>';
            return;
        }

        if ( empty( $agents ) ) {
            echo '<p class="description">' . esc_html__( 'No agents found. We will create one automatically after you save.', 'chatolia' ) . '</p>';
            return;
        }

        $options = '';
        $has_current = false;
        foreach ( $agents as $agent ) {
            if ( ! isset( $agent['publicId'] ) ) {
                continue;
            }
            $id   = (string) $agent['publicId'];
            $name = isset( $agent['name'] ) ? $agent['name'] : $id;
            if ( $current_id === $id ) {
                $has_current = true;
            }
            $options .= '<option value="' . esc_attr( $id ) . '"' . selected( $current_id, $id, false ) . '>' . esc_html( $name ) . '</option>';
        }

        // Preserve a stored agent ID that is not in the fetched list (token mismatch or removed).
        if ( $current_id && ! $has_current ) {
            $options = '<option value="' . esc_attr( $current_id ) . '" selected>' . esc_html( sprintf( __( 'Current agent (%s)', 'chatolia' ), $current_id ) ) . '</option>' . $options;
        }

        echo '<select name="' . esc_attr( $this->option_name ) . '[agent_id]" id="chatolia-agent-select">' . $options . '</select>';
        echo '<p class="description">' . esc_html__( 'Choose an existing agent for your site. If you prefer a new one, save without changing and we will create it.', 'chatolia' ) . '</p>';
    }

	/**
	 * Field: Theme.
	 */
	public function field_theme() {
		$settings = $this->get_settings();
		echo '<select name="' . esc_attr( $this->option_name ) . '[theme]">';
		echo '<option value="light"' . selected( 'light', $settings['theme'], false ) . '>' . esc_html__( 'Light', 'chatolia' ) . '</option>';
		echo '<option value="dark"' . selected( 'dark', $settings['theme'], false ) . '>' . esc_html__( 'Dark', 'chatolia' ) . '</option>';
		echo '<option value="auto"' . selected( 'auto', $settings['theme'], false ) . '>' . esc_html__( 'Auto', 'chatolia' ) . '</option>';
		echo '</select>';
	}

	/**
	 * Field: Position.
	 */
    public function field_position() {
        $settings = $this->get_settings();
        echo '<select name="' . esc_attr( $this->option_name ) . '[position]">';
        $options = array(
            'bottom-right' => __( 'Bottom Right', 'chatolia' ),
            'bottom-left'  => __( 'Bottom Left', 'chatolia' ),
            'top-right'    => __( 'Top Right', 'chatolia' ),
            'top-left'     => __( 'Top Left', 'chatolia' ),
        );
        foreach ( $options as $val => $label ) {
            echo '<option value="' . esc_attr( $val ) . '"' . selected( $val, $settings['position'], false ) . '>' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
    }

    // Primary color field removed

    /**
     * Field: Site-wide widget enabled.
     */
    public function field_sitewide_enabled() {
        $settings = $this->get_settings();
        $field_id = 'chatolia_sitewide_enabled';
        echo '<label for="' . esc_attr( $field_id ) . '">';
        echo '<input type="checkbox" id="' . esc_attr( $field_id ) . '" name="' . esc_attr( $this->option_name ) . '[sitewide_enabled]" value="1" ' . checked( '1', $settings['sitewide_enabled'], false ) . ' /> ';
        echo esc_html__( 'Display the chatbot widget on all front-end pages', 'chatolia' );
        echo '</label>';    }

	/**
	 * Handle the Create Agent form/action.
	 */
	public function handle_create_agent() {
		if ( ! current_user_can( $this->capability() ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'chatolia' ) );
		}
		check_admin_referer( 'chatolia_create_agent' );

		$settings = $this->get_settings();
		$token    = $settings['access_token'];
		if ( empty( $token ) ) {
			wp_safe_redirect( add_query_arg( 'chatolia_msg', 'missing_token', admin_url( 'admin.php?page=chatolia' ) ) );
			exit;
		}

		$api_base = $this->api_base();
		$url      = trailingslashit( $api_base ) . 'api/agents';
		$args     = array(
			'timeout' => 15,
			'headers' => array(
				'Authorization' => 'Bearer ' . $token,
				'Content-Type'  => 'application/json',
				'Accept'        => 'application/json',
				'User-Agent'    => 'Chatolia-WP/' . $this->version . '; ' . home_url( '/' ),
			),
			'body'    => wp_json_encode( array(
				'name'            => sprintf( /* translators: %s: site title */ __( '%s Assistant', 'chatolia' ), get_bloginfo( 'name' ) ),
				'defaultModelId'  => 'openai/gpt-4o',
				'visibility'      => 'public',
			) ),
		);

		$response = wp_remote_post( $url, $args );
		if ( is_wp_error( $response ) ) {
			wp_safe_redirect( add_query_arg( array( 'chatolia_msg' => 'api_error', 'chatolia_err' => rawurlencode( $response->get_error_message() ) ), admin_url( 'admin.php?page=chatolia' ) ) );
			exit;
		}

		$status = (int) wp_remote_retrieve_response_code( $response );
		$body   = wp_remote_retrieve_body( $response );
		$data   = json_decode( $body, true );
		$agent  = '';
		if ( $status >= 200 && $status < 300 && is_array( $data ) ) {
			if ( isset( $data['agent'] ) && is_array( $data['agent'] ) && isset( $data['agent']['publicId'] ) ) {
				$agent = sanitize_text_field( (string) $data['agent']['publicId'] );
			}
		}

		if ( empty( $agent ) ) {
			// Build a more helpful error message for the admin notice.
			$status_text = wp_remote_retrieve_response_message( $response );
			$detail      = '';
			if ( is_array( $data ) ) {
				if ( isset( $data['error'] ) && is_string( $data['error'] ) ) {
					$detail = $data['error'];
				} elseif ( isset( $data['message'] ) && is_string( $data['message'] ) ) {
					$detail = $data['message'];
				}
			}
			if ( '' === $detail && is_string( $body ) ) {
				$snippet = function_exists( 'mb_substr' ) ? mb_substr( $body, 0, 400 ) : substr( $body, 0, 400 );
				$detail  = wp_strip_all_tags( $snippet );
			}

			$base_msg  = trim( sprintf( '%d%s%s', $status, $status_text ? ' ' . $status_text : '', $detail ? ': ' . $detail : '' ) );
			$error_str = $base_msg;
			wp_safe_redirect( add_query_arg( array( 'chatolia_msg' => 'api_error', 'chatolia_err' => rawurlencode( $error_str ) ), admin_url( 'admin.php?page=chatolia' ) ) );
			exit;
		}

		$settings['agent_id'] = $agent;
		update_option( $this->option_name, $settings );

		wp_safe_redirect( add_query_arg( 'chatolia_msg', 'agent_created', admin_url( 'admin.php?page=chatolia' ) ) );
		exit;
	}

	/**
	 * Maybe render admin notices on our page.
	 */
	public function maybe_render_admin_notices() {
		if ( ! isset( $_GET['page'] ) || 'chatolia' !== sanitize_key( wp_unslash( $_GET['page'] ) ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return;
		}

		$msg = isset( $_GET['chatolia_msg'] ) ? sanitize_key( wp_unslash( $_GET['chatolia_msg'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! $msg ) {
			return;
		}

		$class = 'notice-info';
		$text  = '';
		switch ( $msg ) {
			case 'missing_token':
				$class = 'notice-error';
				$text  = __( 'Please save a valid access token before creating an agent.', 'chatolia' );
				break;
			case 'agent_created':
				$class = 'notice-success';
				$text  = __( 'Agent created successfully and saved.', 'chatolia' );
				break;
            case 'api_error':
                $class = 'notice-error';
                $err   = isset( $_GET['chatolia_err'] ) ? sanitize_text_field( wp_unslash( $_GET['chatolia_err'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $text  = sprintf(
                    /* translators: %s: The error message from the Chatolia API */
                    __( 'API error: %s', 'chatolia' ),
                    $err
                );
                break;
		}

		if ( $text ) {
			echo '<div class="notice ' . esc_attr( $class ) . ' is-dismissible"><p>' . esc_html( $text ) . '</p></div>';
		}
	}

    /**
     * API base URL, filterable.
     *
     * @return string
     */
    private function api_base() {
        // Use www to avoid cross-domain redirects which may strip Authorization headers.
        return apply_filters( 'chatolia_api_base_url', 'https://www.chatolia.com' );
    }

    /**
     * Embed base URL, filterable.
     *
     * @return string
     */
    private function embed_base() {
        return apply_filters( 'chatolia_embed_base_url', 'https://www.chatolia.com' );
    }

    /**
     * Default demo agent ID used for previews when the user has not connected yet.
     *
     * @return string
     */
    private function default_preview_agent_id() {
        $default = 'syty3erj8pa30im5';
        return apply_filters( 'chatolia_preview_agent_id', $default );
    }

    /**
     * Render a minimal preview page for the widget in an iframe.
     * Enqueues the widget via WordPress and prints footer scripts.
     */
    public function render_widget_preview() {
        if ( ! current_user_can( $this->capability() ) ) {
            wp_die( esc_html__( 'You do not have permission to perform this action.', 'chatolia' ) );
        }
        check_admin_referer( 'chatolia_widget_preview' );

        // Fetch settings to configure the widget.
        $settings = $this->get_settings();
        $agent_id = isset( $settings['agent_id'] ) ? $settings['agent_id'] : '';
        $mode     = isset( $_GET['preview_mode'] ) ? sanitize_key( wp_unslash( $_GET['preview_mode'] ) ) : 'agent'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $use_demo = ( 'demo' === $mode ) || empty( $agent_id );

        if ( $use_demo ) {
            $agent_id = $this->default_preview_agent_id();
        }
        if ( empty( $agent_id ) ) {
            wp_die( esc_html__( 'No agent configured for preview.', 'chatolia' ) );
        }
        $theme    = isset( $settings['theme'] ) ? $settings['theme'] : 'light';
        $position = isset( $settings['position'] ) ? $settings['position'] : 'bottom-right';

        // Enqueue the widget script using the public helper.
        if ( ! class_exists( 'Chatolia_Public' ) ) {
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-chatolia-public.php';
        }
        $pub = new Chatolia_Public( $this->plugin_name, $this->version );
        $pub->enqueue_widget_script( $agent_id, $position, $theme );

        // Output a minimal HTML document and print footer scripts.
        nocache_headers();
        header( 'Content-Type: text/html; charset=' . get_bloginfo( 'charset' ) );
        echo '<!doctype html><html><head><meta charset="' . esc_attr( get_bloginfo( 'charset' ) ) . '" />';
        echo '<meta name="viewport" content="width=device-width, initial-scale=1" />';
        echo '<style>html,body{height:100%;margin:0;padding:0;background:#fff;}</style>';
        echo '</head><body>';
        // Print footer scripts to emit the enqueued widget.
        wp_print_footer_scripts();
        echo '</body></html>';
        exit;
    }

}
