<?php
/**
 * Leads Class
 * Handles lead capture and management
 */

if (!defined('ABSPATH')) {
    exit;
}

class Chatly_Leads {
    
    public function __construct() {
        // AJAX handlers
        add_action('wp_ajax_chatly_capture_lead', array($this, 'handle_capture'));
        add_action('wp_ajax_nopriv_chatly_capture_lead', array($this, 'handle_capture'));
        add_action('wp_ajax_chatly_export_leads', array($this, 'handle_export'));
    }
    
    /**
     * Handle lead capture AJAX
     */
    public function handle_capture() {
        check_ajax_referer('chatly_chatbot_nonce', 'nonce');
        
        $name = sanitize_text_field($_POST['name'] ?? '');
        $email = sanitize_email($_POST['email'] ?? '');
        $phone = sanitize_text_field($_POST['phone'] ?? '');
        $interest = sanitize_text_field($_POST['interest'] ?? '');
        $session_id = sanitize_text_field($_POST['session_id'] ?? '');
        $send_transcript = !empty($_POST['send_transcript']);
        $conversation = json_decode(stripslashes($_POST['conversation'] ?? '[]'), true);
        
        if (empty($name) || empty($email)) {
            wp_send_json_error('Name and email are required');
            return;
        }
        
        if (!is_email($email)) {
            wp_send_json_error('Invalid email address');
            return;
        }
        
        // Save lead to database
        $lead_id = $this->save_lead($name, $email, $phone, $interest, $session_id);
        
        if (!$lead_id) {
            wp_send_json_error('Failed to save lead');
            return;
        }
        
        // Send to integrations
        $this->send_to_integrations($name, $email, $phone, $interest);
        
        // Send transcript if requested
        if ($send_transcript && !empty($conversation)) {
            $this->send_transcript($name, $email, $conversation);
        }
        
        wp_send_json_success(array(
            'message' => 'Thank you! We\'ll be in touch soon.',
            'lead_id' => $lead_id
        ));
    }
    
    /**
     * Save lead to database
     */
    private function save_lead($name, $email, $phone, $interest, $session_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_leads';
        
        // Get conversation ID if exists
        $conv_table = $wpdb->prefix . 'chatly_conversations';
        $conversation_id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $conv_table WHERE session_id = %s ORDER BY started_at DESC LIMIT 1",
            $session_id
        ));
        
        $result = $wpdb->insert($table, array(
            'conversation_id' => $conversation_id,
            'name' => $name,
            'email' => $email,
            'phone' => $phone,
            'interest' => $interest,
            'source' => 'chatbot',
            'status' => 'new',
            'created_at' => current_time('mysql')
        ));
        
        if ($result) {
            // Mark conversation as having lead
            if ($conversation_id) {
                $wpdb->update($conv_table, 
                    array('lead_captured' => 1), 
                    array('id' => $conversation_id)
                );
            }
            
            return $wpdb->insert_id;
        }
        
        return false;
    }
    
    /**
     * Send to email integrations
     */
    private function send_to_integrations($name, $email, $phone, $interest) {
        $integration = Chatly_Main::get_option('email_integration', 'none');
        
        switch ($integration) {
            case 'mailpoet':
                $this->send_to_mailpoet($name, $email, $phone);
                break;
            case 'mailchimp':
                $this->send_to_mailchimp($name, $email);
                break;
            case 'constantcontact':
                $this->send_to_constantcontact($name, $email, $phone);
                break;
            case 'aweber':
                $this->send_to_aweber($name, $email);
                break;
            case 'webhook':
                $this->send_to_webhook($name, $email, $phone, $interest);
                break;
        }
    }
    
    /**
     * Send to MailPoet
     */
    private function send_to_mailpoet($name, $email, $phone) {
        if (!class_exists(\MailPoet\API\API::class)) {
            return false;
        }
        
        try {
            $mailpoet_api = \MailPoet\API\API::MP('v1');
            $list_id = Chatly_Main::get_option('mailpoet_list_id', '');
            
            if (empty($list_id)) {
                return false;
            }
            
            $name_parts = explode(' ', trim($name), 2);
            
            $subscriber_data = array(
                'email' => $email,
                'first_name' => $name_parts[0],
                'last_name' => isset($name_parts[1]) ? $name_parts[1] : '',
                'cf_1' => 'Chatly' // Custom field for source - creates field if doesn't exist
            );
            
            $mailpoet_api->addSubscriber($subscriber_data, array($list_id));
            
            return true;
        } catch (\Exception $e) {
            error_log('ACA MailPoet Error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send to Mailchimp
     */
    private function send_to_mailchimp($name, $email) {
        $api_key = Chatly_Main::get_option('mailchimp_api_key', '');
        $list_id = Chatly_Main::get_option('mailchimp_list_id', '');
        
        if (empty($api_key) || empty($list_id)) {
            return false;
        }
        
        // Extract datacenter from API key
        $dc = substr($api_key, strpos($api_key, '-') + 1);
        $url = "https://{$dc}.api.mailchimp.com/3.0/lists/{$list_id}/members";
        
        $name_parts = explode(' ', trim($name), 2);
        
        $response = wp_remote_post($url, array(
            'headers' => array(
                'Authorization' => 'Basic ' . base64_encode('user:' . $api_key),
                'Content-Type' => 'application/json'
            ),
            'body' => wp_json_encode(array(
                'email_address' => $email,
                'status' => 'subscribed',
                'merge_fields' => array(
                    'FNAME' => $name_parts[0],
                    'LNAME' => isset($name_parts[1]) ? $name_parts[1] : ''
                ),
                'tags' => array('Chatly') // Add source tag
            )),
            'timeout' => 15
        ));
        
        return !is_wp_error($response);
    }
    
    /**
     * Send to Constant Contact
     */
    private function send_to_constantcontact($name, $email, $phone) {
        $access_token = Chatly_Main::get_option('constantcontact_access_token', '');
        $list_id = Chatly_Main::get_option('constantcontact_list_id', '');
        
        if (empty($access_token) || empty($list_id)) {
            return false;
        }
        
        $name_parts = explode(' ', trim($name), 2);
        
        $url = 'https://api.cc.email/v3/contacts';
        
        $response = wp_remote_post($url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ),
            'body' => wp_json_encode(array(
                'email_address' => array(
                    'address' => $email,
                    'permission_to_send' => 'implicit'
                ),
                'first_name' => $name_parts[0],
                'last_name' => isset($name_parts[1]) ? $name_parts[1] : '',
                'phone_numbers' => !empty($phone) ? array(
                    array(
                        'phone_number' => $phone,
                        'kind' => 'other'
                    )
                ) : array(),
                'list_memberships' => array($list_id),
                'create_source' => 'Contact',
                'notes' => array(
                    array(
                        'content' => 'Source: Chatly'
                    )
                )
            )),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            error_log('Chatly Constant Contact Error: ' . $response->get_error_message());
            return false;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        return ($code >= 200 && $code < 300);
    }
    
    /**
     * Send to AWeber
     */
    private function send_to_aweber($name, $email) {
        $api_key = Chatly_Main::get_option('aweber_api_key', '');
        $account_id = Chatly_Main::get_option('aweber_account_id', '');
        $list_id = Chatly_Main::get_option('aweber_list_id', '');
        
        if (empty($api_key) || empty($account_id) || empty($list_id)) {
            return false;
        }
        
        $url = "https://api.aweber.com/1.0/accounts/{$account_id}/lists/{$list_id}/subscribers";
        
        $response = wp_remote_post($url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ),
            'body' => wp_json_encode(array(
                'email' => $email,
                'name' => $name,
                'tags' => array('chatly', 'website-lead')
            )),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            error_log('Chatly AWeber Error: ' . $response->get_error_message());
            return false;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        return ($code >= 200 && $code < 300);
    }
    
    /**
     * Send to webhook
     */
    private function send_to_webhook($name, $email, $phone, $interest) {
        $webhook_url = Chatly_Main::get_option('webhook_url', '');
        
        if (empty($webhook_url)) {
            return false;
        }
        
        $response = wp_remote_post($webhook_url, array(
            'headers' => array('Content-Type' => 'application/json'),
            'body' => wp_json_encode(array(
                'name' => $name,
                'email' => $email,
                'phone' => $phone,
                'interest' => $interest,
                'source' => 'Chatly',
                'timestamp' => current_time('mysql'),
                'site_url' => get_site_url()
            )),
            'timeout' => 15
        ));
        
        return !is_wp_error($response);
    }
    
    /**
     * Send chat transcript
     */
    private function send_transcript($name, $email, $conversation) {
        $site_name = get_bloginfo('name');
        $subject = sprintf(esc_html__('Your conversation with %s', 'chatly-ai-chatbot'), $site_name);
        
        $html = $this->build_transcript_html($name, $conversation);
        
        $headers = array('Content-Type: text/html; charset=UTF-8');
        
        // Send to visitor
        $result = wp_mail($email, $subject, $html, $headers);
        
        // Get settings
        $options = get_option('chatly_settings', array());
        
        // Send to admin if enabled
        if (!empty($options['send_admin_transcript']) && $options['send_admin_transcript'] === '1') {
            $admin_email = get_option('admin_email');
            $admin_subject = sprintf(esc_html__('[Chatly] Conversation transcript from %s', 'chatly-ai-chatbot'), $name);
            wp_mail($admin_email, $admin_subject, $html, $headers);
        }
        
        // Send to additional email if provided
        if (!empty($options['additional_transcript_email']) && is_email($options['additional_transcript_email'])) {
            $additional_subject = sprintf(esc_html__('[Chatly] Conversation transcript from %s', 'chatly-ai-chatbot'), $name);
            wp_mail($options['additional_transcript_email'], $additional_subject, $html, $headers);
        }
        
        return $result;
    }
    
    /**
     * Build HTML transcript
     */
    private function build_transcript_html($name, $conversation) {
        $site_name = get_bloginfo('name');
        $site_url = get_site_url();
        $first_name = explode(' ', trim($name))[0];
        
        $primary = Chatly_Main::get_option('primary_color', '#667eea');
        $secondary = Chatly_Main::get_option('secondary_color', '#764ba2');
        
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
</head>
<body style="margin:0;padding:0;font-family:Arial,sans-serif;background:#f4f4f4;">
    <table width="100%" cellpadding="0" cellspacing="0" style="background:#f4f4f4;padding:20px 0;">
        <tr>
            <td align="center">
                <table width="600" cellpadding="0" cellspacing="0" style="background:#fff;border-radius:8px;overflow:hidden;box-shadow:0 2px 8px rgba(0,0,0,0.1);">
                    <tr>
                        <td style="background:linear-gradient(135deg,' . esc_attr($primary) . ' 0%,' . esc_attr($secondary) . ' 100%);padding:30px 40px;text-align:center;">
                            <h1 style="margin:0;color:#fff;font-size:24px;">Your Conversation</h1>
                        </td>
                    </tr>
                    <tr>
                        <td style="padding:30px 40px 20px;">
                            <p style="margin:0 0 20px;font-size:16px;color:#333;">Hi ' . esc_html($first_name) . ',</p>
                            <p style="margin:0 0 20px;font-size:16px;color:#333;">Here\'s a copy of your chat conversation for your records.</p>
                        </td>
                    </tr>
                    <tr>
                        <td style="padding:0 40px 30px;">
                            <div style="background:#f8f9fa;border-radius:8px;padding:20px;">';
        
        foreach ($conversation as $msg) {
            if (!empty($msg['user'])) {
                $html .= '<div style="margin-bottom:15px;text-align:right;">
                    <div style="display:inline-block;max-width:80%;background:linear-gradient(135deg,' . esc_attr($primary) . ' 0%,' . esc_attr($secondary) . ' 100%);color:#fff;padding:12px 16px;border-radius:18px;border-bottom-right-radius:4px;text-align:left;">
                        <strong style="font-size:12px;opacity:0.9;">You:</strong><br>' . nl2br(esc_html($msg['user'])) . '
                    </div>
                </div>';
            }
            if (!empty($msg['assistant'])) {
                $html .= '<div style="margin-bottom:15px;">
                    <div style="display:inline-block;max-width:80%;background:#fff;border:1px solid #e1e4e8;color:#333;padding:12px 16px;border-radius:18px;border-bottom-left-radius:4px;">
                        <strong style="font-size:12px;color:' . esc_attr($primary) . ';">Assistant:</strong><br>' . nl2br(esc_html($msg['assistant'])) . '
                    </div>
                </div>';
            }
        }
        
        $html .= '</div>
                        </td>
                    </tr>
                    <tr>
                        <td style="background:#f8f9fa;padding:20px 40px;text-align:center;border-top:1px solid #e1e4e8;">
                            <p style="margin:0 0 10px;font-size:14px;color:#666;">' . esc_html($site_name) . '<br>
                            <a href="' . esc_url($site_url) . '" style="color:' . esc_attr($primary) . ';text-decoration:none;">' . esc_url($site_url) . '</a></p>
                        </td>
                    </tr>
                </table>
            </td>
        </tr>
    </table>
</body>
</html>';
        
        return $html;
    }
    
    /**
     * Handle CSV export
     */
    public function handle_export() {
        check_ajax_referer('chatly_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_leads';
        $leads = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC");
        
        $filename = 'leads-' . date('Y-m-d') . '.csv';
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');
        
        $output = fopen('php://output', 'w');
        
        // Header row
        fputcsv($output, array('Date', 'Name', 'Email', 'Phone', 'Interest', 'Status'));
        
        foreach ($leads as $lead) {
            fputcsv($output, array(
                $lead->created_at,
                $lead->name,
                $lead->email,
                $lead->phone,
                $lead->interest,
                $lead->status
            ));
        }
        
        fclose($output);
        exit;
    }
    
    /**
     * Get lead by ID
     */
    public static function get($id) {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_leads';
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));
    }
    
    /**
     * Update lead status
     */
    public static function update_status($id, $status) {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_leads';
        
        return $wpdb->update($table, 
            array('status' => $status), 
            array('id' => $id)
        );
    }
    
    /**
     * Get leads count by status
     */
    public static function count_by_status($status = null) {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_leads';
        
        if ($status) {
            return $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE status = %s",
                $status
            ));
        }
        
        return $wpdb->get_var("SELECT COUNT(*) FROM $table");
    }
}
