<?php
/**
 * Knowledge Base Class
 * Manages custom Q&A pairs for the AI
 */

if (!defined('ABSPATH')) {
    exit;
}

class Chatly_Knowledge_Base {
    
    /**
     * Find relevant Q&A for a message
     */
    public function find_relevant($message) {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '$table'") !== $table) {
            return '';
        }
        
        $message_lower = strtolower($message);
        $words = array_filter(explode(' ', $message_lower), function($w) {
            return strlen($w) > 2;
        });
        
        if (empty($words)) {
            return '';
        }
        
        // Build search query
        $like_clauses = array();
        $params = array();
        
        foreach ($words as $word) {
            $like_clauses[] = "(LOWER(question) LIKE %s OR LOWER(keywords) LIKE %s)";
            $params[] = '%' . $wpdb->esc_like($word) . '%';
            $params[] = '%' . $wpdb->esc_like($word) . '%';
        }
        
        $where = implode(' OR ', $like_clauses);
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT question, answer FROM $table WHERE enabled = 1 AND ($where) ORDER BY priority DESC LIMIT 3",
            ...$params
        ));
        
        if (empty($results)) {
            return '';
        }
        
        $context = '';
        foreach ($results as $row) {
            $context .= "Q: " . $row->question . "\nA: " . $row->answer . "\n\n";
            
            // Increment use count
            $wpdb->query($wpdb->prepare(
                "UPDATE $table SET use_count = use_count + 1 WHERE question = %s",
                $row->question
            ));
        }
        
        return $context;
    }
    
    /**
     * Add a Q&A pair
     */
    public function add($question, $answer, $keywords = '', $category = '', $priority = 0) {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        
        return $wpdb->insert($table, array(
            'question' => $question,
            'answer' => $answer,
            'keywords' => $keywords,
            'category' => $category,
            'priority' => $priority,
            'enabled' => 1,
            'created_at' => current_time('mysql')
        ));
    }
    
    /**
     * Update a Q&A pair
     */
    public function update($id, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        
        $data['updated_at'] = current_time('mysql');
        
        return $wpdb->update($table, $data, array('id' => $id));
    }
    
    /**
     * Delete a Q&A pair
     */
    public function delete($id) {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        
        return $wpdb->delete($table, array('id' => $id));
    }
    
    /**
     * Get all Q&A pairs
     */
    public function get_all($category = null) {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        
        if ($category) {
            return $wpdb->get_results($wpdb->prepare(
                "SELECT * FROM $table WHERE category = %s ORDER BY priority DESC, created_at DESC",
                $category
            ));
        }
        
        return $wpdb->get_results(
            "SELECT * FROM $table ORDER BY priority DESC, created_at DESC"
        );
    }
    
    /**
     * Get categories
     */
    public function get_categories() {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        
        return $wpdb->get_col(
            "SELECT DISTINCT category FROM $table WHERE category IS NOT NULL AND category != '' ORDER BY category"
        );
    }
}
