<?php
/**
 * Chatbot Frontend Class
 * Handles frontend display and AJAX interactions
 */

if (!defined('ABSPATH')) {
    exit;
}

class Chatly_Chatbot {
    
    public function __construct() {
        // Enqueue scripts
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // Render chatbot in footer
        add_action('wp_footer', array($this, 'render_chatbot'), 999);
        
        // AJAX handlers
        add_action('wp_ajax_chatly_send_message', array($this, 'handle_message'));
        add_action('wp_ajax_nopriv_chatly_send_message', array($this, 'handle_message'));
    }
    
    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts() {
        // Check if enabled
        if (Chatly_Main::get_option('enabled', '1') !== '1') {
            return;
        }
        
        // Don't show in admin
        if (is_admin()) {
            return;
        }
        
        // React (local)
        wp_enqueue_script(
            'chatly-react',
            CHATLY_PLUGIN_URL . 'assets/js/react.production.min.js',
            array(),
            '18.2.0',
            true
        );
        
        wp_enqueue_script(
            'chatly-react-dom',
            CHATLY_PLUGIN_URL . 'assets/js/react-dom.production.min.js',
            array('chatly-react'),
            '18.2.0',
            true
        );
        
        // Chatbot script
        wp_enqueue_script(
            'chatly-chatbot',
            CHATLY_PLUGIN_URL . 'assets/js/chatbot.js',
            array('chatly-react', 'chatly-react-dom'),
            CHATLY_VERSION,
            true
        );
        
        // Localize script with settings
        $options = Chatly_Main::get_all_options();
        wp_localize_script('chatly-chatbot', 'chatlyChatbot', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('chatly_chatbot_nonce'),
            'settings' => array(
                'chatTitle' => $options['chat_title'],
                'welcomeMessage' => $options['welcome_message'],
                'showWelcomeBubble' => $options['show_welcome_bubble'] === '1',
                'initialMessage' => $options['initial_bot_message'],
                'enableLeadCapture' => $options['enable_lead_capture'] === '1',
                'leadFormTitle' => $options['lead_form_title'],
                'leadFormTrigger' => $options['lead_form_trigger'],
                'triggerKeywords' => array_map('trim', explode(',', $options['lead_trigger_keywords'])),
                'collectPhone' => $options['collect_phone'] === '1',
                'sendTranscriptOption' => $options['send_transcript_option'] === '1',
                'position' => $options['position']
            )
        ));
        
        // Register empty style handle so we can add inline CSS
        wp_register_style('chatly-chatbot-styles', false);
        wp_enqueue_style('chatly-chatbot-styles');
        
        // Add base CSS (from render_inline_styles)
        $base_css = $this->get_base_css();
        wp_add_inline_style('chatly-chatbot-styles', $base_css);
        
        // Add dynamic CSS
        $dynamic_css = $this->generate_dynamic_css($options);
        wp_add_inline_style('chatly-chatbot-styles', $dynamic_css);
        
        // Custom CSS removed per WordPress.org guidelines
        // Users can add custom CSS via WordPress Customizer → Additional CSS
    }
    
    /**
     * Render the chatbot HTML container
     */
    public function render_chatbot() {
        // Check if enabled
        if (Chatly_Main::get_option('enabled', '1') !== '1') {
            return;
        }
        
        // Don't show in admin
        if (is_admin()) {
            return;
        }
        
        // Just output the container div - scripts are enqueued via wp_enqueue_scripts
        echo '<div id="chatly-chatbot-root"></div>';
    }
    
    /**
     * Generate dynamic CSS based on settings
     */
    private function generate_dynamic_css($options) {
        $primary = esc_attr($options['primary_color']);
        $secondary = esc_attr($options['secondary_color']);
        $text_color = esc_attr($options['text_color']);
        $position = $options['position'];
        $bottom = intval($options['bottom_offset']);
        $side = intval($options['side_offset']);
        $button_size = intval($options['button_size']);
        $window_width = intval($options['window_width']);
        $window_height = intval($options['window_height']);
        
        $side_prop = $position === 'left' ? 'left' : 'right';
        
        $css = ":root {
            --chatly-primary: {$primary};
            --chatly-secondary: {$secondary};
            --chatly-text: {$text_color};
            --chatly-gradient: linear-gradient(135deg, {$primary} 0%, {$secondary} 100%);
        }
        
        #chatly-chatbot-root {
            position: fixed !important;
            bottom: {$bottom}px !important;
            {$side_prop}: {$side}px !important;
            z-index: 2147483647 !important;
        }
        
        .chatly-chat-toggle {
            width: {$button_size}px !important;
            height: {$button_size}px !important;
            background: var(--chatly-gradient) !important;
            color: var(--chatly-text) !important;
        }
        
        .chatly-chat-window {
            width: {$window_width}px !important;
            height: {$window_height}px !important;
        }
        
        .chatly-chat-header {
            background: var(--chatly-gradient) !important;
            color: var(--chatly-text) !important;
        }
        
        .chatly-message-user .chatly-message-content {
            background: var(--chatly-gradient) !important;
            color: var(--chatly-text) !important;
        }
        
        .chatly-send-btn {
            background: var(--chatly-gradient) !important;
            color: var(--chatly-text) !important;
        }
        
        .chatly-form-submit {
            background: var(--chatly-gradient) !important;
            color: var(--chatly-text) !important;
        }
        
        .chatly-welcome-bubble {
            {$side_prop}: 0 !important;
        }
        
        @media (max-width: 480px) {
            .chatly-chat-window {
                width: 100% !important;
                height: 100% !important;
                max-height: 100% !important;
                border-radius: 0 !important;
                position: fixed !important;
                top: 0 !important;
                left: 0 !important;
                right: 0 !important;
                bottom: 0 !important;
            }
        }";
        
        return $css;
    }
    
    /**
     * Get base CSS styles for chatbot
     */
    private function get_base_css() {
        return "
        /* Reset for chatbot elements */
        #chatly-chatbot-root,
        #chatly-chatbot-root *:not(svg):not(svg *),
        #chatly-chatbot-root *:not(svg):not(svg *)::before,
        #chatly-chatbot-root *:not(svg):not(svg *)::after {
            all: initial;
            box-sizing: border-box !important;
        }
        
        #chatly-chatbot-root {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
        }
        
        .chatly-chatbot-container {
            position: relative !important;
            display: block !important;
        }
        
        /* Welcome Bubble */
        .chatly-welcome-bubble {
            position: absolute !important;
            bottom: 70px !important;
            background: #ffffff !important;
            color: #333333 !important;
            padding: 14px 36px 14px 16px !important;
            border-radius: 12px !important;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15) !important;
            font-size: 14px !important;
            line-height: 1.4 !important;
            min-width: 220px !important;
            max-width: 320px !important;
            width: max-content !important;
            cursor: pointer !important;
            display: block !important;
            animation: acaFadeIn 0.5s ease-out !important;
            border: 1px solid #e0e0e0 !important;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
            white-space: normal !important;
        }
        
        .chatly-welcome-bubble span {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
            display: inline !important;
        }
        
        .chatly-welcome-bubble::after {
            content: '' !important;
            position: absolute !important;
            bottom: -8px !important;
            right: 24px !important;
            width: 0 !important;
            height: 0 !important;
            border-left: 8px solid transparent !important;
            border-right: 8px solid transparent !important;
            border-top: 8px solid #ffffff !important;
            display: block !important;
        }
        
        .chatly-welcome-close {
            position: absolute !important;
            top: 8px !important;
            right: 10px !important;
            background: none !important;
            border: none !important;
            font-size: 18px !important;
            color: #999 !important;
            cursor: pointer !important;
            padding: 0 !important;
            line-height: 1 !important;
            display: block !important;
        }
        
        /* Toggle Button */
        .chatly-chat-toggle {
            border-radius: 50% !important;
            border: none !important;
            cursor: pointer !important;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15) !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            transition: transform 0.2s, box-shadow 0.2s !important;
        }
        
        .chatly-chat-toggle:hover {
            transform: scale(1.05) !important;
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.2) !important;
        }
        
        .chatly-chat-toggle svg {
            width: 28px !important;
            height: 28px !important;
            display: block !important;
        }
        
        .chatly-chat-toggle svg path {
            fill: none !important;
            stroke: currentColor !important;
            stroke-width: 2 !important;
            stroke-linecap: round !important;
            stroke-linejoin: round !important;
        }
        
        /* Chat Window */
        .chatly-chat-window {
            max-height: 80vh !important;
            background: #ffffff !important;
            border-radius: 12px !important;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.15) !important;
            display: flex !important;
            flex-direction: column !important;
            overflow: hidden !important;
            animation: acaSlideUp 0.3s ease-out !important;
            position: absolute !important;
            bottom: 70px !important;
            right: 0 !important;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
        }
        
        @keyframes acaSlideUp {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        @keyframes acaFadeIn {
            from { opacity: 0; transform: translateY(10px) scale(0.95); }
            to { opacity: 1; transform: translateY(0) scale(1); }
        }
        
        /* Header */
        .chatly-chat-header {
            padding: 16px 20px !important;
            display: flex !important;
            justify-content: space-between !important;
            align-items: center !important;
            flex-shrink: 0 !important;
        }
        
        .chatly-chat-header h3 {
            margin: 0 !important;
            font-size: 18px !important;
            font-weight: 600 !important;
            color: inherit !important;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
            display: block !important;
        }
        
        .chatly-close-btn {
            background: none !important;
            border: none !important;
            color: inherit !important;
            font-size: 28px !important;
            line-height: 1 !important;
            cursor: pointer !important;
            padding: 0 !important;
            opacity: 0.8 !important;
        }
        
        .chatly-close-btn:hover {
            opacity: 1 !important;
        }
        
        /* Messages */
        .chatly-chat-messages {
            flex: 1 !important;
            overflow-y: auto !important;
            padding: 20px !important;
            background: #f7f8fa !important;
            display: block !important;
        }
        
        .chatly-message {
            margin-bottom: 16px !important;
            display: flex !important;
        }
        
        .chatly-message-user {
            justify-content: flex-end !important;
        }
        
        .chatly-message-bot {
            justify-content: flex-start !important;
        }
        
        .chatly-message-content {
            max-width: 75% !important;
            padding: 12px 16px !important;
            border-radius: 18px !important;
            word-wrap: break-word !important;
            line-height: 1.4 !important;
            font-size: 14px !important;
            display: inline-block !important;
            white-space: pre-line !important;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
        }
        
        .chatly-message-bot .chatly-message-content {
            background: #ffffff !important;
            color: #333333 !important;
            border-bottom-left-radius: 4px !important;
            box-shadow: 0 1px 2px rgba(0, 0, 0, 0.1) !important;
        }
        
        .chatly-message-user .chatly-message-content {
            border-bottom-right-radius: 4px !important;
        }
        
        /* Typing Indicator */
        .chatly-typing {
            display: flex !important;
            gap: 4px !important;
            align-items: center !important;
            padding: 12px 16px !important;
        }
        
        .chatly-typing span {
            width: 8px !important;
            height: 8px !important;
            background: #999 !important;
            border-radius: 50% !important;
            animation: acaTyping 1.4s infinite !important;
            display: inline-block !important;
        }
        
        .chatly-typing span:nth-child(2) { animation-delay: 0.2s !important; }
        .chatly-typing span:nth-child(3) { animation-delay: 0.4s !important; }
        
        @keyframes acaTyping {
            0%, 60%, 100% { transform: translateY(0); }
            30% { transform: translateY(-8px); }
        }
        
        /* Lead Form */
        .chatly-lead-form {
            padding: 16px 20px !important;
            background: #ffffff !important;
            border-top: 1px solid #e1e4e8 !important;
            display: block !important;
        }
        
        .chatly-lead-form h4 {
            margin: 0 0 12px 0 !important;
            font-size: 16px !important;
            font-weight: 600 !important;
            color: #333 !important;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
            display: block !important;
        }
        
        .chatly-lead-form input[type=\"text\"],
        .chatly-lead-form input[type=\"email\"],
        .chatly-lead-form input[type=\"tel\"] {
            width: 100% !important;
            padding: 10px 12px !important;
            border: 1px solid #d1d5db !important;
            border-radius: 6px !important;
            margin-bottom: 10px !important;
            font-size: 14px !important;
            background: #ffffff !important;
            color: #333333 !important;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
            display: block !important;
        }
        
        .chatly-lead-form input:focus {
            outline: none !important;
            border-color: var(--chatly-primary, #667eea) !important;
        }
        
        .chatly-checkbox-label {
            display: flex !important;
            align-items: center !important;
            gap: 8px !important;
            margin: 10px 0 !important;
            font-size: 14px !important;
            cursor: pointer !important;
            color: #333 !important;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
        }
        
        .chatly-form-buttons {
            display: flex !important;
            gap: 8px !important;
            margin-top: 12px !important;
        }
        
        .chatly-form-buttons button {
            flex: 1 !important;
            padding: 10px !important;
            border-radius: 6px !important;
            border: none !important;
            font-size: 14px !important;
            font-weight: 600 !important;
            cursor: pointer !important;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
            text-align: center !important;
        }
        
        .chatly-form-cancel {
            background: #f3f4f6 !important;
            color: #333 !important;
        }
        
        /* Input Area */
        .chatly-chat-input-wrapper {
            display: flex !important;
            gap: 8px !important;
            padding: 16px 20px !important;
            background: #ffffff !important;
            border-top: 1px solid #e1e4e8 !important;
            align-items: flex-end !important;
        }
        
        .chatly-chat-input {
            flex: 1 !important;
            padding: 10px 12px !important;
            border: 1px solid #d1d5db !important;
            border-radius: 20px !important;
            font-size: 14px !important;
            resize: none !important;
            max-height: 100px !important;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif !important;
            background: #ffffff !important;
            color: #333333 !important;
            line-height: 1.4 !important;
        }
        
        .chatly-chat-input:focus {
            outline: none !important;
            border-color: var(--chatly-primary, #667eea) !important;
        }
        
        .chatly-send-btn {
            width: 40px !important;
            height: 40px !important;
            border-radius: 50% !important;
            border: none !important;
            cursor: pointer !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            flex-shrink: 0 !important;
        }
        
        .chatly-send-btn:disabled {
            opacity: 0.5 !important;
            cursor: not-allowed !important;
        }
        
        .chatly-send-btn svg {
            width: 20px !important;
            height: 20px !important;
        }
        ";
    }
    
    /**
     * Render inline styles (comprehensive CSS)
     */
    /**
     * Handle chat message AJAX
     */
    public function handle_message() {
        check_ajax_referer('chatly_chatbot_nonce', 'nonce');
        
        $message = sanitize_text_field($_POST['message'] ?? '');
        $session_id = sanitize_text_field($_POST['session_id'] ?? '');
        $history = json_decode(stripslashes($_POST['history'] ?? '[]'), true);
        
        if (empty($message)) {
            wp_send_json_error('Message is required');
            return;
        }
        
        // Get system prompt
        $system_prompt = Chatly_Main::get_option('system_prompt', '');
        
        // Include knowledge base context if available
        $knowledge = new Chatly_Knowledge_Base();
        $relevant_qa = $knowledge->find_relevant($message);
        if (!empty($relevant_qa)) {
            $system_prompt .= "\n\nRelevant information from knowledge base:\n" . $relevant_qa;
        }
        
        // Build messages array
        $messages = array();
        foreach ($history as $h) {
            if (!empty($h['user'])) {
                $messages[] = array('role' => 'user', 'content' => $h['user']);
            }
            if (!empty($h['assistant'])) {
                $messages[] = array('role' => 'assistant', 'content' => $h['assistant']);
            }
        }
        $messages[] = array('role' => 'user', 'content' => $message);
        
        // Send to AI
        $ai = new Chatly_AI_Provider();
        $result = $ai->send_message($system_prompt, $messages);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
            return;
        }
        
        // Check for lead capture trigger
        $trigger_lead = $this->should_trigger_lead_capture($message);
        
        // Log conversation
        $this->log_conversation($session_id, $message, $result['response']);
        
        wp_send_json_success(array(
            'response' => $result['response'],
            'trigger_lead_capture' => $trigger_lead
        ));
    }
    
    /**
     * Check if should trigger lead capture
     */
    private function should_trigger_lead_capture($message) {
        $trigger = Chatly_Main::get_option('lead_form_trigger', 'interest');
        
        if ($trigger !== 'interest') {
            return false;
        }
        
        $keywords = Chatly_Main::get_option('lead_trigger_keywords', '');
        $keywords = array_map('trim', explode(',', strtolower($keywords)));
        $message_lower = strtolower($message);
        
        foreach ($keywords as $keyword) {
            if (!empty($keyword) && strpos($message_lower, $keyword) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Log conversation to database
     */
    private function log_conversation($session_id, $user_message, $bot_response) {
        global $wpdb;
        
        // Get or create conversation
        $conv_table = $wpdb->prefix . 'chatly_conversations';
        $msg_table = $wpdb->prefix . 'chatly_messages';
        
        $conversation = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $conv_table WHERE session_id = %s AND ended_at IS NULL",
            $session_id
        ));
        
        if (!$conversation) {
            $wpdb->insert($conv_table, array(
                'session_id' => $session_id,
                'visitor_ip' => $this->get_client_ip(),
                'page_url' => wp_get_referer(),
                'started_at' => current_time('mysql')
            ));
            $conversation_id = $wpdb->insert_id;
        } else {
            $conversation_id = $conversation->id;
        }
        
        // Insert messages
        $wpdb->insert($msg_table, array(
            'conversation_id' => $conversation_id,
            'role' => 'user',
            'content' => $user_message,
            'created_at' => current_time('mysql')
        ));
        
        $wpdb->insert($msg_table, array(
            'conversation_id' => $conversation_id,
            'role' => 'assistant',
            'content' => $bot_response,
            'created_at' => current_time('mysql')
        ));
        
        // Update message count
        $wpdb->query($wpdb->prepare(
            "UPDATE $conv_table SET message_count = message_count + 2 WHERE id = %d",
            $conversation_id
        ));
    }
    
    /**
     * Get client IP
     */
    private function get_client_ip() {
        $ip = '';
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        }
        return sanitize_text_field($ip);
    }
}
