<?php
/**
 * AI Provider Class
 * Handles communication with Anthropic and OpenAI APIs
 */

if (!defined('ABSPATH')) {
    exit;
}

class Chatly_AI_Provider {
    
    private $provider;
    private $api_key;
    private $model;
    
    public function __construct() {
        $this->provider = Chatly_Main::get_option('ai_provider', 'anthropic');
        
        if ($this->provider === 'anthropic') {
            $this->api_key = Chatly_Main::get_option('anthropic_api_key', '');
            $this->model = Chatly_Main::get_option('anthropic_model', 'claude-sonnet-4-20250514');
        } else {
            $this->api_key = Chatly_Main::get_option('openai_api_key', '');
            $this->model = Chatly_Main::get_option('openai_model', 'gpt-4o-mini');
        }
    }
    
    /**
     * Send a message and get a response
     */
    public function send_message($system_prompt, $messages) {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', 'API key not configured');
        }
        
        if ($this->provider === 'anthropic') {
            return $this->send_anthropic_message($system_prompt, $messages);
        } else {
            return $this->send_openai_message($system_prompt, $messages);
        }
    }
    
    /**
     * Send message to Anthropic Claude
     */
    private function send_anthropic_message($system_prompt, $messages) {
        $api_url = 'https://api.anthropic.com/v1/messages';
        
        // Format messages for Anthropic
        $formatted_messages = array();
        foreach ($messages as $msg) {
            $formatted_messages[] = array(
                'role' => $msg['role'] === 'bot' ? 'assistant' : $msg['role'],
                'content' => $msg['content']
            );
        }
        
        $body = array(
            'model' => $this->model,
            'max_tokens' => 1024,
            'system' => $system_prompt,
            'messages' => $formatted_messages
        );
        
        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $this->api_key,
                'anthropic-version' => '2023-06-01'
            ),
            'body' => wp_json_encode($body),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        if ($response_code !== 200) {
            $error_msg = isset($data['error']['message']) ? $data['error']['message'] : 'Unknown error';
            return new WP_Error('api_error', $error_msg);
        }
        
        if (empty($data['content'][0]['text'])) {
            return new WP_Error('parse_error', 'Unable to parse response');
        }
        
        return array(
            'response' => $data['content'][0]['text'],
            'tokens' => array(
                'input' => $data['usage']['input_tokens'] ?? 0,
                'output' => $data['usage']['output_tokens'] ?? 0
            )
        );
    }
    
    /**
     * Send message to OpenAI
     */
    private function send_openai_message($system_prompt, $messages) {
        $api_url = 'https://api.openai.com/v1/chat/completions';
        
        // Format messages for OpenAI (add system message first)
        $formatted_messages = array(
            array(
                'role' => 'system',
                'content' => $system_prompt
            )
        );
        
        foreach ($messages as $msg) {
            $formatted_messages[] = array(
                'role' => $msg['role'] === 'bot' ? 'assistant' : $msg['role'],
                'content' => $msg['content']
            );
        }
        
        $body = array(
            'model' => $this->model,
            'messages' => $formatted_messages,
            'max_tokens' => 1024,
            'temperature' => 0.7
        );
        
        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->api_key
            ),
            'body' => wp_json_encode($body),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        if ($response_code !== 200) {
            $error_msg = isset($data['error']['message']) ? $data['error']['message'] : 'Unknown error';
            return new WP_Error('api_error', $error_msg);
        }
        
        if (empty($data['choices'][0]['message']['content'])) {
            return new WP_Error('parse_error', 'Unable to parse response');
        }
        
        return array(
            'response' => $data['choices'][0]['message']['content'],
            'tokens' => array(
                'input' => $data['usage']['prompt_tokens'] ?? 0,
                'output' => $data['usage']['completion_tokens'] ?? 0
            )
        );
    }
    
    /**
     * Test API connection
     */
    public function test_connection($provider, $api_key) {
        if (empty($api_key)) {
            return array(
                'success' => false,
                'message' => 'API key is required'
            );
        }
        
        if ($provider === 'anthropic') {
            return $this->test_anthropic_connection($api_key);
        } else {
            return $this->test_openai_connection($api_key);
        }
    }
    
    /**
     * Test Anthropic connection
     */
    private function test_anthropic_connection($api_key) {
        $api_url = 'https://api.anthropic.com/v1/messages';
        
        $body = array(
            'model' => 'claude-3-haiku-20240307', // Use cheapest model for test
            'max_tokens' => 10,
            'messages' => array(
                array('role' => 'user', 'content' => 'Hi')
            )
        );
        
        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01'
            ),
            'body' => wp_json_encode($body),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code === 200) {
            return array(
                'success' => true,
                'message' => 'Successfully connected to Anthropic API'
            );
        }
        
        $response_body = json_decode(wp_remote_retrieve_body($response), true);
        $error_msg = isset($response_body['error']['message']) ? $response_body['error']['message'] : 'Connection failed';
        
        return array(
            'success' => false,
            'message' => $error_msg
        );
    }
    
    /**
     * Test OpenAI connection
     */
    private function test_openai_connection($api_key) {
        $api_url = 'https://api.openai.com/v1/chat/completions';
        
        $body = array(
            'model' => 'gpt-3.5-turbo', // Use cheapest model for test
            'messages' => array(
                array('role' => 'user', 'content' => 'Hi')
            ),
            'max_tokens' => 10
        );
        
        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'body' => wp_json_encode($body),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code === 200) {
            return array(
                'success' => true,
                'message' => 'Successfully connected to OpenAI API'
            );
        }
        
        $response_body = json_decode(wp_remote_retrieve_body($response), true);
        $error_msg = isset($response_body['error']['message']) ? $response_body['error']['message'] : 'Connection failed';
        
        return array(
            'success' => false,
            'message' => $error_msg
        );
    }
    
    /**
     * Check if provider is configured
     */
    public function is_configured() {
        return !empty($this->api_key);
    }
    
    /**
     * Get current provider name
     */
    public function get_provider_name() {
        return $this->provider === 'anthropic' ? 'Anthropic (Claude)' : 'OpenAI (GPT)';
    }
}
