<?php
/**
 * Admin Settings Class
 * Handles all admin functionality and settings pages
 */

if (!defined('ABSPATH')) {
    exit;
}

class Chatly_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_ajax_chatly_test_api', array($this, 'ajax_test_api'));
        add_action('wp_ajax_chatly_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_chatly_generate_messaging', array($this, 'ajax_generate_messaging'));
        
        // Knowledge Base AJAX
        add_action('wp_ajax_chatly_kb_save', array($this, 'ajax_kb_save'));
        add_action('wp_ajax_chatly_kb_get', array($this, 'ajax_kb_get'));
        add_action('wp_ajax_chatly_kb_delete', array($this, 'ajax_kb_delete'));
        
        // Conversations AJAX
        add_action('wp_ajax_chatly_get_conversation', array($this, 'ajax_get_conversation'));
        
        // Activation redirect
        add_action('admin_init', array($this, 'activation_redirect'));
    }
    
    /**
     * Redirect to settings page on activation
     */
    public function activation_redirect() {
        if (get_transient('chatly_activation_redirect')) {
            delete_transient('chatly_activation_redirect');
            if (!isset($_GET['activate-multi'])) {
                wp_safe_redirect(admin_url('admin.php?page=chatly'));
                exit;
            }
        }
    }
    
    /**
     * Add admin menu pages
     */
    public function add_admin_menu() {
        // Main menu page
        add_menu_page(
            __('Chatly', 'chatly-ai-chatbot'),
            __('Chatly', 'chatly-ai-chatbot'),
            'manage_options',
            'chatly',
            array($this, 'render_settings_page'),
            'dashicons-format-chat',
            30
        );
        
        // Submenu pages
        add_submenu_page(
            'chatly',
            __('Settings', 'chatly-ai-chatbot'),
            __('Settings', 'chatly-ai-chatbot'),
            'manage_options',
            'chatly',
            array($this, 'render_settings_page')
        );
        
        add_submenu_page(
            'chatly',
            __('Knowledge Base', 'chatly-ai-chatbot'),
            __('Knowledge Base', 'chatly-ai-chatbot'),
            'manage_options',
            'chatly-knowledge',
            array($this, 'render_knowledge_page')
        );
        
        add_submenu_page(
            'chatly',
            __('Leads', 'chatly-ai-chatbot'),
            __('Leads', 'chatly-ai-chatbot'),
            'manage_options',
            'chatly-leads',
            array($this, 'render_leads_page')
        );
        
        add_submenu_page(
            'chatly',
            __('Conversations', 'chatly-ai-chatbot'),
            __('Conversations', 'chatly-ai-chatbot'),
            'manage_options',
            'chatly-conversations',
            array($this, 'render_conversations_page')
        );
        
        add_submenu_page(
            'chatly',
            __('Analytics', 'chatly-ai-chatbot'),
            __('Analytics', 'chatly-ai-chatbot'),
            'manage_options',
            'chatly-analytics',
            array($this, 'render_analytics_page')
        );
        
        add_submenu_page(
            'chatly',
            __('SEO Insights', 'chatly-ai-chatbot'),
            __('SEO Insights', 'chatly-ai-chatbot'),
            'manage_options',
            'chatly-insights',
            array($this, 'render_insights_page')
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('chatly_settings_group', 'chatly_settings', array(
            'sanitize_callback' => array($this, 'sanitize_settings')
        ));
    }
    
    /**
     * Sanitize settings
     */
    public function sanitize_settings($input) {
        // IMPORTANT: Start with existing settings, not empty array
        // This preserves settings from other tabs
        $existing = get_option('chatly_settings', array());
        $defaults = Chatly_Main::get_default_options();
        $sanitized = array_merge($defaults, $existing);
        
        // Determine which tab we're on based on submitted fields
        $current_tab = $this->detect_current_tab($input);
        
        // Define which fields belong to which tab
        $tab_fields = array(
            'general' => array('enabled', 'chat_title'),
            'ai' => array('ai_provider', 'anthropic_api_key', 'anthropic_model', 'openai_api_key', 'openai_model'),
            'appearance' => array('primary_color', 'secondary_color', 'text_color', 'position', 'bottom_offset', 'side_offset', 'button_size', 'window_width', 'window_height'),
            'messages' => array('show_welcome_bubble', 'welcome_message', 'initial_bot_message', 'system_prompt'),
            'leads' => array('enable_lead_capture', 'lead_form_title', 'lead_form_trigger', 'lead_trigger_keywords', 'collect_phone', 'email_integration', 'mailpoet_list_id', 'mailchimp_api_key', 'mailchimp_list_id', 'constantcontact_api_key', 'constantcontact_api_secret', 'constantcontact_access_token', 'constantcontact_list_id', 'aweber_api_key', 'aweber_account_id', 'aweber_list_id', 'webhook_url', 'send_transcript_option', 'send_admin_transcript', 'additional_transcript_email'),
        );
        
        // Get fields for current tab (or all fields if we can't detect)
        $fields_to_process = isset($tab_fields[$current_tab]) ? $tab_fields[$current_tab] : array_keys($defaults);
        
        // Checkbox fields that should be '0' when not submitted (only for current tab)
        $checkbox_fields = array('enabled', 'show_welcome_bubble', 'enable_lead_capture', 'collect_phone', 'send_transcript_option', 'send_admin_transcript', 'enable_analytics');
        
        // Process only fields from the current tab
        foreach ($fields_to_process as $key) {
            if (!isset($defaults[$key])) continue;
            
            $default = $defaults[$key];
            
            if (isset($input[$key])) {
                switch ($key) {
                    case 'anthropic_api_key':
                    case 'openai_api_key':
                    case 'mailchimp_api_key':
                    case 'constantcontact_api_key':
                    case 'constantcontact_api_secret':
                    case 'constantcontact_access_token':
                    case 'aweber_api_key':
                        $sanitized[$key] = sanitize_text_field($input[$key]);
                        break;
                    case 'system_prompt':
                    case 'initial_bot_message':
                        $sanitized[$key] = wp_kses_post($input[$key]);
                        break;
                    case 'webhook_url':
                        $sanitized[$key] = esc_url_raw($input[$key]);
                        break;
                    case 'primary_color':
                    case 'secondary_color':
                    case 'text_color':
                        $sanitized[$key] = sanitize_hex_color($input[$key]) ?: $default;
                        break;
                    case 'bottom_offset':
                    case 'side_offset':
                    case 'button_size':
                    case 'window_width':
                    case 'window_height':
                        $sanitized[$key] = absint($input[$key]);
                        break;
                    default:
                        $sanitized[$key] = sanitize_text_field($input[$key]);
                }
            } else {
                // Only set checkbox to '0' if it's a checkbox field in the current tab
                if (in_array($key, $checkbox_fields) && in_array($key, $fields_to_process)) {
                    $sanitized[$key] = '0';
                }
                // For non-checkbox fields that aren't submitted, keep existing value
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Detect which tab the form was submitted from
     */
    private function detect_current_tab($input) {
        // Check for unique fields in each tab
        if (isset($input['chat_title']) || isset($input['enabled'])) {
            return 'general';
        }
        if (isset($input['ai_provider']) || isset($input['anthropic_api_key']) || isset($input['openai_api_key'])) {
            return 'ai';
        }
        if (isset($input['primary_color']) || isset($input['position']) || isset($input['button_size'])) {
            return 'appearance';
        }
        if (isset($input['system_prompt']) || isset($input['welcome_message']) || isset($input['initial_bot_message'])) {
            return 'messages';
        }
        if (isset($input['lead_form_title']) || isset($input['email_integration']) || isset($input['lead_form_trigger'])) {
            return 'leads';
        }
        return 'general';
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Debug: Uncomment next line to see what $hook value is
        // error_log('Chatly Hook: ' . $hook);
        
        // Only load on our plugin pages
        // Hook could be various formats depending on folder name:
        // - "toplevel_page_chatly"
        // - "chatly_page_chatly-leads"  
        // - "toplevel_page_chatly-ai-chatbot"
        // Just check if hook contains "chatly" anywhere
        if (stripos($hook, 'chatly') === false) {
            return;
        }
        
        // WordPress color picker
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        // Our admin styles
        wp_enqueue_style(
            'chatly-admin-styles',
            CHATLY_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            CHATLY_VERSION
        );
        
        // Add insights page CSS
        $insights_css = "
            .chatly-insights-intro { font-size: 14px; color: #666; margin-bottom: 20px; }
            .chatly-insights-period { margin-bottom: 20px; }
            .chatly-insights-period label { margin-right: 10px; font-weight: 600; }
            .chatly-insights-period select { padding: 5px 10px; }
            .chatly-yoast-badge { display: inline-block; background: linear-gradient(135deg, #a4286a 0%, #00a0d2 100%); color: #fff; padding: 4px 12px; border-radius: 20px; font-size: 12px; margin-left: 15px; vertical-align: middle; }
            .chatly-yoast-badge .dashicons { font-size: 14px; width: 14px; height: 14px; vertical-align: middle; margin-right: 3px; }
            .chatly-insights-content { margin-top: 20px; }
            .chatly-insight-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin-bottom: 30px; }
            .chatly-insight-card { background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 20px; }
            .chatly-insight-card h3 { margin: 0 0 15px; padding-bottom: 10px; border-bottom: 2px solid #667eea; color: #333; display: flex; align-items: center; gap: 8px; }
            .chatly-insight-card h3 .dashicons { color: #667eea; }
            .chatly-keyword-list { display: flex; flex-wrap: wrap; gap: 8px; }
            .chatly-keyword-tag { background: #f0f0f1; padding: 4px 12px; border-radius: 15px; font-size: 13px; display: inline-flex; align-items: center; gap: 5px; }
            .chatly-keyword-tag .count { background: #667eea; color: #fff; padding: 2px 6px; border-radius: 10px; font-size: 11px; }
            .chatly-question-list { list-style: none; padding: 0; margin: 0; }
            .chatly-question-list li { padding: 10px 0; border-bottom: 1px solid #eee; display: flex; justify-content: space-between; align-items: center; }
            .chatly-question-list li:last-child { border-bottom: none; }
            .chatly-question-text { flex: 1; }
            .chatly-question-count { background: #667eea; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 12px; margin-left: 10px; }
            .chatly-gap-item { background: #fff8e5; border-left: 4px solid #ffb900; padding: 12px 15px; margin-bottom: 10px; border-radius: 0 4px 4px 0; }
            .chatly-gap-item .question { font-weight: 600; margin-bottom: 5px; }
            .chatly-gap-item .suggestion { font-size: 13px; color: #666; }
            .chatly-code-block { background: #1e1e1e; color: #d4d4d4; padding: 20px; border-radius: 8px; overflow-x: auto; font-family: 'Monaco', 'Consolas', monospace; font-size: 13px; line-height: 1.5; }
            .chatly-copy-btn { margin-top: 10px; }
            .chatly-stat-highlight { font-size: 36px; font-weight: 700; color: #667eea; }
            .chatly-stat-label { font-size: 14px; color: #666; }
            .chatly-action-btn { display: inline-block; background: #667eea; color: #fff; padding: 8px 16px; border-radius: 4px; text-decoration: none; margin-top: 10px; }
            .chatly-action-btn:hover { background: #5a6fd6; color: #fff; }
            .chatly-opportunity-good { color: #00a32a; }
            .chatly-opportunity-new { color: #d63638; }
            .chatly-empty-state { text-align: center; padding: 40px; color: #666; }
            .chatly-empty-state .dashicons { font-size: 48px; width: 48px; height: 48px; color: #ddd; }
            .chatly-suggestion-item { background: #f0f6fc; border-left: 4px solid #2271b1; padding: 12px 15px; margin-bottom: 10px; border-radius: 0 4px 4px 0; }
            .chatly-add-to-kb { float: right; font-size: 12px; }
        ";
        wp_add_inline_style('chatly-admin-styles', $insights_css);
        
        // Our admin scripts
        wp_enqueue_script(
            'chatly-admin-scripts',
            CHATLY_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery', 'wp-color-picker'),
            CHATLY_VERSION,
            true
        );
        
        // Localize script
        wp_localize_script('chatly-admin-scripts', 'chatlyAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('chatly_admin_nonce'),
            'strings' => array(
                'testing' => __('Testing...', 'chatly-ai-chatbot'),
                'success' => __('Connection successful!', 'chatly-ai-chatbot'),
                'error' => __('Connection failed:', 'chatly-ai-chatbot'),
                'saving' => __('Saving...', 'chatly-ai-chatbot'),
                'saved' => __('Settings saved!', 'chatly-ai-chatbot'),
                'saveError' => __('Error saving settings:', 'chatly-ai-chatbot'),
            )
        ));
    }
    
    /**
     * Render main settings page
     */
    public function render_settings_page() {
        $options = Chatly_Main::get_all_options();
        $active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'general';
        ?>
        <div class="wrap chatly-admin-wrap">
            <h1>
                <span class="dashicons dashicons-format-chat"></span>
                <?php esc_html_e('Chatly', 'chatly-ai-chatbot'); ?>
            </h1>
            
            <nav class="nav-tab-wrapper chatly-nav-tabs">
                <a href="?page=chatly&tab=general" class="nav-tab <?php echo $active_tab === 'general' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-admin-generic"></span>
                    <?php esc_html_e('General', 'chatly-ai-chatbot'); ?>
                </a>
                <a href="?page=chatly&tab=license" class="nav-tab <?php echo $active_tab === 'license' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-admin-network"></span>
                    <?php esc_html_e('License', 'chatly-ai-chatbot'); ?>
                </a>
                <a href="?page=chatly&tab=ai" class="nav-tab <?php echo $active_tab === 'ai' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-cloud"></span>
                    <?php esc_html_e('AI Provider', 'chatly-ai-chatbot'); ?>
                </a>
                <a href="?page=chatly&tab=appearance" class="nav-tab <?php echo $active_tab === 'appearance' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-art"></span>
                    <?php esc_html_e('Appearance', 'chatly-ai-chatbot'); ?>
                </a>
                <a href="?page=chatly&tab=messages" class="nav-tab <?php echo $active_tab === 'messages' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-admin-comments"></span>
                    <?php esc_html_e('Messages', 'chatly-ai-chatbot'); ?>
                </a>
                <a href="?page=chatly&tab=leads" class="nav-tab <?php echo $active_tab === 'leads' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-groups"></span>
                    <?php esc_html_e('Lead Capture', 'chatly-ai-chatbot'); ?>
                </a>
            </nav>
            
            <form method="post" action="options.php" id="chatly-settings-form">
                <?php settings_fields('chatly_settings_group'); ?>
                
                <div class="chatly-settings-content">
                    <?php
                    switch ($active_tab) {
                        case 'license':
                            $this->render_license_tab();
                            break;
                        case 'ai':
                            $this->render_ai_tab($options);
                            break;
                        case 'appearance':
                            $this->render_appearance_tab($options);
                            break;
                        case 'messages':
                            $this->render_messages_tab($options);
                            break;
                        case 'leads':
                            $this->render_leads_tab($options);
                            break;
                        default:
                            $this->render_general_tab($options);
                    }
                    ?>
                </div>
                
                <?php if ($active_tab !== 'license') : ?>
                <div class="chatly-settings-footer">
                    <?php submit_button(__('Save Settings', 'chatly-ai-chatbot'), 'primary', 'submit', false); ?>
                    <span class="chatly-save-status"></span>
                </div>
                <?php endif; ?>
            </form>
        </div>
        <?php
    }
    
    /**
     * License Tab
     */
    private function render_license_tab() {
        // License functionality removed in free version
        // This tab should not be accessible, but just in case:
        ?>
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('License Not Required', 'chatly-ai-chatbot'); ?></h2>
            <p><?php esc_html_e('This is the free version of Chatly. No license key is required.', 'chatly-ai-chatbot'); ?></p>
            <p><?php esc_html_e('To unlock premium features including unlimited conversations, all email integrations, SEO insights, and priority support:', 'chatly-ai-chatbot'); ?> <a href="https://hellochatly.com/product/chatly-pro/" target="_blank"><?php esc_html_e('Upgrade to Chatly Pro', 'chatly-ai-chatbot'); ?></a></p>
        </div>
        <?php
        return;
    }
    
    /**
     * General Tab
     */
    private function render_general_tab($options) {
        ?>
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('General Settings', 'chatly-ai-chatbot'); ?></h2>
            
            <table class="form-table">
                <tr>
                    <th scope="row"><?php esc_html_e('Enable Chatbot', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <label class="chatly-switch">
                            <input type="checkbox" name="chatly_settings[enabled]" value="1" <?php checked($options['enabled'], '1'); ?>>
                            <span class="chatly-slider"></span>
                        </label>
                        <p class="description"><?php esc_html_e('Turn the chatbot on or off on your website.', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="chat_title"><?php esc_html_e('Chat Window Title', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <input type="text" id="chat_title" name="chatly_settings[chat_title]" value="<?php echo esc_attr($options['chat_title']); ?>" class="regular-text">
                        <p class="description"><?php esc_html_e('The title shown in the chat window header.', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
            </table>
        </div>
        
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('Quick Status', 'chatly-ai-chatbot'); ?></h2>
            
            <div class="chatly-status-grid">
                <div class="chatly-status-card">
                    <div class="chatly-status-icon <?php echo !empty($options['anthropic_api_key']) || !empty($options['openai_api_key']) ? 'status-ok' : 'status-warning'; ?>">
                        <span class="dashicons dashicons-cloud"></span>
                    </div>
                    <div class="chatly-status-info">
                        <h4><?php esc_html_e('AI Provider', 'chatly-ai-chatbot'); ?></h4>
                        <p>
                            <?php 
                            if (!empty($options['anthropic_api_key']) || !empty($options['openai_api_key'])) {
                                echo esc_html(ucfirst($options['ai_provider'])) . ' ' . esc_html__('configured', 'chatly-ai-chatbot');
                            } else {
                                esc_html_e('Not configured', 'chatly-ai-chatbot');
                            }
                            ?>
                        </p>
                    </div>
                </div>
                
                <div class="chatly-status-card">
                    <div class="chatly-status-icon <?php echo $options['enabled'] === '1' ? 'status-ok' : 'status-off'; ?>">
                        <span class="dashicons dashicons-visibility"></span>
                    </div>
                    <div class="chatly-status-info">
                        <h4><?php esc_html_e('Chatbot Status', 'chatly-ai-chatbot'); ?></h4>
                        <p><?php echo $options['enabled'] === '1' ? esc_html__('Active', 'chatly-ai-chatbot') : esc_html__('Disabled', 'chatly-ai-chatbot'); ?></p>
                    </div>
                </div>
                
                <div class="chatly-status-card">
                    <div class="chatly-status-icon status-info">
                        <span class="dashicons dashicons-chart-bar"></span>
                    </div>
                    <div class="chatly-status-info">
                        <h4><?php esc_html_e('Today\'s Chats', 'chatly-ai-chatbot'); ?></h4>
                        <p><?php echo $this->get_today_stats(); ?></p>
                    </div>
                </div>
                
                <div class="chatly-status-card">
                    <div class="chatly-status-icon status-info">
                        <span class="dashicons dashicons-groups"></span>
                    </div>
                    <div class="chatly-status-info">
                        <h4><?php esc_html_e('Total Leads', 'chatly-ai-chatbot'); ?></h4>
                        <p><?php echo $this->get_total_leads(); ?></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * AI Provider Tab
     */
    private function render_ai_tab($options) {
        ?>
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('AI Provider Settings', 'chatly-ai-chatbot'); ?></h2>
            
            <table class="form-table">
                <tr>
                    <th scope="row"><?php esc_html_e('Select AI Provider', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <div class="chatly-provider-selector">
                            <label class="chatly-provider-option <?php echo $options['ai_provider'] === 'anthropic' ? 'selected' : ''; ?>">
                                <input type="radio" name="chatly_settings[ai_provider]" value="anthropic" <?php checked($options['ai_provider'], 'anthropic'); ?>>
                                <div class="chatly-provider-card">
                                    <strong>Anthropic (Claude)</strong>
                                    <span>Recommended for nuanced conversations</span>
                                </div>
                            </label>
                            <label class="chatly-provider-option <?php echo $options['ai_provider'] === 'openai' ? 'selected' : ''; ?>">
                                <input type="radio" name="chatly_settings[ai_provider]" value="openai" <?php checked($options['ai_provider'], 'openai'); ?>>
                                <div class="chatly-provider-card">
                                    <strong>OpenAI (GPT)</strong>
                                    <span>Popular choice with flexible pricing</span>
                                </div>
                            </label>
                        </div>
                    </td>
                </tr>
            </table>
        </div>
        
        <!-- Anthropic Settings -->
        <div class="chatly-settings-section chatly-provider-settings" id="anthropic-settings" style="<?php echo $options['ai_provider'] !== 'anthropic' ? 'display:none;' : ''; ?>">
            <h2><?php esc_html_e('Anthropic (Claude) Settings', 'chatly-ai-chatbot'); ?></h2>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="anthropic_api_key"><?php esc_html_e('API Key', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <input type="password" id="anthropic_api_key" name="chatly_settings[anthropic_api_key]" value="<?php echo esc_attr($options['anthropic_api_key']); ?>" class="regular-text">
                        <button type="button" class="button chatly-toggle-password" data-target="anthropic_api_key">
                            <span class="dashicons dashicons-visibility"></span>
                        </button>
                        <button type="button" class="button chatly-test-api" data-provider="anthropic">
                            <?php esc_html_e('Test Connection', 'chatly-ai-chatbot'); ?>
                        </button>
                        <span class="chatly-test-result"></span>
                        <p class="description">
                            <?php esc_html_e('Get your API key from', 'chatly-ai-chatbot'); ?>
                            <a href="https://console.anthropic.com/" target="_blank">console.anthropic.com</a>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="anthropic_model"><?php esc_html_e('Model', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <select id="anthropic_model" name="chatly_settings[anthropic_model]">
                            <option value="claude-sonnet-4-20250514" <?php selected($options['anthropic_model'], 'claude-sonnet-4-20250514'); ?>>Claude Sonnet 4 (Recommended)</option>
                            <option value="claude-3-5-sonnet-20241022" <?php selected($options['anthropic_model'], 'claude-3-5-sonnet-20241022'); ?>>Claude 3.5 Sonnet</option>
                            <option value="claude-3-haiku-20240307" <?php selected($options['anthropic_model'], 'claude-3-haiku-20240307'); ?>>Claude 3 Haiku (Faster/Cheaper)</option>
                        </select>
                    </td>
                </tr>
            </table>
        </div>
        
        <!-- OpenAI Settings -->
        <div class="chatly-settings-section chatly-provider-settings" id="openai-settings" style="<?php echo $options['ai_provider'] !== 'openai' ? 'display:none;' : ''; ?>">
            <h2><?php esc_html_e('OpenAI (GPT) Settings', 'chatly-ai-chatbot'); ?></h2>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="openai_api_key"><?php esc_html_e('API Key', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <input type="password" id="openai_api_key" name="chatly_settings[openai_api_key]" value="<?php echo esc_attr($options['openai_api_key']); ?>" class="regular-text">
                        <button type="button" class="button chatly-toggle-password" data-target="openai_api_key">
                            <span class="dashicons dashicons-visibility"></span>
                        </button>
                        <button type="button" class="button chatly-test-api" data-provider="openai">
                            <?php esc_html_e('Test Connection', 'chatly-ai-chatbot'); ?>
                        </button>
                        <span class="chatly-test-result"></span>
                        <p class="description">
                            <?php esc_html_e('Get your API key from', 'chatly-ai-chatbot'); ?>
                            <a href="https://platform.openai.com/api-keys" target="_blank">platform.openai.com</a>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="openai_model"><?php esc_html_e('Model', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <select id="openai_model" name="chatly_settings[openai_model]">
                            <option value="gpt-4o" <?php selected($options['openai_model'], 'gpt-4o'); ?>>GPT-4o (Recommended)</option>
                            <option value="gpt-4o-mini" <?php selected($options['openai_model'], 'gpt-4o-mini'); ?>>GPT-4o Mini (Faster/Cheaper)</option>
                            <option value="gpt-4-turbo" <?php selected($options['openai_model'], 'gpt-4-turbo'); ?>>GPT-4 Turbo</option>
                            <option value="gpt-3.5-turbo" <?php selected($options['openai_model'], 'gpt-3.5-turbo'); ?>>GPT-3.5 Turbo (Budget)</option>
                        </select>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }
    
    /**
     * Appearance Tab
     */
    private function render_appearance_tab($options) {
        ?>
        <div class="chatly-appearance-layout">
            <div class="chatly-appearance-settings">
                <div class="chatly-settings-section">
                    <h2><?php esc_html_e('Colors', 'chatly-ai-chatbot'); ?></h2>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="primary_color"><?php esc_html_e('Primary Color', 'chatly-ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="text" id="primary_color" name="chatly_settings[primary_color]" value="<?php echo esc_attr($options['primary_color']); ?>" class="chatly-color-picker">
                                <p class="description"><?php esc_html_e('Main button and header color', 'chatly-ai-chatbot'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="secondary_color"><?php esc_html_e('Secondary Color', 'chatly-ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="text" id="secondary_color" name="chatly_settings[secondary_color]" value="<?php echo esc_attr($options['secondary_color']); ?>" class="chatly-color-picker">
                                <p class="description"><?php esc_html_e('Gradient end color', 'chatly-ai-chatbot'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="text_color"><?php esc_html_e('Button Text Color', 'chatly-ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="text" id="text_color" name="chatly_settings[text_color]" value="<?php echo esc_attr($options['text_color']); ?>" class="chatly-color-picker">
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="chatly-settings-section">
                    <h2><?php esc_html_e('Position & Size', 'chatly-ai-chatbot'); ?></h2>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Button Position', 'chatly-ai-chatbot'); ?></th>
                            <td>
                                <label>
                                    <input type="radio" name="chatly_settings[position]" value="right" <?php checked($options['position'], 'right'); ?>>
                                    <?php esc_html_e('Bottom Right', 'chatly-ai-chatbot'); ?>
                                </label>
                                &nbsp;&nbsp;
                                <label>
                                    <input type="radio" name="chatly_settings[position]" value="left" <?php checked($options['position'], 'left'); ?>>
                                    <?php esc_html_e('Bottom Left', 'chatly-ai-chatbot'); ?>
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="bottom_offset"><?php esc_html_e('Bottom Offset', 'chatly-ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="number" id="bottom_offset" name="chatly_settings[bottom_offset]" value="<?php echo esc_attr($options['bottom_offset']); ?>" min="0" max="200" class="small-text"> px
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="side_offset"><?php esc_html_e('Side Offset', 'chatly-ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="number" id="side_offset" name="chatly_settings[side_offset]" value="<?php echo esc_attr($options['side_offset']); ?>" min="0" max="200" class="small-text"> px
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="button_size"><?php esc_html_e('Button Size', 'chatly-ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="number" id="button_size" name="chatly_settings[button_size]" value="<?php echo esc_attr($options['button_size']); ?>" min="40" max="100" class="small-text"> px
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="window_width"><?php esc_html_e('Chat Window Width', 'chatly-ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="number" id="window_width" name="chatly_settings[window_width]" value="<?php echo esc_attr($options['window_width']); ?>" min="300" max="600" class="small-text"> px
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="window_height"><?php esc_html_e('Chat Window Height', 'chatly-ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="number" id="window_height" name="chatly_settings[window_height]" value="<?php echo esc_attr($options['window_height']); ?>" min="300" max="800" class="small-text"> px
                            </td>
                        </tr>
                    </table>
                </div>
                
            </div>
            
            <div class="chatly-appearance-preview">
                <h3><?php esc_html_e('Preview', 'chatly-ai-chatbot'); ?></h3>
                <div class="chatly-preview-container">
                    <div class="chatly-preview-chat" id="chatly-preview">
                        <!-- Preview will be rendered by JS -->
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Messages Tab
     */
    private function render_messages_tab($options) {
        ?>
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('AI Messaging Generator', 'chatly-ai-chatbot'); ?></h2>
            <p class="description"><?php esc_html_e('Use AI to automatically generate your chatbot messaging. Describe your business and the AI will create a system prompt, welcome message, and initial bot message for you.', 'chatly-ai-chatbot'); ?></p>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="ai_generator_input"><?php esc_html_e('Describe Your Business', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <textarea id="ai_generator_input" rows="4" class="large-text" placeholder="<?php esc_attr_e('Example: We are a plumbing company in Boston. We offer emergency repairs, bathroom renovations, and water heater installation. Our hours are Mon-Fri 8am-6pm. We want the chatbot to be friendly and helpful, and to collect contact info from potential customers.', 'chatly-ai-chatbot'); ?>"></textarea>
                        <p class="description"><?php esc_html_e('Include: what your business does, your services/products, business hours, location, and how you want the bot to behave.', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"></th>
                    <td>
                        <button type="button" class="button button-primary" id="chatly-generate-messaging">
                            <span class="dashicons dashicons-admin-generic" style="margin-top:4px;"></span>
                            <?php esc_html_e('Generate Messaging with AI', 'chatly-ai-chatbot'); ?>
                        </button>
                        <span class="spinner" id="ai-generator-spinner" style="float:none;margin-top:4px;"></span>
                        <span id="ai-generator-status" style="margin-left:10px;"></span>
                    </td>
                </tr>
            </table>
        </div>
        
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('Welcome Bubble', 'chatly-ai-chatbot'); ?></h2>
            
            <table class="form-table">
                <tr>
                    <th scope="row"><?php esc_html_e('Show Welcome Bubble', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <label class="chatly-switch">
                            <input type="checkbox" name="chatly_settings[show_welcome_bubble]" value="1" <?php checked($options['show_welcome_bubble'], '1'); ?>>
                            <span class="chatly-slider"></span>
                        </label>
                        <p class="description"><?php esc_html_e('Show a welcome message bubble above the chat button on page load.', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="welcome_message"><?php esc_html_e('Welcome Message', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <input type="text" id="welcome_message" name="chatly_settings[welcome_message]" value="<?php echo esc_attr($options['welcome_message']); ?>" class="large-text">
                        <p class="description"><?php esc_html_e('The message shown in the welcome bubble. Emojis are supported!', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
            </table>
        </div>
        
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('Initial Bot Message', 'chatly-ai-chatbot'); ?></h2>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="initial_bot_message"><?php esc_html_e('First Message', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <textarea id="initial_bot_message" name="chatly_settings[initial_bot_message]" rows="4" class="large-text"><?php echo esc_textarea($options['initial_bot_message']); ?></textarea>
                        <p class="description"><?php esc_html_e('The first message visitors see when they open the chat. Use line breaks to format. You can include bullet points with •', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
            </table>
        </div>
        
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('System Prompt (AI Instructions)', 'chatly-ai-chatbot'); ?></h2>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="system_prompt"><?php esc_html_e('System Prompt', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <textarea id="system_prompt" name="chatly_settings[system_prompt]" rows="10" class="large-text"><?php echo esc_textarea($options['system_prompt']); ?></textarea>
                        <p class="description">
                            <?php esc_html_e('Instructions that tell the AI how to behave. Include information about your business, services, and how you want the AI to respond.', 'chatly-ai-chatbot'); ?>
                        </p>
                        <div class="chatly-prompt-tips">
                            <strong><?php esc_html_e('Tips for a good system prompt:', 'chatly-ai-chatbot'); ?></strong>
                            <ul>
                                <li><?php esc_html_e('Describe your business and what you offer', 'chatly-ai-chatbot'); ?></li>
                                <li><?php esc_html_e('List your main services or products', 'chatly-ai-chatbot'); ?></li>
                                <li><?php esc_html_e('Include pricing information if appropriate', 'chatly-ai-chatbot'); ?></li>
                                <li><?php esc_html_e('Specify the tone (friendly, professional, casual)', 'chatly-ai-chatbot'); ?></li>
                                <li><?php esc_html_e('Tell it when to suggest contacting you', 'chatly-ai-chatbot'); ?></li>
                            </ul>
                        </div>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }
    
    /**
     * Leads Tab
     */
    private function render_leads_tab($options) {
        ?>
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('Lead Capture Settings', 'chatly-ai-chatbot'); ?></h2>
            
            <table class="form-table">
                <tr>
                    <th scope="row"><?php esc_html_e('Enable Lead Capture', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <label class="chatly-switch">
                            <input type="checkbox" name="chatly_settings[enable_lead_capture]" value="1" <?php checked($options['enable_lead_capture'], '1'); ?>>
                            <span class="chatly-slider"></span>
                        </label>
                        <p class="description"><?php esc_html_e('Show a lead capture form to collect visitor information.', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="lead_form_title"><?php esc_html_e('Form Title', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <input type="text" id="lead_form_title" name="chatly_settings[lead_form_title]" value="<?php echo esc_attr($options['lead_form_title']); ?>" class="regular-text">
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('When to Show Form', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <select name="chatly_settings[lead_form_trigger]">
                            <option value="interest" <?php selected($options['lead_form_trigger'], 'interest'); ?>><?php esc_html_e('When visitor shows interest (keywords)', 'chatly-ai-chatbot'); ?></option>
                            <option value="messages" <?php selected($options['lead_form_trigger'], 'messages'); ?>><?php esc_html_e('After 3 messages', 'chatly-ai-chatbot'); ?></option>
                            <option value="manual" <?php selected($options['lead_form_trigger'], 'manual'); ?>><?php esc_html_e('Never (let AI decide)', 'chatly-ai-chatbot'); ?></option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="lead_trigger_keywords"><?php esc_html_e('Trigger Keywords', 'chatly-ai-chatbot'); ?></label>
                    </th>
                    <td>
                        <textarea id="lead_trigger_keywords" name="chatly_settings[lead_trigger_keywords]" rows="2" class="large-text"><?php echo esc_textarea($options['lead_trigger_keywords']); ?></textarea>
                        <p class="description"><?php esc_html_e('Comma-separated keywords that trigger the lead form.', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Collect Phone Number', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <label class="chatly-switch">
                            <input type="checkbox" name="chatly_settings[collect_phone]" value="1" <?php checked($options['collect_phone'], '1'); ?>>
                            <span class="chatly-slider"></span>
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Send Transcript Option', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <label class="chatly-switch">
                            <input type="checkbox" name="chatly_settings[send_transcript_option]" value="1" <?php checked($options['send_transcript_option'], '1'); ?>>
                            <span class="chatly-slider"></span>
                        </label>
                        <p class="description"><?php esc_html_e('Allow visitors to receive a copy of the chat via email.', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Send Transcript to Admin', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <label class="chatly-switch">
                            <input type="checkbox" name="chatly_settings[send_admin_transcript]" value="1" <?php checked($options['send_admin_transcript'], '1'); ?>>
                            <span class="chatly-slider"></span>
                        </label>
                        <p class="description"><?php printf(esc_html__('Send chat transcripts to the WordPress admin email (%s).', 'chatly-ai-chatbot'), get_option('admin_email')); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Additional Transcript Email', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <input type="email" name="chatly_settings[additional_transcript_email]" value="<?php echo esc_attr($options['additional_transcript_email']); ?>" class="regular-text">
                        <p class="description"><?php esc_html_e('Optional: Send transcripts to this additional email address.', 'chatly-ai-chatbot'); ?></p>
                    </td>
                </tr>
            </table>
        </div>
        
        <div class="chatly-settings-section">
            <h2><?php esc_html_e('Email Integration', 'chatly-ai-chatbot'); ?></h2>
            
            <table class="form-table">
                <tr>
                    <th scope="row"><?php esc_html_e('Integration', 'chatly-ai-chatbot'); ?></th>
                    <td>
                        <select name="chatly_settings[email_integration]" id="email_integration">
                            <option value="none" <?php selected($options['email_integration'], 'none'); ?>><?php esc_html_e('None (store locally only)', 'chatly-ai-chatbot'); ?></option>
                            <option value="mailpoet" <?php selected($options['email_integration'], 'mailpoet'); ?>><?php esc_html_e('MailPoet', 'chatly-ai-chatbot'); ?></option>
                            <option value="mailchimp" <?php selected($options['email_integration'], 'mailchimp'); ?>><?php esc_html_e('Mailchimp', 'chatly-ai-chatbot'); ?></option>
                            <option value="constantcontact" <?php selected($options['email_integration'], 'constantcontact'); ?>><?php esc_html_e('Constant Contact', 'chatly-ai-chatbot'); ?></option>
                            <option value="aweber" <?php selected($options['email_integration'], 'aweber'); ?>><?php esc_html_e('AWeber', 'chatly-ai-chatbot'); ?></option>
                            <option value="webhook" <?php selected($options['email_integration'], 'webhook'); ?>><?php esc_html_e('Custom Webhook', 'chatly-ai-chatbot'); ?></option>
                        </select>
                    </td>
                </tr>
            </table>
            
            <!-- MailPoet Settings -->
            <div class="chatly-integration-settings" id="mailpoet-settings" style="<?php echo $options['email_integration'] !== 'mailpoet' ? 'display:none;' : ''; ?>">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="mailpoet_list_id"><?php esc_html_e('MailPoet List ID', 'chatly-ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="mailpoet_list_id" name="chatly_settings[mailpoet_list_id]" value="<?php echo esc_attr($options['mailpoet_list_id']); ?>" class="regular-text">
                            <p class="description"><?php esc_html_e('The list ID to add subscribers to.', 'chatly-ai-chatbot'); ?></p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- Mailchimp Settings -->
            <div class="chatly-integration-settings" id="mailchimp-settings" style="<?php echo $options['email_integration'] !== 'mailchimp' ? 'display:none;' : ''; ?>">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="mailchimp_api_key"><?php esc_html_e('Mailchimp API Key', 'chatly-ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="password" id="mailchimp_api_key" name="chatly_settings[mailchimp_api_key]" value="<?php echo esc_attr($options['mailchimp_api_key']); ?>" class="regular-text">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="mailchimp_list_id"><?php esc_html_e('Mailchimp List ID', 'chatly-ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="mailchimp_list_id" name="chatly_settings[mailchimp_list_id]" value="<?php echo esc_attr($options['mailchimp_list_id']); ?>" class="regular-text">
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- Constant Contact Settings -->
            <div class="chatly-integration-settings" id="constantcontact-settings" style="<?php echo $options['email_integration'] !== 'constantcontact' ? 'display:none;' : ''; ?>">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="constantcontact_access_token"><?php esc_html_e('Access Token', 'chatly-ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="password" id="constantcontact_access_token" name="chatly_settings[constantcontact_access_token]" value="<?php echo esc_attr($options['constantcontact_access_token']); ?>" class="regular-text">
                            <p class="description"><?php esc_html_e('Your Constant Contact API access token. Get this from the Constant Contact V3 API.', 'chatly-ai-chatbot'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="constantcontact_list_id"><?php esc_html_e('List ID', 'chatly-ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="constantcontact_list_id" name="chatly_settings[constantcontact_list_id]" value="<?php echo esc_attr($options['constantcontact_list_id']); ?>" class="regular-text">
                            <p class="description"><?php esc_html_e('The UUID of the list to add contacts to.', 'chatly-ai-chatbot'); ?></p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- AWeber Settings -->
            <div class="chatly-integration-settings" id="aweber-settings" style="<?php echo $options['email_integration'] !== 'aweber' ? 'display:none;' : ''; ?>">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="aweber_api_key"><?php esc_html_e('Access Token', 'chatly-ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="password" id="aweber_api_key" name="chatly_settings[aweber_api_key]" value="<?php echo esc_attr($options['aweber_api_key']); ?>" class="regular-text">
                            <p class="description"><?php esc_html_e('Your AWeber API access token.', 'chatly-ai-chatbot'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="aweber_account_id"><?php esc_html_e('Account ID', 'chatly-ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="aweber_account_id" name="chatly_settings[aweber_account_id]" value="<?php echo esc_attr($options['aweber_account_id']); ?>" class="regular-text">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="aweber_list_id"><?php esc_html_e('List ID', 'chatly-ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="aweber_list_id" name="chatly_settings[aweber_list_id]" value="<?php echo esc_attr($options['aweber_list_id']); ?>" class="regular-text">
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- Webhook Settings -->
            <div class="chatly-integration-settings" id="webhook-settings" style="<?php echo $options['email_integration'] !== 'webhook' ? 'display:none;' : ''; ?>">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="webhook_url"><?php esc_html_e('Webhook URL', 'chatly-ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="url" id="webhook_url" name="chatly_settings[webhook_url]" value="<?php echo esc_attr($options['webhook_url']); ?>" class="large-text">
                            <p class="description"><?php esc_html_e('Lead data will be POSTed to this URL as JSON.', 'chatly-ai-chatbot'); ?></p>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Knowledge Base page
     */
    public function render_knowledge_page() {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        $entries = $wpdb->get_results("SELECT * FROM $table ORDER BY priority DESC, id DESC");
        ?>
        <div class="wrap chatly-admin-wrap">
            <h1>
                <span class="dashicons dashicons-book"></span>
                <?php esc_html_e('Knowledge Base', 'chatly-ai-chatbot'); ?>
                <a href="#" class="page-title-action" id="chatly-add-knowledge"><?php esc_html_e('Add New', 'chatly-ai-chatbot'); ?></a>
            </h1>
            <p><?php esc_html_e('Add custom questions and answers. The AI will use these to provide more accurate responses.', 'chatly-ai-chatbot'); ?></p>
            
            <?php if (empty($entries)) : ?>
            <div class="chatly-empty-state">
                <span class="dashicons dashicons-book"></span>
                <h3><?php esc_html_e('No knowledge entries yet', 'chatly-ai-chatbot'); ?></h3>
                <p><?php esc_html_e('Add questions and answers to help the AI provide better responses.', 'chatly-ai-chatbot'); ?></p>
            </div>
            <?php else : ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width:30%"><?php esc_html_e('Question / Topic', 'chatly-ai-chatbot'); ?></th>
                        <th style="width:40%"><?php esc_html_e('Answer', 'chatly-ai-chatbot'); ?></th>
                        <th style="width:10%"><?php esc_html_e('Category', 'chatly-ai-chatbot'); ?></th>
                        <th style="width:8%"><?php esc_html_e('Priority', 'chatly-ai-chatbot'); ?></th>
                        <th style="width:12%"><?php esc_html_e('Actions', 'chatly-ai-chatbot'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($entries as $entry) : 
                        $priority_labels = array(0 => 'Normal', 1 => 'High', 2 => 'Highest');
                    ?>
                    <tr>
                        <td><strong><?php echo esc_html(wp_trim_words($entry->question, 10)); ?></strong></td>
                        <td><?php echo esc_html(wp_trim_words($entry->answer, 15)); ?></td>
                        <td><?php echo esc_html($entry->category ?: '—'); ?></td>
                        <td><?php echo esc_html($priority_labels[$entry->priority] ?? 'Normal'); ?></td>
                        <td>
                            <button type="button" class="button button-small chatly-kb-edit" data-id="<?php echo esc_attr($entry->id); ?>"><?php esc_html_e('Edit', 'chatly-ai-chatbot'); ?></button>
                            <button type="button" class="button button-small chatly-kb-delete" data-id="<?php echo esc_attr($entry->id); ?>"><?php esc_html_e('Delete', 'chatly-ai-chatbot'); ?></button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
        
        <!-- Knowledge Base Modal -->
        <div id="chatly-kb-modal" class="chatly-modal" style="display:none;">
            <div class="chatly-modal-content">
                <div class="chatly-modal-header">
                    <h3 id="chatly-kb-modal-title"><?php esc_html_e('Add Knowledge Entry', 'chatly-ai-chatbot'); ?></h3>
                    <button type="button" class="chatly-modal-close">&times;</button>
                </div>
                <div class="chatly-modal-body">
                    <input type="hidden" id="chatly-kb-id" value="">
                    <p><label for="chatly-kb-question"><?php esc_html_e('Question / Topic', 'chatly-ai-chatbot'); ?></label>
                    <input type="text" id="chatly-kb-question" class="large-text" placeholder="e.g., What are your business hours?"></p>
                    <p><label for="chatly-kb-answer"><?php esc_html_e('Answer', 'chatly-ai-chatbot'); ?></label>
                    <textarea id="chatly-kb-answer" rows="4" class="large-text" placeholder="The answer the AI should provide..."></textarea></p>
                    <p><label for="chatly-kb-keywords"><?php esc_html_e('Keywords (comma-separated)', 'chatly-ai-chatbot'); ?></label>
                    <input type="text" id="chatly-kb-keywords" class="large-text" placeholder="e.g., hours, open, schedule"></p>
                    <p><label for="chatly-kb-category"><?php esc_html_e('Category', 'chatly-ai-chatbot'); ?></label>
                    <input type="text" id="chatly-kb-category" class="regular-text" placeholder="e.g., General, Pricing"></p>
                    <p><label for="chatly-kb-priority"><?php esc_html_e('Priority', 'chatly-ai-chatbot'); ?></label>
                    <select id="chatly-kb-priority"><option value="0">Normal</option><option value="1">High</option><option value="2">Highest</option></select></p>
                </div>
                <div class="chatly-modal-footer">
                    <button type="button" class="button" id="chatly-kb-cancel"><?php esc_html_e('Cancel', 'chatly-ai-chatbot'); ?></button>
                    <button type="button" class="button button-primary" id="chatly-kb-save"><?php esc_html_e('Save Entry', 'chatly-ai-chatbot'); ?></button>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Leads page
     */
    public function render_leads_page() {
        ?>
        <div class="wrap chatly-admin-wrap">
            <h1>
                <span class="dashicons dashicons-groups"></span>
                <?php esc_html_e('Leads', 'chatly-ai-chatbot'); ?>
                <a href="#" class="page-title-action" id="chatly-export-leads"><?php esc_html_e('Export CSV', 'chatly-ai-chatbot'); ?></a>
            </h1>
            
            <?php $this->render_leads_table(); ?>
        </div>
        <?php
    }
    
    /**
     * Render leads table
     */
    private function render_leads_table() {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_leads';
        $leads = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC LIMIT 100");
        
        if (empty($leads)) {
            ?>
            <div class="chatly-empty-state">
                <span class="dashicons dashicons-groups"></span>
                <h3><?php esc_html_e('No leads yet', 'chatly-ai-chatbot'); ?></h3>
                <p><?php esc_html_e('Leads captured through the chatbot will appear here.', 'chatly-ai-chatbot'); ?></p>
            </div>
            <?php
            return;
        }
        ?>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th><?php esc_html_e('Date', 'chatly-ai-chatbot'); ?></th>
                    <th><?php esc_html_e('Name', 'chatly-ai-chatbot'); ?></th>
                    <th><?php esc_html_e('Email', 'chatly-ai-chatbot'); ?></th>
                    <th><?php esc_html_e('Phone', 'chatly-ai-chatbot'); ?></th>
                    <th><?php esc_html_e('Interest', 'chatly-ai-chatbot'); ?></th>
                    <th><?php esc_html_e('Status', 'chatly-ai-chatbot'); ?></th>
                    <th><?php esc_html_e('Actions', 'chatly-ai-chatbot'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($leads as $lead) : ?>
                <tr>
                    <td><?php echo esc_html(date('M j, Y g:i a', strtotime($lead->created_at))); ?></td>
                    <td><?php echo esc_html($lead->name); ?></td>
                    <td><a href="mailto:<?php echo esc_attr($lead->email); ?>"><?php echo esc_html($lead->email); ?></a></td>
                    <td><?php echo esc_html($lead->phone); ?></td>
                    <td><?php echo esc_html(wp_trim_words($lead->interest, 10)); ?></td>
                    <td><span class="chatly-status-badge status-<?php echo esc_attr($lead->status); ?>"><?php echo esc_html(ucfirst($lead->status)); ?></span></td>
                    <td>
                        <a href="#" class="chatly-view-lead" data-id="<?php echo esc_attr($lead->id); ?>"><?php esc_html_e('View', 'chatly-ai-chatbot'); ?></a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php
    }
    
    /**
     * Render Conversations page
     */
    public function render_conversations_page() {
        global $wpdb;
        $conv_table = $wpdb->prefix . 'chatly_conversations';
        $msg_table = $wpdb->prefix . 'chatly_messages';
        $conversations = $wpdb->get_results("SELECT c.*, (SELECT COUNT(*) FROM $msg_table WHERE conversation_id = c.id) as msg_count FROM $conv_table c ORDER BY c.started_at DESC LIMIT 100");
        ?>
        <div class="wrap chatly-admin-wrap">
            <h1>
                <span class="dashicons dashicons-admin-comments"></span>
                <?php esc_html_e('Conversations', 'chatly-ai-chatbot'); ?>
            </h1>
            
            <?php if (empty($conversations)) : ?>
            <div class="chatly-empty-state">
                <span class="dashicons dashicons-admin-comments"></span>
                <h3><?php esc_html_e('No conversations yet', 'chatly-ai-chatbot'); ?></h3>
                <p><?php esc_html_e('Conversations from your chatbot will appear here.', 'chatly-ai-chatbot'); ?></p>
            </div>
            <?php else : ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width:18%"><?php esc_html_e('Date', 'chatly-ai-chatbot'); ?></th>
                        <th style="width:10%"><?php esc_html_e('Messages', 'chatly-ai-chatbot'); ?></th>
                        <th style="width:10%"><?php esc_html_e('Lead?', 'chatly-ai-chatbot'); ?></th>
                        <th style="width:47%"><?php esc_html_e('Page URL', 'chatly-ai-chatbot'); ?></th>
                        <th style="width:15%"><?php esc_html_e('Actions', 'chatly-ai-chatbot'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($conversations as $conv) : ?>
                    <tr>
                        <td><?php echo esc_html(date_i18n('M j, Y g:i a', strtotime($conv->started_at))); ?></td>
                        <td><?php echo intval($conv->msg_count); ?></td>
                        <td><?php echo $conv->lead_captured ? '<span class="dashicons dashicons-yes" style="color:#059669"></span>' : '—'; ?></td>
                        <td><small><?php echo esc_html(wp_trim_words($conv->page_url, 10)); ?></small></td>
                        <td><button type="button" class="button button-small chatly-view-conv" data-id="<?php echo esc_attr($conv->id); ?>"><?php esc_html_e('View', 'chatly-ai-chatbot'); ?></button></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
        
        <!-- Conversation Modal -->
        <div id="chatly-conv-modal" class="chatly-modal" style="display:none;">
            <div class="chatly-modal-content" style="max-width:600px">
                <div class="chatly-modal-header">
                    <h3><?php esc_html_e('Conversation', 'chatly-ai-chatbot'); ?></h3>
                    <button type="button" class="chatly-modal-close">&times;</button>
                </div>
                <div class="chatly-modal-body">
                    <div id="chatly-conv-messages" style="max-height:400px;overflow-y:auto;"></div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Analytics page
     */
    public function render_analytics_page() {
        global $wpdb;
        $conv_table = $wpdb->prefix . 'chatly_conversations';
        $msg_table = $wpdb->prefix . 'chatly_messages';
        $leads_table = $wpdb->prefix . 'chatly_leads';
        
        $total_convs = $wpdb->get_var("SELECT COUNT(*) FROM $conv_table") ?: 0;
        $total_msgs = $wpdb->get_var("SELECT COUNT(*) FROM $msg_table") ?: 0;
        $total_leads = $wpdb->get_var("SELECT COUNT(*) FROM $leads_table") ?: 0;
        $today_convs = $wpdb->get_var("SELECT COUNT(*) FROM $conv_table WHERE DATE(started_at) = CURDATE()") ?: 0;
        $week_convs = $wpdb->get_var("SELECT COUNT(*) FROM $conv_table WHERE started_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)") ?: 0;
        $week_leads = $wpdb->get_var("SELECT COUNT(*) FROM $leads_table WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)") ?: 0;
        $conversion_rate = $total_convs > 0 ? round(($total_leads / $total_convs) * 100, 1) : 0;
        ?>
        <div class="wrap chatly-admin-wrap">
            <h1>
                <span class="dashicons dashicons-chart-bar"></span>
                <?php esc_html_e('Analytics', 'chatly-ai-chatbot'); ?>
            </h1>
            
            <div class="chatly-stats-grid">
                <div class="chatly-stat-box">
                    <div class="chatly-stat-number"><?php echo number_format($total_convs); ?></div>
                    <div class="chatly-stat-label"><?php esc_html_e('Total Conversations', 'chatly-ai-chatbot'); ?></div>
                </div>
                <div class="chatly-stat-box">
                    <div class="chatly-stat-number"><?php echo number_format($total_leads); ?></div>
                    <div class="chatly-stat-label"><?php esc_html_e('Total Leads', 'chatly-ai-chatbot'); ?></div>
                </div>
                <div class="chatly-stat-box">
                    <div class="chatly-stat-number"><?php echo $conversion_rate; ?>%</div>
                    <div class="chatly-stat-label"><?php esc_html_e('Conversion Rate', 'chatly-ai-chatbot'); ?></div>
                </div>
                <div class="chatly-stat-box">
                    <div class="chatly-stat-number"><?php echo number_format($total_msgs); ?></div>
                    <div class="chatly-stat-label"><?php esc_html_e('Total Messages', 'chatly-ai-chatbot'); ?></div>
                </div>
            </div>
            
            <div class="chatly-settings-section" style="margin-top:20px">
                <h2><?php esc_html_e('This Week', 'chatly-ai-chatbot'); ?></h2>
                <div class="chatly-stats-grid">
                    <div class="chatly-stat-box">
                        <div class="chatly-stat-number"><?php echo number_format($week_convs); ?></div>
                        <div class="chatly-stat-label"><?php esc_html_e('Conversations (7 days)', 'chatly-ai-chatbot'); ?></div>
                    </div>
                    <div class="chatly-stat-box">
                        <div class="chatly-stat-number"><?php echo number_format($week_leads); ?></div>
                        <div class="chatly-stat-label"><?php esc_html_e('Leads (7 days)', 'chatly-ai-chatbot'); ?></div>
                    </div>
                    <div class="chatly-stat-box">
                        <div class="chatly-stat-number"><?php echo number_format($today_convs); ?></div>
                        <div class="chatly-stat-label"><?php esc_html_e('Today', 'chatly-ai-chatbot'); ?></div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Get today's stats
     */
    private function get_today_stats() {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_conversations';
        $count = $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE DATE(started_at) = CURDATE()");
        return $count ?: '0';
    }
    
    /**
     * Get total leads
     */
    private function get_total_leads() {
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_leads';
        $count = $wpdb->get_var("SELECT COUNT(*) FROM $table");
        return $count ?: '0';
    }
    
    /**
     * AJAX: Test API connection
     */
    public function ajax_test_api() {
        check_ajax_referer('chatly_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }
        
        $provider = sanitize_text_field($_POST['provider']);
        $api_key = sanitize_text_field($_POST['api_key']);
        
        $ai_provider = new Chatly_AI_Provider();
        $result = $ai_provider->test_connection($provider, $api_key);
        
        if ($result['success']) {
            wp_send_json_success($result['message']);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * AJAX: Save Knowledge Base entry
     */
    public function ajax_kb_save() {
        check_ajax_referer('chatly_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) { wp_send_json_error('Unauthorized'); }
        
        $id = intval($_POST['id'] ?? 0);
        $question = sanitize_text_field($_POST['question'] ?? '');
        $answer = wp_kses_post($_POST['answer'] ?? '');
        $keywords = sanitize_text_field($_POST['keywords'] ?? '');
        $category = sanitize_text_field($_POST['category'] ?? '');
        $priority = intval($_POST['priority'] ?? 0);
        
        if (empty($question) || empty($answer)) { wp_send_json_error('Question and answer required.'); }
        
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        $data = array('question' => $question, 'answer' => $answer, 'keywords' => $keywords, 'category' => $category, 'priority' => $priority, 'enabled' => 1);
        
        if ($id > 0) {
            $wpdb->update($table, $data, array('id' => $id));
        } else {
            $data['created_at'] = current_time('mysql');
            $wpdb->insert($table, $data);
            $id = $wpdb->insert_id;
        }
        
        wp_send_json_success(array('message' => 'Saved!', 'id' => $id));
    }
    
    /**
     * AJAX: Get Knowledge Base entry
     */
    public function ajax_kb_get() {
        check_ajax_referer('chatly_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) { wp_send_json_error('Unauthorized'); }
        
        $id = intval($_POST['id'] ?? 0);
        if ($id <= 0) { wp_send_json_error('Invalid ID'); }
        
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        $entry = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id));
        
        if (!$entry) { wp_send_json_error('Not found'); }
        wp_send_json_success($entry);
    }
    
    /**
     * AJAX: Delete Knowledge Base entry
     */
    public function ajax_kb_delete() {
        check_ajax_referer('chatly_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) { wp_send_json_error('Unauthorized'); }
        
        $id = intval($_POST['id'] ?? 0);
        if ($id <= 0) { wp_send_json_error('Invalid ID'); }
        
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_knowledge_base';
        $wpdb->delete($table, array('id' => $id));
        
        wp_send_json_success(array('message' => 'Deleted!'));
    }
    
    /**
     * AJAX: Get conversation messages
     */
    public function ajax_get_conversation() {
        check_ajax_referer('chatly_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) { wp_send_json_error('Unauthorized'); }
        
        $id = intval($_POST['id'] ?? 0);
        if ($id <= 0) { wp_send_json_error('Invalid ID'); }
        
        global $wpdb;
        $table = $wpdb->prefix . 'chatly_messages';
        $messages = $wpdb->get_results($wpdb->prepare("SELECT role, content, created_at FROM $table WHERE conversation_id = %d ORDER BY created_at ASC", $id));
        
        wp_send_json_success($messages);
    }
    
    /**
     * AJAX: Generate messaging with AI
     */
    public function ajax_generate_messaging() {
        check_ajax_referer('chatly_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) { 
            wp_send_json_error('Unauthorized'); 
        }
        
        $business_description = sanitize_textarea_field($_POST['description'] ?? '');
        
        if (empty($business_description)) {
            wp_send_json_error('Please provide a business description.');
        }
        
        // Get the configured AI provider
        $options = Chatly_Main::get_all_options();
        $provider = $options['ai_provider'];
        
        // Check for API key
        if ($provider === 'anthropic' && empty($options['anthropic_api_key'])) {
            wp_send_json_error('Please configure your Anthropic API key in the AI Provider tab first.');
        }
        if ($provider === 'openai' && empty($options['openai_api_key'])) {
            wp_send_json_error('Please configure your OpenAI API key in the AI Provider tab first.');
        }
        
        // Build the prompt for the AI
        $generation_prompt = "Based on the following business description, create chatbot messaging. Return your response as JSON with exactly these three fields:

1. \"system_prompt\" - A detailed system prompt (200-400 words) that tells the AI chatbot how to behave, including:
   - What the business does and offers
   - Key services/products
   - Business hours and location if mentioned
   - Tone and behavior guidelines
   - When to ask for contact information
   - Any specific policies or FAQs

2. \"welcome_message\" - A short, friendly welcome bubble text (under 50 characters) with an emoji

3. \"initial_bot_message\" - The first message visitors see when opening the chat (50-150 words), including:
   - A greeting
   - Brief intro of what you can help with
   - 3-5 bullet points of topics/services (use • character for bullets)
   - A prompt to ask questions

Business Description:
{$business_description}

Respond with ONLY valid JSON, no markdown, no explanation. Example format:
{\"system_prompt\":\"...\",\"welcome_message\":\"...\",\"initial_bot_message\":\"...\"}";

        // Call the AI provider
        $ai_provider = new Chatly_AI_Provider();
        
        if ($provider === 'anthropic') {
            $result = $this->call_anthropic_for_generation($options['anthropic_api_key'], $generation_prompt);
        } else {
            $result = $this->call_openai_for_generation($options['openai_api_key'], $generation_prompt);
        }
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }
        
        // Parse the JSON response
        $messaging = json_decode($result, true);
        
        if (json_last_error() !== JSON_ERROR_NONE || !isset($messaging['system_prompt'])) {
            // Try to extract JSON from the response if it has extra text
            if (preg_match('/\{[^{}]*"system_prompt"[^{}]*\}/s', $result, $matches)) {
                $messaging = json_decode($matches[0], true);
            }
            
            if (!$messaging || !isset($messaging['system_prompt'])) {
                wp_send_json_error('Failed to parse AI response. Please try again.');
            }
        }
        
        wp_send_json_success(array(
            'system_prompt' => $messaging['system_prompt'] ?? '',
            'welcome_message' => $messaging['welcome_message'] ?? '',
            'initial_bot_message' => $messaging['initial_bot_message'] ?? ''
        ));
    }
    
    /**
     * Call Anthropic API for messaging generation
     */
    private function call_anthropic_for_generation($api_key, $prompt) {
        $response = wp_remote_post('https://api.anthropic.com/v1/messages', array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $api_key,
                'anthropic-version' => '2023-06-01'
            ),
            'body' => wp_json_encode(array(
                'model' => 'claude-sonnet-4-20250514',
                'max_tokens' => 2000,
                'messages' => array(
                    array('role' => 'user', 'content' => $prompt)
                )
            )),
            'timeout' => 60
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['error'])) {
            return new WP_Error('api_error', $body['error']['message'] ?? 'API error');
        }
        
        return $body['content'][0]['text'] ?? '';
    }
    
    /**
     * Call OpenAI API for messaging generation
     */
    private function call_openai_for_generation($api_key, $prompt) {
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'body' => wp_json_encode(array(
                'model' => 'gpt-4o-mini',
                'messages' => array(
                    array('role' => 'user', 'content' => $prompt)
                ),
                'max_tokens' => 2000,
                'temperature' => 0.7
            )),
            'timeout' => 60
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['error'])) {
            return new WP_Error('api_error', $body['error']['message'] ?? 'API error');
        }
        
        return $body['choices'][0]['message']['content'] ?? '';
    }
    
    /**
     * Render SEO Insights page
     */
    public function render_insights_page() {
        $insights = new Chatly_Insights();
        $days = isset($_GET['days']) ? intval($_GET['days']) : 30;
        $active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'overview';
        
        // Get data
        $stats = $insights->get_conversation_stats($days);
        $top_questions = $insights->get_top_questions(15, $days);
        $keywords = $insights->get_keyword_frequency(30, $days);
        $phrases = $insights->get_keyword_phrases(20, $days);
        $content_gaps = $insights->get_content_gaps(10, $days);
        $kb_suggestions = $insights->suggest_kb_entries(5);
        $yoast_active = $insights->is_yoast_active();
        $keyword_opportunities = $yoast_active ? $insights->get_keyword_opportunities() : null;
        ?>
        <div class="wrap chatly-admin-wrap">
            <h1>
                <span class="dashicons dashicons-chart-line"></span>
                <?php esc_html_e('SEO Insights', 'chatly-ai-chatbot'); ?>
                <?php if ($yoast_active) : ?>
                    <span class="chatly-yoast-badge" title="Yoast SEO Detected">
                        <span class="dashicons dashicons-yes-alt"></span> Yoast Connected
                    </span>
                <?php endif; ?>
            </h1>
            
            <p class="chatly-insights-intro">
                <?php esc_html_e('Discover what your visitors are really asking. Use these insights to improve your content, SEO strategy, and Knowledge Base.', 'chatly-ai-chatbot'); ?>
            </p>
            
            <!-- Time Period Selector -->
            <div class="chatly-insights-period">
                <label><?php esc_html_e('Time Period:', 'chatly-ai-chatbot'); ?></label>
                <select id="chatly-insights-days" onchange="location.href='?page=chatly-insights&days='+this.value+'&tab=<?php echo esc_attr($active_tab); ?>'">
                    <option value="7" <?php selected($days, 7); ?>><?php esc_html_e('Last 7 days', 'chatly-ai-chatbot'); ?></option>
                    <option value="30" <?php selected($days, 30); ?>><?php esc_html_e('Last 30 days', 'chatly-ai-chatbot'); ?></option>
                    <option value="90" <?php selected($days, 90); ?>><?php esc_html_e('Last 90 days', 'chatly-ai-chatbot'); ?></option>
                    <option value="365" <?php selected($days, 365); ?>><?php esc_html_e('Last year', 'chatly-ai-chatbot'); ?></option>
                </select>
            </div>
            
            <!-- Tab Navigation -->
            <nav class="nav-tab-wrapper chatly-nav-tabs">
                <a href="?page=chatly-insights&tab=overview&days=<?php echo $days; ?>" class="nav-tab <?php echo $active_tab === 'overview' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-dashboard"></span> <?php esc_html_e('Overview', 'chatly-ai-chatbot'); ?>
                </a>
                <a href="?page=chatly-insights&tab=questions&days=<?php echo $days; ?>" class="nav-tab <?php echo $active_tab === 'questions' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-editor-help"></span> <?php esc_html_e('Top Questions', 'chatly-ai-chatbot'); ?>
                </a>
                <a href="?page=chatly-insights&tab=keywords&days=<?php echo $days; ?>" class="nav-tab <?php echo $active_tab === 'keywords' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-tag"></span> <?php esc_html_e('Keywords', 'chatly-ai-chatbot'); ?>
                display: inline-block; 
                background: linear-gradient(135deg, #a4286a 0%, #00a0d2 100%); 
                color: #fff; 
                padding: 4px 12px; 
                border-radius: 20px; 
                font-size: 12px; 
                margin-left: 15px; 
                vertical-align: middle;
            }
            .chatly-yoast-badge .dashicons { font-size: 14px; width: 14px; height: 14px; vertical-align: middle; margin-right: 3px; }
            .chatly-insights-content { margin-top: 20px; }
            .chatly-insight-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin-bottom: 30px; }
            .chatly-insight-card { background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 20px; }
            .chatly-insight-card h3 { margin: 0 0 15px; padding-bottom: 10px; border-bottom: 2px solid #667eea; color: #333; display: flex; align-items: center; gap: 8px; }
            .chatly-insight-card h3 .dashicons { color: #667eea; }
            .chatly-keyword-list { display: flex; flex-wrap: wrap; gap: 8px; }
            .chatly-keyword-tag { background: #f0f0f1; padding: 4px 12px; border-radius: 15px; font-size: 13px; display: inline-flex; align-items: center; gap: 5px; }
            .chatly-keyword-tag .count { background: #667eea; color: #fff; padding: 2px 6px; border-radius: 10px; font-size: 11px; }
            .chatly-question-list { list-style: none; padding: 0; margin: 0; }
            .chatly-question-list li { padding: 10px 0; border-bottom: 1px solid #eee; display: flex; justify-content: space-between; align-items: center; }
            .chatly-question-list li:last-child { border-bottom: none; }
            .chatly-question-text { flex: 1; }
            .chatly-question-count { background: #667eea; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 12px; margin-left: 10px; }
            .chatly-gap-item { background: #fff8e5; border-left: 4px solid #ffb900; padding: 12px 15px; margin-bottom: 10px; border-radius: 0 4px 4px 0; }
            .chatly-gap-item .question { font-weight: 600; margin-bottom: 5px; }
            .chatly-gap-item .suggestion { font-size: 13px; color: #666; }
            .chatly-code-block { background: #1e1e1e; color: #d4d4d4; padding: 20px; border-radius: 8px; overflow-x: auto; font-family: 'Monaco', 'Consolas', monospace; font-size: 13px; line-height: 1.5; }
            .chatly-copy-btn { margin-top: 10px; }
            .chatly-stat-highlight { font-size: 36px; font-weight: 700; color: #667eea; }
            .chatly-stat-label { font-size: 14px; color: #666; }
            .chatly-action-btn { display: inline-block; background: #667eea; color: #fff; padding: 8px 16px; border-radius: 4px; text-decoration: none; margin-top: 10px; }
            .chatly-action-btn:hover { background: #5a6fd6; color: #fff; }
            .chatly-opportunity-good { color: #00a32a; }
            .chatly-opportunity-new { color: #d63638; }
            .chatly-empty-state { text-align: center; padding: 40px; color: #666; }
            .chatly-empty-state .dashicons { font-size: 48px; width: 48px; height: 48px; color: #ddd; }
            .chatly-suggestion-item { background: #f0f6fc; border-left: 4px solid #2271b1; padding: 12px 15px; margin-bottom: 10px; border-radius: 0 4px 4px 0; }
            .chatly-add-to-kb { float: right; font-size: 12px; }
        </style>
        <?php
    }
    
    /**
     * Render Overview tab
     */
    private function render_insights_overview($stats, $top_questions, $keywords, $content_gaps, $yoast_active) {
        ?>
        <!-- Stats Summary -->
        <div class="chatly-insight-grid">
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-chart-bar"></span> <?php esc_html_e('Conversation Data', 'chatly-ai-chatbot'); ?></h3>
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; text-align: center;">
                    <div>
                        <div class="chatly-stat-highlight"><?php echo number_format($stats['total_conversations']); ?></div>
                        <div class="chatly-stat-label"><?php esc_html_e('Conversations', 'chatly-ai-chatbot'); ?></div>
                    </div>
                    <div>
                        <div class="chatly-stat-highlight"><?php echo number_format($stats['user_messages']); ?></div>
                        <div class="chatly-stat-label"><?php esc_html_e('Visitor Messages', 'chatly-ai-chatbot'); ?></div>
                    </div>
                </div>
            </div>
            
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-lightbulb"></span> <?php esc_html_e('Key Insight', 'chatly-ai-chatbot'); ?></h3>
                <?php if (!empty($top_questions)) : ?>
                    <p><?php esc_html_e('Your visitors most frequently ask about:', 'chatly-ai-chatbot'); ?></p>
                    <p><strong style="font-size: 16px;">"<?php echo esc_html(wp_trim_words($top_questions[0]['question'], 12)); ?>"</strong></p>
                    <p style="font-size: 13px; color: #666;"><?php printf(esc_html__('Asked %d times', 'chatly-ai-chatbot'), $top_questions[0]['count']); ?></p>
                <?php else : ?>
                    <p class="chatly-empty-state"><?php esc_html_e('Not enough conversation data yet.', 'chatly-ai-chatbot'); ?></p>
                <?php endif; ?>
            </div>
            
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-warning"></span> <?php esc_html_e('Content Gaps', 'chatly-ai-chatbot'); ?></h3>
                <?php if (!empty($content_gaps)) : ?>
                    <p><?php printf(esc_html__('%d questions where your chatbot struggled to answer.', 'chatly-ai-chatbot'), count($content_gaps)); ?></p>
                    <a href="?page=chatly-insights&tab=gaps" class="chatly-action-btn"><?php esc_html_e('View Content Gaps', 'chatly-ai-chatbot'); ?></a>
                <?php else : ?>
                    <p style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e('No content gaps detected!', 'chatly-ai-chatbot'); ?></p>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Quick Views -->
        <div class="chatly-insight-grid">
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-editor-help"></span> <?php esc_html_e('Top Questions', 'chatly-ai-chatbot'); ?></h3>
                <?php if (!empty($top_questions)) : ?>
                    <ul class="chatly-question-list">
                        <?php foreach (array_slice($top_questions, 0, 5) as $q) : ?>
                            <li>
                                <span class="chatly-question-text"><?php echo esc_html(wp_trim_words($q['question'], 10)); ?></span>
                                <span class="chatly-question-count"><?php echo esc_html($q['count']); ?></span>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                    <a href="?page=chatly-insights&tab=questions" class="chatly-action-btn"><?php esc_html_e('View All Questions', 'chatly-ai-chatbot'); ?></a>
                <?php else : ?>
                    <p class="chatly-empty-state"><?php esc_html_e('No questions recorded yet.', 'chatly-ai-chatbot'); ?></p>
                <?php endif; ?>
            </div>
            
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-tag"></span> <?php esc_html_e('Top Keywords', 'chatly-ai-chatbot'); ?></h3>
                <?php if (!empty($keywords)) : ?>
                    <div class="chatly-keyword-list">
                        <?php foreach (array_slice($keywords, 0, 15, true) as $word => $count) : ?>
                            <span class="chatly-keyword-tag"><?php echo esc_html($word); ?> <span class="count"><?php echo esc_html($count); ?></span></span>
                        <?php endforeach; ?>
                    </div>
                    <a href="?page=chatly-insights&tab=keywords" class="chatly-action-btn"><?php esc_html_e('View All Keywords', 'chatly-ai-chatbot'); ?></a>
                <?php else : ?>
                    <p class="chatly-empty-state"><?php esc_html_e('No keyword data yet.', 'chatly-ai-chatbot'); ?></p>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if ($yoast_active) : ?>
        <!-- Yoast Teaser -->
        <div class="chatly-insight-card" style="background: linear-gradient(135deg, #f8f0fc 0%, #e8f4fc 100%); border-color: #a4286a;">
            <h3><span class="dashicons dashicons-search"></span> <?php esc_html_e('Yoast SEO Integration', 'chatly-ai-chatbot'); ?></h3>
            <p><?php esc_html_e('Yoast SEO detected! Compare your focus keywords with what visitors are actually searching for.', 'chatly-ai-chatbot'); ?></p>
            <a href="?page=chatly-insights&tab=yoast" class="chatly-action-btn" style="background: #a4286a;"><?php esc_html_e('View SEO Opportunities', 'chatly-ai-chatbot'); ?></a>
        </div>
        <?php endif; ?>
        <?php
    }
    
    /**
     * Render Questions tab
     */
    private function render_insights_questions($top_questions, $kb_suggestions) {
        ?>
        <div class="chatly-insight-grid">
            <div class="chatly-insight-card" style="grid-column: span 2;">
                <h3><span class="dashicons dashicons-editor-help"></span> <?php esc_html_e('Most Asked Questions', 'chatly-ai-chatbot'); ?></h3>
                <p class="description"><?php esc_html_e('These are the questions visitors ask most frequently. Consider adding them to your FAQ page or Knowledge Base.', 'chatly-ai-chatbot'); ?></p>
                
                <?php if (!empty($top_questions)) : ?>
                    <table class="wp-list-table widefat fixed striped" style="margin-top: 15px;">
                        <thead>
                            <tr>
                                <th style="width: 60%;"><?php esc_html_e('Question', 'chatly-ai-chatbot'); ?></th>
                                <th style="width: 15%;"><?php esc_html_e('Times Asked', 'chatly-ai-chatbot'); ?></th>
                                <th style="width: 25%;"><?php esc_html_e('Action', 'chatly-ai-chatbot'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($top_questions as $q) : ?>
                                <tr>
                                    <td><?php echo esc_html($q['question']); ?></td>
                                    <td><span class="chatly-question-count"><?php echo esc_html($q['count']); ?></span></td>
                                    <td>
                                        <a href="<?php echo esc_url(admin_url('admin.php?page=chatly-knowledge')); ?>" class="button button-small"><?php esc_html_e('Add to KB', 'chatly-ai-chatbot'); ?></a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else : ?>
                    <div class="chatly-empty-state">
                        <span class="dashicons dashicons-format-chat"></span>
                        <p><?php esc_html_e('No questions recorded yet. Start conversations to see insights.', 'chatly-ai-chatbot'); ?></p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if (!empty($kb_suggestions)) : ?>
        <div class="chatly-insight-card">
            <h3><span class="dashicons dashicons-lightbulb"></span> <?php esc_html_e('Suggested Knowledge Base Entries', 'chatly-ai-chatbot'); ?></h3>
            <p class="description"><?php esc_html_e('These frequently asked questions are not yet in your Knowledge Base:', 'chatly-ai-chatbot'); ?></p>
            
            <?php foreach ($kb_suggestions as $suggestion) : ?>
                <div class="chatly-suggestion-item">
                    <strong><?php echo esc_html($suggestion['question']); ?></strong>
                    <span style="color: #666; font-size: 12px;"> — <?php printf(esc_html__('Asked %d times', 'chatly-ai-chatbot'), esc_html($suggestion['count'])); ?></span>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=chatly-knowledge')); ?>" class="chatly-add-to-kb button button-small"><?php esc_html_e('Add Answer', 'chatly-ai-chatbot'); ?></a>
                </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
        <?php
    }
    
    /**
     * Render Keywords tab
     */
    private function render_insights_keywords($keywords, $phrases) {
        ?>
        <div class="chatly-insight-grid">
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-tag"></span> <?php esc_html_e('Single Keywords', 'chatly-ai-chatbot'); ?></h3>
                <p class="description"><?php esc_html_e('Most common words used by your visitors:', 'chatly-ai-chatbot'); ?></p>
                
                <?php if (!empty($keywords)) : ?>
                    <div class="chatly-keyword-list" style="margin-top: 15px;">
                        <?php foreach ($keywords as $word => $count) : ?>
                            <span class="chatly-keyword-tag"><?php echo esc_html($word); ?> <span class="count"><?php echo esc_html($count); ?></span></span>
                        <?php endforeach; ?>
                    </div>
                <?php else : ?>
                    <div class="chatly-empty-state">
                        <p><?php esc_html_e('No keyword data yet.', 'chatly-ai-chatbot'); ?></p>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-editor-textcolor"></span> <?php esc_html_e('Keyword Phrases', 'chatly-ai-chatbot'); ?></h3>
                <p class="description"><?php esc_html_e('Multi-word phrases visitors use (great for long-tail SEO):', 'chatly-ai-chatbot'); ?></p>
                
                <?php if (!empty($phrases)) : ?>
                    <div class="chatly-keyword-list" style="margin-top: 15px;">
                        <?php foreach ($phrases as $phrase => $count) : ?>
                            <span class="chatly-keyword-tag"><?php echo esc_html($phrase); ?> <span class="count"><?php echo esc_html($count); ?></span></span>
                        <?php endforeach; ?>
                    </div>
                <?php else : ?>
                    <div class="chatly-empty-state">
                        <p><?php esc_html_e('No phrase data yet.', 'chatly-ai-chatbot'); ?></p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="chatly-insight-card">
            <h3><span class="dashicons dashicons-download"></span> <?php esc_html_e('Export Keywords', 'chatly-ai-chatbot'); ?></h3>
            <p><?php esc_html_e('Copy these keywords to use in your SEO tools, Yoast focus keyphrases, or content planning:', 'chatly-ai-chatbot'); ?></p>
            <textarea readonly style="width: 100%; height: 100px; font-family: monospace; margin-top: 10px;"><?php echo esc_textarea(implode(', ', array_keys($keywords))); ?></textarea>
            <button type="button" class="button chatly-copy-btn" onclick="this.previousElementSibling.select(); document.execCommand('copy'); this.textContent='Copied!';"><?php esc_html_e('Copy to Clipboard', 'chatly-ai-chatbot'); ?></button>
        </div>
        <?php
    }
    
    /**
     * Render Content Gaps tab
     */
    private function render_insights_gaps($content_gaps) {
        ?>
        <div class="chatly-insight-card" style="max-width: 800px;">
            <h3><span class="dashicons dashicons-warning"></span> <?php esc_html_e('Content Gaps Detected', 'chatly-ai-chatbot'); ?></h3>
            <p class="description"><?php esc_html_e('These are questions where your chatbot indicated it didn\'t have enough information to answer well. Consider adding this content to your website or Knowledge Base.', 'chatly-ai-chatbot'); ?></p>
            
            <?php if (!empty($content_gaps)) : ?>
                <?php foreach ($content_gaps as $gap) : ?>
                    <div class="chatly-gap-item">
                        <div class="question">"<?php echo esc_html($gap['question']); ?>"</div>
                        <div class="suggestion">
                            <?php printf(esc_html__('Asked %d times without a good answer', 'chatly-ai-chatbot'), esc_html($gap['count'])); ?>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=chatly-knowledge')); ?>" style="margin-left: 15px;"><?php esc_html_e('Add to Knowledge Base →', 'chatly-ai-chatbot'); ?></a>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else : ?>
                <div class="chatly-empty-state" style="background: #d4edda; border-radius: 8px; padding: 30px;">
                    <span class="dashicons dashicons-yes-alt" style="color: #00a32a;"></span>
                    <p style="color: #00a32a; font-weight: 600;"><?php esc_html_e('Great job! No content gaps detected.', 'chatly-ai-chatbot'); ?></p>
                    <p style="color: #666;"><?php esc_html_e('Your chatbot is answering questions well with the information available.', 'chatly-ai-chatbot'); ?></p>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render FAQ Schema tab
     */
    private function render_insights_faq($insights) {
        $schema = $insights->generate_faq_schema();
        $html = $insights->generate_faq_html();
        ?>
        <div class="chatly-insight-grid">
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-media-code"></span> <?php esc_html_e('FAQ Schema (JSON-LD)', 'chatly-ai-chatbot'); ?></h3>
                <p class="description"><?php esc_html_e('Add this to your page for rich FAQ snippets in Google search results. Works great with Yoast SEO!', 'chatly-ai-chatbot'); ?></p>
                
                <?php if ($schema) : ?>
                    <pre class="chatly-code-block"><code>&lt;script type="application/ld+json"&gt;
<?php echo esc_html($schema); ?>
&lt;/script&gt;</code></pre>
                    <button type="button" class="button chatly-copy-btn" onclick="var code = this.previousElementSibling.querySelector('code').textContent; navigator.clipboard.writeText(code); this.textContent='Copied!';"><?php esc_html_e('Copy Schema', 'chatly-ai-chatbot'); ?></button>
                    
                    <div style="margin-top: 20px; padding: 15px; background: #f0f6fc; border-radius: 4px;">
                        <strong><?php esc_html_e('How to use:', 'chatly-ai-chatbot'); ?></strong>
                        <ol style="margin: 10px 0 0 20px;">
                            <li><?php esc_html_e('Copy the schema code above', 'chatly-ai-chatbot'); ?></li>
                            <li><?php esc_html_e('Add it to your FAQ page using a custom HTML block or your theme\'s header', 'chatly-ai-chatbot'); ?></li>
                            <li><?php esc_html_e('Test with Google\'s Rich Results Test tool', 'chatly-ai-chatbot'); ?></li>
                        </ol>
                    </div>
                <?php else : ?>
                    <div class="chatly-empty-state">
                        <span class="dashicons dashicons-book"></span>
                        <p><?php esc_html_e('Add entries to your Knowledge Base to generate FAQ schema.', 'chatly-ai-chatbot'); ?></p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=chatly-knowledge')); ?>" class="chatly-action-btn"><?php esc_html_e('Go to Knowledge Base', 'chatly-ai-chatbot'); ?></a>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-editor-code"></span> <?php esc_html_e('FAQ HTML Block', 'chatly-ai-chatbot'); ?></h3>
                <p class="description"><?php esc_html_e('Ready-to-use HTML for your FAQ page:', 'chatly-ai-chatbot'); ?></p>
                
                <?php if ($html) : ?>
                    <pre class="chatly-code-block" style="max-height: 300px; overflow-y: auto;"><code><?php echo esc_html($html); ?></code></pre>
                    <button type="button" class="button chatly-copy-btn" onclick="var code = this.previousElementSibling.querySelector('code').textContent; navigator.clipboard.writeText(code); this.textContent='Copied!';"><?php esc_html_e('Copy HTML', 'chatly-ai-chatbot'); ?></button>
                <?php else : ?>
                    <div class="chatly-empty-state">
                        <p><?php esc_html_e('Add entries to your Knowledge Base first.', 'chatly-ai-chatbot'); ?></p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Yoast Integration tab
     */
    private function render_insights_yoast($opportunities, $keywords) {
        ?>
        <div class="chatly-insight-card" style="background: linear-gradient(135deg, #f8f0fc 0%, #e8f4fc 100%); border-color: #a4286a; margin-bottom: 20px;">
            <h3><span class="dashicons dashicons-admin-plugins"></span> <?php esc_html_e('Yoast SEO + Chatly', 'chatly-ai-chatbot'); ?></h3>
            <p><?php esc_html_e('Compare what you\'re optimizing for (Yoast focus keywords) with what visitors actually ask about (Chatly conversations).', 'chatly-ai-chatbot'); ?></p>
        </div>
        
        <div class="chatly-insight-grid">
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-yes-alt" style="color: #00a32a;"></span> <?php esc_html_e('Keywords You\'re Targeting That Visitors Ask About', 'chatly-ai-chatbot'); ?></h3>
                <p class="description"><?php esc_html_e('Great news! These Yoast focus keywords match what visitors are asking:', 'chatly-ai-chatbot'); ?></p>
                
                <?php if (!empty($opportunities['matching'])) : ?>
                    <div class="chatly-keyword-list" style="margin-top: 15px;">
                        <?php foreach ($opportunities['matching'] as $word => $count) : ?>
                            <span class="chatly-keyword-tag" style="background: #d4edda; border: 1px solid #00a32a;">
                                <?php echo esc_html($word); ?> <span class="count" style="background: #00a32a;"><?php echo esc_html($count); ?></span>
                            </span>
                        <?php endforeach; ?>
                    </div>
                <?php else : ?>
                    <p style="color: #666;"><?php esc_html_e('No direct matches found yet.', 'chatly-ai-chatbot'); ?></p>
                <?php endif; ?>
            </div>
            
            <div class="chatly-insight-card">
                <h3><span class="dashicons dashicons-star-filled" style="color: #d63638;"></span> <?php esc_html_e('New Keyword Opportunities', 'chatly-ai-chatbot'); ?></h3>
                <p class="description"><?php esc_html_e('Visitors are asking about these topics, but you\'re not targeting them in Yoast:', 'chatly-ai-chatbot'); ?></p>
                
                <?php if (!empty($opportunities['opportunities'])) : ?>
                    <div class="chatly-keyword-list" style="margin-top: 15px;">
                        <?php foreach ($opportunities['opportunities'] as $word => $count) : ?>
                            <span class="chatly-keyword-tag" style="background: #fcf0f0; border: 1px solid #d63638;">
                                <?php echo esc_html($word); ?> <span class="count" style="background: #d63638;"><?php echo esc_html($count); ?></span>
                            </span>
                        <?php endforeach; ?>
                    </div>
                    <p style="margin-top: 15px; font-size: 13px; color: #666;">
                        <span class="dashicons dashicons-info"></span>
                        <?php esc_html_e('Consider creating content targeting these keywords and adding them as Yoast focus keyphrases.', 'chatly-ai-chatbot'); ?>
                    </p>
                <?php else : ?>
                    <p style="color: #666;"><?php esc_html_e('No new opportunities detected yet. Keep gathering conversation data!', 'chatly-ai-chatbot'); ?></p>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="chatly-insight-card">
            <h3><span class="dashicons dashicons-clipboard"></span> <?php esc_html_e('Suggested Actions', 'chatly-ai-chatbot'); ?></h3>
            <ol style="line-height: 2;">
                <li><?php esc_html_e('Review the <strong>New Keyword Opportunities</strong> above', 'chatly-ai-chatbot'); ?></li>
                <li><?php esc_html_e('Create new content or pages targeting these topics', 'chatly-ai-chatbot'); ?></li>
                <li><?php esc_html_e('Add them as focus keyphrases in Yoast SEO', 'chatly-ai-chatbot'); ?></li>
                <li><?php 
                    /* translators: %s: URL to Knowledge Base page */
                    printf(
                        wp_kses(
                            __('Add answers to your <a href="%s">Knowledge Base</a> so the chatbot can better answer these questions', 'chatly-ai-chatbot'),
                            array('a' => array('href' => array()))
                        ),
                        esc_url(admin_url('admin.php?page=chatly-knowledge'))
                    ); 
                ?></li>
                <li><?php 
                    /* translators: %s: URL to FAQ Schema page */
                    printf(
                        wp_kses(
                            __('Use the <a href="%s">FAQ Schema</a> generator to get rich snippets', 'chatly-ai-chatbot'),
                            array('a' => array('href' => array()))
                        ),
                        esc_url(admin_url('admin.php?page=chatly-insights&tab=faq'))
                    ); 
                ?></li>
            </ol>
        </div>
        <?php
    }
}
