<?php
/**
 * Plugin Activator
 * Handles activation, deactivation, and database setup
 */

if (!defined('ABSPATH')) {
    exit;
}

class Chatly_Activator {
    
    /**
     * Run on plugin activation
     */
    public static function activate() {
        self::create_tables();
        self::set_default_options();
        
        // Clear any cached data
        wp_cache_flush();
        
        // Set activation flag for welcome notice
        set_transient('chatly_activation_redirect', true, 30);
    }
    
    /**
     * Run on plugin deactivation
     */
    public static function deactivate() {
        // Clear scheduled events if any
        wp_clear_scheduled_hook('chatly_daily_cleanup');
    }
    
    /**
     * Create database tables
     */
    private static function create_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        // Conversations table
        $table_conversations = $wpdb->prefix . 'chatly_conversations';
        $sql_conversations = "CREATE TABLE IF NOT EXISTS $table_conversations (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            session_id varchar(255) NOT NULL,
            visitor_ip varchar(45),
            visitor_user_agent text,
            page_url text,
            started_at datetime DEFAULT CURRENT_TIMESTAMP,
            ended_at datetime,
            message_count int DEFAULT 0,
            lead_captured tinyint(1) DEFAULT 0,
            PRIMARY KEY (id),
            KEY session_id (session_id),
            KEY started_at (started_at)
        ) $charset_collate;";
        dbDelta($sql_conversations);
        
        // Messages table
        $table_messages = $wpdb->prefix . 'chatly_messages';
        $sql_messages = "CREATE TABLE IF NOT EXISTS $table_messages (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            conversation_id bigint(20) NOT NULL,
            role varchar(20) NOT NULL,
            content text NOT NULL,
            tokens_used int DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY conversation_id (conversation_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        dbDelta($sql_messages);
        
        // Leads table
        $table_leads = $wpdb->prefix . 'chatly_leads';
        $sql_leads = "CREATE TABLE IF NOT EXISTS $table_leads (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            conversation_id bigint(20),
            name varchar(255) NOT NULL,
            email varchar(255) NOT NULL,
            phone varchar(50),
            interest text,
            source varchar(100) DEFAULT 'chatbot',
            status varchar(50) DEFAULT 'new',
            notes text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            exported_at datetime,
            PRIMARY KEY (id),
            KEY email (email),
            KEY status (status),
            KEY created_at (created_at)
        ) $charset_collate;";
        dbDelta($sql_leads);
        
        // Knowledge base table
        $table_knowledge = $wpdb->prefix . 'chatly_knowledge_base';
        $sql_knowledge = "CREATE TABLE IF NOT EXISTS $table_knowledge (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            question text NOT NULL,
            answer text NOT NULL,
            keywords text,
            category varchar(100),
            priority int DEFAULT 0,
            enabled tinyint(1) DEFAULT 1,
            use_count int DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY category (category),
            KEY enabled (enabled)
        ) $charset_collate;";
        dbDelta($sql_knowledge);
        
        // Analytics table (daily aggregates)
        $table_analytics = $wpdb->prefix . 'chatly_analytics';
        $sql_analytics = "CREATE TABLE IF NOT EXISTS $table_analytics (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            date date NOT NULL,
            conversations int DEFAULT 0,
            messages int DEFAULT 0,
            leads int DEFAULT 0,
            tokens_used int DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY date (date)
        ) $charset_collate;";
        dbDelta($sql_analytics);
        
        // Store database version
        update_option('chatly_db_version', CHATLY_VERSION);
    }
    
    /**
     * Set default options if not exists
     */
    private static function set_default_options() {
        if (!get_option('chatly_settings')) {
            update_option('chatly_settings', Chatly_Main::get_default_options());
        }
    }
}
