<?php
/**
 * Plugin Name: Chatly
 * Plugin URI: https://hellochatly.com
 * Description: AI-powered chatbot for WordPress with lead capture, SEO insights, and multi-provider support (OpenAI & Anthropic)
 * Version: 2.3.2
 * Author: MediaManagement Marketing Services LLC
 * Author URI: https://mediamanagementcorp.com
 * License: GPL v2 or later
 * Text Domain: chatly-ai-chatbot
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CHATLY_VERSION', '2.3.2');
define('CHATLY_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CHATLY_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CHATLY_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include required files
require_once CHATLY_PLUGIN_DIR . 'includes/class-activator.php';
require_once CHATLY_PLUGIN_DIR . 'includes/class-admin.php';
require_once CHATLY_PLUGIN_DIR . 'includes/class-chatbot.php';
require_once CHATLY_PLUGIN_DIR . 'includes/class-ai-provider.php';
require_once CHATLY_PLUGIN_DIR . 'includes/class-knowledge-base.php';
require_once CHATLY_PLUGIN_DIR . 'includes/class-leads.php';
require_once CHATLY_PLUGIN_DIR . 'includes/class-insights.php';

/**
 * Main Plugin Class
 */
class Chatly_Main {
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Activation/deactivation hooks
        register_activation_hook(__FILE__, array('Chatly_Activator', 'activate'));
        register_deactivation_hook(__FILE__, array('Chatly_Activator', 'deactivate'));
        
        // Initialize plugin
        add_action('plugins_loaded', array($this, 'init'));
    }
    
    public function init() {
        // Initialize admin
        if (is_admin()) {
            new Chatly_Admin();
        }
        
        // Initialize frontend chatbot
        new Chatly_Chatbot();
        
        // Initialize leads handler
        new Chatly_Leads();
    }
    
    /**
     * Get plugin option with default
     */
    public static function get_option($key, $default = '') {
        $options = get_option('chatly_settings', array());
        return isset($options[$key]) ? $options[$key] : $default;
    }
    
    /**
     * Update plugin option
     */
    public static function update_option($key, $value) {
        $options = get_option('chatly_settings', array());
        $options[$key] = $value;
        update_option('chatly_settings', $options);
    }
    
    /**
     * Get all options
     */
    public static function get_all_options() {
        return get_option('chatly_settings', self::get_default_options());
    }
    
    /**
     * Default options
     */
    public static function get_default_options() {
        return array(
            // General
            'enabled' => '1',
            'chat_title' => 'Chat Assistant',
            
            // AI Provider
            'ai_provider' => 'anthropic', // 'anthropic' or 'openai'
            'anthropic_api_key' => '',
            'anthropic_model' => 'claude-sonnet-4-20250514',
            'openai_api_key' => '',
            'openai_model' => 'gpt-4o-mini',
            
            // Appearance
            'primary_color' => '#667eea',
            'secondary_color' => '#764ba2',
            'text_color' => '#ffffff',
            'position' => 'right', // 'left' or 'right'
            'bottom_offset' => '20',
            'side_offset' => '20',
            'button_size' => '60',
            'window_width' => '380',
            'window_height' => '550',
            
            // Welcome Message
            'show_welcome_bubble' => '1',
            'welcome_message' => '👋 Hi! How can I help you today?',
            'initial_bot_message' => "Hi! I'm here to help. What would you like to know?",
            
            // System Prompt
            'system_prompt' => "You are a helpful assistant for this website. Be friendly, professional, and concise in your responses. Help visitors find information and answer their questions.",
            
            // Lead Capture
            'enable_lead_capture' => '1',
            'lead_form_title' => 'Get More Information',
            'lead_form_trigger' => 'interest', // 'always', 'interest', 'manual'
            'lead_trigger_keywords' => 'pricing, cost, quote, contact, demo, trial, buy, purchase, interested, help, service',
            'collect_phone' => '1',
            'email_integration' => 'none', // 'none', 'mailpoet', 'mailchimp', 'constantcontact', 'aweber', 'webhook'
            'mailpoet_list_id' => '',
            'mailchimp_api_key' => '',
            'mailchimp_list_id' => '',
            'constantcontact_api_key' => '',
            'constantcontact_api_secret' => '',
            'constantcontact_access_token' => '',
            'constantcontact_list_id' => '',
            'aweber_api_key' => '',
            'aweber_account_id' => '',
            'aweber_list_id' => '',
            'webhook_url' => '',
            'send_transcript_option' => '1',
            'send_admin_transcript' => '0',
            'additional_transcript_email' => '',
            
            // Analytics
            'enable_analytics' => '1',
        );
    }
}

// Initialize plugin
function chatly_init() {
    return Chatly_Main::get_instance();
}

chatly_init();
