/**
 * Chatly - Frontend Chatbot
 * React-based chat widget
 */

(function() {
    'use strict';
    
    const { useState, useEffect, useRef } = React;
    
    // Generate unique session ID
    function generateSessionId() {
        return 'aca_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
    }
    
    function getSessionId() {
        let sessionId = sessionStorage.getItem('aca_session_id');
        if (!sessionId) {
            sessionId = generateSessionId();
            sessionStorage.setItem('aca_session_id', sessionId);
        }
        return sessionId;
    }
    
    function wasWelcomeDismissed() {
        return sessionStorage.getItem('aca_welcome_dismissed') === 'true';
    }
    
    function setWelcomeDismissed() {
        sessionStorage.setItem('aca_welcome_dismissed', 'true');
    }
    
    function ChatWidget() {
        const settings = window.chatlyChatbot.settings;
        
        const [isOpen, setIsOpen] = useState(false);
        const [showWelcome, setShowWelcome] = useState(settings.showWelcomeBubble && !wasWelcomeDismissed());
        const [messages, setMessages] = useState([
            {
                type: 'bot',
                text: settings.initialMessage,
                timestamp: new Date()
            }
        ]);
        const [inputValue, setInputValue] = useState('');
        const [isLoading, setIsLoading] = useState(false);
        const [showLeadForm, setShowLeadForm] = useState(false);
        const [messageCount, setMessageCount] = useState(0);
        const [leadFormData, setLeadFormData] = useState({
            name: '',
            email: '',
            phone: '',
            interest: '',
            sendTranscript: true
        });
        const [sessionId] = useState(getSessionId());
        
        const messagesEndRef = useRef(null);
        const inputRef = useRef(null);
        
        useEffect(() => {
            if (messagesEndRef.current) {
                messagesEndRef.current.scrollIntoView({ behavior: 'smooth' });
            }
        }, [messages]);
        
        useEffect(() => {
            if (isOpen && inputRef.current) {
                inputRef.current.focus();
            }
        }, [isOpen]);
        
        const openChat = () => {
            setIsOpen(true);
            setShowWelcome(false);
            setWelcomeDismissed();
        };
        
        const dismissWelcome = (e) => {
            e.stopPropagation();
            setShowWelcome(false);
            setWelcomeDismissed();
        };
        
        const sendMessage = async () => {
            if (!inputValue.trim() || isLoading) return;
            
            const userMessage = inputValue.trim();
            setInputValue('');
            
            setMessages(prev => [...prev, {
                type: 'user',
                text: userMessage,
                timestamp: new Date()
            }]);
            
            setMessageCount(prev => prev + 1);
            setIsLoading(true);
            
            try {
                const history = messages.map(msg => ({
                    user: msg.type === 'user' ? msg.text : null,
                    assistant: msg.type === 'bot' ? msg.text : null
                })).filter(msg => msg.user || msg.assistant);
                
                const response = await fetch(window.chatlyChatbot.ajaxUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'chatly_send_message',
                        nonce: window.chatlyChatbot.nonce,
                        message: userMessage,
                        session_id: sessionId,
                        history: JSON.stringify(history)
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    setMessages(prev => [...prev, {
                        type: 'bot',
                        text: data.data.response,
                        timestamp: new Date()
                    }]);
                    
                    // Check if should show lead form
                    if (settings.enableLeadCapture && !showLeadForm) {
                        if (data.data.trigger_lead_capture || 
                            (settings.leadFormTrigger === 'messages' && messageCount >= 2)) {
                            setTimeout(() => {
                                setShowLeadForm(true);
                                setLeadFormData(prev => ({ ...prev, interest: userMessage }));
                            }, 1000);
                        }
                    }
                } else {
                    setMessages(prev => [...prev, {
                        type: 'bot',
                        text: 'Sorry, I encountered an error. Please try again.',
                        timestamp: new Date()
                    }]);
                }
            } catch (error) {
                console.error('Chat error:', error);
                setMessages(prev => [...prev, {
                    type: 'bot',
                    text: 'Sorry, something went wrong. Please try again.',
                    timestamp: new Date()
                }]);
            } finally {
                setIsLoading(false);
            }
        };
        
        const handleKeyPress = (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        };
        
        const submitLeadForm = async (e) => {
            e.preventDefault();
            
            if (!leadFormData.name || !leadFormData.email) {
                alert('Please provide your name and email');
                return;
            }
            
            setIsLoading(true);
            
            try {
                const conversation = messages.map(msg => ({
                    user: msg.type === 'user' ? msg.text : null,
                    assistant: msg.type === 'bot' ? msg.text : null
                })).filter(msg => msg.user || msg.assistant);
                
                const response = await fetch(window.chatlyChatbot.ajaxUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'chatly_capture_lead',
                        nonce: window.chatlyChatbot.nonce,
                        name: leadFormData.name,
                        email: leadFormData.email,
                        phone: leadFormData.phone,
                        interest: leadFormData.interest,
                        session_id: sessionId,
                        send_transcript: leadFormData.sendTranscript ? '1' : '0',
                        conversation: JSON.stringify(conversation)
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    setMessages(prev => [...prev, {
                        type: 'bot',
                        text: data.data.message || "Thank you! We'll be in touch soon.",
                        timestamp: new Date()
                    }]);
                    setShowLeadForm(false);
                    setLeadFormData({ name: '', email: '', phone: '', interest: '', sendTranscript: true });
                } else {
                    alert(data.data || 'Error submitting form. Please try again.');
                }
            } catch (error) {
                console.error('Lead capture error:', error);
                alert('Error submitting form. Please try again.');
            } finally {
                setIsLoading(false);
            }
        };
        
        return React.createElement('div', { className: 'chatly-chatbot-container' },
            // Welcome bubble
            !isOpen && showWelcome && React.createElement('div', {
                className: 'chatly-welcome-bubble',
                onClick: openChat
            },
                React.createElement('button', {
                    className: 'chatly-welcome-close',
                    onClick: dismissWelcome,
                    'aria-label': 'Dismiss'
                }, '×'),
                React.createElement('span', null, settings.welcomeMessage)
            ),
            
            // Toggle button
            !isOpen && React.createElement('button', {
                className: 'chatly-chat-toggle',
                onClick: openChat,
                'aria-label': 'Open chat'
            },
                React.createElement('svg', {
                    width: '24',
                    height: '24',
                    viewBox: '0 0 24 24',
                    fill: 'none',
                    stroke: 'currentColor',
                    strokeWidth: '2'
                },
                    React.createElement('path', {
                        d: 'M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z'
                    })
                )
            ),
            
            // Chat window
            isOpen && React.createElement('div', { className: 'chatly-chat-window' },
                // Header
                React.createElement('div', { className: 'chatly-chat-header' },
                    React.createElement('h3', null, settings.chatTitle),
                    React.createElement('button', {
                        className: 'chatly-close-btn',
                        onClick: () => setIsOpen(false),
                        'aria-label': 'Close chat'
                    }, '×')
                ),
                
                // Messages
                React.createElement('div', { className: 'chatly-chat-messages' },
                    messages.map((msg, index) =>
                        React.createElement('div', {
                            key: index,
                            className: `chatly-message chatly-message-${msg.type}`
                        },
                            React.createElement('div', {
                                className: 'chatly-message-content'
                            }, msg.text)
                        )
                    ),
                    isLoading && React.createElement('div', { className: 'chatly-message chatly-message-bot' },
                        React.createElement('div', { className: 'chatly-message-content chatly-typing' },
                            React.createElement('span'),
                            React.createElement('span'),
                            React.createElement('span')
                        )
                    ),
                    React.createElement('div', { ref: messagesEndRef })
                ),
                
                // Lead form
                showLeadForm && React.createElement('div', { className: 'chatly-lead-form' },
                    React.createElement('form', { onSubmit: submitLeadForm },
                        React.createElement('h4', null, settings.leadFormTitle),
                        React.createElement('input', {
                            type: 'text',
                            placeholder: 'Your Name *',
                            value: leadFormData.name,
                            onChange: (e) => setLeadFormData(prev => ({ ...prev, name: e.target.value })),
                            required: true
                        }),
                        React.createElement('input', {
                            type: 'email',
                            placeholder: 'Your Email *',
                            value: leadFormData.email,
                            onChange: (e) => setLeadFormData(prev => ({ ...prev, email: e.target.value })),
                            required: true
                        }),
                        settings.collectPhone && React.createElement('input', {
                            type: 'tel',
                            placeholder: 'Phone (optional)',
                            value: leadFormData.phone,
                            onChange: (e) => setLeadFormData(prev => ({ ...prev, phone: e.target.value }))
                        }),
                        settings.sendTranscriptOption && React.createElement('label', {
                            className: 'chatly-checkbox-label'
                        },
                            React.createElement('input', {
                                type: 'checkbox',
                                checked: leadFormData.sendTranscript,
                                onChange: (e) => setLeadFormData(prev => ({ ...prev, sendTranscript: e.target.checked }))
                            }),
                            React.createElement('span', null, 'Email me a copy of this conversation')
                        ),
                        React.createElement('div', { className: 'chatly-form-buttons' },
                            React.createElement('button', {
                                type: 'submit',
                                className: 'chatly-form-submit',
                                disabled: isLoading
                            }, isLoading ? 'Submitting...' : 'Submit'),
                            React.createElement('button', {
                                type: 'button',
                                className: 'chatly-form-cancel',
                                onClick: () => setShowLeadForm(false)
                            }, 'Cancel')
                        )
                    )
                ),
                
                // Input area
                React.createElement('div', { className: 'chatly-chat-input-wrapper' },
                    React.createElement('textarea', {
                        ref: inputRef,
                        className: 'chatly-chat-input',
                        placeholder: 'Type your message...',
                        value: inputValue,
                        onChange: (e) => setInputValue(e.target.value),
                        onKeyPress: handleKeyPress,
                        rows: 1,
                        disabled: isLoading
                    }),
                    React.createElement('button', {
                        className: 'chatly-send-btn',
                        onClick: sendMessage,
                        disabled: isLoading || !inputValue.trim(),
                        'aria-label': 'Send message'
                    },
                        React.createElement('svg', {
                            width: '20',
                            height: '20',
                            viewBox: '0 0 24 24',
                            fill: 'currentColor'
                        },
                            React.createElement('path', {
                                d: 'M2.01 21L23 12 2.01 3 2 10l15 2-15 2z'
                            })
                        )
                    )
                )
            )
        );
    }
    
    // Initialize
    function init() {
        const container = document.getElementById('chatly-chatbot-root');
        
        if (!container) {
            console.error('Chatly: Container not found');
            return;
        }
        
        if (typeof React === 'undefined' || typeof ReactDOM === 'undefined') {
            console.error('Chatly: React not loaded');
            return;
        }
        
        try {
            const root = ReactDOM.createRoot(container);
            root.render(React.createElement(ChatWidget));
        } catch (error) {
            console.error('Chatly: Init error', error);
        }
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
