/**
 * Chatly - Admin Scripts
 */

(function($) {
    'use strict';
    
    $(document).ready(function() {
        
        // Initialize color pickers
        $('.chatly-color-picker').wpColorPicker({
            change: function(event, ui) {
                updatePreview();
            }
        });
        
        // Provider selection
        $('.chatly-provider-option input').on('change', function() {
            var provider = $(this).val();
            
            // Update selected class
            $('.chatly-provider-option').removeClass('selected');
            $(this).closest('.chatly-provider-option').addClass('selected');
            
            // Show/hide provider settings
            $('.chatly-provider-settings').hide();
            $('#' + provider + '-settings').show();
        });
        
        // Toggle password visibility
        $('.chatly-toggle-password').on('click', function() {
            var targetId = $(this).data('target');
            var input = $('#' + targetId);
            var icon = $(this).find('.dashicons');
            
            if (input.attr('type') === 'password') {
                input.attr('type', 'text');
                icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            } else {
                input.attr('type', 'password');
                icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            }
        });
        
        // Test API connection
        $('.chatly-test-api').on('click', function() {
            var $button = $(this);
            var provider = $button.data('provider');
            var apiKey = $('#' + provider + '_api_key').val();
            var $result = $button.siblings('.chatly-test-result');
            
            if (!apiKey) {
                $result.removeClass('success').addClass('error').text('Please enter an API key first');
                return;
            }
            
            $button.prop('disabled', true);
            $result.removeClass('success error').text(chatlyAdmin.strings.testing);
            
            $.ajax({
                url: chatlyAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'chatly_test_api',
                    nonce: chatlyAdmin.nonce,
                    provider: provider,
                    api_key: apiKey
                },
                success: function(response) {
                    if (response.success) {
                        $result.removeClass('error').addClass('success').text('✓ ' + response.data);
                    } else {
                        $result.removeClass('success').addClass('error').text('✗ ' + response.data);
                    }
                },
                error: function() {
                    $result.removeClass('success').addClass('error').text('Connection error');
                },
                complete: function() {
                    $button.prop('disabled', false);
                }
            });
        });
        
        // Email integration toggle
        $('#email_integration').on('change', function() {
            var integration = $(this).val();
            
            $('.chatly-integration-settings').hide();
            
            if (integration !== 'none') {
                $('#' + integration + '-settings').show();
            }
        });
        
        // AI Messaging Generator
        $('#chatly-generate-messaging').on('click', function() {
            var $btn = $(this);
            var description = $('#ai_generator_input').val().trim();
            var $spinner = $('#ai-generator-spinner');
            var $status = $('#ai-generator-status');
            
            if (!description) {
                $status.removeClass('success').addClass('error').text('Please describe your business first.');
                return;
            }
            
            $btn.prop('disabled', true);
            $spinner.addClass('is-active');
            $status.removeClass('success error').text('Generating messaging...');
            
            $.ajax({
                url: chatlyAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'chatly_generate_messaging',
                    nonce: chatlyAdmin.nonce,
                    description: description
                },
                success: function(response) {
                    if (response.success) {
                        // Populate the form fields
                        $('#system_prompt').val(response.data.system_prompt);
                        $('#welcome_message').val(response.data.welcome_message);
                        $('#initial_bot_message').val(response.data.initial_bot_message);
                        
                        $status.removeClass('error').addClass('success').text('✓ Messaging generated! Review and save your settings.');
                        
                        // Scroll to show the generated content
                        $('html, body').animate({
                            scrollTop: $('#welcome_message').closest('.chatly-settings-section').offset().top - 100
                        }, 500);
                    } else {
                        $status.removeClass('success').addClass('error').text('✗ ' + response.data);
                    }
                },
                error: function() {
                    $status.removeClass('success').addClass('error').text('Connection error. Please try again.');
                },
                complete: function() {
                    $btn.prop('disabled', false);
                    $spinner.removeClass('is-active');
                }
            });
        });
        
        // Update preview on input change
        $('input, select, textarea').on('change input', function() {
            updatePreview();
        });
        
        // Preview update function
        function updatePreview() {
            var primary = $('#primary_color').val() || '#667eea';
            var secondary = $('#secondary_color').val() || '#764ba2';
            var textColor = $('#text_color').val() || '#ffffff';
            var buttonSize = $('#button_size').val() || 60;
            var position = $('input[name="aca_settings[position]"]:checked').val() || 'right';
            
            var previewHtml = `
                <div style="position: absolute; bottom: 20px; ${position}: 20px;">
                    <div style="
                        width: ${buttonSize}px;
                        height: ${buttonSize}px;
                        border-radius: 50%;
                        background: linear-gradient(135deg, ${primary} 0%, ${secondary} 100%);
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        color: ${textColor};
                        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                        cursor: pointer;
                    ">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
                        </svg>
                    </div>
                </div>
            `;
            
            $('#chatly-preview').html(previewHtml);
        }
        
        // Initial preview
        updatePreview();
        
        // Export leads
        $('#chatly-export-leads').on('click', function(e) {
            e.preventDefault();
            window.location.href = chatlyAdmin.ajaxUrl + '?action=aca_export_leads&nonce=' + chatlyAdmin.nonce;
        });
        
        // License activation
        $('#chatly-activate-license').on('click', function() {
            var $btn = $(this), licenseKey = $('#aca_license_key').val().trim(), $result = $('.chatly-license-result');
            if (!licenseKey) { $result.removeClass('success').addClass('error').text('Please enter a license key'); return; }
            $btn.prop('disabled', true).text('Activating...');
            $.ajax({
                url: chatlyAdmin.ajaxUrl, type: 'POST',
                data: { action: 'chatly_activate_license', nonce: chatlyAdmin.nonce, license_key: licenseKey },
                success: function(r) {
                    if (r.success) { $result.removeClass('error').addClass('success').text('✓ License activated!'); setTimeout(function(){ location.reload(); }, 1500); }
                    else { $result.removeClass('success').addClass('error').text('✗ ' + r.data); $btn.prop('disabled', false).text('Activate License'); }
                },
                error: function() { $result.addClass('error').text('Connection error'); $btn.prop('disabled', false).text('Activate License'); }
            });
        });
        
        // License deactivation
        $('#chatly-deactivate-license').on('click', function() {
            if (!confirm('Are you sure? The chatbot will stop working.')) return;
            var $btn = $(this), $result = $('.chatly-license-result');
            $btn.prop('disabled', true).text('Deactivating...');
            $.ajax({
                url: chatlyAdmin.ajaxUrl, type: 'POST',
                data: { action: 'chatly_deactivate_license', nonce: chatlyAdmin.nonce },
                success: function(r) { if (r.success) { location.reload(); } else { $result.addClass('error').text('✗ ' + r.data); $btn.prop('disabled', false).text('Deactivate'); } },
                error: function() { $result.addClass('error').text('Connection error'); $btn.prop('disabled', false).text('Deactivate'); }
            });
        });
        
        // License verification
        $('#chatly-check-license').on('click', function() {
            var $btn = $(this), $result = $('.chatly-license-result');
            $btn.prop('disabled', true).text('Verifying...');
            $.ajax({
                url: chatlyAdmin.ajaxUrl, type: 'POST',
                data: { action: 'chatly_check_license', nonce: chatlyAdmin.nonce },
                success: function(r) { if (r.success) { $result.addClass('success').text('✓ Verified'); setTimeout(function(){ location.reload(); }, 1500); } else { $result.addClass('error').text('✗ ' + r.data); $btn.prop('disabled', false).text('Verify'); } },
                error: function() { $result.addClass('error').text('Connection error'); $btn.prop('disabled', false).text('Verify'); }
            });
        });
        
        // === Knowledge Base ===
        $('#chatly-add-knowledge').on('click', function(e) {
            e.preventDefault();
            $('#chatly-kb-id').val('');
            $('#chatly-kb-question, #chatly-kb-answer, #chatly-kb-keywords, #chatly-kb-category').val('');
            $('#chatly-kb-priority').val('0');
            $('#chatly-kb-modal-title').text('Add Knowledge Entry');
            $('#chatly-kb-modal').fadeIn(200);
        });
        
        $(document).on('click', '.chatly-kb-edit', function() {
            var id = $(this).data('id');
            $('#chatly-kb-modal-title').text('Edit Knowledge Entry');
            $.post(chatlyAdmin.ajaxUrl, { action: 'chatly_kb_get', nonce: chatlyAdmin.nonce, id: id }, function(r) {
                if (r.success) {
                    $('#chatly-kb-id').val(r.data.id);
                    $('#chatly-kb-question').val(r.data.question);
                    $('#chatly-kb-answer').val(r.data.answer);
                    $('#chatly-kb-keywords').val(r.data.keywords);
                    $('#chatly-kb-category').val(r.data.category);
                    $('#chatly-kb-priority').val(r.data.priority);
                    $('#chatly-kb-modal').fadeIn(200);
                }
            });
        });
        
        $(document).on('click', '.chatly-kb-delete', function() {
            if (!confirm('Delete this entry?')) return;
            var id = $(this).data('id'), $row = $(this).closest('tr');
            $.post(chatlyAdmin.ajaxUrl, { action: 'chatly_kb_delete', nonce: chatlyAdmin.nonce, id: id }, function(r) {
                if (r.success) $row.fadeOut(300, function() { $(this).remove(); });
            });
        });
        
        $('#chatly-kb-save').on('click', function() {
            var $btn = $(this);
            if (!$('#chatly-kb-question').val() || !$('#chatly-kb-answer').val()) { alert('Question and answer required.'); return; }
            $btn.prop('disabled', true).text('Saving...');
            $.post(chatlyAdmin.ajaxUrl, {
                action: 'chatly_kb_save', nonce: chatlyAdmin.nonce,
                id: $('#chatly-kb-id').val(), question: $('#chatly-kb-question').val(), answer: $('#chatly-kb-answer').val(),
                keywords: $('#chatly-kb-keywords').val(), category: $('#chatly-kb-category').val(), priority: $('#chatly-kb-priority').val()
            }, function(r) {
                if (r.success) location.reload();
                else { alert(r.data); $btn.prop('disabled', false).text('Save Entry'); }
            });
        });
        
        $('#chatly-kb-cancel, #chatly-kb-modal .chatly-modal-close').on('click', function() { $('#chatly-kb-modal').fadeOut(200); });
        $('#chatly-kb-modal').on('click', function(e) { if (e.target === this) $(this).fadeOut(200); });
        
        // === Conversations ===
        $(document).on('click', '.chatly-view-conv', function() {
            var id = $(this).data('id');
            $('#chatly-conv-messages').html('<p>Loading...</p>');
            $('#chatly-conv-modal').fadeIn(200);
            $.post(chatlyAdmin.ajaxUrl, { action: 'chatly_get_conversation', nonce: chatlyAdmin.nonce, id: id }, function(r) {
                if (r.success && r.data.length) {
                    var html = '';
                    r.data.forEach(function(m) {
                        var cls = m.role === 'user' ? 'user' : 'bot';
                        html += '<div class="chatly-msg chatly-msg-' + cls + '"><strong>' + (m.role === 'user' ? 'Visitor' : 'AI') + ':</strong> ' + m.content + '</div>';
                    });
                    $('#chatly-conv-messages').html(html);
                } else {
                    $('#chatly-conv-messages').html('<p>No messages.</p>');
                }
            });
        });
        
        $('#chatly-conv-modal .chatly-modal-close').on('click', function() { $('#chatly-conv-modal').fadeOut(200); });
        $('#chatly-conv-modal').on('click', function(e) { if (e.target === this) $(this).fadeOut(200); });
        
    });
    
})(jQuery);
