<?php
/**
 * Classe para criar a tabela de leads capturados na administração do WordPress.
 * Este arquivo é uma parte do plugin Chat Button & Lead Capture.
 */

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class CBLC_Leads_List_Table extends WP_List_Table {

	/**
	 * Construtor da classe. Define os nomes singular e plural da tabela.
	 */
	public function __construct() {
		parent::__construct(
			[
				'singular' => __( 'Lead', 'chat-button-lead-capture' ),
				'plural'   => __( 'Leads', 'chat-button-lead-capture' ),
				'ajax'     => false,
			]
		);
	}

	/**
	 * Define as colunas que serão exibidas na tabela.
	 *
	 * @return array Um array associativo de colunas.
	 */
	public function get_columns() {
		$columns = [
			'cb'            => '<input type="checkbox" />',
			'name'          => __( 'Name', 'chat-button-lead-capture' ),
			'email'         => __( 'E-mail', 'chat-button-lead-capture' ),
			'lead_whatsapp' => __( 'WhatsApp', 'chat-button-lead-capture' ),
			'time'          => __( 'Date', 'chat-button-lead-capture' ),
		];
		return $columns;
	}

	/**
	 * Define como a coluna 'name' deve ser renderizada, incluindo as ações (links).
	 *
	 * @param array $item Os dados do item atual.
	 * @return string O HTML para a coluna.
	 */
	public function column_name( $item ) {
		$delete_nonce = wp_create_nonce( 'cblc_delete_lead_nonce' );
		$actions      = [
			'delete' => sprintf(
				'<a href="?page=%s&action=%s&id=%s&_wpnonce=%s">%s</a>',
				esc_attr( sanitize_key( $_REQUEST['page'] ) ),
				'cblc_delete_lead',
				absint( $item['id'] ),
				$delete_nonce,
				__( 'Delete', 'chat-button-lead-capture' )
			),
		];
		return sprintf( '%1$s %2$s', esc_html( $item['name'] ), $this->row_actions( $actions ) );
	}

	/**
	 * Define as ações em massa disponíveis no menu dropdown.
	 *
	 * @return array Um array de ações em massa.
	 */
	public function get_bulk_actions() {
		return [
			'bulk-delete' => __( 'Bulk Delete', 'chat-button-lead-capture' ),
		];
	}

	/**
	 * Define os links de visualização acima da tabela (Todos, Hoje, etc.).
	 *
	 * @return array Um array de links de visualização.
	 */
	protected function get_views() {
		$views   = [];
		$current = ( ! empty( $_GET['period'] ) ? $_GET['period'] : 'all' );
		$all_url = remove_query_arg( 'period' );
		$today_url = add_query_arg( 'period', 'today' );
		$week_url  = add_query_arg( 'period', 'week' );
		$month_url = add_query_arg( 'period', 'month' );

		$views['all']   = "<a href='{$all_url}' class='" . ( 'all' === $current ? 'current' : '' ) . "'>" . __( 'All', 'chat-button-lead-capture' ) . '</a>';
		$views['today'] = "<a href='{$today_url}' class='" . ( 'today' === $current ? 'current' : '' ) . "'>" . __( 'Today', 'chat-button-lead-capture' ) . '</a>';
		$views['week']  = "<a href='{$week_url}' class='" . ( 'week' === $current ? 'current' : '' ) . "'>" . __( 'This Week', 'chat-button-lead-capture' ) . '</a>';
		$views['month'] = "<a href='{$month_url}' class='" . ( 'month' === $current ? 'current' : '' ) . "'>" . __( 'This Month', 'chat-button-lead-capture' ) . '</a>';

		return $views;
	}

	/**
	 * Prepara os itens para serem exibidos. Busca os dados do banco, lida com paginação e ordenação.
	 */
	public function prepare_items() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'cblc_leads';
		$per_page   = 20;

		$this->_column_headers = [ $this->get_columns(), [], $this->get_sortable_columns() ];

		// Lógica para filtrar por período de tempo.
		$where_clause = 'WHERE 1=1';
		if ( ! empty( $_GET['period'] ) ) {
			$period = sanitize_key( $_GET['period'] );
			if ( 'today' === $period ) {
				$where_clause = $wpdb->prepare( ' WHERE DATE(time) = CURDATE()' );
			} elseif ( 'week' === $period ) {
				$where_clause = $wpdb->prepare( ' WHERE time >= %s', date( 'Y-m-d H:i:s', strtotime( '-7 days' ) ) );
			} elseif ( 'month' === $period ) {
				$where_clause = $wpdb->prepare( ' WHERE MONTH(time) = MONTH(CURDATE()) AND YEAR(time) = YEAR(CURDATE())' );
			}
		}

		$total_items = $wpdb->get_var( "SELECT COUNT(id) FROM $table_name $where_clause" );

		$paged  = isset( $_REQUEST['paged'] ) ? max( 0, intval( $_REQUEST['paged'] ) - 1 ) : 0;
		$offset = $paged * $per_page;

		$this->items = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM $table_name $where_clause ORDER BY time DESC LIMIT %d OFFSET %d", $per_page, $offset ), ARRAY_A );

		$this->set_pagination_args(
			[
				'total_items' => $total_items,
				'per_page'    => $per_page,
			]
		);
	}

	/**
	 * Define a renderização padrão para colunas sem uma função específica.
	 */
	public function column_default( $item, $column_name ) {
		return esc_html( $item[ $column_name ] );
	}

	/**
	 * Formata a coluna 'time' para um formato de data legível.
	 */
	public function column_time( $item ) {
		return date_i18n( 'd/m/Y H:i:s', strtotime( $item['time'] ) );
	}

	/**
	 * Renderiza a caixa de seleção para cada item.
	 */
	public function column_cb( $item ) {
		return sprintf( '<input type="checkbox" name="ids[]" value="%s" />', $item['id'] );
	}

	/**
	 * Define quais colunas podem ser ordenadas.
	 */
	public function get_sortable_columns() {
		return [
			'time' => [ 'time', true ], // true significa que a ordenação padrão é descendente.
			'name' => [ 'name', false ],
		];
	}
}