<?php
/**
 * Plugin Name:       Chat Button & Lead Capture
 * Description:       Adds a floating chat button with a lead capture form to your site.
 * Version:           2.0.2
 * Author:            DropDesk
 * Author URI:        https://buy-dropdesk.com.br
 * Text Domain:       chat-button-lead-capture
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

// Define a versão do banco de dados para a versão gratuita.
define( 'CBLC_DB_VERSION', '1.0' );

// Carrega o arquivo da classe WP_List_Table para a área de administração.
require_once plugin_dir_path( __FILE__ ) . 'includes/class-cblc-leads-list-table.php';

/**
 * Função executada na ativação do plugin. Cria ou atualiza a tabela no banco de dados.
 */
function cblc_activate_plugin() {
	global $wpdb;
	$table_name      = $wpdb->prefix . 'cblc_leads';
	$charset_collate = $wpdb->get_charset_collate();

	// Estrutura da tabela simplificada para a versão gratuita.
	$sql = "CREATE TABLE $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        time datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
        name varchar(100) NOT NULL,
        email varchar(100) NOT NULL,
        cpf_cnpj varchar(20) DEFAULT '' NOT NULL,
        lead_whatsapp varchar(20) DEFAULT '' NOT NULL,
        PRIMARY KEY  (id)
    ) $charset_collate;";

	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	dbDelta( $sql );

	update_option( 'cblc_db_version', CBLC_DB_VERSION );
}
register_activation_hook( __FILE__, 'cblc_activate_plugin' );

/**
 * Verifica a versão do banco de dados ao carregar os plugins e atualiza se necessário.
 */
function cblc_update_db_check() {
	$current_db_version = get_option( 'cblc_db_version', '0' );
	if ( version_compare( $current_db_version, CBLC_DB_VERSION, '<' ) ) {
		cblc_activate_plugin();
	}
}
add_action( 'plugins_loaded', 'cblc_update_db_check' );

/**
 * Adiciona as páginas do plugin ao menu de administração do WordPress.
 */
function cblc_add_admin_menu() {
	add_menu_page(
		__( 'Chat Button', 'chat-button-lead-capture' ),
		__( 'Chat Button', 'chat-button-lead-capture' ),
		'manage_options',
		'cblc_main_menu',
		'cblc_settings_page_html',
		'dashicons-format-chat',
		6
	);
	add_submenu_page(
		'cblc_main_menu',
		__( 'Captured Leads', 'chat-button-lead-capture' ),
		__( 'Captured Leads', 'chat-button-lead-capture' ),
		'manage_options',
		'cblc_leads_list',
		'cblc_leads_page_html'
	);
}
add_action( 'admin_menu', 'cblc_add_admin_menu' );

/**
 * Enfileira os scripts e estilos para as páginas de administração do plugin.
 */
function cblc_admin_enqueue_assets( $hook ) {
	// Garante que os scripts só carreguem nas páginas do nosso plugin.
	if ( 'toplevel_page_cblc_main_menu' !== $hook && 'chat-button_page_cblc_leads_list' !== $hook ) {
		return;
	}

	// Carrega o arquivo de estilo para a administração.
	wp_enqueue_style( 'cblc-admin-style', plugin_dir_url( __FILE__ ) . 'assets/css/admin-style.css', array(), '5.1.6' );

	// Carrega o script de upload de mídia do WordPress apenas na página de configurações.
	if ( 'toplevel_page_cblc_main_menu' === $hook ) {
		wp_enqueue_media();
		wp_enqueue_script( 'cblc-admin-script', plugin_dir_url( __FILE__ ) . 'assets/js/admin-script.js', array( 'jquery' ), '1.0.0', true );
	}
}
add_action( 'admin_enqueue_scripts', 'cblc_admin_enqueue_assets' );

/**
 * Renderiza o HTML da página de configurações principal.
 */
function cblc_settings_page_html() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}
	global $wpdb;
	$table_name  = $wpdb->prefix . 'cblc_leads';
	$total_leads = $wpdb->get_var( "SELECT COUNT(id) FROM $table_name" );
	$options     = get_option( 'cblc_options' );
	$is_active   = ! empty( $options['cblc_active'] );
	?>
	<div class="wrap">
		<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

		<?php cblc_render_donation_meta_box(); ?>

		<div class="cblc-status-panel">
			<div class="status-box <?php echo $is_active ? 'active' : 'inactive'; ?>">
				<h4><?php esc_html_e( 'Button Status', 'chat-button-lead-capture' ); ?></h4>
				<p><?php echo $is_active ? esc_html__( 'ACTIVE', 'chat-button-lead-capture' ) : esc_html__( 'INACTIVE', 'chat-button-lead-capture' ); ?></p>
			</div>
			<div class="status-box">
				<h4><?php esc_html_e( 'Total Leads', 'chat-button-lead-capture' ); ?></h4>
				<p><?php echo (int) $total_leads; ?></p>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=cblc_leads_list' ) ); ?>"><?php esc_html_e( 'View Leads', 'chat-button-lead-capture' ); ?></a>
			</div>
		</div>
		
		<form action="options.php" method="post">
			<?php
			settings_fields( 'cblc_settings_group' );
			do_settings_sections( 'cblc_main_menu' );
			submit_button( __( 'Save Changes', 'chat-button-lead-capture' ) );
			?>
		</form>
		<hr>
		<div id="cblc-tracking-guide">
			<h2><span class="dashicons dashicons-chart-line" style="vertical-align: middle;"></span> <?php esc_html_e( 'Conversion Tracking (Optional)', 'chat-button-lead-capture' ); ?></h2>
			<p><?php esc_html_e( 'To measure your button\'s performance, this plugin sends an event to the Data Layer each time a lead is generated. You can use this information to create goals in Google Analytics, Google Ads, Meta Pixel, etc.', 'chat-button-lead-capture' ); ?></p>
			<details>
				<summary><?php esc_html_e( 'Click to see configuration instructions for Google Tag Manager (GTM)', 'chat-button-lead-capture' ); ?></summary>
				<div class="instructions-content">
					<p><?php esc_html_e( 'In your GTM container, follow the 3 steps below to track conversions and send data for Meta\'s Advanced Matching.', 'chat-button-lead-capture' ); ?></p>
					<h4><?php esc_html_e( 'Step 1: Create Data Layer Variables', 'chat-button-lead-capture' ); ?></h4>
					<p><?php esc_html_e( 'First, teach GTM to read the data that our plugin sends. In `Variables > User-Defined Variables`, create a new variable for each item below:', 'chat-button-lead-capture' ); ?></p>
					<ul>
						<li><strong><?php esc_html_e( 'Variable 1 (Email):', 'chat-button-lead-capture' ); ?></strong>
							<ul>
								<li><?php esc_html_e( 'Variable Name in GTM:', 'chat-button-lead-capture' ); ?> <code>DLV - user_data.email</code></li>
								<li><?php esc_html_e( 'Type:', 'chat-button-lead-capture' ); ?> <code><?php esc_html_e( 'Data Layer Variable', 'chat-button-lead-capture' ); ?></code></li>
								<li><?php esc_html_e( 'Data Layer Variable Name:', 'chat-button-lead-capture' ); ?> <code>user_data.email</code></li>
							</ul>
						</li>
						<li><strong><?php esc_html_e( 'Variable 2 (Phone):', 'chat-button-lead-capture' ); ?></strong>
							<ul>
								<li><?php esc_html_e( 'Variable Name in GTM:', 'chat-button-lead-capture' ); ?> <code>DLV - user_data.phone</code></li>
								<li><?php esc_html_e( 'Type:', 'chat-button-lead-capture' ); ?> <code><?php esc_html_e( 'Data Layer Variable', 'chat-button-lead-capture' ); ?></code></li>
								<li><?php esc_html_e( 'Data Layer Variable Name:', 'chat-button-lead-capture' ); ?> <code>user_data.phone_number</code></li>
							</ul>
						</li>
						<li><strong><?php esc_html_e( 'Variable 3 (First Name):', 'chat-button-lead-capture' ); ?></strong>
							<ul>
								<li><?php esc_html_e( 'Variable Name in GTM:', 'chat-button-lead-capture' ); ?> <code>DLV - user_data.firstname</code></li>
								<li><?php esc_html_e( 'Type:', 'chat-button-lead-capture' ); ?> <code><?php esc_html_e( 'Data Layer Variable', 'chat-button-lead-capture' ); ?></code></li>
								<li><?php esc_html_e( 'Data Layer Variable Name:', 'chat-button-lead-capture' ); ?> <code>user_data.first_name</code></li>
							</ul>
						</li>
					</ul>
					<h4><?php esc_html_e( 'Step 2: Create the Trigger', 'chat-button-lead-capture' ); ?></h4>
					<p><?php esc_html_e( 'Now, create the trigger that will fire your conversion tags.', 'chat-button-lead-capture' ); ?></p>
					<ul>
						<li><?php esc_html_e( 'In `Triggers > New`:', 'chat-button-lead-capture' ); ?>
							<ul>
								<li><?php esc_html_e( 'Trigger Name:', 'chat-button-lead-capture' ); ?> <code><?php esc_html_e( 'Event - WhatsApp Lead', 'chat-button-lead-capture' ); ?></code></li>
								<li><?php esc_html_e( 'Trigger Type:', 'chat-button-lead-capture' ); ?> <strong><?php esc_html_e( 'Custom Event', 'chat-button-lead-capture' ); ?></strong></li>
								<li><?php esc_html_e( 'Event Name:', 'chat-button-lead-capture' ); ?> <code>whatsapp_lead_submitted</code></li>
							</ul>
						</li>
					</ul>
					<h4><?php esc_html_e( 'Step 3: Configure Your Conversion Tag', 'chat-button-lead-capture' ); ?></h4>
					<p><?php esc_html_e( 'Finally, configure your tag (e.g., Meta Pixel) to use the data and the trigger.', 'chat-button-lead-capture' ); ?></p>
					<ul>
						<li><?php esc_html_e( 'Edit your Meta Pixel tag.', 'chat-button-lead-capture' ); ?></li>
						<li><?php esc_html_e( 'Enable "Advanced Matching".', 'chat-button-lead-capture' ); ?></li>
						<li><?php esc_html_e( 'Map the fields with the variables you created in Step 1.', 'chat-button-lead-capture' ); ?></li>
						<li><?php esc_html_e( 'In the "Triggering" section, select the trigger you created:', 'chat-button-lead-capture' ); ?> <strong>"<?php esc_html_e( 'Event - WhatsApp Lead', 'chat-button-lead-capture' ); ?>"</strong>.</li>
					</ul>
				</div>
			</details>
		</div>
	</div>
	<?php
}

/**
 * Renderiza o HTML da página de leads capturados.
 */
function cblc_leads_page_html() {
	$leads_table = new CBLC_Leads_List_Table();
	$leads_table->prepare_items();
	$export_url = add_query_arg(
		array(
			'action'   => 'cblc_export_leads',
			'period'   => ( $_GET['period'] ?? 'all' ),
			'_wpnonce' => wp_create_nonce( 'cblc_export_nonce' ),
		)
	);
	?>
	<div class="wrap">
		<h1 class="wp-heading-inline"><?php esc_html_e( 'Captured Leads', 'chat-button-lead-capture' ); ?></h1>
		<a href="<?php echo esc_url( $export_url ); ?>" class="page-title-action"><?php esc_html_e( 'Export to CSV', 'chat-button-lead-capture' ); ?></a>
		<hr class="wp-header-end">

		<?php cblc_render_donation_meta_box(); ?>

		<form id="leads-filter" method="post">
			<input type="hidden" name="page" value="<?php echo isset( $_REQUEST['page'] ) ? esc_attr( sanitize_key( $_REQUEST['page'] ) ) : ''; ?>" />
			<?php
			$leads_table->views();
			$leads_table->display();
			?>
		</form>
	</div>
	<?php
}

/**
 * Lida com as ações da tabela de leads (excluir, exportar).
 * VERSÃO CORRIGIDA E SEGURA
 */
function cblc_handle_list_table_actions() {
	if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
		return;
	}
	global $wpdb;
	$table_name = $wpdb->prefix . 'cblc_leads';

	// Ação de exclusão individual.
	if ( isset( $_GET['action'] ) && 'cblc_delete_lead' === $_GET['action'] ) {
		// CORREÇÃO: Sanitizando o nonce antes de verificar.
		if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'cblc_delete_lead_nonce' ) ) {
			wp_die( 'Unauthorized action.' );
		}
		$lead_id = isset( $_GET['id'] ) ? absint( $_GET['id'] ) : 0;
		if ( $lead_id > 0 ) {
			$wpdb->delete( $table_name, array( 'id' => $lead_id ), array( '%d' ) );
		}
		wp_safe_redirect( admin_url( 'admin.php?page=cblc_leads_list' ) );
		exit;
	}

	// Ação de exclusão em massa.
	$current_action = isset( $_POST['action'] ) && -1 !== $_POST['action'] ? $_POST['action'] : ( $_POST['action2'] ?? '' );
	if ( 'bulk-delete' === $current_action ) {
		check_admin_referer( 'bulk-leads' );

		// CORREÇÃO: Lógica segura para query com IN().
		$ids_to_delete = isset( $_POST['ids'] ) && is_array( $_POST['ids'] ) ? array_map( 'absint', $_POST['ids'] ) : array();

		if ( ! empty( $ids_to_delete ) ) {
			// 1. Cria os placeholders (%d, %d, %d...)
			$placeholders = implode( ', ', array_fill( 0, count( $ids_to_delete ), '%d' ) );
			
			// 2. Prepara e executa a query de forma segura.
			$query = $wpdb->prepare( "DELETE FROM $table_name WHERE id IN ( $placeholders )", $ids_to_delete );
			$wpdb->query( $query );
		}
		
		wp_safe_redirect( admin_url( 'admin.php?page=cblc_leads_list' ) );
		exit;
	}

	// Ação de exportação para CSV.
	if ( isset( $_GET['action'] ) && 'cblc_export_leads' === $_GET['action'] ) {
		// CORREÇÃO: Sanitizando o nonce antes de verificar.
		if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'cblc_export_nonce' ) ) {
			wp_die( 'Unauthorized action.' );
		}
		
		// A sanitização aqui já estava correta, mas mantemos para clareza.
		$period = isset( $_GET['period'] ) ? sanitize_key( $_GET['period'] ) : 'all';
		
		$where_clause = 'WHERE 1=1';
		if ( 'today' === $period ) {
			$where_clause = ' WHERE DATE(time) = CURDATE()';
		} elseif ( 'week' === $period ) {
			$where_clause = $wpdb->prepare( ' WHERE time >= %s', date_i18n( 'Y-m-d H:i:s', strtotime( '-7 days' ) ) );
		} elseif ( 'month' === $period ) {
			$where_clause = $wpdb->prepare( ' WHERE MONTH(time) = %d AND YEAR(time) = %d', date_i18n( 'm' ), date_i18n( 'Y' ) );
		}
		
		// A query aqui não usa input direto do usuário na estrutura, então é segura.
		$leads = $wpdb->get_results( "SELECT id, time, name, email, cpf_cnpj, lead_whatsapp FROM $table_name $where_clause ORDER BY time DESC", ARRAY_A );

		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename="cblc-leads-' . date_i18n( 'Y-m-d' ) . '.csv"' );
		$output = fopen( 'php://output', 'w' );
		fputcsv( $output, array( 'ID', 'Date', 'Name', 'Email', 'ID (CPF/CNPJ)', 'WhatsApp' ) );
		if ( $leads ) {
			foreach ( $leads as $lead ) {
				fputcsv( $output, $lead );
			}
		}
		fclose( $output );
		exit;
	}
}
add_action( 'admin_init', 'cblc_handle_list_table_actions' );

/**
 * Registra as configurações do plugin, seções e campos.
 */
function cblc_settings_init() {
	register_setting( 'cblc_settings_group', 'cblc_options', 'cblc_options_sanitize' );
	add_settings_section( 'cblc_settings_section', __( 'Main Settings', 'chat-button-lead-capture' ), '__return_false', 'cblc_main_menu' );
	add_settings_field( 'cblc_active', __( 'Activate Button', 'chat-button-lead-capture' ), 'cblc_active_callback', 'cblc_main_menu', 'cblc_settings_section' );
	add_settings_field( 'cblc_phone_number', __( 'Your WhatsApp Number', 'chat-button-lead-capture' ), 'cblc_phone_number_callback', 'cblc_main_menu', 'cblc_settings_section' );
	add_settings_field( 'cblc_agent_name', __( 'Attendant Name (Optional)', 'chat-button-lead-capture' ), 'cblc_agent_name_callback', 'cblc_main_menu', 'cblc_settings_section' );
	add_settings_field( 'cblc_agent_image', __( 'Attendant Photo (Optional)', 'chat-button-lead-capture' ), 'cblc_agent_image_callback', 'cblc_main_menu', 'cblc_settings_section' );
	add_settings_field( 'cblc_default_message', __( 'Default Message', 'chat-button-lead-capture' ), 'cblc_default_message_callback', 'cblc_main_menu', 'cblc_settings_section' );
	add_settings_field( 'cblc_fields_to_capture', __( 'Fields to Capture', 'chat-button-lead-capture' ), 'cblc_fields_to_capture_callback', 'cblc_main_menu', 'cblc_settings_section' );
}
add_action( 'admin_init', 'cblc_settings_init' );

/**
 * Sanitiza as opções do plugin antes de salvá-las no banco de dados.
 */
function cblc_options_sanitize( $input ) {
	$new_input                        = array();
	$new_input['cblc_active']         = isset( $input['cblc_active'] ) ? 1 : 0;
	$new_input['cblc_phone_number']   = isset( $input['cblc_phone_number'] ) ? sanitize_text_field( $input['cblc_phone_number'] ) : '';
	$new_input['cblc_default_message'] = isset( $input['cblc_default_message'] ) ? sanitize_textarea_field( $input['cblc_default_message'] ) : '';
	$new_input['cblc_agent_name']     = isset( $input['cblc_agent_name'] ) ? sanitize_text_field( $input['cblc_agent_name'] ) : '';
	$new_input['cblc_agent_image_id'] = isset( $input['cblc_agent_image_id'] ) ? absint( $input['cblc_agent_image_id'] ) : 0;
	$new_input['cblc_fields_name']    = isset( $input['cblc_fields_name'] ) ? 1 : 0;
	$new_input['cblc_fields_email']   = isset( $input['cblc_fields_email'] ) ? 1 : 0;
	$new_input['cblc_fields_cpf_cnpj'] = isset( $input['cblc_fields_cpf_cnpj'] ) ? 1 : 0;
	$new_input['cblc_fields_phone']   = isset( $input['cblc_fields_phone'] ) ? 1 : 0;
	return $new_input;
}

// Funções de callback para renderizar os campos de configuração.
function cblc_active_callback() {
	$options = get_option( 'cblc_options' );
	?>
	<input type="checkbox" name="cblc_options[cblc_active]" value="1" <?php checked( 1, $options['cblc_active'] ?? 0, true ); ?> />
	<label><?php esc_html_e( 'Check to activate the button on your site.', 'chat-button-lead-capture' ); ?></label>
	<?php
}

function cblc_phone_number_callback() {
	$options = get_option( 'cblc_options' );
	$value   = $options['cblc_phone_number'] ?? '';
	echo '<input type="text" name="cblc_options[cblc_phone_number]" value="' . esc_attr( $value ) . '" placeholder="5521999998888" class="regular-text" />';
}

function cblc_default_message_callback() {
	$options = get_option( 'cblc_options' );
	$value   = $options['cblc_default_message'] ?? '';
	echo '<textarea name="cblc_options[cblc_default_message]" rows="4" class="large-text" placeholder="' . esc_attr__( 'Hello! I\'m interested in your services.', 'chat-button-lead-capture' ) . '">' . esc_textarea( $value ) . '</textarea>';
}

function cblc_fields_to_capture_callback() {
	$options = get_option( 'cblc_options' );
	?>
	<input type="checkbox" name="cblc_options[cblc_fields_name]" value="1" <?php checked( 1, $options['cblc_fields_name'] ?? 0, true ); ?>>
	<label><?php esc_html_e( 'Name', 'chat-button-lead-capture' ); ?></label><br>

	<input type="checkbox" name="cblc_options[cblc_fields_email]" value="1" <?php checked( 1, $options['cblc_fields_email'] ?? 0, true ); ?>>
	<label><?php esc_html_e( 'E-mail', 'chat-button-lead-capture' ); ?></label><br>

	<input type="checkbox" name="cblc_options[cblc_fields_cpf_cnpj]" value="1" <?php checked( 1, $options['cblc_fields_cpf_cnpj'] ?? 0, true ); ?>>
	<label><?php esc_html_e( 'ID (CPF/CNPJ)', 'chat-button-lead-capture' ); ?></label><br>

	<input type="checkbox" name="cblc_options[cblc_fields_phone]" value="1" <?php checked( 1, $options['cblc_fields_phone'] ?? 0, true ); ?>>
	<label><?php esc_html_e( 'Lead\'s Phone (with mask)', 'chat-button-lead-capture' ); ?></label>
	<?php
}

function cblc_agent_name_callback() {
	$options = get_option( 'cblc_options' );
	$value   = $options['cblc_agent_name'] ?? '';
	echo '<input type="text" name="cblc_options[cblc_agent_name]" value="' . esc_attr( $value ) . '" placeholder="Annalú" class="regular-text" />';
	echo '<p class="description">' . esc_html__( 'Leave blank to use the default name "Annalú".', 'chat-button-lead-capture' ) . '</p>';
}

function cblc_agent_image_callback() {
	$options           = get_option( 'cblc_options' );
	$image_id          = $options['cblc_agent_image_id'] ?? 0;
	$default_image_url = plugin_dir_url( __FILE__ ) . 'assets/images/default-avatar.png';
	$image_url         = ( $image_id > 0 ) ? wp_get_attachment_image_url( $image_id, 'thumbnail' ) : $default_image_url;
	?>
	<img src="<?php echo esc_url( $image_url ); ?>" class="cblc-agent-photo-preview" width="60" height="60" style="border-radius: 50%; border: 1px solid #ccc; display: inline-block; vertical-align: middle; margin-right: 15px;">
	<button type="button" class="button cblc-upload-button"><?php esc_html_e( 'Select/Upload Image', 'chat-button-lead-capture' ); ?></button>
	<input type="hidden" class="cblc-agent-image-id" name="cblc_options[cblc_agent_image_id]" value="<?php echo esc_attr( $image_id ); ?>">
	<p class="description"><?php esc_html_e( 'For a better appearance, use a square image.', 'chat-button-lead-capture' ); ?></p>
	<?php
}

/**
 * Renderiza o HTML do botão e do modal no frontend.
 */
function cblc_render_frontend_html() {
	$options = get_option( 'cblc_options' );
	if ( empty( $options['cblc_active'] ) || empty( $options['cblc_phone_number'] ) ) {
		return;
	}

	$button_text = __( 'Contact Us', 'chat-button-lead-capture' );
	echo '<div id="wa-float-button"><svg fill="white" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512" width="20" height="20"><path d="M380.9 97.1C339 55.1 283.2 32 223.9 32c-122.4 0-222 99.6-222 222 0 39.1 10.2 77.3 29.6 111L0 480l117.7-30.9c32.4 17.7 68.9 27 106.1 27h.1c122.3 0 224.1-99.6 224.1-222 0-59.3-25.2-115-67.1-157zm-157 341.6c-33.8 0-67.6-9.5-97.8-26.7l-7-4.1-72.5 19 19.3-71.1-4.5-7.4c-18.5-30.7-29.9-66.5-29.9-103.3 0-101.7 82.8-184.5 184.6-184.5 49.3 0 95.6 19.2 130.4 54.1 34.8 34.9 56.2 81.2 56.1 130.5 0 101.8-84.9 184.6-186.6 184.6zm101.2-138.2c-5.5-2.8-32.8-16.2-37.9-18-5.1-1.9-8.8-2.8-12.5 2.8-3.7 5.6-14.3 18-17.6 21.8-3.2 3.7-6.5 4.2-12 1.4-32.6-16.3-54-29.1-75.5-66-5.7-9.8 5.7-9.1 16.3-30.3 1.8-3.7.9-6.9-.5-9.7-1.4-2.8-12.5-30.1-17.1-41.2-4.5-10.8-9.1-9.3-12.5-9.5-3.2-.2-6.9-.2-10.6-.2-3.7 0-9.7 1.4-14.8 6.9-5.1 5.6-19.4 19-19.4 46.3 0 27.3 19.9 53.7 22.6 57.4 2.8 3.7 39.1 59.7 94.8 83.8 35.2 15.2 49 16.5 66.6 13.9 10.7-1.6 32.8-13.4 37.4-26.4 4.6-13 4.6-24.1 3.2-26.4-1.3-2.5-5-3.9-10.5-6.6z"/></svg> <span>' . esc_html( $button_text ) . '</span></div>';

	// Modal do formulário de captura de leads.
	$agent_name        = ! empty( $options['cblc_agent_name'] ) ? $options['cblc_agent_name'] : 'Annalú';
	$modal_status_text = sprintf( __( 'Usually replies in %s', 'chat-button-lead-capture' ), __( 'a few minutes', 'chat-button-lead-capture' ) );
	$image_id          = ! empty( $options['cblc_agent_image_id'] ) ? $options['cblc_agent_image_id'] : 0;
	$agent_image_url   = $image_id ? wp_get_attachment_image_url( $image_id, 'thumbnail' ) : plugin_dir_url( __FILE__ ) . 'assets/images/default-avatar.png';
	$form_fields_html  = '';
	if ( ! empty( $options['cblc_fields_name'] ) ) {
		$form_fields_html .= '<input type="text" name="name" placeholder="' . esc_attr__( 'Your name', 'chat-button-lead-capture' ) . '" required>';
	}
	if ( ! empty( $options['cblc_fields_email'] ) ) {
		$form_fields_html .= '<input type="email" name="email" placeholder="' . esc_attr__( 'Your e-mail', 'chat-button-lead-capture' ) . '" required>';
	}
	if ( ! empty( $options['cblc_fields_cpf_cnpj'] ) ) {
		$form_fields_html .= '<input type="text" name="cpf_cnpj" placeholder="' . esc_attr__( 'Your ID (CPF or CNPJ)', 'chat-button-lead-capture' ) . '" required>';
	}
	if (!empty($options['cblc_fields_phone'])) { $form_fields_html .= '<input type="tel" name="phone" placeholder="'.esc_attr__('Your phone number', 'chat-button-lead-capture').'" required>';
	}
	$modal_subtitle    = __( 'To start the conversation, please provide your details.', 'chat-button-lead-capture' );
	$modal_button_text = __( 'Start Conversation', 'chat-button-lead-capture' );

	echo '<div id="wa-lead-modal" style="display: none;">
            <div class="wa-modal-content">
                <div class="wa-modal-header">
                    <img src="' . esc_url( $agent_image_url ) . '" class="cblc-agent-photo" />
                    <div class="cblc-agent-info">
                        <h3>' . esc_html( $agent_name ) . '</h3>
                        <p>' . esc_html( $modal_status_text ) . '</p>
                    </div>
                    <span id="wa-modal-close" class="close">&times;</span>
                </div>
                <div class="wa-modal-body">
                    <p>' . esc_html( $modal_subtitle ) . '</p>
                    <form id="wa-lead-form">' . $form_fields_html . '<button type="submit">' . esc_html( $modal_button_text ) . '</button></form>
                    <div class="cblc-branding"><a href="https://buy-dropdesk.com.br" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Developed by DropDesk', 'chat-button-lead-capture' ) . '</a></div>
                </div>
            </div>
          </div>';
}
add_action( 'wp_footer', 'cblc_render_frontend_html' );

/**
 * Enfileira os scripts e estilos do frontend.
 */
function cblc_enqueue_frontend_assets() {
	$options = get_option( 'cblc_options' );
	if ( empty( $options['cblc_active'] ) ) {
		return;
	}

	wp_enqueue_style( 'cblc-frontend-style', plugin_dir_url( __FILE__ ) . 'assets/css/frontend-style.css', array(), '5.1.0' );
	wp_enqueue_script( 'cblc-frontend-script', plugin_dir_url( __FILE__ ) . 'assets/js/frontend-script.js', array(), '5.1.0', true );

	// Passa dados do PHP para o JavaScript de forma segura e traduzível.
	$params = array(
		'ajax_url'          => admin_url( 'admin-ajax.php' ),
		'nonce'             => wp_create_nonce( 'cblc_ajax_nonce' ),
		'phoneNumber'       => $options['cblc_phone_number'] ?? '',
		'defaultMessage'    => $options['cblc_default_message'] ?? '',
		'sending'           => __( 'Sending...', 'chat-button-lead-capture' ),
		'startConversation' => __( 'Start Conversation', 'chat-button-lead-capture' ),
	);
	wp_localize_script( 'cblc-frontend-script', 'cblc_params', $params );
}
add_action( 'wp_enqueue_scripts', 'cblc_enqueue_frontend_assets' );

/**
 * Callback AJAX para salvar o lead capturado no banco de dados.
 */
function cblc_save_lead_callback() {
	// Verifica o nonce de segurança para proteger contra ataques CSRF.
	check_ajax_referer( 'cblc_ajax_nonce', '_wpnonce' );

	global $wpdb;
	$table_name = $wpdb->prefix . 'cblc_leads';

	// Sanitiza todos os dados recebidos do formulário.
	$lead_data = array(
		'time'          => current_time( 'mysql' ),
		'name'          => isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '',
		'email'         => isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '',
		'cpf_cnpj'      => isset( $_POST['cpf_cnpj'] ) ? sanitize_text_field( wp_unslash( $_POST['cpf_cnpj'] ) ) : '',
		'lead_whatsapp' => isset( $_POST['phone'] ) ? sanitize_text_field( wp_unslash( $_POST['phone'] ) ) : '',
	);

	// Insere o lead no banco de dados.
	$wpdb->insert( $table_name, $lead_data );

	// Retorna uma resposta de sucesso.
	wp_send_json_success( 'Lead saved successfully.' );
}
add_action( 'wp_ajax_cblc_save_lead', 'cblc_save_lead_callback' );
add_action( 'wp_ajax_nopriv_cblc_save_lead', 'cblc_save_lead_callback' );

/**
 * Renderiza a caixa de doação nas páginas de administração.
 */
function cblc_render_donation_meta_box() {
	$donation_url = 'https://www.paypal.com/donate/?business=R28GR9XDBULK8&no_recurring=0&item_name=Me+ajude+a+continuar+oferecendo+o+Plugin+Gratuito+contribuindo+com+qualquer+valor.+&currency_code=BRL';
	?>
	<div class="cblc-donation-panel">
		<h2><span class="dashicons dashicons-heart" style="color: #e25555;"></span> <?php esc_html_e( 'Liked the Plugin?', 'chat-button-lead-capture' ); ?></h2>
		<p class="description"><?php esc_html_e( 'If this plugin helps you generate more business, please consider making a small donation to support continuous development.', 'chat-button-lead-capture' ); ?></p>
		<p><a href="<?php echo esc_url( $donation_url ); ?>" target="_blank" class="button button-primary">
			<?php esc_html_e( 'Make a Donation via PayPal', 'chat-button-lead-capture' ); ?>
		</a></p>
	</div>
	<?php
	// NOTA: O CSS para esta caixa de doação foi movido para o arquivo 'admin-style.css'.
	// Por favor, certifique-se de que o seguinte CSS esteja no seu arquivo:
	// .cblc-donation-panel { background: #fff; padding: 1px 20px 10px; margin-top: 20px; margin-bottom: 20px; border: 1px solid #c3c4c7; box-shadow: 0 1px 1px rgba(0,0,0,.04); }
	// .cblc-donation-panel h2 { font-size: 1.1em; }
}