<?php
/**
 * Plugin Name:  Chat Bot Alpha Interface
 * Plugin URI: https://www.npmjs.com/package/alpha-interface
 * Description: Embeds the Alpha Interface chatbot into your site using a custom web component.
 * Version: 2.1.6
 * Author: alphainterface
 * Author URI: https://alphainterface.ai
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Tested up to: 6.8
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Enqueue chatbot module script with type="module".
 */
function ai_chatbot_enqueue_scripts() {
    wp_register_script(
        'ai-chatbot-module',
        'https://cdn.jsdelivr.net/npm/alpha-interface@1.0.2/dist/hash-bot/hash-bot.esm.js',
        [],
        null,
        true
    );

    add_filter( 'script_loader_tag', function( $tag, $handle, $src ) {
        if ( 'ai-chatbot-module' === $handle ) {
            return '<script type="module" src="' . esc_url( $src ) . '"></script>';
        }
        return $tag;
    }, 10, 3 );

    wp_enqueue_script( 'ai-chatbot-module' );
}
add_action( 'wp_enqueue_scripts', 'ai_chatbot_enqueue_scripts' );

/**
 * Render chatbot in footer.
 */
function ai_chatbot_render() {
    $options = get_option( 'ai_chatbot_settings', [] );
    $bot_name = esc_attr( $options['bot_name'] ?? 'Chat Bot' );
    $agent_uuid    = esc_attr( $options['agent_uuid'] ?? '' );
    $welcome_message = esc_attr( $options['welcome_message'] ?? '' );
    $iconsize      = esc_attr( $options['iconsize'] ?? '70' );
    $chatbotwidth  = esc_attr( $options['chatbotwidth'] ?? '300' );
    $chatbotheight = esc_attr( $options['chatbotheight'] ?? '400' );

    echo '<hash-bot 
            bot_name="' . $bot_name . '"
            agent_uuid="'. $agent_uuid . '" 
            welcome_message="' . $welcome_message . '"
            iconsize="' . $iconsize . '" 
            chatbotwidth="' . $chatbotwidth . '" 
            chatbotheight="' . $chatbotheight . '">
        </hash-bot>';
}
add_action( 'wp_footer', 'ai_chatbot_render' );

/**
 * Register settings.
 */
function ai_chatbot_register_settings() {
    register_setting( 'ai_chatbot_settings_group', 'ai_chatbot_settings' );
}
add_action( 'admin_init', 'ai_chatbot_register_settings' );

/**
 * Add "Settings" link under plugin name in Plugins list.
 */
function ai_chatbot_add_settings_link( $links ) {
    $settings_link = '<a href="options-general.php?page=ai-chatbot-settings">Settings</a>';
    array_unshift( $links, $settings_link );
    return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'ai_chatbot_add_settings_link' );

/**
 * Add hidden settings page (only accessed via link in Plugins list).
 */
function ai_chatbot_register_settings_page() {
    add_options_page(
        'Chat Bot Alpha Interface Settings',
        'Chat Bot Alpha Interface',
        'manage_options',
        'ai-chatbot-settings',
        'ai_chatbot_settings_page_html'
    );
}
add_action( 'admin_menu', 'ai_chatbot_register_settings_page' );

/**
 * Settings page HTML.
 */
function ai_chatbot_settings_page_html() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $options = get_option( 'ai_chatbot_settings', [] );
    ?>
    <div class="wrap">
        <img src="<?php echo esc_url( plugins_url( 'assets/icon-128x128.png', __FILE__ ) ); ?>" 
             alt="Chat Bot Alpha Logo" style="width:64px;height:64px;margin-bottom:10px;">
        <h1>Chat Bot Alpha Interface Settings</h1>
        <form method="post" action="options.php">
            <?php settings_fields( 'ai_chatbot_settings_group' ); ?>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">Bot Name</th>
                    <td><input type="text" name="ai_chatbot_settings[bot_name]" value="<?php echo esc_attr( $options['bot_name'] ?? '' ); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row">Agent UUID</th>
                    <td><input type="text" name="ai_chatbot_settings[agent_uuid]" value="<?php echo esc_attr( $options['agent_uuid'] ?? '' ); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row">Welcome Message</th>
                    <td><input type="text" name="ai_chatbot_settings[welcome_message]" value="<?php echo esc_attr( $options['welcome_message'] ?? "Hi. Ask me anything!" ); ?>" class="regular-text" /></td>
                </tr>
                <tr>
                    <th scope="row">Icon Size</th>
                    <td><input type="number" name="ai_chatbot_settings[iconsize]" value="<?php echo esc_attr( $options['iconsize'] ?? '70' ); ?>" /></td>
                </tr>
                <tr>
                    <th scope="row">Chatbot Width</th>
                    <td><input type="number" name="ai_chatbot_settings[chatbotwidth]" value="<?php echo esc_attr( $options['chatbotwidth'] ?? '300' ); ?>" /></td>
                </tr>
                <tr>
                    <th scope="row">Chatbot Height</th>
                    <td><input type="number" name="ai_chatbot_settings[chatbotheight]" value="<?php echo esc_attr( $options['chatbotheight'] ?? '400' ); ?>" /></td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}