<?php
/**
 * Simple Stripe Integration (WordPress.org Compatible)
 *
 * @package Charity_Campaigns
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Simple Stripe Integration Class
 *
 * Uses WordPress HTTP API instead of Stripe SDK for WordPress.org compatibility
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Stripe_Simple {

	/**
	 * Stripe API base URL
	 *
	 * @since 1.0.0
	 * @var string
	 */
	private $api_base = 'https://api.stripe.com/v1/';

	/**
	 * Stripe secret key
	 *
	 * @since 1.0.0
	 * @var string
	 */
	private $secret_key;

	/**
	 * Initialize Stripe integration
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->secret_key = $this->get_secret_key();
	}

	/**
	 * Get Stripe publishable key
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function get_publishable_key() {
		// Try environment variable first
		if ( defined( 'STRIPE_PUBLISHABLE_KEY' ) ) {
			return STRIPE_PUBLISHABLE_KEY;
		}

		// Fallback to settings
		return Charity_Campaigns_Settings::get_setting( 'stripe_publishable_key', '' );
	}

	/**
	 * Get Stripe secret key
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function get_secret_key() {
		// Try environment variable first
		if ( defined( 'STRIPE_SECRET_KEY' ) ) {
			return STRIPE_SECRET_KEY;
		}

		// Fallback to settings
		return Charity_Campaigns_Settings::get_setting( 'stripe_secret_key', '' );
	}

	/**
	 * Check if test mode is enabled
	 *
	 * @since 1.0.0
	 * @return bool
	 */
	public function is_test_mode() {
		return 'true' === Charity_Campaigns_Settings::get_setting( 'stripe_test_mode', 'true' );
	}

	/**
	 * Validate that the correct keys are being used for the current mode
	 *
	 * @since 1.0.0
	 * @return bool|WP_Error
	 */
	public function validate_keys_for_mode() {
		$publishable_key = $this->get_publishable_key();
		$secret_key      = $this->get_secret_key();
		$is_test_mode    = $this->is_test_mode();

		// Check if keys are provided
		if ( empty( $publishable_key ) || empty( $secret_key ) ) {
			return new WP_Error( 'missing_keys', esc_html__( 'Stripe API keys are not configured.', 'charity-campaigns' ) );
		}

		// Check publishable key matches mode
		$publishable_is_test = strpos( $publishable_key, 'pk_test_' ) === 0;
		$publishable_is_live = strpos( $publishable_key, 'pk_live_' ) === 0;

		if ( ! $publishable_is_test && ! $publishable_is_live ) {
			return new WP_Error( 'invalid_publishable_key', esc_html__( 'Invalid publishable key format.', 'charity-campaigns' ) );
		}

		// Check secret key matches mode
		$secret_is_test = strpos( $secret_key, 'sk_test_' ) === 0;
		$secret_is_live = strpos( $secret_key, 'sk_live_' ) === 0;

		if ( ! $secret_is_test && ! $secret_is_live ) {
			return new WP_Error( 'invalid_secret_key', esc_html__( 'Invalid secret key format.', 'charity-campaigns' ) );
		}

		// Check if keys match the test mode setting
		if ( $is_test_mode && ( ! $publishable_is_test || ! $secret_is_test ) ) {
			return new WP_Error( 'mode_key_mismatch', esc_html__( 'Test mode is enabled but live API keys are being used. Please use test keys or disable test mode.', 'charity-campaigns' ) );
		}

		if ( ! $is_test_mode && ( ! $publishable_is_live || ! $secret_is_live ) ) {
			return new WP_Error( 'mode_key_mismatch', esc_html__( 'Test mode is disabled but test API keys are being used. Please use live keys or enable test mode.', 'charity-campaigns' ) );
		}

		return true;
	}

	/**
	 * Save Stripe keys
	 *
	 * @since 1.0.0
	 * @param string $publishable_key Publishable key.
	 * @param string $secret_key Secret key.
	 */
	public function save_keys( $publishable_key, $secret_key ) {
		// Get current settings
		$options = get_option( 'charity_campaigns_options', array() );

		// Update keys
		$options['stripe_publishable_key'] = sanitize_text_field( $publishable_key );
		$options['stripe_secret_key']      = sanitize_text_field( $secret_key );

		// Save settings
		update_option( 'charity_campaigns_options', $options );
	}

	/**
	 * Make API request to Stripe
	 *
	 * @since 1.0.0
	 * @param string $endpoint API endpoint.
	 * @param array  $data Request data.
	 * @param string $method HTTP method.
	 * @return array|WP_Error
	 */
	private function make_api_request( $endpoint, $data = array(), $method = 'POST' ) {
		if ( empty( $this->secret_key ) ) {
			return new WP_Error( 'no_secret_key', esc_html__( 'Stripe secret key is not configured.', 'charity-campaigns' ) );
		}

		$url  = $this->api_base . $endpoint;
		$args = array(
			'method'  => $method,
			'headers' => array(
				'Authorization' => 'Bearer ' . $this->secret_key,
				'Content-Type'  => 'application/x-www-form-urlencoded',
			),
			'timeout' => 30,
		);

		if ( 'POST' === $method && ! empty( $data ) ) {
			$args['body'] = http_build_query( $data );
		}

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$body        = wp_remote_retrieve_body( $response );
		$status_code = wp_remote_retrieve_response_code( $response );

		$decoded_response = json_decode( $body, true );

		if ( $status_code >= 400 ) {
			$error_message = isset( $decoded_response['error']['message'] )
				? $decoded_response['error']['message']
				: esc_html__( 'Stripe API error.', 'charity-campaigns' );

			return new WP_Error( 'stripe_api_error', $error_message );
		}

		return $decoded_response;
	}

	/**
	 * Create payment intent
	 *
	 * @since 1.0.0
	 * @param array $donation_data Donation data.
	 * @return array|WP_Error
	 */
	public function create_payment_intent( $donation_data ) {
		$amount   = floatval( $donation_data['amount'] ) * 100; // Convert to cents
		$currency = $this->get_currency();

		$data = array(
			'amount'                 => $amount,
			'currency'               => $currency,
			'payment_method_types[]' => 'card',
			'metadata[campaign_id]'  => $donation_data['campaign_id'],
			'metadata[donor_name]'   => $donation_data['name'],
			'metadata[donor_email]'  => $donation_data['email'],
			'metadata[plugin]'       => 'charity-campaigns',
			'metadata[version]'      => CCFS_VERSION,
		);

		$result = $this->make_api_request( 'payment_intents', $data );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return array(
			'success'        => true,
			'client_secret'  => $result['client_secret'],
			'payment_intent' => $result['id'],
		);
	}

	/**
	 * Retrieve payment intent
	 *
	 * @since 1.0.0
	 * @param string $payment_intent_id Payment intent ID.
	 * @return array|WP_Error
	 */
	public function retrieve_payment_intent( $payment_intent_id ) {
		$result = $this->make_api_request( 'payment_intents/' . $payment_intent_id, array(), 'GET' );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return array(
			'success'        => true,
			'payment_intent' => $result,
		);
	}

	/**
	 * Get currency from settings
	 *
	 * @since 1.0.0
	 * @return string
	 */
	private function get_currency() {
		$currency = Charity_Campaigns_Settings::get_default_currency();

		// Convert WordPress currency codes to Stripe format
		$currency_map = array(
			'USD' => 'usd',
			'EUR' => 'eur',
			'GBP' => 'gbp',
			'CAD' => 'cad',
			'AUD' => 'aud',
			'JPY' => 'jpy',
		);

		return isset( $currency_map[ $currency ] ) ? $currency_map[ $currency ] : 'usd';
	}

	/**
	 * Check if Stripe is properly configured
	 *
	 * @since 1.0.0
	 * @return bool
	 */
	public function is_configured() {
		$validation = $this->validate_keys_for_mode();
		return ! is_wp_error( $validation );
	}

	/**
	 * Get Stripe configuration for frontend
	 *
	 * @since 1.0.0
	 * @return array
	 */
	public function get_frontend_config() {
		return array(
			'publishable_key' => $this->get_publishable_key(),
			'is_configured'   => $this->is_configured(),
			'is_test_mode'    => $this->is_test_mode(),
		);
	}

	/**
	 * Get configuration validation error message
	 *
	 * @since 1.0.0
	 * @return string|false
	 */
	public function get_configuration_error() {
		$validation = $this->validate_keys_for_mode();
		return is_wp_error( $validation ) ? $validation->get_error_message() : false;
	}
}
