<?php
/**
 * Shortcodes for Charity Campaigns
 *
 * @package Charity_Campaigns
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class responsible for creating and managing shortcodes.
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Shortcodes {

	/**
	 * Initialize the class and register hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'register_shortcodes' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_shortcode_assets' ) );
		add_action( 'wp_ajax_ccfs_process_donation', array( $this, 'process_donation_ajax' ) );
		add_action( 'wp_ajax_nopriv_charity_process_donation', array( $this, 'process_donation_ajax' ) );
		add_action( 'wp_ajax_ccfs_create_payment_intent', array( $this, 'create_payment_intent_ajax' ) );
		add_action( 'wp_ajax_nopriv_charity_create_payment_intent', array( $this, 'create_payment_intent_ajax' ) );
		add_action( 'wp_ajax_ccfs_confirm_payment', array( $this, 'confirm_payment_ajax' ) );
		add_action( 'wp_ajax_nopriv_charity_confirm_payment', array( $this, 'confirm_payment_ajax' ) );
	}

	/**
	 * Register all shortcodes.
	 *
	 * @since 1.0.0
	 */
	public function register_shortcodes() {
		add_shortcode( 'ccfs_donation_form', array( $this, 'donation_form_shortcode' ) );
	}

	/**
	 * Enqueue assets for shortcodes.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_shortcode_assets() {
		// Only enqueue if shortcodes are present on the page
		global $post;
		if ( is_a( $post, 'WP_Post' ) && has_shortcode( $post->post_content, 'ccfs_donation_form' ) ) {
			wp_enqueue_style(
				'ccfs-shortcode',
				CCFS_PLUGIN_URL . 'assets/css/shortcode.css',
				array(),
				CCFS_VERSION
			);

			wp_enqueue_script(
				'ccfs-shortcode',
				CCFS_PLUGIN_URL . 'assets/js/shortcode.js',
				array( 'jquery' ),
				CCFS_VERSION,
				true
			);

			// Get Stripe configuration
			$stripe        = new Charity_Campaigns_Stripe_Simple();
			$stripe_config = $stripe->get_frontend_config();

			// Enqueue Stripe.js if Stripe is configured
			if ( $stripe_config['is_configured'] ) {
				wp_enqueue_script( 'stripe-js', 'https://js.stripe.com/v3/', array(), '3.0', true );
			}

			// Localize script with settings
			wp_localize_script(
				'ccfs-shortcode',
				'charityCampaignsShortcode',
				array(
					'ajaxUrl'               => admin_url( 'admin-ajax.php' ),
					'nonce'                 => wp_create_nonce( 'ccfs_shortcode' ),
					'currency'              => Charity_Campaigns_Settings::get_default_currency(),
					'currencySymbol'        => Charity_Campaigns_Settings::get_currency_symbol(),
					'currencyPosition'      => Charity_Campaigns_Settings::get_currency_symbol_position(),
					'decimalPlaces'         => Charity_Campaigns_Settings::get_decimal_places(),
					'predefinedAmounts'     => Charity_Campaigns_Settings::get_predefined_amounts(),
					'showPredefinedAmounts' => Charity_Campaigns_Settings::show_predefined_amounts(),
					'stripe'                => $stripe_config,
				)
			);
		}
	}

	/**
	 * Donation form shortcode.
	 *
	 * @since 1.0.0
	 * @param array $atts Shortcode attributes.
	 * @return string Shortcode output.
	 */
	public function donation_form_shortcode( $atts ) {
		// Parse shortcode attributes
		$atts = shortcode_atts(
			array(
				'campaign_id'             => '',
				'campaign'                => '', // Alternative to campaign_id
				'amount'                  => '',
				'button_text'             => esc_html__( 'Donate Now', 'charity-campaigns' ),
				'show_goal'               => 'true',
				'show_progress'           => 'true',
				'show_predefined_amounts' => '', // Empty means use setting, 'true'/'false' to override
				'style'                   => 'default', // default, minimal, featured
				'class'                   => '',
			),
			$atts,
			'ccfs_donation_form'
		);

		// Determine campaign ID
		$campaign_id = $atts['campaign_id'] ?: $atts['campaign'];

		// If no campaign specified, get the first active campaign
		if ( empty( $campaign_id ) ) {
			$campaigns = get_posts(
				array(
					'post_type'      => 'ccfs_campaign',
					'posts_per_page' => 1,
					'post_status'    => 'publish',
					'meta_query'     => array(
						array(
							'key'     => 'campaign_status',
							'value'   => 'active',
							'compare' => '=',
						),
					),
				)
			);

			if ( ! empty( $campaigns ) ) {
				$campaign_id = $campaigns[0]->ID;
			}
		}

		// If still no campaign, show error message
		if ( empty( $campaign_id ) ) {
			return '<div class="charity-campaigns-error">' . esc_html__( 'No active campaigns found.', 'charity-campaigns' ) . '</div>';
		}

		// Get campaign data
		$campaign = get_post( $campaign_id );
		if ( ! $campaign || $campaign->post_type !== 'ccfs_campaign' ) {
			return '<div class="charity-campaigns-error">' . esc_html__( 'Invalid campaign specified.', 'charity-campaigns' ) . '</div>';
		}

		$campaign_goal    = get_post_meta( $campaign_id, 'campaign_goal', true );
		$campaign_raised  = get_post_meta( $campaign_id, 'campaign_raised', true );
		$campaign_status  = get_post_meta( $campaign_id, 'campaign_status', true );
		$excess_amount    = get_post_meta( $campaign_id, 'campaign_excess_amount', true );
		$completion_date  = get_post_meta( $campaign_id, 'campaign_completion_date', true );

		// Calculate progress
		$progress_percentage = 0;
		if ( $campaign_goal > 0 ) {
			$progress_percentage = min( 100, ( $campaign_raised / $campaign_goal ) * 100 );
		}

		// Don't show form for completed or cancelled campaigns - show summary instead
		if ( in_array( $campaign_status, array( 'completed', 'cancelled' ) ) ) {
			$status_label = 'completed' === $campaign_status ? esc_html__( 'Completed', 'charity-campaigns' ) : esc_html__( 'Cancelled', 'charity-campaigns' );
			$status_class = 'completed' === $campaign_status ? 'charity-status-completed' : 'charity-status-cancelled';
			
			ob_start();
			?>
			<div class="charity-campaign-summary charity-campaign-status-<?php echo esc_attr( $campaign_status ); ?>">
				<div class="charity-campaign-info">
					<h3 class="charity-campaign-title"><?php echo esc_html( $campaign->post_title ); ?></h3>
					<div class="charity-campaign-status <?php echo esc_attr( $status_class ); ?>">
						<strong><?php echo esc_html( $status_label ); ?></strong>
						<?php if ( 'completed' === $campaign_status && $completion_date ) : ?>
							<span class="charity-completion-date">
								<?php
								// translators: %s is the completion date
								echo esc_html( sprintf( __( 'Completed on %s', 'charity-campaigns' ), date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $completion_date ) ) ) );
								?>
							</span>
						<?php endif; ?>
					</div>
				</div>

				<div class="charity-campaign-notice">
					<p><?php esc_html_e( 'This campaign is no longer accepting donations.', 'charity-campaigns' ); ?></p>
				</div>

				<div class="charity-campaign-goal">
					<span class="charity-goal-label"><?php esc_html_e( 'Goal:', 'charity-campaigns' ); ?></span>
					<span class="charity-goal-amount"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $campaign_goal ) ); ?></span>
				</div>

				<div class="charity-campaign-progress">
					<div class="charity-progress-bar">
						<div class="charity-progress-fill" data-progress="<?php echo esc_attr( $progress_percentage ); ?>"></div>
					</div>
					<div class="charity-progress-text">
						<span class="charity-raised"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $campaign_raised ) ); ?></span>
						<span class="charity-progress-percentage">(<?php echo esc_html( number_format( $progress_percentage, 1 ) ); ?>%)</span>
					</div>
				</div>

				<?php if ( $excess_amount && $excess_amount > 0 ) : ?>
					<div class="charity-excess-funds">
						<span class="charity-excess-label"><?php esc_html_e( 'Excess Funds:', 'charity-campaigns' ); ?></span>
						<span class="charity-excess-amount"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $excess_amount ) ); ?></span>
					</div>
				<?php endif; ?>
			</div>
			<?php
			return ob_get_clean();
		}

		// Calculate progress for active campaigns
		$progress_percentage = 0;
		if ( $campaign_goal > 0 ) {
			$progress_percentage = min( 100, ( $campaign_raised / $campaign_goal ) * 100 );
		}

		// Determine if predefined amounts should be shown
		$show_predefined = $atts['show_predefined_amounts'];
		if ( $show_predefined === '' ) {
			$show_predefined = Charity_Campaigns_Settings::show_predefined_amounts() ? 'true' : 'false';
		}

		// Build CSS classes
		$css_classes = array(
			'charity-donation-form',
			'charity-donation-form-' . esc_attr( $atts['style'] ),
		);

		if ( $atts['class'] ) {
			$css_classes[] = esc_attr( $atts['class'] );
		}

		ob_start();
		?>
		<div class="<?php echo esc_attr( implode( ' ', $css_classes ) ); ?>" 
			data-campaign-id="<?php echo esc_attr( $campaign_id ); ?>"
			data-show-predefined-amounts="<?php echo esc_attr( $show_predefined ); ?>">
			
			<?php if ( $atts['show_goal'] === 'true' || $atts['show_progress'] === 'true' ) : ?>
			<div class="charity-campaign-info">
				<h3 class="charity-campaign-title"><?php echo esc_html( $campaign->post_title ); ?></h3>
				
				<?php if ( $atts['show_goal'] === 'true' ) : ?>
				<div class="charity-campaign-goal">
					<span class="charity-goal-label"><?php esc_html_e( 'Goal:', 'charity-campaigns' ); ?></span>
					<span class="charity-goal-amount"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $campaign_goal ) ); ?></span>
				</div>
				<?php endif; ?>
				
				<?php if ( $atts['show_progress'] === 'true' ) : ?>
                <div class="charity-campaign-progress">
                    <div class="charity-progress-bar">
                        <div class="charity-progress-fill" data-progress="<?php echo esc_attr( $progress_percentage ); ?>"></div>
                    </div>
					<div class="charity-progress-text">
						<span class="charity-raised"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $campaign_raised ) ); ?></span>
						<span class="charity-progress-percentage">(<?php echo esc_html( number_format( $progress_percentage, 1 ) ); ?>%)</span>
					</div>
				</div>
				<?php endif; ?>
			</div>
			<?php endif; ?>

			<form class="charity-donation-form-fields" method="post" action="">
				<?php wp_nonce_field( 'ccfs_donation_form', 'ccfs_donation_nonce' ); ?>
				<input type="hidden" name="charity_campaign_id" value="<?php echo esc_attr( $campaign_id ); ?>" />
				
				<div class="charity-form-group">
					<label for="charity_donation_amount"><?php esc_html_e( 'Donation Amount', 'charity-campaigns' ); ?></label>
					<input type="number" id="charity_donation_amount" name="charity_donation_amount" 
						value="<?php echo esc_attr( $atts['amount'] ); ?>" 
						step="0.01" min="1" required />
					<span class="charity-currency-symbol"><?php echo esc_html( Charity_Campaigns_Settings::get_currency_symbol() ); ?></span>
				</div>

				<div class="charity-form-group">
					<label for="charity_donor_name"><?php esc_html_e( 'Your Name', 'charity-campaigns' ); ?></label>
					<input type="text" id="charity_donor_name" name="charity_donor_name" required />
				</div>

				<div class="charity-form-group">
					<label for="charity_donor_email"><?php esc_html_e( 'Email Address', 'charity-campaigns' ); ?></label>
					<input type="email" id="charity_donor_email" name="charity_donor_email" required />
				</div>

				<div class="charity-form-group">
					<label for="charity_donor_message"><?php esc_html_e( 'Message (Optional)', 'charity-campaigns' ); ?></label>
					<textarea id="charity_donor_message" name="charity_donor_message" rows="3"></textarea>
				</div>

				<div class="charity-form-group">
					<label>
						<input type="checkbox" id="charity_donor_anonymous" name="charity_donor_anonymous" value="1" />
						<?php esc_html_e( 'Make this donation anonymous', 'charity-campaigns' ); ?>
					</label>
				</div>

		<?php
		// Check if Stripe is configured
		$stripe = new Charity_Campaigns_Stripe_Simple();
		if ( $stripe->is_configured() ) :
			?>
            <?php if ( $stripe->is_test_mode() ) : ?>
            <div class="charity-test-mode-notice">
				<strong><?php esc_html_e( 'Test Mode', 'charity-campaigns' ); ?>:</strong> <?php esc_html_e( 'This is a test environment. No real payments will be processed.', 'charity-campaigns' ); ?>
			</div>
		<?php endif; ?>
		<div class="charity-form-group">
			<label for="charity-card-element"><?php esc_html_e( 'Card Information', 'charity-campaigns' ); ?></label>
			<div id="charity-card-element" class="charity-stripe-element">
				<!-- Stripe Elements will create form elements here -->
			</div>
			<div id="charity-card-errors" role="alert" class="charity-stripe-error"></div>
		</div>
		<?php endif; ?>

				<div class="charity-form-group">
					<?php if ( $stripe->is_configured() ) : ?>
						<button type="submit" class="charity-donate-button" id="charity-submit-button">
							<?php echo esc_html( $atts['button_text'] ); ?>
						</button>
                    <?php else : ?>
                        <button type="button" class="charity-donate-button is-disabled" disabled>
							<?php echo esc_html( $atts['button_text'] ); ?>
						</button>
                        <p class="charity-stripe-error">
							<?php esc_html_e( 'Payment processing is not configured. Please contact the administrator.', 'charity-campaigns' ); ?>
						</p>
					<?php endif; ?>
				</div>
			</form>

            <div class="charity-donation-messages hidden"></div>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Process donation form submission via AJAX (manual donations only).
	 *
	 * @since 1.0.0
	 */
	public function process_donation_ajax() {
		// Check if Stripe is configured - if so, redirect to Stripe flow
		$stripe = new Charity_Campaigns_Stripe_Simple();
		if ( $stripe->is_configured() ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Stripe is configured. Please use the payment form.', 'charity-campaigns' ) ) );
		}

		// Verify nonce
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'ccfs_shortcode' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed. Please try again.', 'charity-campaigns' ) ) );
		}

		// Parse form data
		if ( ! isset( $_POST['form_data'] ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid form data.', 'charity-campaigns' ) ) );
		}
		parse_str( wp_unslash( $_POST['form_data'] ), $form_data );

		// Validate required fields
		$campaign_id = isset( $form_data['charity_campaign_id'] ) ? intval( $form_data['charity_campaign_id'] ) : 0;
		$amount      = isset( $form_data['charity_donation_amount'] ) ? floatval( $form_data['charity_donation_amount'] ) : 0;
		$message     = isset( $form_data['charity_donor_message'] ) ? sanitize_textarea_field( $form_data['charity_donor_message'] ) : '';
		$anonymous   = isset( $form_data['charity_donor_anonymous'] ) ? $form_data['charity_donor_anonymous'] : '';

		// Handle anonymous donations
		if ( $anonymous === '1' ) {
			$name  = esc_html__( 'Anonymous Donor', 'charity-campaigns' );
			$email = '';
		} else {
			$name  = isset( $form_data['charity_donor_name'] ) ? sanitize_text_field( $form_data['charity_donor_name'] ) : '';
			$email = isset( $form_data['charity_donor_email'] ) ? sanitize_email( $form_data['charity_donor_email'] ) : '';
		}

		// Validate inputs
		if ( empty( $campaign_id ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid campaign selected.', 'charity-campaigns' ) ) );
		}

		if ( $amount <= 0 ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Please enter a valid donation amount.', 'charity-campaigns' ) ) );
		}
		
		// Only validate name and email if not anonymous
		if ( $anonymous !== '1' ) {
			if ( empty( $name ) || empty( $email ) ) {
				wp_send_json_error( array( 'message' => esc_html__( 'Please fill in all required fields.', 'charity-campaigns' ) ) );
			}
			
			if ( ! is_email( $email ) ) {
				wp_send_json_error( array( 'message' => esc_html__( 'Please enter a valid email address.', 'charity-campaigns' ) ) );
			}
		}

		// Verify campaign exists
		$campaign = get_post( $campaign_id );
		if ( ! $campaign || $campaign->post_type !== 'ccfs_campaign' ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid campaign selected.', 'charity-campaigns' ) ) );
		}

		// Check campaign status
		$campaign_status = get_post_meta( $campaign_id, 'campaign_status', true );
		if ( in_array( $campaign_status, array( 'completed', 'cancelled' ) ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'This campaign is no longer accepting donations.', 'charity-campaigns' ) ) );
		}

		// Create donation post
		$donation_data = array(
			// translators: %s is the donor's name
			'post_title'   => sprintf( esc_html__( 'Donation from %s', 'charity-campaigns' ), $name ),
			'post_content' => $message,
			'post_status'  => 'publish',
			'post_type'      => 'ccfs_donation',
			'post_author'  => 1, // Admin user
		);

		$donation_id = wp_insert_post( $donation_data );

		if ( is_wp_error( $donation_id ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to process donation. Please try again.', 'charity-campaigns' ) ) );
		}

		// Save donation meta data
		update_post_meta( $donation_id, 'donation_campaign_id', $campaign_id );
		update_post_meta( $donation_id, 'donation_amount', $amount );
		update_post_meta( $donation_id, 'donation_name', $name );
		update_post_meta( $donation_id, 'donation_email', $email );
		update_post_meta( $donation_id, 'donation_message', $message );
		update_post_meta( $donation_id, 'donation_anon', $anonymous );
		update_post_meta( $donation_id, 'donation_status', 'pending' );
		update_post_meta( $donation_id, 'donation_date', current_time( 'mysql' ) );
		update_post_meta( $donation_id, 'donation_gateway', 'manual' );

		// Update campaign raised amount
		$current_raised = get_post_meta( $campaign_id, 'campaign_raised', true );
		$new_raised     = floatval( $current_raised ) + $amount;
		update_post_meta( $campaign_id, 'campaign_raised', $new_raised );

		// Check if campaign goal is reached
		$campaign_goal = get_post_meta( $campaign_id, 'campaign_goal', true );
		if ( $new_raised >= $campaign_goal && $campaign_status !== 'completed' ) {
			update_post_meta( $campaign_id, 'campaign_status', 'completed' );
			update_post_meta( $campaign_id, 'campaign_completion_date', current_time( 'mysql' ) );
		}

		// Send success response
		$success_message = sprintf(
			// translators: %1$s is the donor's name, %2$s is the formatted donation amount
			esc_html__( 'Thank you, %1$s! Your donation of %2$s has been received and is pending processing.', 'charity-campaigns' ),
			$name,
			Charity_Campaigns_Functions::format_currency( $amount )
		);

		wp_send_json_success( array( 'message' => $success_message ) );
	}

	/**
	 * Create payment intent via AJAX.
	 *
	 * @since 1.0.0
	 */
	public function create_payment_intent_ajax() {
		// Verify nonce
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'ccfs_shortcode' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed. Please try again.', 'charity-campaigns' ) ) );
		}

		// Parse form data
		if ( ! isset( $_POST['form_data'] ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid form data.', 'charity-campaigns' ) ) );
		}
		parse_str( wp_unslash( $_POST['form_data'] ), $form_data );

		// Validate required fields
		$campaign_id = isset( $form_data['charity_campaign_id'] ) ? intval( $form_data['charity_campaign_id'] ) : 0;
		$amount      = isset( $form_data['charity_donation_amount'] ) ? floatval( $form_data['charity_donation_amount'] ) : 0;
		$anonymous   = isset( $form_data['charity_donor_anonymous'] ) ? $form_data['charity_donor_anonymous'] : '';
		
		// Validate inputs
		if ( empty( $campaign_id ) || $amount <= 0 ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Please fill in all required fields.', 'charity-campaigns' ) ) );
		}
		
		// Check if donation is anonymous
		if ( $anonymous !== '1' ) {
			// For non-anonymous donations, validate name and email
			$name  = isset( $form_data['charity_donor_name'] ) ? sanitize_text_field( $form_data['charity_donor_name'] ) : '';
			$email = isset( $form_data['charity_donor_email'] ) ? sanitize_email( $form_data['charity_donor_email'] ) : '';
			
			if ( empty( $name ) || empty( $email ) ) {
				wp_send_json_error( array( 'message' => esc_html__( 'Please fill in all required fields.', 'charity-campaigns' ) ) );
			}
		} else {
			// For anonymous donations, use placeholder values
			$name  = esc_html__( 'Anonymous Donor', 'charity-campaigns' );
			$email = '';
		}

		// Verify campaign exists
		$campaign = get_post( $campaign_id );
		if ( ! $campaign || $campaign->post_type !== 'ccfs_campaign' ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid campaign selected.', 'charity-campaigns' ) ) );
		}

		// Check campaign status
		$campaign_status = get_post_meta( $campaign_id, 'campaign_status', true );
		if ( in_array( $campaign_status, array( 'completed', 'cancelled' ) ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'This campaign is no longer accepting donations.', 'charity-campaigns' ) ) );
		}

		// Create payment intent
		$stripe        = new Charity_Campaigns_Stripe_Simple();
		$donation_data = array(
			'campaign_id' => $campaign_id,
			'amount'      => $amount,
			'name'        => $name,
			'email'       => $email ? $email : ( $anonymous === '1' ? 'anonymous@example.com' : '' ), // Stripe requires email, use placeholder for anonymous
		);

		$result = $stripe->create_payment_intent( $donation_data );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success( $result );
	}

	/**
	 * Confirm payment via AJAX.
	 *
	 * @since 1.0.0
	 */
	public function confirm_payment_ajax() {
		// Verify nonce
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'ccfs_shortcode' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Security check failed. Please try again.', 'charity-campaigns' ) ) );
		}

		if ( ! isset( $_POST['payment_intent_id'] ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid payment intent.', 'charity-campaigns' ) ) );
		}
		$payment_intent_id = sanitize_text_field( wp_unslash( $_POST['payment_intent_id'] ) );

		if ( empty( $payment_intent_id ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid payment intent.', 'charity-campaigns' ) ) );
		}

		// Parse form data to get message and other fields
		if ( ! isset( $_POST['form_data'] ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid form data.', 'charity-campaigns' ) ) );
		}
		parse_str( wp_unslash( $_POST['form_data'] ), $form_data );
		$message   = sanitize_textarea_field( $form_data['charity_donor_message'] );
		$anonymous = isset( $form_data['charity_donor_anonymous'] ) ? '1' : '';

		// Retrieve payment intent from Stripe
		$stripe = new Charity_Campaigns_Stripe_Simple();
		$result = $stripe->retrieve_payment_intent( $payment_intent_id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		$payment_intent = $result['payment_intent'];

		// Check if payment was successful
		if ( $payment_intent['status'] !== 'succeeded' ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Payment was not successful.', 'charity-campaigns' ) ) );
		}

		// Get donation data from metadata
		$campaign_id = $payment_intent['metadata']['campaign_id'];
		$amount      = $payment_intent['amount'] / 100; // Convert from cents

		// Check if donation is anonymous
		if ( $anonymous === '1' ) {
			$name  = esc_html__( 'Anonymous Donor', 'charity-campaigns' );
			$email = '';
		} else {
			$name  = $payment_intent['metadata']['donor_name'];
			$email = $payment_intent['metadata']['donor_email'];
		}

		// Create donation post
		$donation_data = array(
			// translators: %s is the donor's name
			'post_title'   => sprintf( esc_html__( 'Donation from %s', 'charity-campaigns' ), $name ),
			'post_content' => $message,
			'post_status'  => 'publish',
			'post_type'      => 'ccfs_donation',
			'post_author'  => 1, // Admin user
		);

		$donation_id = wp_insert_post( $donation_data );

		if ( is_wp_error( $donation_id ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to process donation. Please try again.', 'charity-campaigns' ) ) );
		}

		// Save donation meta data
		update_post_meta( $donation_id, 'donation_campaign_id', $campaign_id );
		update_post_meta( $donation_id, 'donation_amount', $amount );
		update_post_meta( $donation_id, 'donation_name', $name );
		update_post_meta( $donation_id, 'donation_email', $email );
		update_post_meta( $donation_id, 'donation_message', $message );
		update_post_meta( $donation_id, 'donation_anon', $anonymous );
		update_post_meta( $donation_id, 'donation_status', 'completed' );
		update_post_meta( $donation_id, 'donation_date', current_time( 'mysql' ) );
		update_post_meta( $donation_id, 'donation_gateway', 'stripe' );
		update_post_meta( $donation_id, 'donation_txn_id', $payment_intent['id'] );

		// Update campaign raised amount
		$current_raised = get_post_meta( $campaign_id, 'campaign_raised', true );
		$new_raised     = floatval( $current_raised ) + $amount;
		update_post_meta( $campaign_id, 'campaign_raised', $new_raised );

		// Check if campaign goal is reached
		$campaign_goal   = get_post_meta( $campaign_id, 'campaign_goal', true );
		$campaign_status = get_post_meta( $campaign_id, 'campaign_status', true );
		if ( $new_raised >= $campaign_goal && $campaign_status !== 'completed' ) {
			update_post_meta( $campaign_id, 'campaign_status', 'completed' );
			update_post_meta( $campaign_id, 'campaign_completion_date', current_time( 'mysql' ) );
		}

		// Send success response
		$success_message = sprintf(
			// translators: %1$s is the donor's name, %2$s is the formatted donation amount
			esc_html__( 'Thank you, %1$s! Your donation of %2$s has been processed successfully.', 'charity-campaigns' ),
			$name,
			Charity_Campaigns_Functions::format_currency( $amount )
		);

		wp_send_json_success( array( 'message' => $success_message ) );
	}
}
