<?php
/**
 * Plugin Settings Management
 *
 * @package Charity_Campaigns
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class responsible for managing plugin settings.
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Settings {

	/**
	 * Settings group name.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	private $settings_group = 'charity_campaigns_settings';

	/**
	 * Settings option name.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	private $settings_option = 'charity_campaigns_options';

	/**
	 * Initialize the class and register hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_settings_scripts' ) );
	}

	/**
	 * Add settings page to admin menu.
	 *
	 * @since 1.0.0
	 */
	public function add_settings_page() {
		add_submenu_page(
			'ccfs-dashboard',
			__( 'Settings', 'charity-campaigns' ),
			__( 'Settings', 'charity-campaigns' ),
			'manage_options',
			'ccfs-settings',
			array( $this, 'settings_page_callback' )
		);
	}

	/**
	 * Register settings and fields.
	 *
	 * @since 1.0.0
	 */
	public function register_settings() {
		register_setting(
			$this->settings_group,
			$this->settings_option,
			array( $this, 'sanitize_settings' )
		);

		// Currency Settings Section
		add_settings_section(
			'currency_settings',
			__( 'Currency Settings', 'charity-campaigns' ),
			array( $this, 'currency_section_callback' ),
			'ccfs-settings'
		);

		add_settings_field(
			'default_currency',
			__( 'Default Currency', 'charity-campaigns' ),
			array( $this, 'default_currency_callback' ),
			'ccfs-settings',
			'currency_settings'
		);

		add_settings_field(
			'currency_symbol_position',
			__( 'Currency Symbol Position', 'charity-campaigns' ),
			array( $this, 'currency_symbol_position_callback' ),
			'ccfs-settings',
			'currency_settings'
		);

		add_settings_field(
			'decimal_places',
			__( 'Decimal Places', 'charity-campaigns' ),
			array( $this, 'decimal_places_callback' ),
			'ccfs-settings',
			'currency_settings'
		);

		// Organization Settings Section
		add_settings_section(
			'organization_settings',
			__( 'Organization Settings', 'charity-campaigns' ),
			array( $this, 'organization_section_callback' ),
			'ccfs-settings'
		);

		add_settings_field(
			'organization_name',
			__( 'Organization Name', 'charity-campaigns' ),
			array( $this, 'organization_name_callback' ),
			'ccfs-settings',
			'organization_settings'
		);

		add_settings_field(
			'organization_logo',
			__( 'Organization Logo', 'charity-campaigns' ),
			array( $this, 'organization_logo_callback' ),
			'ccfs-settings',
			'organization_settings'
		);

		// Default Settings Section
		add_settings_section(
			'default_settings',
			__( 'Default Settings', 'charity-campaigns' ),
			array( $this, 'default_section_callback' ),
			'ccfs-settings'
		);

		add_settings_field(
			'default_campaign_status',
			__( 'Default Campaign Status', 'charity-campaigns' ),
			array( $this, 'default_campaign_status_callback' ),
			'ccfs-settings',
			'default_settings'
		);

		add_settings_field(
			'default_donation_status',
			__( 'Default Donation Status', 'charity-campaigns' ),
			array( $this, 'default_donation_status_callback' ),
			'ccfs-settings',
			'default_settings'
		);

		// Campaign Categories & Tags Section
		add_settings_section(
			'campaign_taxonomies',
			__( 'Campaign Categories & Tags', 'charity-campaigns' ),
			array( $this, 'taxonomies_section_callback' ),
			'ccfs-settings'
		);

		// Donation Form Settings Section
		add_settings_section(
			'donation_form_settings',
			__( 'Donation Form Settings', 'charity-campaigns' ),
			array( $this, 'donation_form_section_callback' ),
			'ccfs-settings'
		);

		// Stripe Payment Settings Section
		add_settings_section(
			'stripe_settings',
			__( 'Stripe Payment Settings', 'charity-campaigns' ),
			array( $this, 'stripe_section_callback' ),
			'ccfs-settings'
		);

		add_settings_field(
			'predefined_amounts',
			__( 'Predefined Donation Amounts', 'charity-campaigns' ),
			array( $this, 'predefined_amounts_callback' ),
			'ccfs-settings',
			'donation_form_settings'
		);

		add_settings_field(
			'show_predefined_amounts',
			__( 'Show Predefined Amounts', 'charity-campaigns' ),
			array( $this, 'show_predefined_amounts_callback' ),
			'ccfs-settings',
			'donation_form_settings'
		);

		// Stripe Settings Fields
		add_settings_field(
			'stripe_publishable_key',
			__( 'Stripe Publishable Key', 'charity-campaigns' ),
			array( $this, 'stripe_publishable_key_callback' ),
			'ccfs-settings',
			'stripe_settings'
		);

		add_settings_field(
			'stripe_secret_key',
			__( 'Stripe Secret Key', 'charity-campaigns' ),
			array( $this, 'stripe_secret_key_callback' ),
			'ccfs-settings',
			'stripe_settings'
		);

		add_settings_field(
			'stripe_test_mode',
			__( 'Test Mode', 'charity-campaigns' ),
			array( $this, 'stripe_test_mode_callback' ),
			'ccfs-settings',
			'stripe_settings'
		);
	}

	/**
	 * Settings page callback.
	 *
	 * @since 1.0.0
	 */
	public function settings_page_callback() {
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Charity Campaigns Settings', 'charity-campaigns' ); ?></h1>
			
			<form method="post" action="options.php">
				<?php
				settings_fields( $this->settings_group );
				do_settings_sections( 'ccfs-settings' );
				submit_button();
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Currency section callback.
	 *
	 * @since 1.0.0
	 */
	public function currency_section_callback() {
		echo '<p>' . esc_html( __( 'Configure currency settings for all campaigns and donations.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Default currency field callback.
	 *
	 * @since 1.0.0
	 */
	public function default_currency_callback() {
		$options = get_option( $this->settings_option );
		$value   = isset( $options['default_currency'] ) ? $options['default_currency'] : 'USD';

		$currencies = array(
			'USD' => 'USD - US Dollar',
			'EUR' => 'EUR - Euro',
			'GBP' => 'GBP - British Pound',
			'CAD' => 'CAD - Canadian Dollar',
			'AUD' => 'AUD - Australian Dollar',
			'JPY' => 'JPY - Japanese Yen',
			'CHF' => 'CHF - Swiss Franc',
			'CNY' => 'CNY - Chinese Yuan',
			'INR' => 'INR - Indian Rupee',
			'BRL' => 'BRL - Brazilian Real',
		);

		echo '<select name="' . esc_attr( $this->settings_option ) . '[default_currency]" id="default_currency">';
		foreach ( $currencies as $code => $label ) {
			echo '<option value="' . esc_attr( $code ) . '"' . selected( $value, $code, false ) . '>' . esc_html( $label ) . '</option>';
		}
		echo '</select>';
		echo '<p class="description">' . esc_html( __( 'Default currency for new campaigns and donations.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Currency symbol position field callback.
	 *
	 * @since 1.0.0
	 */
	public function currency_symbol_position_callback() {
		$options = get_option( $this->settings_option );
		$value   = isset( $options['currency_symbol_position'] ) ? $options['currency_symbol_position'] : 'before';

		echo '<select name="' . esc_attr( $this->settings_option ) . '[currency_symbol_position]" id="currency_symbol_position">';
		echo '<option value="before"' . selected( $value, 'before', false ) . '>' . esc_html( __( 'Before amount ($100)', 'charity-campaigns' ) ) . '</option>';
		echo '<option value="after"' . selected( $value, 'after', false ) . '>' . esc_html( __( 'After amount (100$)', 'charity-campaigns' ) ) . '</option>';
		echo '</select>';
		echo '<p class="description">' . esc_html( __( 'Position of currency symbol relative to the amount.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Decimal places field callback.
	 *
	 * @since 1.0.0
	 */
	public function decimal_places_callback() {
		$options = get_option( $this->settings_option );
		$value   = isset( $options['decimal_places'] ) ? $options['decimal_places'] : '2';

		echo '<input type="number" name="' . esc_attr( $this->settings_option ) . '[decimal_places]" id="decimal_places" value="' . esc_attr( $value ) . '" min="0" max="4" />';
		echo '<p class="description">' . esc_html( __( 'Number of decimal places to display (0-4).', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Organization section callback.
	 *
	 * @since 1.0.0
	 */
	public function organization_section_callback() {
		echo '<p>' . esc_html( __( 'Configure your organization information.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Organization name field callback.
	 *
	 * @since 1.0.0
	 */
	public function organization_name_callback() {
		$options = get_option( $this->settings_option );
		$value   = isset( $options['organization_name'] ) ? $options['organization_name'] : get_bloginfo( 'name' );

		echo '<input type="text" name="' . esc_attr( $this->settings_option ) . '[organization_name]" id="organization_name" value="' . esc_attr( $value ) . '" class="regular-text" />';
		echo '<p class="description">' . esc_html( __( 'Name of your organization.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Organization logo field callback.
	 *
	 * @since 1.0.0
	 */
	public function organization_logo_callback() {
		$options = get_option( $this->settings_option );
		$value   = isset( $options['organization_logo'] ) ? $options['organization_logo'] : '';

		echo '<input type="url" name="' . esc_attr( $this->settings_option ) . '[organization_logo]" id="organization_logo" value="' . esc_attr( $value ) . '" class="regular-text" />';
		echo '<button type="button" class="button" id="upload-logo">' . esc_html( __( 'Upload Logo', 'charity-campaigns' ) ) . '</button>';
		echo '<p class="description">' . esc_html( __( 'URL of your organization logo.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Default section callback.
	 *
	 * @since 1.0.0
	 */
	public function default_section_callback() {
		echo '<p>' . esc_html( __( 'Set default values for new campaigns and donations.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Default campaign status field callback.
	 *
	 * @since 1.0.0
	 */
	public function default_campaign_status_callback() {
		$options = get_option( $this->settings_option );
		$value   = isset( $options['default_campaign_status'] ) ? $options['default_campaign_status'] : 'active';

		$statuses = array(
			'active'    => __( 'Active', 'charity-campaigns' ),
			'completed' => __( 'Completed', 'charity-campaigns' ),
			'paused'    => __( 'Paused', 'charity-campaigns' ),
			'cancelled' => __( 'Cancelled', 'charity-campaigns' ),
		);

		echo '<select name="' . esc_attr( $this->settings_option ) . '[default_campaign_status]" id="default_campaign_status">';
		foreach ( $statuses as $code => $label ) {
			echo '<option value="' . esc_attr( $code ) . '"' . selected( $value, $code, false ) . '>' . esc_html( $label ) . '</option>';
		}
		echo '</select>';
		echo '<p class="description">' . esc_html( __( 'Default status for new campaigns.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Default donation status field callback.
	 *
	 * @since 1.0.0
	 */
	public function default_donation_status_callback() {
		$options = get_option( $this->settings_option );
		$value   = isset( $options['default_donation_status'] ) ? $options['default_donation_status'] : 'pending';

		$statuses = array(
			'pending'   => __( 'Pending', 'charity-campaigns' ),
			'completed' => __( 'Completed', 'charity-campaigns' ),
			'failed'    => __( 'Failed', 'charity-campaigns' ),
			'refunded'  => __( 'Refunded', 'charity-campaigns' ),
		);

		echo '<select name="' . esc_attr( $this->settings_option ) . '[default_donation_status]" id="default_donation_status">';
		foreach ( $statuses as $code => $label ) {
			echo '<option value="' . esc_attr( $code ) . '"' . selected( $value, $code, false ) . '>' . esc_html( $label ) . '</option>';
		}
		echo '</select>';
		echo '<p class="description">' . esc_html( __( 'Default status for new donations.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Taxonomies section callback.
	 *
	 * @since 1.0.0
	 */
	public function taxonomies_section_callback() {
		echo '<p>' . esc_html( __( 'Manage campaign categories and tags.', 'charity-campaigns' ) ) . '</p>';

		// Campaign Categories
		echo '<h3>' . esc_html( __( 'Campaign Categories', 'charity-campaigns' ) ) . '</h3>';
		echo '<p><a href="' . esc_url( admin_url( 'edit-tags.php?taxonomy=ccfs_campaign_category&post_type=ccfs_campaign' ) ) . '" class="button">' . esc_html( __( 'Manage Categories', 'charity-campaigns' ) ) . '</a></p>';

		// Campaign Tags
		echo '<h3>' . esc_html( __( 'Campaign Tags', 'charity-campaigns' ) ) . '</h3>';
		echo '<p><a href="' . esc_url( admin_url( 'edit-tags.php?taxonomy=ccfs_campaign_tag&post_type=ccfs_campaign' ) ) . '" class="button">' . esc_html( __( 'Manage Tags', 'charity-campaigns' ) ) . '</a></p>';
	}

	/**
	 * Donation form section callback.
	 *
	 * @since 1.0.0
	 */
	public function donation_form_section_callback() {
		echo '<p>' . esc_html( __( 'Configure donation form settings and predefined amounts.', 'charity-campaigns' ) ) . '</p>';
	}

	/**
	 * Predefined amounts callback.
	 *
	 * @since 1.0.0
	 */
	public function predefined_amounts_callback() {
		$predefined_amounts = $this->get_setting( 'predefined_amounts', array( 25, 50, 100, 250, 500 ) );
		?>
		<div class="charity-predefined-amounts">
			<p class="description"><?php esc_html_e( 'Enter predefined donation amounts (one per line). These will appear as quick-select buttons on donation forms.', 'charity-campaigns' ); ?></p>
			<textarea 
				id="predefined_amounts" 
				name="<?php echo esc_attr( $this->settings_option ); ?>[predefined_amounts]" 
				rows="5" 
				cols="50" 
				class="large-text"
			><?php echo esc_textarea( implode( "\n", $predefined_amounts ) ); ?></textarea>
			<p class="description">
				<?php esc_html_e( 'Example:', 'charity-campaigns' ); ?><br>
				25<br>
				50<br>
				100<br>
				250<br>
				500
			</p>
		</div>
		<?php
	}

	/**
	 * Show predefined amounts callback.
	 *
	 * @since 1.0.0
	 */
	public function show_predefined_amounts_callback() {
		$show_predefined = $this->get_setting( 'show_predefined_amounts', 'true' );
		?>
		<label>
			<input 
				type="checkbox" 
				name="<?php echo esc_attr( $this->settings_option ); ?>[show_predefined_amounts]" 
				value="true" 
				<?php checked( $show_predefined, 'true' ); ?> 
			/>
			<?php esc_html_e( 'Show predefined amount buttons on donation forms', 'charity-campaigns' ); ?>
		</label>
		<p class="description"><?php esc_html_e( 'When enabled, quick-select buttons will appear above the amount input field.', 'charity-campaigns' ); ?></p>
		<?php
	}

	/**
	 * Stripe section callback.
	 *
	 * @since 1.0.0
	 */
	public function stripe_section_callback() {
		$stripe        = new Charity_Campaigns_Stripe_Simple();
		$is_configured = $stripe->is_configured();
		$is_test_mode  = $stripe->is_test_mode();
		$error_message = $stripe->get_configuration_error();
		?>
		<p><?php esc_html_e( 'Configure Stripe payment processing for donation forms. You can get your API keys from your Stripe Dashboard.', 'charity-campaigns' ); ?></p>
		<p>
			<a href="https://dashboard.stripe.com/apikeys" target="_blank" class="button button-secondary">
				<?php esc_html_e( 'Get Stripe API Keys', 'charity-campaigns' ); ?>
			</a>
		</p>
		
		<?php if ( $is_configured ) : ?>
			<div class="notice notice-success inline">
				<p>
					<strong><?php esc_html_e( 'Stripe is configured and ready!', 'charity-campaigns' ); ?></strong>
					<?php if ( $is_test_mode ) : ?>
						<br><?php esc_html_e( 'Currently in TEST MODE - no real payments will be processed.', 'charity-campaigns' ); ?>
					<?php else : ?>
						<br><?php esc_html_e( 'Currently in LIVE MODE - real payments will be processed.', 'charity-campaigns' ); ?>
					<?php endif; ?>
				</p>
			</div>
		<?php elseif ( $error_message ) : ?>
			<div class="notice notice-error inline">
				<p><strong><?php esc_html_e( 'Configuration Error:', 'charity-campaigns' ); ?></strong> <?php echo esc_html( $error_message ); ?></p>
			</div>
		<?php endif; ?>
		<?php
	}

	/**
	 * Stripe publishable key callback.
	 *
	 * @since 1.0.0
	 */
	public function stripe_publishable_key_callback() {
		$publishable_key = $this->get_setting( 'stripe_publishable_key', '' );
		?>
		<input 
			type="text" 
			id="stripe_publishable_key" 
			name="<?php echo esc_attr( $this->settings_option ); ?>[stripe_publishable_key]" 
			value="<?php echo esc_attr( $publishable_key ); ?>" 
			class="regular-text" 
			placeholder="pk_test_..."
		/>
		<p class="description">
			<?php esc_html_e( 'Your Stripe publishable key (starts with pk_test_ or pk_live_)', 'charity-campaigns' ); ?>
		</p>
		<?php
	}

	/**
	 * Stripe secret key callback.
	 *
	 * @since 1.0.0
	 */
	public function stripe_secret_key_callback() {
		$secret_key = $this->get_setting( 'stripe_secret_key', '' );
		?>
		<input 
			type="password" 
			id="stripe_secret_key" 
			name="<?php echo esc_attr( $this->settings_option ); ?>[stripe_secret_key]" 
			value="<?php echo esc_attr( $secret_key ); ?>" 
			class="regular-text" 
			placeholder="sk_test_..."
		/>
		<p class="description">
			<?php esc_html_e( 'Your Stripe secret key (starts with sk_test_ or sk_live_). This will be encrypted when saved.', 'charity-campaigns' ); ?>
		</p>
		<?php
	}

	/**
	 * Stripe test mode callback.
	 *
	 * @since 1.0.0
	 */
	public function stripe_test_mode_callback() {
		$test_mode = $this->get_setting( 'stripe_test_mode', 'true' );
		?>
		<label>
			<input 
				type="checkbox" 
				name="<?php echo esc_attr( $this->settings_option ); ?>[stripe_test_mode]" 
				value="true" 
				<?php checked( $test_mode, 'true' ); ?> 
			/>
			<?php esc_html_e( 'Enable test mode (use test API keys)', 'charity-campaigns' ); ?>
		</label>
		<p class="description">
			<?php esc_html_e( 'When enabled, the plugin will use Stripe test mode. Uncheck this when you\'re ready to accept real payments.', 'charity-campaigns' ); ?>
		</p>
		<?php
	}

	/**
	 * Sanitize settings input.
	 *
	 * @since 1.0.0
	 * @param array $input Raw input data.
	 * @return array Sanitized data.
	 */
	public function sanitize_settings( $input ) {
		$sanitized = array();

		// Sanitize currency settings
		if ( isset( $input['default_currency'] ) ) {
			$sanitized['default_currency'] = sanitize_text_field( $input['default_currency'] );
		}

		if ( isset( $input['currency_symbol_position'] ) ) {
			$sanitized['currency_symbol_position'] = sanitize_text_field( $input['currency_symbol_position'] );
		}

		if ( isset( $input['decimal_places'] ) ) {
			$sanitized['decimal_places'] = absint( $input['decimal_places'] );
		}

		// Sanitize organization settings
		if ( isset( $input['organization_name'] ) ) {
			$sanitized['organization_name'] = sanitize_text_field( $input['organization_name'] );
		}

		if ( isset( $input['organization_logo'] ) ) {
			$sanitized['organization_logo'] = esc_url_raw( $input['organization_logo'] );
		}

		// Sanitize default settings
		if ( isset( $input['default_campaign_status'] ) ) {
			$sanitized['default_campaign_status'] = sanitize_text_field( $input['default_campaign_status'] );
		}

		if ( isset( $input['default_donation_status'] ) ) {
			$sanitized['default_donation_status'] = sanitize_text_field( $input['default_donation_status'] );
		}

		// Sanitize donation form settings
		if ( isset( $input['predefined_amounts'] ) ) {
			$amounts           = explode( "\n", $input['predefined_amounts'] );
			$sanitized_amounts = array();
			foreach ( $amounts as $amount ) {
				$amount = trim( $amount );
				if ( is_numeric( $amount ) && $amount > 0 ) {
					$sanitized_amounts[] = floatval( $amount );
				}
			}
			$sanitized['predefined_amounts'] = $sanitized_amounts;
		}

		if ( isset( $input['show_predefined_amounts'] ) ) {
			$sanitized['show_predefined_amounts'] = sanitize_text_field( $input['show_predefined_amounts'] );
		}

		// Sanitize Stripe settings
		if ( isset( $input['stripe_publishable_key'] ) ) {
			$sanitized['stripe_publishable_key'] = sanitize_text_field( $input['stripe_publishable_key'] );
		}

		if ( isset( $input['stripe_secret_key'] ) ) {
			$sanitized['stripe_secret_key'] = sanitize_text_field( $input['stripe_secret_key'] );
		}

		if ( isset( $input['stripe_test_mode'] ) ) {
			$sanitized['stripe_test_mode'] = sanitize_text_field( $input['stripe_test_mode'] );
		}

		return $sanitized;
	}

	/**
	 * Enqueue settings page scripts.
	 *
	 * @since 1.0.0
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_settings_scripts( $hook ) {
		if ( $hook === 'ccfs_page_ccfs-settings' ) {
			wp_enqueue_media();
			wp_enqueue_script(
				'ccfs-settings',
				CCFS_PLUGIN_URL . 'assets/js/settings.js',
				array( 'jquery' ),
				CCFS_VERSION,
				true
			);
		}
	}

	/**
	 * Get setting value.
	 *
	 * @since 1.0.0
	 * @param string $key Setting key.
	 * @param mixed  $default Default value.
	 * @return mixed Setting value.
	 */
	public static function get_setting( $key, $default = null ) {
		$options = get_option( 'charity_campaigns_options' );
		return isset( $options[ $key ] ) ? $options[ $key ] : $default;
	}

	/**
	 * Get default currency.
	 *
	 * @since 1.0.0
	 * @return string Currency code.
	 */
	public static function get_default_currency() {
		return self::get_setting( 'default_currency', 'USD' );
	}

	/**
	 * Get currency symbol.
	 *
	 * @since 1.0.0
	 * @return string Currency symbol.
	 */
	public static function get_currency_symbol() {
		$currency = self::get_default_currency();
		$symbols  = array(
			'USD' => '$',
			'EUR' => '€',
			'GBP' => '£',
			'CAD' => 'C$',
			'AUD' => 'A$',
			'JPY' => '¥',
			'CHF' => 'CHF',
			'CNY' => '¥',
			'INR' => '₹',
			'BRL' => 'R$',
		);
		return isset( $symbols[ $currency ] ) ? $symbols[ $currency ] : $currency . ' ';
	}

	/**
	 * Get currency symbol position.
	 *
	 * @since 1.0.0
	 * @return string Position (before/after).
	 */
	public static function get_currency_symbol_position() {
		return self::get_setting( 'currency_symbol_position', 'before' );
	}

	/**
	 * Get decimal places.
	 *
	 * @since 1.0.0
	 * @return int Number of decimal places.
	 */
	public static function get_decimal_places() {
		return self::get_setting( 'decimal_places', 2 );
	}

	/**
	 * Get default campaign status.
	 *
	 * @since 1.0.0
	 * @return string Campaign status.
	 */
	public static function get_default_campaign_status() {
		return self::get_setting( 'default_campaign_status', 'active' );
	}

	/**
	 * Get default donation status.
	 *
	 * @since 1.0.0
	 * @return string Donation status.
	 */
	public static function get_default_donation_status() {
		return self::get_setting( 'default_donation_status', 'pending' );
	}

	/**
	 * Get organization name.
	 *
	 * @since 1.0.0
	 * @return string Organization name.
	 */
	public static function get_organization_name() {
		return self::get_setting( 'organization_name', get_bloginfo( 'name' ) );
	}

	/**
	 * Get organization logo.
	 *
	 * @since 1.0.0
	 * @return string Organization logo URL.
	 */
	public static function get_organization_logo() {
		return self::get_setting( 'organization_logo', '' );
	}

	/**
	 * Get predefined donation amounts.
	 *
	 * @since 1.0.0
	 * @return array Array of predefined amounts.
	 */
	public static function get_predefined_amounts() {
		return self::get_setting( 'predefined_amounts', array( 25, 50, 100, 250, 500 ) );
	}

	/**
	 * Check if predefined amounts should be shown.
	 *
	 * @since 1.0.0
	 * @return bool True if predefined amounts should be shown.
	 */
	public static function show_predefined_amounts() {
		return self::get_setting( 'show_predefined_amounts', 'true' ) === 'true';
	}
}
