<?php
/**
 * Post Types and Taxonomies Registration
 *
 * @package Charity_Campaigns
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class responsible for registering custom post types and taxonomies.
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Post_Types {

	/**
	 * Initialize the class and register hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'register_post_types' ) );
		add_action( 'init', array( $this, 'register_taxonomies' ) );
	}

	/**
	 * Register custom post types.
	 *
	 * @since 1.0.0
	 */
	public function register_post_types() {
		$this->register_campaign_post_type();
		$this->register_donation_post_type();
	}

	/**
	 * Register campaign custom post type.
	 *
	 * @since 1.0.0
	 */
	private function register_campaign_post_type() {
		$labels = array(
			'name'                  => _x( 'Campaigns', 'Post type general name', 'charity-campaigns' ),
			'singular_name'         => _x( 'Campaign', 'Post type singular name', 'charity-campaigns' ),
			'menu_name'             => _x( 'Campaigns', 'Admin Menu text', 'charity-campaigns' ),
			'name_admin_bar'        => _x( 'Campaign', 'Add New on Toolbar', 'charity-campaigns' ),
			'add_new'               => __( 'Add New', 'charity-campaigns' ),
			'add_new_item'          => __( 'Add New Campaign', 'charity-campaigns' ),
			'new_item'              => __( 'New Campaign', 'charity-campaigns' ),
			'edit_item'             => __( 'Edit Campaign', 'charity-campaigns' ),
			'view_item'             => __( 'View Campaign', 'charity-campaigns' ),
			'all_items'             => __( 'All Campaigns', 'charity-campaigns' ),
			'search_items'          => __( 'Search Campaigns', 'charity-campaigns' ),
			'parent_item_colon'     => __( 'Parent Campaigns:', 'charity-campaigns' ),
			'not_found'             => __( 'No campaigns found.', 'charity-campaigns' ),
			'not_found_in_trash'    => __( 'No campaigns found in Trash.', 'charity-campaigns' ),
			'featured_image'        => _x( 'Campaign Image', 'Overrides the "Featured Image" phrase', 'charity-campaigns' ),
			'set_featured_image'    => _x( 'Set campaign image', 'Overrides the "Set featured image" phrase', 'charity-campaigns' ),
			'remove_featured_image' => _x( 'Remove campaign image', 'Overrides the "Remove featured image" phrase', 'charity-campaigns' ),
			'use_featured_image'    => _x( 'Use as campaign image', 'Overrides the "Use as featured image" phrase', 'charity-campaigns' ),
			'archives'              => _x( 'Campaign archives', 'The post type archive label', 'charity-campaigns' ),
			'insert_into_item'      => _x( 'Insert into campaign', 'Overrides the "Insert into post" phrase', 'charity-campaigns' ),
			'uploaded_to_this_item' => _x( 'Uploaded to this campaign', 'Overrides the "Uploaded to this post" phrase', 'charity-campaigns' ),
			'filter_items_list'     => _x( 'Filter campaigns list', 'Screen reader text for the filter links', 'charity-campaigns' ),
			'items_list_navigation' => _x( 'Campaigns list navigation', 'Screen reader text for the pagination', 'charity-campaigns' ),
			'items_list'            => _x( 'Campaigns list', 'Screen reader text for the items list', 'charity-campaigns' ),
		);

		$args = array(
			'labels'             => $labels,
			'public'             => true,
			'publicly_queryable' => true,
			'show_ui'            => true,
			'show_in_menu'       => 'ccfs-dashboard',
			'show_in_rest'       => true,
			'query_var'          => true,
			'rewrite'            => array( 'slug' => 'campaigns' ),
			'capability_type'    => 'post',
			'has_archive'        => true,
			'hierarchical'       => false,
			'menu_position'      => 26,
			'menu_icon'          => 'dashicons-heart',
			'supports'           => array( 'title', 'thumbnail' ),
			'taxonomies'         => array( 'ccfs_campaign_category', 'ccfs_campaign_tag' ),
		);

		register_post_type( 'ccfs_campaign', $args );
	}

	/**
	 * Register donation custom post type.
	 *
	 * @since 1.0.0
	 */
	private function register_donation_post_type() {
		$labels = array(
			'name'                  => _x( 'Donations', 'Post type general name', 'charity-campaigns' ),
			'singular_name'         => _x( 'Donation', 'Post type singular name', 'charity-campaigns' ),
			'menu_name'             => _x( 'Donations', 'Admin Menu text', 'charity-campaigns' ),
			'name_admin_bar'        => _x( 'Donation', 'Add New on Toolbar', 'charity-campaigns' ),
			'add_new'               => __( 'Add New', 'charity-campaigns' ),
			'add_new_item'          => __( 'Add New Donation', 'charity-campaigns' ),
			'new_item'              => __( 'New Donation', 'charity-campaigns' ),
			'edit_item'             => __( 'Edit Donation', 'charity-campaigns' ),
			'view_item'             => __( 'View Donation', 'charity-campaigns' ),
			'all_items'             => __( 'All Donations', 'charity-campaigns' ),
			'search_items'          => __( 'Search Donations', 'charity-campaigns' ),
			'parent_item_colon'     => __( 'Parent Donations:', 'charity-campaigns' ),
			'not_found'             => __( 'No donations found.', 'charity-campaigns' ),
			'not_found_in_trash'    => __( 'No donations found in Trash.', 'charity-campaigns' ),
			'archives'              => _x( 'Donation archives', 'The post type archive label', 'charity-campaigns' ),
			'insert_into_item'      => _x( 'Insert into donation', 'Overrides the "Insert into post" phrase', 'charity-campaigns' ),
			'uploaded_to_this_item' => _x( 'Uploaded to this donation', 'Overrides the "Uploaded to this post" phrase', 'charity-campaigns' ),
			'filter_items_list'     => _x( 'Filter donations list', 'Screen reader text for the filter links', 'charity-campaigns' ),
			'items_list_navigation' => _x( 'Donations list navigation', 'Screen reader text for the pagination', 'charity-campaigns' ),
			'items_list'            => _x( 'Donations list', 'Screen reader text for the items list', 'charity-campaigns' ),
		);

		$args = array(
			'labels'             => $labels,
			'public'             => false,
			'publicly_queryable' => false,
			'show_ui'            => true,
			'show_in_menu'       => 'ccfs-dashboard',
			'show_in_rest'       => true,
			'query_var'          => true,
			'rewrite'            => false,
			'capability_type'    => 'post',
			'has_archive'        => false,
			'hierarchical'       => false,
			'menu_position'      => 27,
			'menu_icon'          => 'dashicons-money-alt',
			'supports'           => array( 'title' ),
		);

		register_post_type( 'ccfs_donation', $args );
	}

	/**
	 * Register taxonomies.
	 *
	 * @since 1.0.0
	 */
	public function register_taxonomies() {
		$this->register_campaign_category_taxonomy();
		$this->register_campaign_tag_taxonomy();
	}

	/**
	 * Register campaign category taxonomy.
	 *
	 * @since 1.0.0
	 */
	private function register_campaign_category_taxonomy() {
		$labels = array(
			'name'              => _x( 'Campaign Categories', 'taxonomy general name', 'charity-campaigns' ),
			'singular_name'     => _x( 'Campaign Category', 'taxonomy singular name', 'charity-campaigns' ),
			'search_items'      => __( 'Search Categories', 'charity-campaigns' ),
			'all_items'         => __( 'All Categories', 'charity-campaigns' ),
			'parent_item'       => __( 'Parent Category', 'charity-campaigns' ),
			'parent_item_colon' => __( 'Parent Category:', 'charity-campaigns' ),
			'edit_item'         => __( 'Edit Category', 'charity-campaigns' ),
			'update_item'       => __( 'Update Category', 'charity-campaigns' ),
			'add_new_item'      => __( 'Add New Category', 'charity-campaigns' ),
			'new_item_name'     => __( 'New Category Name', 'charity-campaigns' ),
			'menu_name'         => __( 'Categories', 'charity-campaigns' ),
		);

		$args = array(
			'hierarchical'      => true,
			'labels'            => $labels,
			'show_ui'           => true,
			'show_admin_column' => true,
			'show_in_rest'      => true,
			'query_var'         => true,
			'rewrite'           => array( 'slug' => 'campaign-category' ),
		);

		register_taxonomy( 'ccfs_campaign_category', array( 'ccfs_campaign' ), $args );
	}

	/**
	 * Register campaign tag taxonomy.
	 *
	 * @since 1.0.0
	 */
	private function register_campaign_tag_taxonomy() {
		$labels = array(
			'name'                       => _x( 'Campaign Tags', 'taxonomy general name', 'charity-campaigns' ),
			'singular_name'              => _x( 'Campaign Tag', 'taxonomy singular name', 'charity-campaigns' ),
			'search_items'               => __( 'Search Tags', 'charity-campaigns' ),
			'popular_items'              => __( 'Popular Tags', 'charity-campaigns' ),
			'all_items'                  => __( 'All Tags', 'charity-campaigns' ),
			'edit_item'                  => __( 'Edit Tag', 'charity-campaigns' ),
			'update_item'                => __( 'Update Tag', 'charity-campaigns' ),
			'add_new_item'               => __( 'Add New Tag', 'charity-campaigns' ),
			'new_item_name'              => __( 'New Tag Name', 'charity-campaigns' ),
			'separate_items_with_commas' => __( 'Separate tags with commas', 'charity-campaigns' ),
			'add_or_remove_items'        => __( 'Add or remove tags', 'charity-campaigns' ),
			'choose_from_most_used'      => __( 'Choose from the most used tags', 'charity-campaigns' ),
			'not_found'                  => __( 'No tags found.', 'charity-campaigns' ),
			'menu_name'                  => __( 'Tags', 'charity-campaigns' ),
		);

		$args = array(
			'hierarchical'      => false,
			'labels'            => $labels,
			'show_ui'           => true,
			'show_admin_column' => true,
			'show_in_rest'      => true,
			'query_var'         => true,
			'rewrite'           => array( 'slug' => 'campaign-tag' ),
		);

		register_taxonomy( 'ccfs_campaign_tag', array( 'ccfs_campaign' ), $args );
	}


	/**
	 * Flush rewrite rules on plugin activation.
	 *
	 * @since 1.0.0
	 */
	public function flush_rewrite_rules() {
		$this->register_post_types();
		$this->register_taxonomies();
		flush_rewrite_rules();
	}
}
