<?php
/**
 * Metaboxes for Custom Post Types
 *
 * @package Charity_Campaigns
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class responsible for creating and managing metaboxes.
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Metaboxes {

	/**
	 * Initialize the class and register hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'add_meta_boxes', array( $this, 'add_metaboxes' ) );
		add_action( 'save_post', array( $this, 'save_metabox_data' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
		add_action( 'wp_insert_post', array( $this, 'set_default_campaign_values' ), 10, 3 );
		add_action( 'wp_insert_post', array( $this, 'set_default_donation_values' ), 10, 3 );
		add_action( 'admin_notices', array( $this, 'show_goal_increase_notice' ) );

		// Calculation hooks moved to Charity_Campaigns_Donations_Service for portability
	}

	/**
	 * Add metaboxes to post types.
	 *
	 * @since 1.0.0
	 */
	public function add_metaboxes() {
		// Campaign metabox
		add_meta_box(
			'ccfs_campaign_details',
			__( 'Campaign Details', 'charity-campaigns' ),
			array( $this, 'campaign_metabox_callback' ),
			'ccfs_campaign',
			'normal',
			'high'
		);

		// Campaign donors metabox
		add_meta_box(
			'ccfs_campaign_donors',
			__( 'Campaign Donors', 'charity-campaigns' ),
			array( $this, 'campaign_donors_metabox_callback' ),
			'ccfs_campaign',
			'side',
			'default'
		);

		// Donation metabox
		add_meta_box(
			'ccfs_donation_details',
			__( 'Donation Details', 'charity-campaigns' ),
			array( $this, 'donation_metabox_callback' ),
			'ccfs_donation',
			'normal',
			'high'
		);
	}

	/**
	 * Campaign metabox callback.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post The post object.
	 */
	public function campaign_metabox_callback( $post ) {
		// Add nonce field for security
		wp_nonce_field( 'ccfs_campaign_metabox', 'ccfs_campaign_metabox_nonce' );

		// Get current values
		$goal            = get_post_meta( $post->ID, 'campaign_goal', true );
		$raised          = get_post_meta( $post->ID, 'campaign_raised', true );
		$start_date      = get_post_meta( $post->ID, 'campaign_start_date', true );
		$end_date        = get_post_meta( $post->ID, 'campaign_end_date', true );
		$completion_date = get_post_meta( $post->ID, 'campaign_completion_date', true );
		$status          = get_post_meta( $post->ID, 'campaign_status', true ) ?: Charity_Campaigns_Settings::get_default_campaign_status();
		$excess_amount   = get_post_meta( $post->ID, 'campaign_excess_amount', true );
		$excess_date     = get_post_meta( $post->ID, 'campaign_excess_date', true );
		$external_url    = get_post_meta( $post->ID, 'campaign_external_url', true );
		$highlight       = get_post_meta( $post->ID, 'campaign_highlight', true );
		$color           = get_post_meta( $post->ID, 'campaign_color', true );

		// Calculate actual raised amount from donations
		if ( class_exists( 'Charity_Campaigns_Donations_Service' ) ) {
			$calculated_raised = Charity_Campaigns_Donations_Service::calculate_campaign_raised_amount( $post->ID );
		} else {
			$calculated_raised = floatval( $raised );
		}

		// Status options
		$status_options = array(
			'active'    => __( 'Active', 'charity-campaigns' ),
			'completed' => __( 'Completed', 'charity-campaigns' ),
			'paused'    => __( 'Paused', 'charity-campaigns' ),
			'cancelled' => __( 'Cancelled', 'charity-campaigns' ),
		);
		?>
		<div class="campaign-progress-container"></div>
		<table class="form-table">
			<tr>
				<th scope="row">
					<label for="campaign_goal"><?php esc_html_e( 'Fundraising Goal', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="number" id="campaign_goal" name="campaign_goal" value="<?php echo esc_attr( $goal ); ?>" class="regular-text" step="0.01" min="0" />
					<p class="description"><?php esc_html_e( 'Enter the target amount to raise for this campaign.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="campaign_raised"><?php esc_html_e( 'Amount Raised', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="number" id="campaign_raised" name="campaign_raised" value="<?php echo esc_attr( $raised ); ?>" class="regular-text" step="0.01" min="0" />
					<p class="description">
						<?php esc_html_e( 'Current amount raised for this campaign.', 'charity-campaigns' ); ?>
						<br>
						<strong><?php esc_html_e( 'Calculated from donations:', 'charity-campaigns' ); ?> <?php echo esc_html( Charity_Campaigns_Functions::format_currency( $calculated_raised ) ); ?></strong>
						<br>
						<button type="button" id="sync-raised-amount" class="button button-secondary">
							<?php esc_html_e( 'Sync with Donations', 'charity-campaigns' ); ?>
						</button>
					</p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="campaign_start_date"><?php esc_html_e( 'Start Date', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="date" id="campaign_start_date" name="campaign_start_date" value="<?php echo esc_attr( $start_date ); ?>" class="regular-text" />
					<p class="description"><?php esc_html_e( 'When this campaign starts.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="campaign_end_date"><?php esc_html_e( 'End Date', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="date" id="campaign_end_date" name="campaign_end_date" value="<?php echo esc_attr( $end_date ); ?>" class="regular-text" />
					<p class="description"><?php esc_html_e( 'When this campaign ends.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="campaign_completion_date"><?php esc_html_e( 'Completion Date', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="datetime-local" id="campaign_completion_date" name="campaign_completion_date" value="<?php echo esc_attr( $completion_date ); ?>" class="regular-text" readonly />
					<p class="description"><?php esc_html_e( 'Automatically set when campaign reaches 100% funding.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label><?php esc_html_e( 'Excess Funds', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="number" value="<?php echo esc_attr( $excess_amount ); ?>" class="regular-text" step="0.01" min="0" readonly />
					<p class="description">
						<?php esc_html_e( 'Amount collected over the goal.', 'charity-campaigns' ); ?>
						<?php if ( $excess_date ) : ?>
							<br>
							<strong><?php esc_html_e( 'First exceeded on:', 'charity-campaigns' ); ?> <?php echo esc_html( $excess_date ); ?></strong>
						<?php endif; ?>
					</p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="campaign_status"><?php esc_html_e( 'Status', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<select id="campaign_status" name="campaign_status" class="regular-text">
						<option value=""><?php esc_html_e( 'Select Status', 'charity-campaigns' ); ?></option>
						<?php foreach ( $status_options as $value => $label ) : ?>
							<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $status, $value ); ?>><?php echo esc_html( $label ); ?></option>
						<?php endforeach; ?>
					</select>
					<p class="description"><?php esc_html_e( 'Current status of this campaign.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="campaign_external_url"><?php esc_html_e( 'External URL', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="url" id="campaign_external_url" name="campaign_external_url" value="<?php echo esc_attr( $external_url ); ?>" class="regular-text" />
					<p class="description"><?php esc_html_e( 'External campaign page URL (optional).', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="campaign_highlight"><?php esc_html_e( 'Highlight Campaign', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<label>
						<input type="checkbox" id="campaign_highlight" name="campaign_highlight" value="1" <?php checked( $highlight, '1' ); ?> />
						<?php esc_html_e( 'Highlight this campaign on the frontend', 'charity-campaigns' ); ?>
					</label>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="campaign_color"><?php esc_html_e( 'Theme Color', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="color" id="campaign_color" name="campaign_color" value="<?php echo esc_attr( $color ); ?>" class="regular-text" />
					<p class="description"><?php esc_html_e( 'Choose a color theme for this campaign.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Campaign donors metabox callback.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post The post object.
	 */
	public function campaign_donors_metabox_callback( $post ) {
		// Get all donations for this campaign
		$donations = get_posts(
			array(
				'post_type'      => 'ccfs_donation',
				'posts_per_page' => -1,
				'meta_query'     => array(
					array(
						'key'     => 'donation_campaign_id',
						'value'   => $post->ID,
						'compare' => '=',
					),
				),
				'orderby'        => 'meta_value',
				'meta_key'       => 'donation_date',
				'order'          => 'DESC',
			)
		);

		if ( empty( $donations ) ) {
			echo '<p>' . esc_html__( 'No donations yet for this campaign.', 'charity-campaigns' ) . '</p>';
			return;
		}

		// translators: %d is the number of donations
		echo '<p><strong>' . esc_html( sprintf( _n( '%d donation', '%d donations', count( $donations ), 'charity-campaigns' ), count( $donations ) ) ) . '</strong></p>';

        echo '<div class="campaign-donors-list scroll-max-300">';
		echo '<table class="widefat fixed" cellspacing="0">';
		echo '<thead>';
		echo '<tr>';
		echo '<th>' . esc_html__( 'Donor', 'charity-campaigns' ) . '</th>';
		echo '<th>' . esc_html__( 'Amount', 'charity-campaigns' ) . '</th>';
		echo '<th>' . esc_html__( 'Date', 'charity-campaigns' ) . '</th>';
		echo '</tr>';
		echo '</thead>';
		echo '<tbody>';

		foreach ( $donations as $donation ) {
			$donation_id = $donation->ID;
			$amount      = get_post_meta( $donation_id, 'donation_amount', true );
			$name        = get_post_meta( $donation_id, 'donation_name', true );
			$email       = get_post_meta( $donation_id, 'donation_email', true );
			$anon        = get_post_meta( $donation_id, 'donation_anon', true );
			$date        = get_post_meta( $donation_id, 'donation_date', true );
			$status      = get_post_meta( $donation_id, 'donation_status', true );

			// Skip failed donations
			if ( 'failed' === $status ) {
				continue;
			}

			// Determine display name
			$display_name = $anon ? esc_html__( 'Anonymous', 'charity-campaigns' ) : ( $name ?: esc_html__( 'Unknown Donor', 'charity-campaigns' ) );

			// Format date
			$formatted_date = $date ? gmdate( 'M j, Y', strtotime( $date ) ) : esc_html__( 'Unknown', 'charity-campaigns' );

			echo '<tr>';
			echo '<td>';
			echo '<strong>' . esc_html( $display_name ) . '</strong>';
			if ( $email && ! $anon ) {
				echo '<br><small>' . esc_html( $email ) . '</small>';
			}
			echo '</td>';
			echo '<td><strong>' . esc_html( Charity_Campaigns_Functions::format_currency( $amount ) ) . '</strong></td>';
			echo '<td>' . esc_html( $formatted_date ) . '</td>';
			echo '</tr>';
		}

		echo '</tbody>';
		echo '</table>';
		echo '</div>';

		// Add link to view all donations
		$donations_url = admin_url( 'edit.php?post_type=ccfs_donation&donation_campaign=' . $post->ID );
        echo '<p class="mt-10">';
		echo '<a href="' . esc_url( $donations_url ) . '" class="button button-secondary">' . esc_html__( 'View All Donations', 'charity-campaigns' ) . '</a>';
		echo '</p>';
	}

	/**
	 * Donation metabox callback.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post The post object.
	 */
	public function donation_metabox_callback( $post ) {
		// Add nonce field for security
		wp_nonce_field( 'ccfs_donation_metabox', 'ccfs_donation_metabox_nonce' );

		// Get current values
		$campaign_id = get_post_meta( $post->ID, 'donation_campaign_id', true );
		$amount      = get_post_meta( $post->ID, 'donation_amount', true );
		$name        = get_post_meta( $post->ID, 'donation_name', true );
		$email       = get_post_meta( $post->ID, 'donation_email', true );
		$message     = get_post_meta( $post->ID, 'donation_message', true );
		$anon        = get_post_meta( $post->ID, 'donation_anon', true );
		$gateway     = get_post_meta( $post->ID, 'donation_gateway', true );
		$txn_id      = get_post_meta( $post->ID, 'donation_txn_id', true );
		$status      = get_post_meta( $post->ID, 'donation_status', true ) ?: Charity_Campaigns_Settings::get_default_donation_status();
		$date        = get_post_meta( $post->ID, 'donation_date', true );

		// Get campaigns for dropdown
		$campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
			)
		);

		// Status options
		$status_options = array(
			'pending'   => __( 'Pending', 'charity-campaigns' ),
			'completed' => __( 'Completed', 'charity-campaigns' ),
			'failed'    => __( 'Failed', 'charity-campaigns' ),
			'refunded'  => __( 'Refunded', 'charity-campaigns' ),
		);

		?>
		<table class="form-table">
			<tr>
				<th scope="row">
					<label for="donation_campaign_id"><?php esc_html_e( 'Campaign', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<select id="donation_campaign_id" name="donation_campaign_id" class="regular-text">
						<option value=""><?php esc_html_e( 'Select Campaign', 'charity-campaigns' ); ?></option>
						<?php foreach ( $campaigns as $campaign ) : ?>
							<option value="<?php echo esc_attr( $campaign->ID ); ?>" <?php selected( $campaign_id, $campaign->ID ); ?>><?php echo esc_html( $campaign->post_title ); ?></option>
						<?php endforeach; ?>
					</select>
					<p class="description"><?php esc_html_e( 'Select the campaign this donation is for.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="donation_amount"><?php esc_html_e( 'Amount', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="number" id="donation_amount" name="donation_amount" value="<?php echo esc_attr( $amount ); ?>" class="regular-text" step="0.01" min="0" />
					<p class="description"><?php esc_html_e( 'Donation amount.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="donation_name"><?php esc_html_e( 'Donor Name', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="text" id="donation_name" name="donation_name" value="<?php echo esc_attr( $name ); ?>" class="regular-text" />
					<p class="description"><?php esc_html_e( 'Name of the donor.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="donation_email"><?php esc_html_e( 'Donor Email', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="email" id="donation_email" name="donation_email" value="<?php echo esc_attr( $email ); ?>" class="regular-text" />
					<p class="description"><?php esc_html_e( 'Email address of the donor (private).', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="donation_message"><?php esc_html_e( 'Message', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<textarea id="donation_message" name="donation_message" rows="3" cols="50" class="large-text"><?php echo esc_textarea( $message ); ?></textarea>
					<p class="description"><?php esc_html_e( 'Message from the donor (optional).', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="donation_anon"><?php esc_html_e( 'Anonymous Donation', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<label>
						<input type="checkbox" id="donation_anon" name="donation_anon" value="1" <?php checked( $anon, '1' ); ?> />
						<?php esc_html_e( 'This is an anonymous donation', 'charity-campaigns' ); ?>
					</label>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="donation_gateway"><?php esc_html_e( 'Payment Gateway', 'charity-campaigns' ); ?></label>
				</th>
				<td>
                    <input type="text" id="donation_gateway" name="donation_gateway" value="<?php echo esc_attr( $gateway ); ?>" class="regular-text input-readonly" readonly="readonly" />
					<p class="description"><?php esc_html_e( 'Payment gateway used for this donation (automatically set).', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="donation_txn_id"><?php esc_html_e( 'Transaction ID', 'charity-campaigns' ); ?></label>
				</th>
				<td>
                    <input type="text" id="donation_txn_id" name="donation_txn_id" value="<?php echo esc_attr( $txn_id ); ?>" class="regular-text input-readonly" readonly="readonly" />
					<p class="description"><?php esc_html_e( 'Transaction ID from the payment gateway (automatically set).', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="donation_status"><?php esc_html_e( 'Status', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<select id="donation_status" name="donation_status" class="regular-text">
						<option value=""><?php esc_html_e( 'Select Status', 'charity-campaigns' ); ?></option>
						<?php foreach ( $status_options as $value => $label ) : ?>
							<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $status, $value ); ?>><?php echo esc_html( $label ); ?></option>
						<?php endforeach; ?>
					</select>
					<p class="description"><?php esc_html_e( 'Current status of this donation.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="donation_date"><?php esc_html_e( 'Donation Date', 'charity-campaigns' ); ?></label>
				</th>
				<td>
					<input type="datetime-local" id="donation_date" name="donation_date" value="<?php echo esc_attr( $date ); ?>" class="regular-text" />
					<p class="description"><?php esc_html_e( 'Date and time of the donation.', 'charity-campaigns' ); ?></p>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Save metabox data.
	 *
	 * @since 1.0.0
	 * @param int $post_id The post ID.
	 */
	public function save_metabox_data( $post_id ) {
		// Check if this is an autosave
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Check user permissions
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Save campaign data
		if ( get_post_type( $post_id ) === 'ccfs_campaign' ) {
			$this->save_campaign_data( $post_id );
		}

		// Save donation data
		if ( get_post_type( $post_id ) === 'ccfs_donation' ) {
			$this->save_donation_data( $post_id );
		}
	}

	/**
	 * Save campaign metabox data.
	 *
	 * @since 1.0.0
	 * @param int $post_id The post ID.
	 */
	private function save_campaign_data( $post_id ) {
		// Verify nonce
		if ( ! isset( $_POST['ccfs_campaign_metabox_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['ccfs_campaign_metabox_nonce'] ) ), 'ccfs_campaign_metabox' ) ) {
			return;
		}

		// Check if goal is being increased and reset completion status if needed
		if ( isset( $_POST['campaign_goal'] ) ) {
			$new_goal        = absint( $_POST['campaign_goal'] );
			$current_goal    = get_post_meta( $post_id, 'campaign_goal', true );
			$current_raised  = get_post_meta( $post_id, 'campaign_raised', true );
			$completion_date = get_post_meta( $post_id, 'campaign_completion_date', true );
			$excess_date     = get_post_meta( $post_id, 'campaign_excess_date', true );

			// If goal is being increased and campaign was completed, reset completion status
			if ( $new_goal > $current_goal && $completion_date ) {
				// Reset completion date
				delete_post_meta( $post_id, 'campaign_completion_date' );

				// Reset excess date and amount if goal is now higher than raised amount
				if ( $new_goal > $current_raised ) {
					delete_post_meta( $post_id, 'campaign_excess_date' );
					update_post_meta( $post_id, 'campaign_excess_amount', 0 );
				}

				// Update status to active if it was completed
				$current_status = get_post_meta( $post_id, 'campaign_status', true );
				if ( $current_status === 'completed' ) {
					update_post_meta( $post_id, 'campaign_status', 'active' );
				}

				// Force clear any cached post data
				clean_post_cache( $post_id );

				// Redirect to refresh the page and show updated data
				wp_redirect(
					add_query_arg(
						array(
							'post'           => $post_id,
							'action'         => 'edit',
							'goal_increased' => '1',
						),
						admin_url( 'post.php' )
					)
				);
				exit;
			}
		}

		// Save meta fields
		$fields = array(
			'campaign_goal'            => 'absint',
			'campaign_raised'          => 'absint',
			'campaign_start_date'      => 'sanitize_text_field',
			'campaign_end_date'        => 'sanitize_text_field',
			'campaign_completion_date' => 'sanitize_text_field',
			'campaign_status'          => 'sanitize_text_field',
			'campaign_external_url'    => 'esc_url_raw',
			'campaign_highlight'       => 'rest_sanitize_boolean',
			'campaign_color'           => 'sanitize_hex_color',
		);

		foreach ( $fields as $field => $sanitize_callback ) {
			if ( isset( $_POST[ $field ] ) ) {
				// phpcs:disable WordPress.Security.ValidatedSanitizedInput.MissingUnslash
				$raw_value = $_POST[ $field ];
				$value     = wp_unslash( $raw_value );
				if ( $sanitize_callback === 'rest_sanitize_boolean' ) {
					$value = $value ? '1' : '';
				} elseif ( $sanitize_callback === 'absint' ) {
					$value = absint( $value );
				} else {
					$value = call_user_func( $sanitize_callback, $value );
				}
				update_post_meta( $post_id, $field, $value );
			} else {
				delete_post_meta( $post_id, $field );
			}
		}

		// After saving, recalculate campaign status based on new goal vs raised amount
		$this->recalculate_campaign_status( $post_id );
	}

	/**
	 * Recalculate campaign status based on goal vs raised amount.
	 *
	 * @since 1.0.0
	 * @param int $post_id Campaign post ID.
	 */
	private function recalculate_campaign_status( $post_id ) {
		$goal            = get_post_meta( $post_id, 'campaign_goal', true );
		$raised          = get_post_meta( $post_id, 'campaign_raised', true );
		$status          = get_post_meta( $post_id, 'campaign_status', true );
		$completion_date = get_post_meta( $post_id, 'campaign_completion_date', true );
		$excess_date     = get_post_meta( $post_id, 'campaign_excess_date', true );

		// Only recalculate if we have valid goal and raised amounts
		if ( empty( $goal ) || empty( $raised ) ) {
			return;
		}

		$goal   = floatval( $goal );
		$raised = floatval( $raised );

		// Check if campaign reached 100% and set completion date
		if ( $goal > 0 && $raised >= $goal && ! $completion_date ) {
			$current_time = current_time( 'Y-m-d\TH:i' );
			update_post_meta( $post_id, 'campaign_completion_date', $current_time );

			// Update status to completed if it's not already
			if ( $status !== 'completed' ) {
				update_post_meta( $post_id, 'campaign_status', 'completed' );
			}
		}

		// Calculate and store excess funds (amount over goal)
		if ( $goal > 0 && $raised > $goal ) {
			$excess_amount = max( $raised - $goal, 0 );
			update_post_meta( $post_id, 'campaign_excess_amount', $excess_amount );

			// Set first time exceeded date if not set
			if ( ! $excess_date ) {
				$current_time = current_time( 'Y-m-d\TH:i' );
				update_post_meta( $post_id, 'campaign_excess_date', $current_time );
			}
		} else {
			// Ensure excess amount is zero if not exceeding
			update_post_meta( $post_id, 'campaign_excess_amount', 0 );
		}
	}

	/**
	 * Save donation metabox data.
	 *
	 * @since 1.0.0
	 * @param int $post_id The post ID.
	 */
	private function save_donation_data( $post_id ) {
		// Verify nonce
		if ( ! isset( $_POST['ccfs_donation_metabox_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['ccfs_donation_metabox_nonce'] ) ), 'ccfs_donation_metabox' ) ) {
			return;
		}

		// Fields that can be edited from admin
		$editable_fields = array(
			'donation_campaign_id' => 'absint',
			'donation_amount'      => 'floatval',
			'donation_name'        => 'sanitize_text_field',
			'donation_email'       => 'sanitize_email',
			'donation_message'     => 'sanitize_textarea_field',
			'donation_anon'        => 'rest_sanitize_boolean',
			'donation_status'      => 'sanitize_text_field',
			'donation_date'        => 'sanitize_text_field',
		);

		// Fields that are read-only (set by payment processing)
		$readonly_fields = array(
			'donation_gateway',
			'donation_txn_id',
		);

		// Save editable fields
		foreach ( $editable_fields as $field => $sanitize_callback ) {
			if ( isset( $_POST[ $field ] ) ) {
				// phpcs:disable WordPress.Security.ValidatedSanitizedInput.MissingUnslash
				$raw_value = $_POST[ $field ];
				$value     = wp_unslash( $raw_value );
				if ( $sanitize_callback === 'rest_sanitize_boolean' ) {
					$value = $value ? '1' : '';
				} else {
					$value = call_user_func( $sanitize_callback, $value );
				}
				update_post_meta( $post_id, $field, $value );
			} else {
				delete_post_meta( $post_id, $field );
			}
		}

		// Prevent modification of read-only fields
		foreach ( $readonly_fields as $field ) {
			// Only allow these fields to be set if they don't already exist
			// This prevents admin from overriding payment gateway data
			if ( ! get_post_meta( $post_id, $field, true ) ) {
				if ( isset( $_POST[ $field ] ) && ! empty( $_POST[ $field ] ) ) {
					update_post_meta( $post_id, $field, sanitize_text_field( wp_unslash( $_POST[ $field ] ) ) );
				}
			}
		}
	}

	/**
	 * Show goal increase notice.
	 *
	 * @since 1.0.0
	 */
	public function show_goal_increase_notice() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for displaying notice
		if ( isset( $_GET['goal_increased'] ) && sanitize_text_field( wp_unslash( $_GET['goal_increased'] ) ) === '1' ) {
			echo '<div class="notice notice-success is-dismissible">';
			echo '<p><strong>' . esc_html__( 'Campaign Goal Increased Successfully!', 'charity-campaigns' ) . '</strong> - ';
			echo esc_html__( 'The campaign completion status has been reset to allow for continued fundraising.', 'charity-campaigns' );
			echo '</p></div>';
		}
	}

	/**
	 * Enqueue admin scripts.
	 *
	 * @since 1.0.0
	 * @param string $hook The current admin page hook.
	 */
	public function enqueue_admin_scripts( $hook ) {
		global $post_type;

		// Only load on post edit screens for our custom post types
		if ( in_array( $post_type, array( 'ccfs_campaign', 'ccfs_donation' ) ) && in_array( $hook, array( 'post.php', 'post-new.php' ) ) ) {
			wp_enqueue_style(
				'ccfs-admin',
				CCFS_PLUGIN_URL . 'assets/css/admin.css',
				array(),
				CCFS_VERSION
			);

			wp_enqueue_script(
				'ccfs-admin',
				CCFS_PLUGIN_URL . 'assets/js/admin.js',
				array( 'jquery' ),
				CCFS_VERSION,
				true
			);

			// Pass currency settings to JavaScript for campaign pages
			wp_localize_script(
				'ccfs-admin',
				'charityCampaignsSettings',
				array(
					'defaultCurrency'        => Charity_Campaigns_Settings::get_default_currency(),
					'currencySymbolPosition' => Charity_Campaigns_Settings::get_currency_symbol_position(),
					'decimalPlaces'          => Charity_Campaigns_Settings::get_decimal_places(),
				)
			);
		}
	}

	/**
	 * Set default values for new campaigns.
	 *
	 * @since 1.0.0
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post Post object.
	 * @param bool    $update Whether this is an update.
	 */
	public function set_default_campaign_values( $post_id, $post, $update ) {
		// Only for new campaigns
		if ( $update || $post->post_type !== 'ccfs_campaign' ) {
			return;
		}

		// Set default campaign status if not already set
		if ( ! get_post_meta( $post_id, 'campaign_status', true ) ) {
			update_post_meta( $post_id, 'campaign_status', Charity_Campaigns_Settings::get_default_campaign_status() );
		}
	}

	/**
	 * Set default values for new donations.
	 *
	 * @since 1.0.0
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post Post object.
	 * @param bool    $update Whether this is an update.
	 */
	public function set_default_donation_values( $post_id, $post, $update ) {
		// Only for new donations
		if ( $update || $post->post_type !== 'ccfs_donation' ) {
			return;
		}

		// Set default donation status if not already set
		if ( ! get_post_meta( $post_id, 'donation_status', true ) ) {
			update_post_meta( $post_id, 'donation_status', Charity_Campaigns_Settings::get_default_donation_status() );
		}
	}
}
