<?php
/**
 * Help System Management
 *
 * @package Charity_Campaigns
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class responsible for managing help content and documentation.
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Help {

	/**
	 * Initialize the class and register hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// No hooks needed - this is a utility class
	}

	/**
	 * Render the help tab content.
	 *
	 * @since 1.0.0
	 */
	public function render_help_content() {
		?>
		<div class="charity-dashboard-help">
			<?php $this->render_help_header(); ?>
			<div class="help-sections">
				<?php
				$this->render_quick_start_guide();
				$this->render_shortcodes_reference();
				$this->render_payment_setup();
				$this->render_campaign_management();
				$this->render_donation_management();
				$this->render_troubleshooting();
				$this->render_support_resources();
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render help header section.
	 *
	 * @since 1.0.0
	 */
	private function render_help_header() {
		?>
		<div class="help-header">
			<h2><?php esc_html_e( 'Charity Campaigns Help Center', 'charity-campaigns' ); ?></h2>
			<p><?php esc_html_e( 'Everything you need to know to get started with Charity Campaigns.', 'charity-campaigns' ); ?></p>
		</div>
		<?php
	}

	/**
	 * Render quick start guide section.
	 *
	 * @since 1.0.0
	 */
	private function render_quick_start_guide() {
		?>
		<div class="help-section">
			<h3><span class="dashicons dashicons-rocket"></span> <?php esc_html_e( 'Quick Start Guide', 'charity-campaigns' ); ?></h3>
			<div class="help-content">
				<ol>
					<li>
						<strong><?php esc_html_e( 'Configure Settings', 'charity-campaigns' ); ?></strong>
						<p><?php esc_html_e( 'Go to Settings → Charity Campaigns to set up your organization details and Stripe payment processing.', 'charity-campaigns' ); ?></p>
					</li>
					<li>
						<strong><?php esc_html_e( 'Create Your First Campaign', 'charity-campaigns' ); ?></strong>
						<p><?php esc_html_e( 'Add a new Campaign from the WordPress admin menu. Set your fundraising goal, description, and campaign details.', 'charity-campaigns' ); ?></p>
					</li>
					<li>
						<strong><?php esc_html_e( 'Add Donation Form', 'charity-campaigns' ); ?></strong>
						<p><?php esc_html_e( 'Use the [ccfs_donation_form] shortcode on any page or post to display the donation form.', 'charity-campaigns' ); ?></p>
					</li>
					<li>
						<strong><?php esc_html_e( 'Track Progress', 'charity-campaigns' ); ?></strong>
						<p><?php esc_html_e( 'Monitor donations and campaign progress from this dashboard.', 'charity-campaigns' ); ?></p>
					</li>
				</ol>
			</div>
		</div>
		<?php
	}

	/**
	 * Render shortcodes reference section.
	 *
	 * @since 1.0.0
	 */
	private function render_shortcodes_reference() {
		?>
		<div class="help-section">
			<h3><span class="dashicons dashicons-editor-code"></span> <?php esc_html_e( 'Shortcodes Reference', 'charity-campaigns' ); ?></h3>
			<div class="help-content">
				<div class="shortcode-item">
					<code>[ccfs_donation_form]</code>
					<p><?php esc_html_e( 'Displays the donation form. Automatically shows all active campaigns for selection.', 'charity-campaigns' ); ?></p>
					<p><strong><?php esc_html_e( 'Parameters:', 'charity-campaigns' ); ?></strong></p>
					<ul>
						<li><code>campaign_id="123"</code> - <?php esc_html_e( 'Show form for specific campaign only', 'charity-campaigns' ); ?></li>
						<li><code>campaign="123"</code> - <?php esc_html_e( 'Alternative to campaign_id', 'charity-campaigns' ); ?></li>
						<li><code>amount="50"</code> - <?php esc_html_e( 'Pre-fill donation amount', 'charity-campaigns' ); ?></li>
						<li><code>button_text="Donate Now"</code> - <?php esc_html_e( 'Customize button text', 'charity-campaigns' ); ?></li>
						<li><code>show_goal="true"</code> - <?php esc_html_e( 'Show campaign goal and progress', 'charity-campaigns' ); ?></li>
						<li><code>show_progress="true"</code> - <?php esc_html_e( 'Show progress bar', 'charity-campaigns' ); ?></li>
						<li><code>show_predefined_amounts="true"</code> - <?php esc_html_e( 'Show predefined amount buttons', 'charity-campaigns' ); ?></li>
						<li><code>style="default"</code> - <?php esc_html_e( 'Form style: default, minimal, or featured', 'charity-campaigns' ); ?></li>
						<li><code>class="custom-class"</code> - <?php esc_html_e( 'Add custom CSS classes', 'charity-campaigns' ); ?></li>
					</ul>
					<p><strong><?php esc_html_e( 'Examples:', 'charity-campaigns' ); ?></strong></p>
					<ul>
						<li><code>[ccfs_donation_form]</code> - <?php esc_html_e( 'Basic donation form', 'charity-campaigns' ); ?></li>
						<li><code>[ccfs_donation_form campaign_id="123" amount="25"]</code> - <?php esc_html_e( 'Form for specific campaign with pre-filled amount', 'charity-campaigns' ); ?></li>
						<li><code>[ccfs_donation_form style="minimal" button_text="Support Us"]</code> - <?php esc_html_e( 'Minimal style with custom button text', 'charity-campaigns' ); ?></li>
					</ul>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Render payment setup section.
	 *
	 * @since 1.0.0
	 */
	private function render_payment_setup() {
		?>
		<div class="help-section">
			<h3><span class="dashicons dashicons-money-alt"></span> <?php esc_html_e( 'Payment Setup', 'charity-campaigns' ); ?></h3>
			<div class="help-content">
				<h4><?php esc_html_e( 'Stripe Integration', 'charity-campaigns' ); ?></h4>
				<ol>
					<li><?php esc_html_e( 'Create a Stripe account at stripe.com', 'charity-campaigns' ); ?></li>
					<li><?php esc_html_e( 'Get your API keys from the Stripe Dashboard', 'charity-campaigns' ); ?></li>
					<li><?php esc_html_e( 'Enter your keys in Settings → Charity Campaigns', 'charity-campaigns' ); ?></li>
					<li><?php esc_html_e( 'Test with Stripe test keys before going live', 'charity-campaigns' ); ?></li>
				</ol>
				<p><strong><?php esc_html_e( 'Test Mode:', 'charity-campaigns' ); ?></strong> <?php esc_html_e( 'Use test keys (pk_test_... and sk_test_...) for development. Switch to live keys (pk_live_... and sk_live_...) for production.', 'charity-campaigns' ); ?></p>
			</div>
		</div>
		<?php
	}

	/**
	 * Render campaign management section.
	 *
	 * @since 1.0.0
	 */
	private function render_campaign_management() {
		?>
		<div class="help-section">
			<h3><span class="dashicons dashicons-megaphone"></span> <?php esc_html_e( 'Campaign Management', 'charity-campaigns' ); ?></h3>
			<div class="help-content">
				<h4><?php esc_html_e( 'Creating Campaigns', 'charity-campaigns' ); ?></h4>
				<ul>
					<li><?php esc_html_e( 'Set a clear, compelling title and description', 'charity-campaigns' ); ?></li>
					<li><?php esc_html_e( 'Upload an engaging featured image', 'charity-campaigns' ); ?></li>
					<li><?php esc_html_e( 'Set realistic fundraising goals', 'charity-campaigns' ); ?></li>
					<li><?php esc_html_e( 'Choose appropriate start and end dates', 'charity-campaigns' ); ?></li>
				</ul>
				
				<h4><?php esc_html_e( 'Campaign Statuses', 'charity-campaigns' ); ?></h4>
				<ul>
					<li><strong><?php esc_html_e( 'Active:', 'charity-campaigns' ); ?></strong> <?php esc_html_e( 'Live and accepting donations', 'charity-campaigns' ); ?></li>
					<li><strong><?php esc_html_e( 'Completed:', 'charity-campaigns' ); ?></strong> <?php esc_html_e( 'Goal reached or end date passed', 'charity-campaigns' ); ?></li>
					<li><strong><?php esc_html_e( 'Paused:', 'charity-campaigns' ); ?></strong> <?php esc_html_e( 'Temporarily stopped, not accepting donations', 'charity-campaigns' ); ?></li>
					<li><strong><?php esc_html_e( 'Cancelled:', 'charity-campaigns' ); ?></strong> <?php esc_html_e( 'Manually stopped by admin', 'charity-campaigns' ); ?></li>
				</ul>
			</div>
		</div>
		<?php
	}

	/**
	 * Render donation management section.
	 *
	 * @since 1.0.0
	 */
	private function render_donation_management() {
		?>
		<div class="help-section">
			<h3><span class="dashicons dashicons-heart"></span> <?php esc_html_e( 'Donation Management', 'charity-campaigns' ); ?></h3>
			<div class="help-content">
				<h4><?php esc_html_e( 'Donation Types', 'charity-campaigns' ); ?></h4>
				<ul>
					<li><strong><?php esc_html_e( 'Stripe Donations:', 'charity-campaigns' ); ?></strong> <?php esc_html_e( 'Processed through Stripe payment gateway', 'charity-campaigns' ); ?></li>
					<li><strong><?php esc_html_e( 'Manual Donations:', 'charity-campaigns' ); ?></strong> <?php esc_html_e( 'Added manually by administrators', 'charity-campaigns' ); ?></li>
				</ul>
				
				<h4><?php esc_html_e( 'Donation Privacy', 'charity-campaigns' ); ?></h4>
				<p><?php esc_html_e( 'Donors can choose to donate anonymously. When selected, their name and email are not recorded, and they appear as "Anonymous Donor" in reports.', 'charity-campaigns' ); ?></p>
				
				<h4><?php esc_html_e( 'Managing Donations', 'charity-campaigns' ); ?></h4>
				<ul>
					<li><?php esc_html_e( 'View all donations in the Donations admin menu', 'charity-campaigns' ); ?></li>
					<li><?php esc_html_e( 'Edit donation details (amount, donor info, status)', 'charity-campaigns' ); ?></li>
					<li><?php esc_html_e( 'Add manual donations for offline contributions', 'charity-campaigns' ); ?></li>
					<li><?php esc_html_e( 'Filter donations by date range and campaign', 'charity-campaigns' ); ?></li>
				</ul>
			</div>
		</div>
		<?php
	}

	/**
	 * Render troubleshooting section.
	 *
	 * @since 1.0.0
	 */
	private function render_troubleshooting() {
		?>
		<div class="help-section">
			<h3><span class="dashicons dashicons-sos"></span> <?php esc_html_e( 'Troubleshooting', 'charity-campaigns' ); ?></h3>
			<div class="help-content">
				<h4><?php esc_html_e( 'Common Issues', 'charity-campaigns' ); ?></h4>
				<div class="troubleshoot-item">
					<strong><?php esc_html_e( 'Donation form not working:', 'charity-campaigns' ); ?></strong>
					<ul>
						<li><?php esc_html_e( 'Check if Stripe keys are properly configured', 'charity-campaigns' ); ?></li>
						<li><?php esc_html_e( 'Verify test mode settings match your API keys', 'charity-campaigns' ); ?></li>
						<li><?php esc_html_e( 'Ensure JavaScript is enabled in the browser', 'charity-campaigns' ); ?></li>
					</ul>
				</div>
				<div class="troubleshoot-item">
					<strong><?php esc_html_e( 'Campaigns not showing:', 'charity-campaigns' ); ?></strong>
					<ul>
						<li><?php esc_html_e( 'Check campaign status is set to "Active"', 'charity-campaigns' ); ?></li>
						<li><?php esc_html_e( 'Verify campaign dates are current', 'charity-campaigns' ); ?></li>
						<li><?php esc_html_e( 'Check if campaign is published', 'charity-campaigns' ); ?></li>
					</ul>
				</div>
				<div class="troubleshoot-item">
					<strong><?php esc_html_e( 'Payment processing errors:', 'charity-campaigns' ); ?></strong>
					<ul>
						<li><?php esc_html_e( 'Verify Stripe API keys are correct', 'charity-campaigns' ); ?></li>
						<li><?php esc_html_e( 'Check Stripe account status and permissions', 'charity-campaigns' ); ?></li>
						<li><?php esc_html_e( 'Review WordPress error logs for detailed error messages', 'charity-campaigns' ); ?></li>
					</ul>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Render support and resources section.
	 *
	 * @since 1.0.0
	 */
	private function render_support_resources() {
		?>
		<div class="help-section">
			<h3><span class="dashicons dashicons-info"></span> <?php esc_html_e( 'Support & Resources', 'charity-campaigns' ); ?></h3>
			<div class="help-content">
				<div class="support-links">
					<div class="support-item">
						<h4><?php esc_html_e( 'Documentation', 'charity-campaigns' ); ?></h4>
						<p><?php esc_html_e( 'Comprehensive guides and API documentation available in the plugin files.', 'charity-campaigns' ); ?></p>
					</div>
					<div class="support-item">
						<h4><?php esc_html_e( 'WordPress.org Support', 'charity-campaigns' ); ?></h4>
						<p><?php esc_html_e( 'Visit the plugin support forum for community help and bug reports.', 'charity-campaigns' ); ?></p>
					</div>
					<div class="support-item">
						<h4><?php esc_html_e( 'Stripe Support', 'charity-campaigns' ); ?></h4>
						<p><?php esc_html_e( 'For payment processing issues, consult Stripe\'s documentation and support.', 'charity-campaigns' ); ?></p>
					</div>
				</div>
			</div>
		</div>
		<?php
	}
}
