<?php
/**
 * General Functions and Utilities
 *
 * @package CCFS
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class containing general utility functions.
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Functions {

	/**
	 * Get recent activities for dashboard.
	 *
	 * @since 1.0.0
	 * @return array Recent activities.
	 */
	public static function get_recent_activities() {
		$activities = array();

		// Get recent donations
		$recent_donations = get_posts(
			array(
				'post_type'      => 'ccfs_donation',
				'posts_per_page' => 5,
				'post_status'    => 'publish',
				'orderby'        => 'date',
				'order'          => 'DESC',
			)
		);

		foreach ( $recent_donations as $donation ) {
			$amount       = get_post_meta( $donation->ID, 'donation_amount', true );
			$campaign_id  = get_post_meta( $donation->ID, 'donation_campaign_id', true );
			$donor_name   = get_post_meta( $donation->ID, 'donation_name', true );
			$is_anonymous = get_post_meta( $donation->ID, 'donation_anon', true );

			$campaign_title = '';
			if ( $campaign_id ) {
				$campaign       = get_post( $campaign_id );
				$campaign_title = $campaign ? $campaign->post_title : '';
			}

			$display_name = $is_anonymous ? __( 'Anonymous', 'charity-campaigns' ) : ( $donor_name ?: __( 'Unknown Donor', 'charity-campaigns' ) );
			$text         = sprintf(
			// translators: %1$s is the donor name, %2$s is the formatted amount, %3$s is the campaign title
				__( '%1$s donated %2$s to %3$s', 'charity-campaigns' ),
				$display_name,
				'$' . number_format( $amount, 2 ),
				$campaign_title ?: __( 'a campaign', 'charity-campaigns' )
			);

			$activities[] = array(
				'type' => 'donation',
				'icon' => '💰',
				'text' => $text,
				'time' => human_time_diff( strtotime( $donation->post_date ), current_time( 'timestamp' ) ) . ' ' . __( 'ago', 'charity-campaigns' ),
				'date' => strtotime( $donation->post_date ),
			);
		}

		// Get recent campaigns
		$recent_campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => 3,
				'post_status'    => 'publish',
				'orderby'        => 'date',
				'order'          => 'DESC',
			)
		);

		foreach ( $recent_campaigns as $campaign ) {
			$activities[] = array(
				'type' => 'campaign',
				'icon' => '🎯',
				// translators: %s is the campaign title
				'text' => sprintf( __( 'New campaign "%s" was created', 'charity-campaigns' ), $campaign->post_title ),
				'time' => human_time_diff( strtotime( $campaign->post_date ), current_time( 'timestamp' ) ) . ' ' . __( 'ago', 'charity-campaigns' ),
				'date' => strtotime( $campaign->post_date ),
			);
		}

		// Get campaign milestones (100% funded campaigns with completion dates)
		$milestone_campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => 5,
				'post_status'    => 'publish',
				'meta_query'     => array(
					array(
						'key'     => 'campaign_completion_date',
						'compare' => 'EXISTS',
					),
				),
				'orderby'        => 'meta_value',
				'meta_key'       => 'campaign_completion_date',
				'order'          => 'DESC',
			)
		);

		foreach ( $milestone_campaigns as $campaign ) {
			$completion_date = get_post_meta( $campaign->ID, 'campaign_completion_date', true );
			$excess_amount   = get_post_meta( $campaign->ID, 'campaign_excess_amount', true );
			$excess_date     = get_post_meta( $campaign->ID, 'campaign_excess_date', true );

			// Only show if campaign has a completion date
			if ( $completion_date ) {
				$activities[] = array(
					'type' => 'milestone',
					'icon' => '🎉',
					// translators: %s is the campaign title
					'text' => sprintf( __( 'Campaign "%s" reached 100%% funding goal!', 'charity-campaigns' ), $campaign->post_title ),
					'time' => human_time_diff( strtotime( $completion_date ), current_time( 'timestamp' ) ) . ' ' . __( 'ago', 'charity-campaigns' ),
					'date' => strtotime( $completion_date ),
				);
			}

			// If there are excess funds, log an excess activity using the first exceeded date
			if ( $excess_amount && $excess_date ) {
				$activities[] = array(
					'type' => 'excess',
					'icon' => '➕',
					// translators: %1$s is the campaign title, %2$s is the formatted currency amount
					'text' => sprintf( __( 'Campaign "%1$s" exceeded goal by %2$s', 'charity-campaigns' ), $campaign->post_title, self::format_currency( $excess_amount ) ),
					'time' => human_time_diff( strtotime( $excess_date ), current_time( 'timestamp' ) ) . ' ' . __( 'ago', 'charity-campaigns' ),
					'date' => strtotime( $excess_date ),
				);
			}
		}

		// Sort by date (newest first)
		usort(
			$activities,
			function ( $a, $b ) {
				return $b['date'] - $a['date'];
			}
		);

		return array_slice( $activities, 0, 8 ); // Limit to 8 most recent
	}

	/**
	 * Format currency amount.
	 *
	 * @since 1.0.0
	 * @param float  $amount The amount to format.
	 * @param string $currency The currency code.
	 * @return string Formatted currency string.
	 */
	public static function format_currency( $amount, $currency = null ) {
		// Use centralized settings if no currency specified
		if ( $currency === null ) {
			$currency = Charity_Campaigns_Settings::get_default_currency();
		}

		$symbols = array(
			'USD' => '$',
			'EUR' => '€',
			'GBP' => '£',
			'CAD' => 'C$',
			'AUD' => 'A$',
			'JPY' => '¥',
			'CHF' => 'CHF',
			'CNY' => '¥',
			'INR' => '₹',
			'BRL' => 'R$',
		);

		$symbol          = isset( $symbols[ $currency ] ) ? $symbols[ $currency ] : $currency . ' ';
		$decimal_places  = Charity_Campaigns_Settings::get_decimal_places();
		$symbol_position = Charity_Campaigns_Settings::get_currency_symbol_position();

		$formatted_amount = number_format( $amount, $decimal_places );

		if ( $symbol_position === 'after' ) {
			return $formatted_amount . $symbol;
		} else {
			return $symbol . $formatted_amount;
		}
	}

	/**
	 * Get campaign progress percentage.
	 *
	 * @since 1.0.0
	 * @param int $campaign_id Campaign post ID.
	 * @return float Progress percentage (0-100).
	 */
	public static function get_campaign_progress( $campaign_id ) {
		$goal   = get_post_meta( $campaign_id, 'campaign_goal', true );
		$raised = get_post_meta( $campaign_id, 'campaign_raised', true );

		if ( ! $goal || $goal <= 0 ) {
			return 0;
		}

		return min( ( $raised / $goal ) * 100, 100 );
	}

	/**
	 * Get campaign statistics.
	 *
	 * @since 1.0.0
	 * @param int $campaign_id Campaign post ID.
	 * @return array Campaign statistics.
	 */
	public static function get_campaign_stats( $campaign_id ) {
		$goal   = floatval( get_post_meta( $campaign_id, 'campaign_goal', true ) );
		$raised = floatval( get_post_meta( $campaign_id, 'campaign_raised', true ) );

		// Count donations for this campaign
		$donations_count = get_posts(
			array(
				'post_type'      => 'ccfs_donation',
				'posts_per_page' => 10,
				'post_status'    => 'publish',
				'meta_query'     => array(
					array(
						'key'     => 'donation_campaign_id',
						'value'   => $campaign_id,
						'compare' => '=',
					),
					array(
						'key'     => 'donation_status',
						'value'   => 'completed',
						'compare' => '=',
					),
				),
				'fields'         => 'ids',
			)
		);

		return array(
			'goal'            => $goal,
			'raised'          => $raised,
			'donations_count' => count( $donations_count ),
			'progress'        => self::get_campaign_progress( $campaign_id ),
			'remaining'       => max( $goal - $raised, 0 ),
		);
	}

	/**
	 * Get total amount raised across all campaigns.
	 *
	 * @since 1.0.0
	 * @param string $filter Date filter type.
	 * @return float Total raised amount.
	 */
	public static function get_total_raised( $filter = 'all' ) {
		$date_range = self::get_date_range( $filter );

		$args = array(
			'post_type'      => 'ccfs_donation',
			'posts_per_page' => -1,
			'post_status'    => 'publish',
			'meta_query'     => array(
				array(
					'key'     => 'donation_status',
					'value'   => 'completed',
					'compare' => '=',
				),
			),
		);

		// Add date filtering if not 'all'
		if ( $date_range ) {
			$args['meta_query'][] = array(
				'key'     => 'donation_date',
				'value'   => array( $date_range['start'], $date_range['end'] ),
				'compare' => 'BETWEEN',
				'type'    => 'DATETIME',
			);
		}

		$donations = get_posts( $args );

		$total = 0;
		foreach ( $donations as $donation ) {
			$amount = get_post_meta( $donation->ID, 'donation_amount', true );
			if ( $amount ) {
				$total += floatval( $amount );
			}
		}

		return $total;
	}

	/**
	 * Get total number of completed donations.
	 *
	 * @since 1.0.0
	 * @param string $filter Date filter type.
	 * @return int Total donations count.
	 */
	public static function get_total_donations_count( $filter = 'all' ) {
		$date_range = self::get_date_range( $filter );

		$args = array(
			'post_type'      => 'ccfs_donation',
			'posts_per_page' => -1,
			'post_status'    => 'publish',
			'meta_query'     => array(
				array(
					'key'     => 'donation_status',
					'value'   => 'completed',
					'compare' => '=',
				),
			),
		);

		// Add date filtering if not 'all'
		if ( $date_range ) {
			$args['meta_query'][] = array(
				'key'     => 'donation_date',
				'value'   => array( $date_range['start'], $date_range['end'] ),
				'compare' => 'BETWEEN',
				'type'    => 'DATETIME',
			);
		}

		$donations = get_posts( $args );

		return count( $donations );
	}

	/**
	 * Get number of active campaigns.
	 *
	 * @since 1.0.0
	 * @return int Active campaigns count.
	 */
	public static function get_active_campaigns_count() {
		$campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
				'meta_query'     => array(
					array(
						'key'     => 'campaign_status',
						'value'   => 'active',
						'compare' => '=',
					),
				),
			)
		);

		return count( $campaigns );
	}

	/**
	 * Get average donation amount.
	 *
	 * @since 1.0.0
	 * @param string $filter Date filter type.
	 * @return float Average donation amount.
	 */
	public static function get_average_donation( $filter = 'all' ) {
		$total_raised = self::get_total_raised( $filter );
		$total_count  = self::get_total_donations_count( $filter );

		if ( $total_count > 0 ) {
			return $total_raised / $total_count;
		}

		return 0;
	}

	/**
	 * Get total goal amount across all campaigns.
	 *
	 * @since 1.0.0
	 * @return float Total goal amount.
	 */
	public static function get_total_goals() {
		$campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
			)
		);

		$total = 0;
		foreach ( $campaigns as $campaign ) {
			$goal = get_post_meta( $campaign->ID, 'campaign_goal', true );
			if ( $goal ) {
				$total += floatval( $goal );
			}
		}

		return $total;
	}

	/**
	 * Get campaign status breakdown.
	 *
	 * @since 1.0.0
	 * @return array Campaign status counts.
	 */
	public static function get_campaign_status_breakdown() {
		$statuses  = array( 'active', 'completed', 'paused', 'cancelled' );
		$breakdown = array();

		foreach ( $statuses as $status ) {
			$campaigns = get_posts(
				array(
					'post_type'      => 'ccfs_campaign',
					'posts_per_page' => -1,
					'post_status'    => 'publish',
					'meta_query'     => array(
						array(
							'key'     => 'campaign_status',
							'value'   => $status,
							'compare' => '=',
						),
					),
				)
			);

			$breakdown[ $status ] = count( $campaigns );
		}

		return $breakdown;
	}

	/**
	 * Get monthly donations amount.
	 *
	 * @since 1.0.0
	 * @return float Monthly donations amount.
	 */
	public static function get_monthly_donations() {
		$current_month = gmdate( 'Y-m' );
		$start_date    = $current_month . '-01 00:00:00';
		$end_date      = gmdate( 'Y-m-t 23:59:59', strtotime( $start_date ) );

		$donations = get_posts(
			array(
				'post_type'      => 'ccfs_donation',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
				'meta_query'     => array(
					array(
						'key'     => 'donation_status',
						'value'   => 'completed',
						'compare' => '=',
					),
					array(
						'key'     => 'donation_date',
						'value'   => array( $start_date, $end_date ),
						'compare' => 'BETWEEN',
						'type'    => 'DATETIME',
					),
				),
			)
		);

		$total = 0;
		foreach ( $donations as $donation ) {
			$amount = get_post_meta( $donation->ID, 'donation_amount', true );
			if ( $amount ) {
				$total += floatval( $amount );
			}
		}

		return $total;
	}

	/**
	 * Get completed campaigns count.
	 *
	 * @since 1.0.0
	 * @return int Completed campaigns count.
	 */
	public static function get_completed_campaigns_count() {
		$campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
				'meta_query'     => array(
					array(
						'key'     => 'campaign_status',
						'value'   => 'completed',
						'compare' => '=',
					),
				),
			)
		);

		return count( $campaigns );
	}

	/**
	 * Get goal completion percentage.
	 *
	 * @since 1.0.0
	 * @param string $filter Date filter type.
	 * @return float Goal completion percentage.
	 */
	public static function get_goal_completion_percentage( $filter = 'all' ) {
		$total_raised = self::get_total_raised( $filter );
		$total_goals  = self::get_total_goals();

		if ( $total_goals > 0 ) {
			return ( $total_raised / $total_goals ) * 100;
		}

		return 0;
	}

	/**
	 * Get total excess funds across all campaigns.
	 *
	 * @since 1.0.0
	 * @return float Total excess funds amount.
	 */
	public static function get_total_excess_funds() {
		$campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
			)
		);

		$total_excess = 0;
		foreach ( $campaigns as $campaign ) {
			$excess_amount = get_post_meta( $campaign->ID, 'campaign_excess_amount', true );
			if ( $excess_amount && $excess_amount > 0 ) {
				$total_excess += floatval( $excess_amount );
			}
		}

		return $total_excess;
	}

	/**
	 * Get date range for filtering.
	 *
	 * @since 1.0.0
	 * @param string $filter Date filter type.
	 * @return array Date range array with 'start' and 'end' keys.
	 */
	public static function get_date_range( $filter ) {
		$now   = new DateTime();
		$start = clone $now;
		$end   = clone $now;

		switch ( $filter ) {
			case 'daily':
				$start->setTime( 0, 0, 0 );
				$end->setTime( 23, 59, 59 );
				break;
			case 'weekly':
				$start->modify( 'monday this week' )->setTime( 0, 0, 0 );
				$end->modify( 'sunday this week' )->setTime( 23, 59, 59 );
				break;
			case 'monthly':
				$start->modify( 'first day of this month' )->setTime( 0, 0, 0 );
				$end->modify( 'last day of this month' )->setTime( 23, 59, 59 );
				break;
			case 'yearly':
				$start->modify( 'first day of january this year' )->setTime( 0, 0, 0 );
				$end->modify( 'last day of december this year' )->setTime( 23, 59, 59 );
				break;
			default: // 'all'
				return null; // No date filtering
		}

		return array(
			'start' => $start->format( 'Y-m-d H:i:s' ),
			'end'   => $end->format( 'Y-m-d H:i:s' ),
		);
	}

	/**
	 * Get period subtitle based on filter.
	 *
	 * @since 1.0.0
	 * @param string $filter Date filter type.
	 * @return string Period subtitle.
	 */
	public static function get_period_subtitle( $filter ) {
		switch ( $filter ) {
			case 'daily':
				return __( 'Today', 'charity-campaigns' );
			case 'weekly':
				return __( 'This week', 'charity-campaigns' );
			case 'monthly':
				// translators: %s is the month and year (e.g., "January 2024")
				return sprintf( __( 'This month (%s)', 'charity-campaigns' ), gmdate( 'F Y' ) );
			case 'yearly':
				// translators: %s is the year (e.g., "2024")
				return sprintf( __( 'This year (%s)', 'charity-campaigns' ), gmdate( 'Y' ) );
			default:
				return __( 'All time', 'charity-campaigns' );
		}
	}

	/**
	 * Get period donations amount.
	 *
	 * @since 1.0.0
	 * @param string $filter Date filter type.
	 * @return float Period donations amount.
	 */
	public static function get_period_donations( $filter ) {
		$date_range = self::get_date_range( $filter );

		$args = array(
			'post_type'      => 'ccfs_donation',
			'posts_per_page' => -1,
			'post_status'    => 'publish',
			'meta_query'     => array(
				array(
					'key'     => 'donation_status',
					'value'   => 'completed',
					'compare' => '=',
				),
			),
		);

		// Add date filtering if not 'all'
		if ( $date_range ) {
			$args['meta_query'][] = array(
				'key'     => 'donation_date',
				'value'   => array( $date_range['start'], $date_range['end'] ),
				'compare' => 'BETWEEN',
				'type'    => 'DATETIME',
			);
		}

		$donations = get_posts( $args );

		$total = 0;
		foreach ( $donations as $donation ) {
			$amount = get_post_meta( $donation->ID, 'donation_amount', true );
			if ( $amount ) {
				$total += floatval( $amount );
			}
		}

		return $total;
	}

	/**
	 * Get recent donations with filtering options.
	 *
	 * @since 1.0.0
	 * @param int    $limit Number of donations to retrieve.
	 * @param string $status Filter by donation status.
	 * @param string $campaign_id Filter by campaign ID.
	 * @param string $date_from Filter from date.
	 * @param string $date_to Filter to date.
	 * @return array Recent donations data.
	 */
	public static function get_recent_donations( $limit = 20, $status = '', $campaign_id = '', $date_from = '', $date_to = '' ) {
		$args = array(
			'post_type'      => 'ccfs_donation',
			'posts_per_page' => $limit,
			'post_status'    => 'publish',
			'orderby'        => 'date',
			'order'          => 'DESC',
			'meta_query'     => array(),
		);

		// Filter by status
		if ( ! empty( $status ) ) {
			$args['meta_query'][] = array(
				'key'     => 'donation_status',
				'value'   => $status,
				'compare' => '=',
			);
		}

		// Filter by campaign
		if ( ! empty( $campaign_id ) ) {
			$args['meta_query'][] = array(
				'key'     => 'donation_campaign_id',
				'value'   => $campaign_id,
				'compare' => '=',
			);
		}

		// Filter by date range
		if ( ! empty( $date_from ) && ! empty( $date_to ) ) {
			$args['meta_query'][] = array(
				'key'     => 'donation_date',
				'value'   => array( $date_from . ' 00:00:00', $date_to . ' 23:59:59' ),
				'compare' => 'BETWEEN',
				'type'    => 'DATETIME',
			);
		}

		$donations      = get_posts( $args );
		$donations_data = array();

		foreach ( $donations as $donation ) {
			$amount       = get_post_meta( $donation->ID, 'donation_amount', true );
			$status       = get_post_meta( $donation->ID, 'donation_status', true );
			$date         = get_post_meta( $donation->ID, 'donation_date', true );
			$campaign_id  = get_post_meta( $donation->ID, 'donation_campaign_id', true );
			$donor_name   = get_post_meta( $donation->ID, 'donation_name', true );
			$donor_email  = get_post_meta( $donation->ID, 'donation_email', true );
			$is_anonymous = get_post_meta( $donation->ID, 'donation_anon', true );

			$campaign_title = '';
			if ( $campaign_id ) {
				$campaign = get_post( $campaign_id );
				if ( $campaign ) {
					$campaign_title = $campaign->post_title;
				}
			}

			$donations_data[] = array(
				'id'             => $donation->ID,
				'title'          => $donation->post_title,
				'amount'         => floatval( $amount ),
				'status'         => $status,
				'date'           => $date,
				'campaign_id'    => $campaign_id,
				'campaign_title' => $campaign_title,
				'donor_name'     => $donor_name,
				'donor_email'    => $donor_email,
				'is_anonymous'   => $is_anonymous,
				'edit_url'       => get_edit_post_link( $donation->ID ),
				'view_url'       => get_permalink( $donation->ID ),
			);
		}

		return $donations_data;
	}

	/**
	 * Get campaigns for filter dropdown.
	 *
	 * @since 1.0.0
	 * @return array Campaigns data.
	 */
	public static function get_campaigns_for_filter() {
		$campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
				'orderby'        => 'title',
				'order'          => 'ASC',
			)
		);

		$campaigns_data = array();
		foreach ( $campaigns as $campaign ) {
			$campaigns_data[] = array(
				'id'    => $campaign->ID,
				'title' => $campaign->post_title,
			);
		}

		return $campaigns_data;
	}

	/**
	 * Get donation status options.
	 *
	 * @since 1.0.0
	 * @return array Status options.
	 */
	public static function get_donation_status_options() {
		return array(
			'completed' => __( 'Completed', 'charity-campaigns' ),
			'pending'   => __( 'Pending', 'charity-campaigns' ),
			'failed'    => __( 'Failed', 'charity-campaigns' ),
			'refunded'  => __( 'Refunded', 'charity-campaigns' ),
		);
	}

	/**
	 * Get monthly donation trends data.
	 *
	 * @since 1.0.0
	 * @param int $months Number of months to retrieve.
	 * @return array Monthly trends data.
	 */
	public static function get_monthly_donation_trends( $months = 12 ) {
		$trends       = array();
		$current_date = new DateTime();

		for ( $i = $months - 1; $i >= 0; $i-- ) {
			$date = clone $current_date;
			$date->modify( "-{$i} months" );

			$month_start = $date->format( 'Y-m-01 00:00:00' );
			$month_end   = $date->format( 'Y-m-t 23:59:59' );
			$month_label = $date->format( 'M Y' );

			$donations = get_posts(
				array(
					'post_type'      => 'ccfs_donation',
					'posts_per_page' => -1,
					'post_status'    => 'publish',
					'meta_query'     => array(
						array(
							'key'     => 'donation_status',
							'value'   => 'completed',
							'compare' => '=',
						),
						array(
							'key'     => 'donation_date',
							'value'   => array( $month_start, $month_end ),
							'compare' => 'BETWEEN',
							'type'    => 'DATETIME',
						),
					),
				)
			);

			$amount = 0;
			$count  = 0;
			foreach ( $donations as $donation ) {
				$donation_amount = get_post_meta( $donation->ID, 'donation_amount', true );
				if ( $donation_amount ) {
					$amount += floatval( $donation_amount );
					++$count;
				}
			}

			$trends[] = array(
				'month'  => $month_label,
				'amount' => $amount,
				'count'  => $count,
			);
		}

		return $trends;
	}

	/**
	 * Get campaign performance data.
	 *
	 * @since 1.0.0
	 * @return array Campaign performance data.
	 */
	public static function get_campaign_performance_data() {
		$campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
				'orderby'        => 'meta_value_num',
				'meta_key'       => 'campaign_raised',
				'order'          => 'DESC',
			)
		);

		$performance_data = array();
		foreach ( $campaigns as $campaign ) {
			$goal   = get_post_meta( $campaign->ID, 'campaign_goal', true );
			$raised = get_post_meta( $campaign->ID, 'campaign_raised', true );

			$goal     = $goal ? floatval( $goal ) : 0;
			$raised   = $raised ? floatval( $raised ) : 0;
			$progress = $goal > 0 ? ( $raised / $goal ) * 100 : 0;

			$performance_data[] = array(
				'title'    => $campaign->post_title,
				'goal'     => $goal,
				'raised'   => $raised,
				'progress' => $progress,
			);
		}

		return $performance_data;
	}

	/**
	 * Get donation status distribution.
	 *
	 * @since 1.0.0
	 * @return array Status distribution data.
	 */
	public static function get_donation_status_distribution() {
		$status_options = self::get_donation_status_options();
		$distribution   = array();

		foreach ( $status_options as $status => $label ) {
			$donations = get_posts(
				array(
					'post_type'      => 'ccfs_donation',
					'posts_per_page' => -1,
					'post_status'    => 'publish',
					'meta_query'     => array(
						array(
							'key'     => 'donation_status',
							'value'   => $status,
							'compare' => '=',
						),
					),
				)
			);

			$amount = 0;
			foreach ( $donations as $donation ) {
				$donation_amount = get_post_meta( $donation->ID, 'donation_amount', true );
				if ( $donation_amount ) {
					$amount += floatval( $donation_amount );
				}
			}

			$distribution[] = array(
				'status' => $status,
				'label'  => $label,
				'amount' => $amount,
				'count'  => count( $donations ),
			);
		}

		return $distribution;
	}

	/**
	 * Get current date filter from URL parameter.
	 *
	 * @since 1.0.0
	 * @return string Current date filter.
	 */
	public static function get_current_date_filter() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering
		return isset( $_GET['date_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['date_filter'] ) ) : 'all';
	}
}
