<?php
/**
 * Donations Service - calculations and hooks
 *
 * @package CCFS
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles donation-related business logic (calculations, syncing, hooks).
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Donations_Service {

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Recalculate when donations are saved or deleted
		add_action( 'save_post', array( $this, 'maybe_recalculate_on_save' ), 20, 1 );
		add_action( 'delete_post', array( $this, 'maybe_recalculate_on_delete' ), 20, 1 );
	}

	/**
	 * Calculate total raised amount for a campaign from completed donations.
	 *
	 * @since 1.0.0
	 * @param int $campaign_id Campaign post ID.
	 * @return float
	 */
	public static function calculate_campaign_raised_amount( $campaign_id ) {
		$donations = get_posts(
			array(
				'post_type'      => 'ccfs_donation',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
				'meta_query'     => array(
					array(
						'key'     => 'donation_campaign_id',
						'value'   => $campaign_id,
						'compare' => '=',
					),
					array(
						'key'     => 'donation_status',
						'value'   => 'completed',
						'compare' => '=',
					),
				),
			)
		);

		$total_raised = 0;
		foreach ( $donations as $donation ) {
			$amount = get_post_meta( $donation->ID, 'donation_amount', true );
			if ( $amount ) {
				$total_raised += floatval( $amount );
			}
		}

		return $total_raised;
	}

	/**
	 * Update the campaign's stored raised amount meta from calculations.
	 *
	 * @since 1.0.0
	 * @param int $campaign_id
	 * @return void
	 */
	public function sync_campaign_raised_amount( $campaign_id ) {
		$new_raised_amount = self::calculate_campaign_raised_amount( $campaign_id );
		$goal              = get_post_meta( $campaign_id, 'campaign_goal', true );
		$completion_date   = get_post_meta( $campaign_id, 'campaign_completion_date', true );
		$excess_date       = get_post_meta( $campaign_id, 'campaign_excess_date', true );

		// Update raised amount
		update_post_meta( $campaign_id, 'campaign_raised', $new_raised_amount );

		// Check if campaign reached 100% and set completion date
		if ( $goal > 0 && $new_raised_amount >= $goal && ! $completion_date ) {
			$current_time = current_time( 'Y-m-d\TH:i' );
			update_post_meta( $campaign_id, 'campaign_completion_date', $current_time );
		}

		// Calculate and store excess funds (amount over goal)
		if ( $goal > 0 && $new_raised_amount > $goal ) {
			$excess_amount = max( floatval( $new_raised_amount ) - floatval( $goal ), 0 );
			update_post_meta( $campaign_id, 'campaign_excess_amount', $excess_amount );
			// Set first time exceeded date if not set
			if ( ! $excess_date ) {
				$current_time = current_time( 'Y-m-d\TH:i' );
				update_post_meta( $campaign_id, 'campaign_excess_date', $current_time );
			}
		} else {
			// Ensure excess amount is zero if not exceeding
			update_post_meta( $campaign_id, 'campaign_excess_amount', 0 );
		}
	}

	/**
	 * Hook: on save_post
	 *
	 * @since 1.0.0
	 * @param int $post_id
	 */
	public function maybe_recalculate_on_save( $post_id ) {
		if ( get_post_type( $post_id ) !== 'ccfs_donation' ) {
			return;
		}
		$campaign_id = get_post_meta( $post_id, 'donation_campaign_id', true );
		if ( $campaign_id ) {
			$this->sync_campaign_raised_amount( $campaign_id );
		}
	}

	/**
	 * Hook: on delete_post
	 *
	 * @since 1.0.0
	 * @param int $post_id
	 */
	public function maybe_recalculate_on_delete( $post_id ) {
		if ( get_post_type( $post_id ) !== 'ccfs_donation' ) {
			return;
		}
		$campaign_id = get_post_meta( $post_id, 'donation_campaign_id', true );
		if ( $campaign_id ) {
			$this->sync_campaign_raised_amount( $campaign_id );
		}
	}
}
