<?php
/**
 * Admin Menu Management
 *
 * @package Charity_Campaigns
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class responsible for managing admin menus and dashboard.
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Admin_Menu {

	/**
	 * Initialize the class and register hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menus' ) );
		add_action( 'admin_menu', array( $this, 'reorder_submenus' ), 999 );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
	}

	/**
	 * Add admin menus.
	 *
	 * @since 1.0.0
	 */
	public function add_admin_menus() {
		// Main dashboard menu
		add_menu_page(
			__( 'Charity Campaigns', 'charity-campaigns' ),
			__( 'Charity Campaigns', 'charity-campaigns' ),
			'manage_options',
			'ccfs-dashboard',
			array( $this, 'dashboard_page' ),
			'dashicons-heart',
			25
		);

		// Dashboard submenu (first item)
		add_submenu_page(
			'ccfs-dashboard',
			__( 'Dashboard', 'charity-campaigns' ),
			__( 'Dashboard', 'charity-campaigns' ),
			'manage_options',
			'ccfs-dashboard',
			array( $this, 'dashboard_page' )
		);
	}

	/**
	 * Reorder submenus to put Dashboard first.
	 *
	 * @since 1.0.0
	 */
	public function reorder_submenus() {
		global $submenu;

		if ( isset( $submenu['ccfs-dashboard'] ) ) {
			$menu_items = $submenu['ccfs-dashboard'];

			// Find Dashboard item
			$dashboard_item = null;
			$other_items    = array();

			foreach ( $menu_items as $key => $item ) {
				if ( $item[2] === 'ccfs-dashboard' ) {
					$dashboard_item = $item;
				} else {
					$other_items[] = $item;
				}
			}

			// Rebuild menu with Dashboard first
			if ( $dashboard_item ) {
				$submenu['ccfs-dashboard'] = array_merge( array( $dashboard_item ), $other_items );
			}
		}
	}

	/**
	 * Dashboard page callback.
	 *
	 * @since 1.0.0
	 */
	public function dashboard_page() {
		// Get current tab from URL parameter (no nonce needed for navigation)
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for navigation
		$current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'overview';
		$tabs        = array(
			'overview' => __( 'Overview', 'charity-campaigns' ),
			'activity' => __( 'Activity', 'charity-campaigns' ),
			'reports'  => __( 'Reports', 'charity-campaigns' ),
			'help'     => __( 'Help', 'charity-campaigns' ),
		);
		?>
		<div class="wrap">
			<h1><?php echo esc_html( Charity_Campaigns_Settings::get_organization_name() ); ?> - <?php esc_html_e( 'Dashboard', 'charity-campaigns' ); ?></h1>
			
			<nav class="nav-tab-wrapper">
				<?php foreach ( $tabs as $tab_key => $tab_name ) : ?>
					<a href="?page=ccfs-dashboard&tab=<?php echo esc_attr( $tab_key ); ?>" 
						class="nav-tab <?php echo $current_tab === $tab_key ? 'nav-tab-active' : ''; ?>">
						<?php echo esc_html( $tab_name ); ?>
					</a>
				<?php endforeach; ?>
			</nav>

			<div class="tab-content">
				<?php $this->render_tab_content( $current_tab ); ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render tab content.
	 *
	 * @since 1.0.0
	 * @param string $tab Current tab.
	 */
	private function render_tab_content( $tab ) {
		switch ( $tab ) {
			case 'overview':
				$this->overview_tab();
				break;
			case 'activity':
				$this->activity_tab();
				break;
			case 'reports':
				$this->reports_tab();
				break;
			case 'help':
				$this->help_tab();
				break;
		}
	}

	/**
	 * Overview tab content.
	 *
	 * @since 1.0.0
	 */
	private function overview_tab() {
		// Get comprehensive dashboard data
		$active_campaigns_count = Charity_Campaigns_Functions::get_active_campaigns_count();
		$total_raised           = Charity_Campaigns_Functions::get_total_raised();
		$total_donations_count  = Charity_Campaigns_Functions::get_total_donations_count();
		$average_donation       = Charity_Campaigns_Functions::get_average_donation();
		$monthly_donations      = Charity_Campaigns_Functions::get_monthly_donations();
		$recent_activities      = Charity_Campaigns_Functions::get_recent_activities();
		$calendar_events        = $this->get_calendar_events();

		// Get additional meaningful data
		$total_goals                = Charity_Campaigns_Functions::get_total_goals();
		$goal_completion_percentage = Charity_Campaigns_Functions::get_goal_completion_percentage();
		$completed_campaigns_count  = Charity_Campaigns_Functions::get_completed_campaigns_count();
		$total_excess_funds         = Charity_Campaigns_Functions::get_total_excess_funds();
		$campaign_status_breakdown  = Charity_Campaigns_Functions::get_campaign_status_breakdown();

		// Get top performing campaign this month
		$monthly_trends = Charity_Campaigns_Functions::get_monthly_donation_trends( 1 );
		$campaigns_data = Charity_Campaigns_Functions::get_campaign_performance_data();
		$top_campaign   = ! empty( $campaigns_data ) ? $campaigns_data[0] : null;

		// Get recent donations for insights
		$recent_donations = Charity_Campaigns_Functions::get_recent_donations( 5 );
		$today_donations  = Charity_Campaigns_Functions::get_total_raised( 'daily' );
		$weekly_donations = Charity_Campaigns_Functions::get_total_raised( 'weekly' );
		?>
		<div class="charity-dashboard-overview">
			<!-- Calendar Section -->
			<div class="dashboard-calendar-section">
				<div class="calendar-header">
					<h2><?php esc_html_e( 'Campaign Calendar', 'charity-campaigns' ); ?></h2>
					<div class="calendar-navigation">
						<button type="button" id="prev-month" class="button">&larr;</button>
						<span id="current-month-year"></span>
						<button type="button" id="next-month" class="button">&rarr;</button>
					</div>
				</div>
				<div class="calendar-container">
					<div id="charity-calendar"></div>
				</div>
			</div>

			<!-- Stats and Activity Section -->
			<div class="dashboard-stats-section">
				<div class="dashboard-widgets">
					<div class="dashboard-widget campaigns-widget">
						<div class="widget-header">
							<h3><?php esc_html_e( 'Campaigns', 'charity-campaigns' ); ?></h3>
							<a href="<?php echo esc_url( admin_url( 'edit.php?post_type=ccfs_campaign' ) ); ?>" class="widget-link"><?php esc_html_e( 'View All', 'charity-campaigns' ); ?></a>
						</div>
						<div class="widget-content">
							<div class="widget-stat">
								<span class="stat-number"><?php echo number_format( $active_campaigns_count ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Active Campaigns', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo number_format( $completed_campaigns_count ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Completed', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo number_format( $goal_completion_percentage, 1 ); ?>%</span>
								<span class="stat-label"><?php esc_html_e( 'Goal Progress', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $total_excess_funds ) ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Excess Funds', 'charity-campaigns' ); ?></span>
							</div>
						</div>
					</div>
					
					<div class="dashboard-widget donations-widget">
						<div class="widget-header">
							<h3><?php esc_html_e( 'Donations', 'charity-campaigns' ); ?></h3>
							<a href="<?php echo esc_url( admin_url( 'edit.php?post_type=ccfs_donation' ) ); ?>" class="widget-link"><?php esc_html_e( 'View All', 'charity-campaigns' ); ?></a>
						</div>
						<div class="widget-content">
							<div class="widget-stat">
								<span class="stat-number"><?php echo number_format( $total_donations_count ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Total Donations', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $total_raised ) ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Total Raised', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $average_donation ) ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Average Donation', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $today_donations ) ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Today\'s Donations', 'charity-campaigns' ); ?></span>
							</div>
						</div>
					</div>
					
					<div class="dashboard-widget monthly-widget">
						<div class="widget-header">
							<h3><?php esc_html_e( 'Performance', 'charity-campaigns' ); ?></h3>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=ccfs-dashboard&tab=reports' ) ); ?>" class="widget-link"><?php esc_html_e( 'View Reports', 'charity-campaigns' ); ?></a>
						</div>
						<div class="widget-content">
							<div class="widget-stat">
								<span class="stat-number"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $monthly_donations ) ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'This Month', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $weekly_donations ) ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'This Week', 'charity-campaigns' ); ?></span>
							</div>
							<?php if ( $top_campaign ) : ?>
								<div class="widget-stat">
									<span class="stat-number"><?php echo number_format( $top_campaign['progress'], 1 ); ?>%</span>
									<span class="stat-label"><?php esc_html_e( 'Top Campaign Progress', 'charity-campaigns' ); ?></span>
								</div>
							<?php endif; ?>
							<div class="widget-stat">
								<span class="stat-number"><?php echo esc_html( Charity_Campaigns_Functions::format_currency( $total_goals ) ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Total Goals', 'charity-campaigns' ); ?></span>
							</div>
						</div>
					</div>
					
					<div class="dashboard-widget insights-widget">
						<div class="widget-header">
							<h3><?php esc_html_e( 'Insights', 'charity-campaigns' ); ?></h3>
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=ccfs-dashboard&tab=reports' ) ); ?>" class="widget-link"><?php esc_html_e( 'View Details', 'charity-campaigns' ); ?></a>
						</div>
						<div class="widget-content">
							<div class="widget-stat">
								<span class="stat-number"><?php echo number_format( $campaign_status_breakdown['active'] ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Active', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo number_format( $campaign_status_breakdown['paused'] ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Paused', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo number_format( $campaign_status_breakdown['cancelled'] ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Cancelled', 'charity-campaigns' ); ?></span>
							</div>
							<div class="widget-stat">
								<span class="stat-number"><?php echo count( $recent_donations ); ?></span>
								<span class="stat-label"><?php esc_html_e( 'Recent Donations', 'charity-campaigns' ); ?></span>
							</div>
						</div>
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Activity tab content.
	 *
	 * @since 1.0.0
	 */
	private function activity_tab() {
		// Get filter parameters from URL (no nonce needed for filtering)
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for filtering
		$activity_type = isset( $_GET['activity_type'] ) ? sanitize_text_field( wp_unslash( $_GET['activity_type'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for filtering
		$campaign_id = isset( $_GET['campaign_id'] ) ? sanitize_text_field( wp_unslash( $_GET['campaign_id'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for filtering
		$date_from = isset( $_GET['date_from'] ) ? sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for filtering
		$date_to = isset( $_GET['date_to'] ) ? sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for filtering
		$search_term = isset( $_GET['search'] ) ? sanitize_text_field( wp_unslash( $_GET['search'] ) ) : '';

		// Get filtered activities
		$activities = $this->get_filtered_activities( $activity_type, $campaign_id, $date_from, $date_to, $search_term );

		// Get filter options
		$campaigns      = Charity_Campaigns_Functions::get_campaigns_for_filter();
		$activity_types = array(
			'campaign'  => __( 'Campaign Activities', 'charity-campaigns' ),
			'donation'  => __( 'Donation Activities', 'charity-campaigns' ),
			'milestone' => __( 'Milestone Activities', 'charity-campaigns' ),
			'excess'    => __( 'Excess Fund Activities', 'charity-campaigns' ),
		);
		?>
		<div class="charity-activity-page">
			<!-- Activity Filters -->
			<div class="activity-filters">
				<form method="get" action="">
					<input type="hidden" name="page" value="ccfs-dashboard">
					<input type="hidden" name="tab" value="activity">
					
					<div class="filter-row fields-row">
						<div class="filter-group">
							<label for="search"><?php esc_html_e( 'Search:', 'charity-campaigns' ); ?></label>
							<input type="text" name="search" id="search" value="<?php echo esc_attr( $search_term ); ?>" placeholder="<?php esc_attr_e( 'Search activities...', 'charity-campaigns' ); ?>">
						</div>
						
						<div class="filter-group">
							<label for="activity_type"><?php esc_html_e( 'Activity Type:', 'charity-campaigns' ); ?></label>
							<select name="activity_type" id="activity_type">
								<option value=""><?php esc_html_e( 'All Types', 'charity-campaigns' ); ?></option>
								<?php foreach ( $activity_types as $type => $label ) : ?>
									<option value="<?php echo esc_attr( $type ); ?>" <?php selected( $activity_type, $type ); ?>>
										<?php echo esc_html( $label ); ?>
									</option>
								<?php endforeach; ?>
							</select>
						</div>
						
						<div class="filter-group">
							<label for="campaign_id"><?php esc_html_e( 'Campaign:', 'charity-campaigns' ); ?></label>
							<select name="campaign_id" id="campaign_id">
								<option value=""><?php esc_html_e( 'All Campaigns', 'charity-campaigns' ); ?></option>
								<?php foreach ( $campaigns as $campaign ) : ?>
									<option value="<?php echo esc_attr( $campaign['id'] ); ?>" <?php selected( $campaign_id, $campaign['id'] ); ?>>
										<?php echo esc_html( $campaign['title'] ); ?>
									</option>
								<?php endforeach; ?>
							</select>
						</div>
						
						<div class="filter-group">
							<label for="date_from"><?php esc_html_e( 'From Date:', 'charity-campaigns' ); ?></label>
							<input type="date" name="date_from" id="date_from" value="<?php echo esc_attr( $date_from ); ?>">
						</div>
						
						<div class="filter-group">
							<label for="date_to"><?php esc_html_e( 'To Date:', 'charity-campaigns' ); ?></label>
							<input type="date" name="date_to" id="date_to" value="<?php echo esc_attr( $date_to ); ?>">
						</div>
					</div>
					
					<div class="filter-row buttons-row">
						<div class="filter-actions">
							<button type="submit" class="button button-primary"><?php esc_html_e( 'Filter', 'charity-campaigns' ); ?></button>
							<a href="?page=ccfs-dashboard&tab=activity" class="button"><?php esc_html_e( 'Clear', 'charity-campaigns' ); ?></a>
						</div>
					</div>
				</form>
			</div>
			
			<!-- Activity Results -->
			<div class="activity-results">
				<div class="activity-header">
					<h2><?php esc_html_e( 'Activity Timeline', 'charity-campaigns' ); ?></h2>
					<div class="activity-count">
						<?php
						// translators: %d is the number of activities found
						echo esc_html( sprintf( _n( '%d activity found', '%d activities found', count( $activities ), 'charity-campaigns' ), count( $activities ) ) );
						?>
					</div>
				</div>
				
				<?php if ( ! empty( $activities ) ) : ?>
					<div class="activity-timeline">
						<?php foreach ( $activities as $activity ) : ?>
							<div class="activity-item activity-<?php echo esc_attr( $activity['type'] ); ?>">
								<div class="activity-icon"><?php echo esc_html( $activity['icon'] ); ?></div>
								<div class="activity-content">
									<div class="activity-text"><?php echo esc_html( $activity['text'] ); ?></div>
									<div class="activity-meta">
										<span class="activity-time"><?php echo esc_html( $activity['time'] ); ?></span>
										<?php if ( ! empty( $activity['campaign'] ) ) : ?>
											<span class="activity-campaign"><?php echo esc_html( $activity['campaign'] ); ?></span>
										<?php endif; ?>
									</div>
								</div>
							</div>
						<?php endforeach; ?>
					</div>
				<?php else : ?>
					<div class="no-activities">
						<p><?php esc_html_e( 'No activities found matching your criteria.', 'charity-campaigns' ); ?></p>
						<a href="?page=ccfs-dashboard&tab=activity" class="button"><?php esc_html_e( 'View All Activities', 'charity-campaigns' ); ?></a>
					</div>
				<?php endif; ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Get filtered activities based on criteria.
	 *
	 * @since 1.0.0
	 * @param string $activity_type Filter by activity type.
	 * @param string $campaign_id Filter by campaign ID.
	 * @param string $date_from Filter from date.
	 * @param string $date_to Filter to date.
	 * @param string $search_term Search term.
	 * @return array Filtered activities.
	 */
	private function get_filtered_activities( $activity_type = '', $campaign_id = '', $date_from = '', $date_to = '', $search_term = '' ) {
		$all_activities      = Charity_Campaigns_Functions::get_recent_activities( 100 ); // Get more activities for filtering
		$filtered_activities = array();

		foreach ( $all_activities as $activity ) {
			// Filter by activity type
			if ( ! empty( $activity_type ) && $activity['type'] !== $activity_type ) {
				continue;
			}

			// Filter by campaign ID (if activity has campaign info)
			if ( ! empty( $campaign_id ) && isset( $activity['campaign_id'] ) && $activity['campaign_id'] != $campaign_id ) {
				continue;
			}

			// Filter by date range (if activity has date info)
			if ( ! empty( $date_from ) && isset( $activity['date'] ) && $activity['date'] < $date_from ) {
				continue;
			}
			if ( ! empty( $date_to ) && isset( $activity['date'] ) && $activity['date'] > $date_to ) {
				continue;
			}

			// Filter by search term
			if ( ! empty( $search_term ) && stripos( $activity['text'], $search_term ) === false ) {
				continue;
			}

			$filtered_activities[] = $activity;
		}

		return $filtered_activities;
	}

	/**
	 * Get calendar events for campaigns and donations.
	 *
	 * @since 1.0.0
	 * @return array Calendar events data.
	 */
	private function get_calendar_events() {
		$events = array();

		// Get campaign events (start dates, end dates, completion dates)
		$campaigns = get_posts(
			array(
				'post_type'      => 'ccfs_campaign',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
			)
		);

		foreach ( $campaigns as $campaign ) {
			$start_date      = get_post_meta( $campaign->ID, 'campaign_start_date', true );
			$end_date        = get_post_meta( $campaign->ID, 'campaign_end_date', true );
			$completion_date = get_post_meta( $campaign->ID, 'campaign_completion_date', true );
			$excess_date     = get_post_meta( $campaign->ID, 'campaign_excess_date', true );

			// Campaign start date
			if ( $start_date ) {
				$events[] = array(
					'date'  => gmdate( 'Y-m-d', strtotime( $start_date ) ),
					// translators: %s is the campaign title
					'title' => sprintf( __( 'Campaign Started: %s', 'charity-campaigns' ), $campaign->post_title ),
					'type'  => 'campaign-start',
					'color' => '#46b450',
				);
			}

			// Campaign end date
			if ( $end_date ) {
				$events[] = array(
					'date'  => gmdate( 'Y-m-d', strtotime( $end_date ) ),
					// translators: %s is the campaign title
					'title' => sprintf( __( 'Campaign Ends: %s', 'charity-campaigns' ), $campaign->post_title ),
					'type'  => 'campaign-end',
					'color' => '#ff6b6b',
				);
			}

			// Campaign completion date
			if ( $completion_date ) {
				$events[] = array(
					'date'  => gmdate( 'Y-m-d', strtotime( $completion_date ) ),
					// translators: %s is the campaign title
					'title' => sprintf( __( 'Campaign Completed: %s', 'charity-campaigns' ), $campaign->post_title ),
					'type'  => 'campaign-completed',
					'color' => '#4caf50',
				);
			}

			// Campaign excess funds date
			if ( $excess_date ) {
				$events[] = array(
					'date'  => gmdate( 'Y-m-d', strtotime( $excess_date ) ),
					// translators: %s is the campaign title
					'title' => sprintf( __( 'Excess Funds: %s', 'charity-campaigns' ), $campaign->post_title ),
					'type'  => 'campaign-excess',
					'color' => '#ff9800',
				);
			}
		}

		// Get donation events (grouped by date)
		$donations = get_posts(
			array(
				'post_type'      => 'ccfs_donation',
				'posts_per_page' => -1,
				'post_status'    => 'publish',
				'meta_query'     => array(
					array(
						'key'     => 'donation_date',
						'compare' => 'EXISTS',
					),
				),
			)
		);

		$donations_by_date = array();
		foreach ( $donations as $donation ) {
			$donation_date = get_post_meta( $donation->ID, 'donation_date', true );
			if ( $donation_date ) {
				$date_key = gmdate( 'Y-m-d', strtotime( $donation_date ) );
				if ( ! isset( $donations_by_date[ $date_key ] ) ) {
					$donations_by_date[ $date_key ] = 0;
				}
				$amount                          = get_post_meta( $donation->ID, 'donation_amount', true );
				$donations_by_date[ $date_key ] += floatval( $amount );
			}
		}

		foreach ( $donations_by_date as $date => $total_amount ) {
			$events[] = array(
				'date'  => $date,
				// translators: %s is the formatted currency amount
				'title' => sprintf( __( 'Donations: %s', 'charity-campaigns' ), Charity_Campaigns_Functions::format_currency( $total_amount ) ),
				'type'  => 'donations',
				'color' => '#2196f3',
			);
		}

		return $events;
	}


	/**
	 * Reports tab content.
	 *
	 * @since 1.0.0
	 */
	private function reports_tab() {
		?>
		<div class="charity-dashboard-reports">
			<!-- Financial Summary Section -->
			<div class="reports-section">
				<div class="reports-header">
					<h2><?php esc_html_e( 'Financial Summary', 'charity-campaigns' ); ?></h2>
					
					<!-- Date Filter Controls -->
					<div class="date-filter-controls">
						<label for="date-filter"><?php esc_html_e( 'Time Period:', 'charity-campaigns' ); ?></label>
						<select id="date-filter" name="date_filter" onchange="filterFinancialSummary(this.value)">
							<?php // No nonce needed for URL parameter filtering ?>
							<?php // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering ?>
							<option value="all" <?php selected( isset( $_GET['date_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['date_filter'] ) ) : 'all', 'all' ); ?>><?php esc_html_e( 'All Time', 'charity-campaigns' ); ?></option>
							<?php // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering ?>
							<option value="daily" <?php selected( isset( $_GET['date_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['date_filter'] ) ) : '', 'daily' ); ?>><?php esc_html_e( 'Today', 'charity-campaigns' ); ?></option>
							<?php // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering ?>
							<option value="weekly" <?php selected( isset( $_GET['date_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['date_filter'] ) ) : '', 'weekly' ); ?>><?php esc_html_e( 'This Week', 'charity-campaigns' ); ?></option>
							<?php // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering ?>
							<option value="monthly" <?php selected( isset( $_GET['date_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['date_filter'] ) ) : '', 'monthly' ); ?>><?php esc_html_e( 'This Month', 'charity-campaigns' ); ?></option>
							<?php // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering ?>
							<option value="yearly" <?php selected( isset( $_GET['date_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['date_filter'] ) ) : '', 'yearly' ); ?>><?php esc_html_e( 'This Year', 'charity-campaigns' ); ?></option>
						</select>
					</div>
				</div>
				
				<!-- Primary Metrics -->
				<div class="reports-grid primary-metrics">
					<div class="report-card primary-card">
						<h3><?php esc_html_e( 'Total Raised', 'charity-campaigns' ); ?></h3>
						<div class="report-value" id="total-raised">
							<?php echo esc_html( Charity_Campaigns_Functions::format_currency( Charity_Campaigns_Functions::get_total_raised( Charity_Campaigns_Functions::get_current_date_filter() ) ) ); ?>
						</div>
						<div class="report-subtitle">
							<?php
							// translators: %s is the formatted currency amount of total goals
							echo esc_html( sprintf( __( 'of %s total goal', 'charity-campaigns' ), Charity_Campaigns_Functions::format_currency( Charity_Campaigns_Functions::get_total_goals() ) ) );
							?>
						</div>
					</div>
					<div class="report-card primary-card">
						<h3><?php esc_html_e( 'Goal Progress', 'charity-campaigns' ); ?></h3>
						<div class="report-value" id="goal-progress">
							<?php echo number_format( Charity_Campaigns_Functions::get_goal_completion_percentage( Charity_Campaigns_Functions::get_current_date_filter() ), 1 ); ?>%
						</div>
                        <div class="progress-bar">
                            <div class="progress-fill" data-progress="<?php echo esc_attr( min( 100, Charity_Campaigns_Functions::get_goal_completion_percentage( Charity_Campaigns_Functions::get_current_date_filter() ) ) ); ?>"></div>
                        </div>
					</div>
					<div class="report-card primary-card">
						<h3><?php esc_html_e( 'Period Donations', 'charity-campaigns' ); ?></h3>
						<div class="report-value" id="period-donations">
							<?php echo esc_html( Charity_Campaigns_Functions::format_currency( Charity_Campaigns_Functions::get_period_donations( Charity_Campaigns_Functions::get_current_date_filter() ) ) ); ?>
						</div>
						<div class="report-subtitle" id="period-subtitle">
							<?php echo esc_html( Charity_Campaigns_Functions::get_period_subtitle( Charity_Campaigns_Functions::get_current_date_filter() ) ); ?>
						</div>
					</div>
				</div>

				<!-- Secondary Metrics -->
				<div class="reports-grid secondary-metrics">
					<div class="report-card">
						<h3><?php esc_html_e( 'Total Donations', 'charity-campaigns' ); ?></h3>
						<div class="report-value" id="total-donations">
							<?php echo number_format( Charity_Campaigns_Functions::get_total_donations_count( Charity_Campaigns_Functions::get_current_date_filter() ) ); ?>
						</div>
						<div class="report-subtitle">
							<?php echo esc_html( Charity_Campaigns_Functions::get_period_subtitle( Charity_Campaigns_Functions::get_current_date_filter() ) ); ?>
						</div>
					</div>
					<div class="report-card">
						<h3><?php esc_html_e( 'Average Donation', 'charity-campaigns' ); ?></h3>
						<div class="report-value" id="average-donation">
							<?php echo esc_html( Charity_Campaigns_Functions::format_currency( Charity_Campaigns_Functions::get_average_donation( Charity_Campaigns_Functions::get_current_date_filter() ) ) ); ?>
						</div>
						<div class="report-subtitle">
							<?php esc_html_e( 'Per donation', 'charity-campaigns' ); ?>
						</div>
					</div>
					<div class="report-card">
						<h3><?php esc_html_e( 'Active Campaigns', 'charity-campaigns' ); ?></h3>
						<div class="report-value" id="active-campaigns">
							<?php echo esc_html( number_format( Charity_Campaigns_Functions::get_active_campaigns_count() ) ); ?>
						</div>
						<div class="report-subtitle">
							<?php esc_html_e( 'Currently running', 'charity-campaigns' ); ?>
						</div>
					</div>
					<div class="report-card">
						<h3><?php esc_html_e( 'Completed Campaigns', 'charity-campaigns' ); ?></h3>
						<div class="report-value" id="completed-campaigns">
							<?php echo esc_html( number_format( Charity_Campaigns_Functions::get_completed_campaigns_count() ) ); ?>
						</div>
						<div class="report-subtitle">
							<?php esc_html_e( 'Goal reached', 'charity-campaigns' ); ?>
						</div>
					</div>
					<div class="report-card">
						<h3><?php esc_html_e( 'Total Excess Funds', 'charity-campaigns' ); ?></h3>
						<div class="report-value" id="total-excess-funds">
							<?php echo esc_html( Charity_Campaigns_Functions::format_currency( Charity_Campaigns_Functions::get_total_excess_funds() ) ); ?>
						</div>
						<div class="report-subtitle">
							<?php esc_html_e( 'Beyond goals', 'charity-campaigns' ); ?>
						</div>
					</div>
				</div>

				<!-- Campaign Status Breakdown -->
				<div class="campaign-status-breakdown">
					<h3><?php esc_html_e( 'Campaign Status Overview', 'charity-campaigns' ); ?></h3>
					<?php
					$status_breakdown = Charity_Campaigns_Functions::get_campaign_status_breakdown();
					$status_labels    = array(
						'active'    => __( 'Active', 'charity-campaigns' ),
						'completed' => __( 'Completed', 'charity-campaigns' ),
						'paused'    => __( 'Paused', 'charity-campaigns' ),
						'cancelled' => __( 'Cancelled', 'charity-campaigns' ),
					);
					$status_tooltips  = array(
						'active'    => __( 'Campaigns currently accepting donations', 'charity-campaigns' ),
						'completed' => __( 'Campaigns that have reached their funding goal', 'charity-campaigns' ),
						'paused'    => __( 'Campaigns temporarily stopped from accepting donations', 'charity-campaigns' ),
						'cancelled' => __( 'Campaigns that have been cancelled or discontinued', 'charity-campaigns' ),
					);
					$status_colors    = array(
						'active'    => '#46b450',
						'completed' => '#4caf50',
						'paused'    => '#ff9800',
						'cancelled' => '#f44336',
					);
					?>
					<div class="status-grid">
						<?php foreach ( $status_breakdown as $status => $count ) : ?>
                            <div class="status-item" title="<?php echo esc_attr( $status_tooltips[ $status ] ); ?>">
                                <div class="status-indicator" data-color="<?php echo esc_attr( $status_colors[ $status ] ); ?>"></div>
								<span class="status-label"><?php echo esc_html( $status_labels[ $status ] ); ?></span>
								<span class="status-count"><?php echo number_format( $count ); ?></span>
							</div>
						<?php endforeach; ?>
					</div>
				</div>
			</div>

			<!-- Recent Donations Section -->
			<div class="reports-section">
				<h2><?php esc_html_e( 'Recent Donations', 'charity-campaigns' ); ?></h2>
				
				<!-- Filter Controls -->
				<div class="donations-filters">
					<form method="get" class="filter-form">
						<input type="hidden" name="page" value="ccfs-dashboard" />
						<input type="hidden" name="tab" value="reports" />
						
						<div class="filter-row fields-row">
							<div class="filter-group">
								<label for="donation_status"><?php esc_html_e( 'Status:', 'charity-campaigns' ); ?></label>
								<select name="donation_status" id="donation_status">
									<option value=""><?php esc_html_e( 'All Statuses', 'charity-campaigns' ); ?></option>
									<?php
									$status_options = Charity_Campaigns_Functions::get_donation_status_options();
									// No nonce needed for URL parameter filtering
									// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering
									$selected_status = isset( $_GET['donation_status'] ) ? sanitize_text_field( wp_unslash( $_GET['donation_status'] ) ) : '';
									foreach ( $status_options as $value => $label ) :
										?>
										<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $selected_status, $value ); ?>>
											<?php echo esc_html( $label ); ?>
										</option>
									<?php endforeach; ?>
								</select>
							</div>
							
							<div class="filter-group">
								<label for="donation_campaign"><?php esc_html_e( 'Campaign:', 'charity-campaigns' ); ?></label>
								<select name="donation_campaign" id="donation_campaign">
									<option value=""><?php esc_html_e( 'All Campaigns', 'charity-campaigns' ); ?></option>
									<?php
									$campaigns = Charity_Campaigns_Functions::get_campaigns_for_filter();
									// No nonce needed for URL parameter filtering
									// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering
									$selected_campaign = isset( $_GET['donation_campaign'] ) ? sanitize_text_field( wp_unslash( $_GET['donation_campaign'] ) ) : '';
									foreach ( $campaigns as $campaign ) :
										?>
										<option value="<?php echo esc_attr( $campaign['id'] ); ?>" <?php selected( $selected_campaign, $campaign['id'] ); ?>>
											<?php echo esc_html( $campaign['title'] ); ?>
										</option>
									<?php endforeach; ?>
								</select>
							</div>
							
							<div class="filter-group">
								<label for="date_from"><?php esc_html_e( 'From:', 'charity-campaigns' ); ?></label>
								<?php // No nonce needed for URL parameter filtering ?>
								<?php // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering ?>
								<input type="date" name="date_from" id="date_from" 
									value="<?php echo isset( $_GET['date_from'] ) ? esc_attr( sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) ) : ''; ?>" />
							</div>
							
							<div class="filter-group">
								<label for="date_to"><?php esc_html_e( 'To:', 'charity-campaigns' ); ?></label>
								<?php // No nonce needed for URL parameter filtering ?>
								<?php // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameter for filtering ?>
								<input type="date" name="date_to" id="date_to" 
									value="<?php echo isset( $_GET['date_to'] ) ? esc_attr( sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) ) : ''; ?>" />
							</div>
						</div>
						
						<div class="filter-row buttons-row">
							<div class="filter-actions">
								<button type="submit" class="button button-primary">
									<?php esc_html_e( 'Filter', 'charity-campaigns' ); ?>
								</button>
								<a href="?page=ccfs-dashboard&tab=reports" class="button button-secondary">
									<?php esc_html_e( 'Clear', 'charity-campaigns' ); ?>
								</a>
							</div>
						</div>
					</form>
				</div>

				<!-- Action Buttons -->
				<div class="reports-actions">
					<a href="<?php echo esc_url( admin_url( 'edit.php?post_type=ccfs_donation' ) ); ?>" 
						class="button button-secondary">
						<?php esc_html_e( 'View All Donations', 'charity-campaigns' ); ?>
					</a>
					<a href="<?php echo esc_url( admin_url( 'post-new.php?post_type=ccfs_donation' ) ); ?>" 
						class="button button-primary">
						<?php esc_html_e( 'Add New Donation', 'charity-campaigns' ); ?>
					</a>
				</div>

				<!-- Donations List -->
				<div class="donations-list">
					<?php
					// Get filter parameters from URL (no nonce needed for filtering)
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for filtering
					$status = isset( $_GET['donation_status'] ) ? sanitize_text_field( wp_unslash( $_GET['donation_status'] ) ) : '';
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for filtering
					$campaign_id = isset( $_GET['donation_campaign'] ) ? sanitize_text_field( wp_unslash( $_GET['donation_campaign'] ) ) : '';
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for filtering
					$date_from = isset( $_GET['date_from'] ) ? sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) : '';
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for filtering
					$date_to = isset( $_GET['date_to'] ) ? sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) : '';

					$donations      = Charity_Campaigns_Functions::get_recent_donations( 30, $status, $campaign_id, $date_from, $date_to );
					$status_options = Charity_Campaigns_Functions::get_donation_status_options();

					if ( ! empty( $donations ) ) :
						?>
						<div class="donations-table-wrapper">
							<table class="wp-list-table widefat fixed striped donations-table">
								<thead>
									<tr>
										<th><?php esc_html_e( 'Donation', 'charity-campaigns' ); ?></th>
										<th><?php esc_html_e( 'Amount', 'charity-campaigns' ); ?></th>
										<th><?php esc_html_e( 'Campaign', 'charity-campaigns' ); ?></th>
										<th><?php esc_html_e( 'Donor', 'charity-campaigns' ); ?></th>
										<th><?php esc_html_e( 'Date', 'charity-campaigns' ); ?></th>
										<th><?php esc_html_e( 'Status', 'charity-campaigns' ); ?></th>
										<th><?php esc_html_e( 'Actions', 'charity-campaigns' ); ?></th>
									</tr>
								</thead>
								<tbody>
									<?php foreach ( $donations as $donation ) : ?>
										<tr>
											<td>
												<strong><?php echo esc_html( $donation['title'] ); ?></strong>
											</td>
											<td>
												<span class="donation-amount">
													<?php echo esc_html( Charity_Campaigns_Functions::format_currency( $donation['amount'] ) ); ?>
												</span>
											</td>
											<td>
												<?php if ( $donation['campaign_title'] ) : ?>
													<a href="<?php echo esc_url( get_edit_post_link( $donation['campaign_id'] ) ); ?>" 
														title="<?php esc_attr_e( 'Edit Campaign', 'charity-campaigns' ); ?>">
														<?php echo esc_html( $donation['campaign_title'] ); ?>
													</a>
												<?php else : ?>
													<span class="no-campaign"><?php esc_html_e( 'No Campaign', 'charity-campaigns' ); ?></span>
												<?php endif; ?>
											</td>
											<td>
												<?php if ( $donation['is_anonymous'] === '1' || $donation['is_anonymous'] === 1 ) : ?>
													<span class="anonymous"><?php esc_html_e( 'Anonymous', 'charity-campaigns' ); ?></span>
												<?php elseif ( $donation['donor_name'] ) : ?>
													<div class="donor-info">
														<strong><?php echo esc_html( $donation['donor_name'] ); ?></strong>
														<?php if ( $donation['donor_email'] ) : ?>
															<br><small><?php echo esc_html( $donation['donor_email'] ); ?></small>
														<?php endif; ?>
													</div>
												<?php else : ?>
													<span class="no-donor"><?php esc_html_e( 'No donor info', 'charity-campaigns' ); ?></span>
												<?php endif; ?>
											</td>
											<td>
												<?php echo esc_html( $donation['date'] ? gmdate( 'M j, Y', strtotime( $donation['date'] ) ) : '—' ); ?>
											</td>
											<td>
												<span class="status-badge status-<?php echo esc_attr( $donation['status'] ); ?>">
													<?php echo isset( $status_options[ $donation['status'] ] ) ? esc_html( $status_options[ $donation['status'] ] ) : esc_html( $donation['status'] ); ?>
												</span>
											</td>
											<td>
												<div class="donation-actions">
													<a href="<?php echo esc_url( $donation['edit_url'] ); ?>" 
														class="button button-small" title="<?php esc_attr_e( 'Edit Donation', 'charity-campaigns' ); ?>">
														<?php esc_html_e( 'Edit', 'charity-campaigns' ); ?>
													</a>
												</div>
											</td>
										</tr>
									<?php endforeach; ?>
								</tbody>
							</table>
						</div>
					<?php else : ?>
						<div class="no-donations">
							<p><?php esc_html_e( 'No donations found matching your criteria.', 'charity-campaigns' ); ?></p>
							<p>
								<a href="<?php echo esc_url( admin_url( 'post-new.php?post_type=ccfs_donation' ) ); ?>" 
									class="button button-primary">
									<?php esc_html_e( 'Add First Donation', 'charity-campaigns' ); ?>
								</a>
							</p>
						</div>
					<?php endif; ?>
				</div>
			</div>

			<!-- Charts & Analytics Section -->
			<div class="reports-section">
				<h2><?php esc_html_e( 'Charts & Analytics', 'charity-campaigns' ); ?></h2>
				
				<!-- Chart Tabs -->
				<div class="chart-tabs">
					<button class="chart-tab active" data-chart="trends"><?php esc_html_e( 'Monthly Trends', 'charity-campaigns' ); ?></button>
					<button class="chart-tab" data-chart="campaigns"><?php esc_html_e( 'Campaign Performance', 'charity-campaigns' ); ?></button>
					<button class="chart-tab" data-chart="status"><?php esc_html_e( 'Status Distribution', 'charity-campaigns' ); ?></button>
				</div>

				<!-- Monthly Trends Chart -->
				<div class="chart-container active" id="trends-chart">
					<h3><?php esc_html_e( 'Monthly Donation Trends', 'charity-campaigns' ); ?></h3>
					<div class="chart-wrapper">
						<canvas id="monthlyTrendsChart" width="400" height="200"></canvas>
					</div>
					<div class="chart-summary">
						<?php
						$trends_data         = Charity_Campaigns_Functions::get_monthly_donation_trends( 12 );
						$total_trends_amount = array_sum( array_column( $trends_data, 'amount' ) );
						$total_trends_count  = array_sum( array_column( $trends_data, 'count' ) );
						?>
						<p><strong><?php esc_html_e( 'Last 12 Months:', 'charity-campaigns' ); ?></strong> 
							<?php echo esc_html( Charity_Campaigns_Functions::format_currency( $total_trends_amount ) ); ?> 
							(<?php echo esc_html( number_format( $total_trends_count ) ); ?> <?php esc_html_e( 'donations', 'charity-campaigns' ); ?>)
						</p>
					</div>
				</div>

				<!-- Campaign Performance Chart -->
				<div class="chart-container" id="campaigns-chart">
					<h3><?php esc_html_e( 'Top Campaign Performance', 'charity-campaigns' ); ?></h3>
					<div class="chart-wrapper">
						<canvas id="campaignPerformanceChart" width="400" height="200"></canvas>
					</div>
					<div class="chart-summary">
						<?php
						$campaigns_data = Charity_Campaigns_Functions::get_campaign_performance_data();
						$top_campaign   = ! empty( $campaigns_data ) ? $campaigns_data[0] : null;
						?>
						<?php if ( $top_campaign ) : ?>
							<p><strong><?php esc_html_e( 'Top Campaign:', 'charity-campaigns' ); ?></strong> 
								<?php echo esc_html( $top_campaign['title'] ); ?> - 
								<?php echo esc_html( Charity_Campaigns_Functions::format_currency( $top_campaign['raised'] ) ); ?>
								(<?php echo esc_html( number_format( $top_campaign['progress'], 1 ) ); ?>% <?php esc_html_e( 'of goal', 'charity-campaigns' ); ?>)
							</p>
						<?php endif; ?>
					</div>
				</div>

				<!-- Status Distribution Chart -->
				<div class="chart-container" id="status-chart">
					<h3><?php esc_html_e( 'Donation Status Distribution', 'charity-campaigns' ); ?></h3>
					<div class="chart-wrapper">
						<canvas id="statusDistributionChart" width="400" height="200"></canvas>
					</div>
					<div class="chart-summary">
						<?php
						$status_data      = Charity_Campaigns_Functions::get_donation_status_distribution();
						$completed_amount = 0;
						foreach ( $status_data as $status ) {
							if ( $status['status'] === 'completed' ) {
								$completed_amount = $status['amount'];
								break;
							}
						}
						?>
						<p><strong><?php esc_html_e( 'Completed Donations:', 'charity-campaigns' ); ?></strong> 
							<?php echo esc_html( Charity_Campaigns_Functions::format_currency( $completed_amount ) ); ?>
						</p>
					</div>
				</div>
			</div>

		</div>

		<?php
	}

	/**
	 * Settings tab content.
	 *
	 * @since 1.0.0
	 */
	private function settings_tab() {
		?>
		<div class="charity-dashboard-settings">
			<div class="dashboard-section">
				<h3><?php esc_html_e( 'Plugin Settings', 'charity-campaigns' ); ?></h3>
				<p><?php esc_html_e( 'Configure your charity campaigns plugin settings here.', 'charity-campaigns' ); ?></p>
			</div>
		</div>
		<?php
	}

	/**
	 * Help tab content.
	 *
	 * @since 1.0.0
	 */
	private function help_tab() {
		$help = new Charity_Campaigns_Help();
		$help->render_help_content();
	}

	/**
	 * Enqueue admin scripts and styles.
	 *
	 * @since 1.0.0
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_scripts( $hook ) {
		if ( strpos( $hook, 'ccfs' ) !== false ) {
			wp_enqueue_style(
				'ccfs-admin',
				CCFS_PLUGIN_URL . 'assets/css/admin.css',
				array(),
				CCFS_VERSION
			);

			// Enqueue JavaScript for dashboard page
			if ( $hook === 'ccfs_page_ccfs-dashboard' || $hook === 'toplevel_page_ccfs-dashboard' ) {
				// Enqueue Chart.js first
				wp_enqueue_script(
					'chart-js',
					CCFS_PLUGIN_URL . 'assets/js/chart.min.js',
					array(),
					'4.5.1',
					true
				);

				// Enqueue admin script with Chart.js as dependency
				wp_enqueue_script(
					'ccfs-admin',
					CCFS_PLUGIN_URL . 'assets/js/admin.js',
					array( 'jquery', 'chart-js' ),
					CCFS_VERSION,
					true
				);

				// Pass currency settings to JavaScript
				wp_localize_script(
					'ccfs-admin',
					'charityCampaignsSettings',
					array(
						'defaultCurrency'        => Charity_Campaigns_Settings::get_default_currency(),
						'currencySymbolPosition' => Charity_Campaigns_Settings::get_currency_symbol_position(),
						'decimalPlaces'          => Charity_Campaigns_Settings::get_decimal_places(),
					)
				);

				// Pass calendar data for overview tab
				$calendar_events = $this->get_calendar_events();
				wp_localize_script( 'ccfs-admin', 'charityCalendarEvents', $calendar_events );

				// Pass chart data for reports tab
				wp_localize_script( 'ccfs-admin', 'ccfsTrendsData', Charity_Campaigns_Functions::get_monthly_donation_trends( 12 ) );
				wp_localize_script( 'ccfs-admin', 'ccfsCampaignsData', Charity_Campaigns_Functions::get_campaign_performance_data() );
				wp_localize_script( 'ccfs-admin', 'ccfsStatusData', Charity_Campaigns_Functions::get_donation_status_distribution() );
			}
		}
	}
}
