<?php
/**
 * Admin Columns for Custom Post Types
 *
 * @package CCFS
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class responsible for managing admin columns for custom post types.
 *
 * @since 1.0.0
 */
class Charity_Campaigns_Admin_Columns {

	/**
	 * Initialize the class and register hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Donation columns
		add_filter( 'manage_ccfs_donation_posts_columns', array( $this, 'add_donation_columns' ) );
		add_action( 'manage_ccfs_donation_posts_custom_column', array( $this, 'populate_donation_columns' ), 10, 2 );
		add_filter( 'manage_edit-ccfs_donation_sortable_columns', array( $this, 'make_donation_columns_sortable' ) );
		add_action( 'pre_get_posts', array( $this, 'handle_donation_column_sorting' ) );

		// Campaign columns
		add_filter( 'manage_ccfs_campaign_posts_columns', array( $this, 'add_campaign_columns' ) );
		add_action( 'manage_ccfs_campaign_posts_custom_column', array( $this, 'populate_campaign_columns' ), 10, 2 );
		add_filter( 'manage_edit-ccfs_campaign_sortable_columns', array( $this, 'make_campaign_columns_sortable' ) );
		add_action( 'pre_get_posts', array( $this, 'handle_campaign_column_sorting' ) );
	}

	/**
	 * Add custom columns to donations list table.
	 *
	 * @since 1.0.0
	 * @param array $columns Existing columns.
	 * @return array Modified columns.
	 */
	public function add_donation_columns( $columns ) {
		// Insert new columns after title
		$new_columns = array();
		foreach ( $columns as $key => $value ) {
			$new_columns[ $key ] = $value;
			if ( 'title' === $key ) {
				$new_columns['campaign']      = __( 'Campaign', 'charity-campaigns' );
				$new_columns['amount']        = __( 'Amount', 'charity-campaigns' );
				$new_columns['status']        = __( 'Status', 'charity-campaigns' );
				$new_columns['donation_date'] = __( 'Donation Date', 'charity-campaigns' );
			}
		}
		return $new_columns;
	}

	/**
	 * Populate custom columns with data.
	 *
	 * @since 1.0.0
	 * @param string $column Column name.
	 * @param int    $post_id Post ID.
	 */
	public function populate_donation_columns( $column, $post_id ) {
		switch ( $column ) {
			case 'campaign':
				$campaign_id = get_post_meta( $post_id, 'donation_campaign_id', true );
				if ( $campaign_id ) {
					$campaign = get_post( $campaign_id );
					if ( $campaign ) {
						$edit_url = admin_url( 'post.php?post=' . $campaign_id . '&action=edit' );
						echo '<a href="' . esc_url( $edit_url ) . '">' . esc_html( $campaign->post_title ) . '</a>';
                    } else {
                        echo '<span class="text-muted">' . esc_html( __( 'Campaign not found', 'charity-campaigns' ) ) . '</span>';
					}
				} else {
                    echo '<span class="text-muted">' . esc_html( __( 'No campaign', 'charity-campaigns' ) ) . '</span>';
				}
				break;

			case 'amount':
				$amount = get_post_meta( $post_id, 'donation_amount', true );
				if ( $amount ) {
                    echo esc_html( Charity_Campaigns_Functions::format_currency( $amount ) );
                } else {
                    echo '<span class="text-muted">' . esc_html( __( 'No amount', 'charity-campaigns' ) ) . '</span>';
				}
				break;

			case 'status':
				$status = get_post_meta( $post_id, 'donation_status', true );
				if ( $status ) {
					$status_labels = array(
						'pending'   => __( 'Pending', 'charity-campaigns' ),
						'completed' => __( 'Completed', 'charity-campaigns' ),
						'failed'    => __( 'Failed', 'charity-campaigns' ),
						'refunded'  => __( 'Refunded', 'charity-campaigns' ),
					);
					$label         = isset( $status_labels[ $status ] ) ? $status_labels[ $status ] : ucfirst( $status );
                    echo '<span class="donation-status-' . esc_attr( $status ) . '">' . esc_html( $label ) . '</span>';
                } else {
                    echo '<span class="text-muted">' . esc_html( __( 'No status', 'charity-campaigns' ) ) . '</span>';
				}
				break;

			case 'donation_date':
				$donation_date = get_post_meta( $post_id, 'donation_date', true );
				if ( $donation_date ) {
					$date = new DateTime( $donation_date );
					echo esc_html( $date->format( 'M j, Y g:i A' ) );
				} else {
					// Fallback to post date
					$post_date = get_the_date( 'M j, Y g:i A', $post_id );
					echo esc_html( $post_date );
				}
				break;
		}
	}

	/**
	 * Make custom columns sortable.
	 *
	 * @since 1.0.0
	 * @param array $columns Sortable columns.
	 * @return array Modified sortable columns.
	 */
	public function make_donation_columns_sortable( $columns ) {
		$columns['campaign']      = 'campaign';
		$columns['amount']        = 'amount';
		$columns['status']        = 'status';
		$columns['donation_date'] = 'donation_date';
		return $columns;
	}

	/**
	 * Handle sorting for custom columns.
	 *
	 * @since 1.0.0
	 * @param WP_Query $query The query object.
	 */
	public function handle_donation_column_sorting( $query ) {
		if ( ! is_admin() || ! $query->is_main_query() ) {
			return;
		}

		// Only apply to donation post type
		if ( 'ccfs_donation' !== $query->get( 'post_type' ) ) {
			return;
		}

		$orderby = $query->get( 'orderby' );

		switch ( $orderby ) {
			case 'campaign':
				$query->set( 'meta_key', 'donation_campaign_id' );
				$query->set( 'orderby', 'meta_value_num' );
				break;

			case 'amount':
				$query->set( 'meta_key', 'donation_amount' );
				$query->set( 'orderby', 'meta_value_num' );
				break;

			case 'status':
				$query->set( 'meta_key', 'donation_status' );
				$query->set( 'orderby', 'meta_value' );
				break;

			case 'donation_date':
				$query->set( 'meta_key', 'donation_date' );
				$query->set( 'orderby', 'meta_value' );
				break;
		}
	}

	/**
	 * Add custom columns to campaigns list table.
	 *
	 * @since 1.0.0
	 * @param array $columns Existing columns.
	 * @return array Modified columns.
	 */
	public function add_campaign_columns( $columns ) {
		// Insert new columns after title
		$new_columns = array();
		foreach ( $columns as $key => $value ) {
			$new_columns[ $key ] = $value;
			if ( 'title' === $key ) {
				$new_columns['status']          = __( 'Status', 'charity-campaigns' );
				$new_columns['progress']        = __( 'Progress', 'charity-campaigns' );
				$new_columns['completion_date'] = __( 'Completed', 'charity-campaigns' );
				$new_columns['excess_funds']    = __( 'Excess Funds', 'charity-campaigns' );
			}
		}
		return $new_columns;
	}

	/**
	 * Populate custom campaign columns with data.
	 *
	 * @since 1.0.0
	 * @param string $column Column name.
	 * @param int    $post_id Post ID.
	 */
	public function populate_campaign_columns( $column, $post_id ) {
		switch ( $column ) {
			case 'status':
				$status = get_post_meta( $post_id, 'campaign_status', true );
				if ( $status ) {
					$status_labels = array(
						'active'    => __( 'Active', 'charity-campaigns' ),
						'completed' => __( 'Completed', 'charity-campaigns' ),
						'paused'    => __( 'Paused', 'charity-campaigns' ),
						'cancelled' => __( 'Cancelled', 'charity-campaigns' ),
					);
					$label         = isset( $status_labels[ $status ] ) ? $status_labels[ $status ] : ucfirst( $status );
					echo '<span class="campaign-status-' . esc_attr( $status ) . '">' . esc_html( $label ) . '</span>';
				} else {
					echo '<span style="color: #999;">' . esc_html( __( 'No status', 'charity-campaigns' ) ) . '</span>';
				}
				break;

			case 'progress':
				$goal   = get_post_meta( $post_id, 'campaign_goal', true );
				$raised = get_post_meta( $post_id, 'campaign_raised', true );
				if ( $goal && $goal > 0 ) {
					$percentage       = min( ( $raised / $goal ) * 100, 100 );
					$formatted_raised = Charity_Campaigns_Functions::format_currency( $raised );
					$formatted_goal   = Charity_Campaigns_Functions::format_currency( $goal );
                    echo '<div class="mini-progress-text">' . esc_html( $formatted_raised ) . ' / ' . esc_html( $formatted_goal ) . '</div>';
                    echo '<div class="mini-progress-bar"><span class="mini-progress-fill" data-progress="' . esc_attr( $percentage ) . '"></span></div>';
                    echo '<div class="mini-progress-note">' . esc_html( round( $percentage ) ) . '%</div>';
                } else {
                    echo '<span class="text-muted">' . esc_html( __( 'No goal set', 'charity-campaigns' ) ) . '</span>';
				}
				break;

			case 'completion_date':
				$completion_date = get_post_meta( $post_id, 'campaign_completion_date', true );
				if ( $completion_date ) {
					$date = new DateTime( $completion_date );
                    echo esc_html( $date->format( 'M j, Y g:i A' ) );
                } else {
                    echo '<span class="text-muted">' . esc_html( __( 'Not completed', 'charity-campaigns' ) ) . '</span>';
				}
				break;

			case 'excess_funds':
				$excess_amount = get_post_meta( $post_id, 'campaign_excess_amount', true );
                if ( $excess_amount && $excess_amount > 0 ) {
                    echo '<span class="text-warning fw-bold">' . esc_html( Charity_Campaigns_Functions::format_currency( $excess_amount ) ) . '</span>';
                } else {
                    echo '<span class="text-muted">' . esc_html( __( 'None', 'charity-campaigns' ) ) . '</span>';
				}
				break;
		}
	}

	/**
	 * Make custom campaign columns sortable.
	 *
	 * @since 1.0.0
	 * @param array $columns Sortable columns.
	 * @return array Modified sortable columns.
	 */
	public function make_campaign_columns_sortable( $columns ) {
		$columns['status']          = 'campaign_status';
		$columns['progress']        = 'campaign_raised';
		$columns['completion_date'] = 'campaign_completion_date';
		$columns['excess_funds']    = 'campaign_excess_amount';
		return $columns;
	}

	/**
	 * Handle sorting for custom campaign columns.
	 *
	 * @since 1.0.0
	 * @param WP_Query $query The query object.
	 */
	public function handle_campaign_column_sorting( $query ) {
		if ( ! is_admin() || ! $query->is_main_query() ) {
			return;
		}

		// Only apply to campaign post type
		if ( 'ccfs_campaign' !== $query->get( 'post_type' ) ) {
			return;
		}

		$orderby = $query->get( 'orderby' );

		switch ( $orderby ) {
			case 'campaign_status':
				$query->set( 'meta_key', 'campaign_status' );
				$query->set( 'orderby', 'meta_value' );
				break;

			case 'campaign_raised':
				$query->set( 'meta_key', 'campaign_raised' );
				$query->set( 'orderby', 'meta_value_num' );
				break;

			case 'campaign_completion_date':
				$query->set( 'meta_key', 'campaign_completion_date' );
				$query->set( 'orderby', 'meta_value' );
				break;

			case 'campaign_excess_amount':
				$query->set( 'meta_key', 'campaign_excess_amount' );
				$query->set( 'orderby', 'meta_value_num' );
				break;
		}
	}
}
