/**
 * Shortcode JavaScript for Charity Campaigns
 *
 * @package CCFS
 * @since 1.0.0
 */

(function($) {
    'use strict';

    var stripe, elements, cardElement;
    var isStripeConfigured = false;

    // Initialize when document is ready
    $(document).ready(function() {
        initDonationForms();
        initProgressBars();
    });

    /**
     * Initialize donation forms
     */
    function initDonationForms() {
        $('.charity-donation-form').each(function() {
            var $form = $(this);
            var $formFields = $form.find('.charity-donation-form-fields');
            var $messages = $form.find('.charity-donation-messages');
            var $amountInput = $form.find('#charity_donation_amount');
            var $button = $form.find('.charity-donate-button');

            // Check if Stripe is configured
            isStripeConfigured = charityCampaignsShortcode.stripe && charityCampaignsShortcode.stripe.is_configured;

            // Initialize Stripe if configured
            if (isStripeConfigured) {
                initStripe();
            }

            // Format currency input
            if ($amountInput.length) {
                $amountInput.on('input', function() {
                    formatCurrencyInput($(this));
                });
            }

            // Store original button text
            $button.data('original-text', $button.text());

            // Handle form submission
            $formFields.on('submit', function(e) {
                e.preventDefault();
                handleDonationSubmission($form, $messages, $button);
            });

            // Add quick amount buttons
            addQuickAmountButtons($form, $amountInput);
        });
    }

    /**
     * Initialize Stripe
     */
    function initStripe() {
        if (typeof Stripe === 'undefined') {
            console.error('Stripe.js not loaded');
            return;
        }

        stripe = Stripe(charityCampaignsShortcode.stripe.publishable_key);
        elements = stripe.elements();

        // Create card element
        cardElement = elements.create('card', {
            style: {
                base: {
                    fontSize: '16px',
                    color: '#424770',
                    '::placeholder': {
                        color: '#aab7c4',
                    },
                },
                invalid: {
                    color: '#9e2146',
                },
            },
        });

        // Mount card element
        cardElement.mount('#charity-card-element');

        // Handle real-time validation errors from the card Element
        cardElement.on('change', function(event) {
            var displayError = document.getElementById('charity-card-errors');
            if (event.error) {
                displayError.textContent = event.error.message;
            } else {
                displayError.textContent = '';
            }
        });
    }

    /**
     * Initialize progress bars animation
     */
    function initProgressBars() {
        $('.charity-progress-fill').each(function() {
            var $bar = $(this);
            var pct = $bar.data('progress');
            if (typeof pct !== 'undefined') {
                $bar.css('width', '0%').animate({
                    width: pct + '%'
                }, 1000, 'easeOutQuart');
            }
        });
    }

    /**
     * Format currency input
     */
    function formatCurrencyInput($input) {
        var value = $input.val();
        var cleanValue = value.replace(/[^\d.]/g, '');
        
        if (cleanValue !== value) {
            $input.val(cleanValue);
        }
    }

    /**
     * Add quick amount buttons
     */
    function addQuickAmountButtons($form, $amountInput) {
        // Check if predefined amounts should be shown
        var showPredefined = $form.data('show-predefined-amounts');
        if (showPredefined === 'false') {
            return;
        }

        // Get predefined amounts from settings or use defaults
        var quickAmounts = charityCampaignsShortcode.predefinedAmounts || [25, 50, 100, 250, 500];
        var currencySymbol = charityCampaignsShortcode.currencySymbol || '$';
        var $quickAmounts = $('<div class="charity-quick-amounts"></div>');
        
        quickAmounts.forEach(function(amount) {
            var $button = $('<button type="button" class="charity-quick-amount-btn">' + currencySymbol + amount + '</button>');
            $button.on('click', function() {
                $amountInput.val(amount);
                $amountInput.trigger('input');
            });
            $quickAmounts.append($button);
        });

        $amountInput.after($quickAmounts);
    }

    /**
     * Handle donation form submission
     */
    function handleDonationSubmission($form, $messages, $button) {
        var formData = $form.find('.charity-donation-form-fields').serialize();
        var campaignId = $form.data('campaign-id');

        // Validate form
        if (!validateDonationForm($form)) {
            return;
        }

        // Show loading state
        $button.prop('disabled', true).text('Processing...');
        $messages.hide();

        if (isStripeConfigured) {
            // Process with Stripe
            processStripePayment($form, $messages, $button, formData, campaignId);
        } else {
            // Process as manual donation
            processManualDonation($form, $messages, $button, formData, campaignId);
        }
    }

    /**
     * Process Stripe payment
     */
    function processStripePayment($form, $messages, $button, formData, campaignId) {
        // First create payment intent
        $.ajax({
            url: charityCampaignsShortcode.ajaxUrl,
            type: 'POST',
            data: {
                action: 'ccfs_create_payment_intent',
                nonce: charityCampaignsShortcode.nonce,
                campaign_id: campaignId,
                form_data: formData
            },
            success: function(response) {
                if (response.success) {
                    // Confirm payment with Stripe
                    confirmStripePayment(response.data.client_secret, $form, $messages, $button, formData, campaignId);
                } else {
                    showMessage($messages, response.data.message, 'error');
                    $button.prop('disabled', false).text($button.data('original-text') || 'Donate Now');
                }
            },
            error: function(xhr, status, error) {
                console.log('Payment Intent Error:', xhr, status, error);
                showMessage($messages, 'Failed to create payment. Please try again.', 'error');
                $button.prop('disabled', false).text($button.data('original-text') || 'Donate Now');
            }
        });
    }

    /**
     * Confirm Stripe payment
     */
    function confirmStripePayment(clientSecret, $form, $messages, $button, formData, campaignId) {
        stripe.confirmCardPayment(clientSecret, {
            payment_method: {
                card: cardElement,
                billing_details: {
                    name: $form.find('#charity_donor_name').val(),
                    email: $form.find('#charity_donor_email').val(),
                },
            }
        }).then(function(result) {
            if (result.error) {
                // Show error to customer
                showMessage($messages, result.error.message, 'error');
                $button.prop('disabled', false).text($button.data('original-text') || 'Donate Now');
            } else {
                // Payment succeeded
                confirmPaymentOnServer(result.paymentIntent.id, $form, $messages, $button, formData, campaignId);
            }
        });
    }

    /**
     * Confirm payment on server
     */
    function confirmPaymentOnServer(paymentIntentId, $form, $messages, $button, formData, campaignId) {
        $.ajax({
            url: charityCampaignsShortcode.ajaxUrl,
            type: 'POST',
            data: {
                action: 'ccfs_confirm_payment',
                nonce: charityCampaignsShortcode.nonce,
                campaign_id: campaignId,
                form_data: formData,
                payment_intent_id: paymentIntentId
            },
            success: function(response) {
                if (response.success) {
                    showMessage($messages, response.data.message, 'success');
                    $form.find('.charity-donation-form-fields')[0].reset();
                    // Clear Stripe card element
                    if (cardElement) {
                        cardElement.clear();
                    }
                } else {
                    showMessage($messages, response.data.message, 'error');
                }
                $button.prop('disabled', false).text($button.data('original-text') || 'Donate Now');
            },
            error: function(xhr, status, error) {
                console.log('Payment Confirmation Error:', xhr, status, error);
                showMessage($messages, 'Payment processed but failed to record. Please contact support.', 'error');
                $button.prop('disabled', false).text($button.data('original-text') || 'Donate Now');
            }
        });
    }

    /**
     * Process manual donation (fallback)
     */
    function processManualDonation($form, $messages, $button, formData, campaignId) {
        $.ajax({
            url: charityCampaignsShortcode.ajaxUrl,
            type: 'POST',
            data: {
                action: 'ccfs_process_donation',
                nonce: charityCampaignsShortcode.nonce,
                campaign_id: campaignId,
                form_data: formData
            },
            success: function(response) {
                console.log('AJAX Success:', response);
                if (response.success) {
                    showMessage($messages, response.data.message, 'success');
                    $form.find('.charity-donation-form-fields')[0].reset();
                } else {
                    showMessage($messages, response.data.message, 'error');
                }
            },
            error: function(xhr, status, error) {
                console.log('AJAX Error:', xhr, status, error);
                showMessage($messages, 'An error occurred. Please try again.', 'error');
            },
            complete: function() {
                $button.prop('disabled', false).text($button.data('original-text') || 'Donate Now');
            }
        });
    }

    /**
     * Validate donation form
     */
    function validateDonationForm($form) {
        var isValid = true;
        var $amountInput = $form.find('#charity_donation_amount');
        var $nameInput = $form.find('#charity_donor_name');
        var $emailInput = $form.find('#charity_donor_email');
        var $anonymousCheckbox = $form.find('#charity_donor_anonymous');

        // Clear previous errors
        $form.find('.charity-field-error').remove();
        $form.find('.charity-form-group').removeClass('charity-has-error');

        // Validate amount
        if ($amountInput.length === 0) {
            console.error('Amount input not found');
            isValid = false;
        } else {
            var amount = parseFloat($amountInput.val());
            if (!amount || amount < 1 || isNaN(amount)) {
                showFieldError($amountInput, 'Please enter a valid donation amount (minimum $1)');
                isValid = false;
            }
        }

        // Check if donation is anonymous
        var isAnonymous = $anonymousCheckbox.length > 0 && $anonymousCheckbox.is(':checked');

        // Only validate name and email if not anonymous
        if (!isAnonymous) {
            // Validate name
            if ($nameInput.length === 0) {
                console.error('Name input not found');
                isValid = false;
            } else if (!$nameInput.val().trim()) {
                showFieldError($nameInput, 'Please enter your name');
                isValid = false;
            }

            // Validate email
            if ($emailInput.length === 0) {
                console.error('Email input not found');
                isValid = false;
            } else {
                var email = $emailInput.val().trim();
                if (!email) {
                    showFieldError($emailInput, 'Please enter your email address');
                    isValid = false;
                } else if (!isValidEmail(email)) {
                    showFieldError($emailInput, 'Please enter a valid email address');
                    isValid = false;
                }
            }
        }

        return isValid;
    }

    /**
     * Show field error
     */
    function showFieldError($field, message) {
        var $group = $field.closest('.charity-form-group');
        $group.addClass('charity-has-error');
        
        var $error = $('<div class="charity-field-error">' + message + '</div>');
        $group.append($error);
    }

    /**
     * Show message
     */
    function showMessage($container, message, type) {
        $container
            .removeClass('success error')
            .addClass(type)
            .html(message)
            .fadeIn();
    }

    /**
     * Validate email format
     */
    function isValidEmail(email) {
        var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }

    /**
     * Animate progress bars on scroll
     */
    function animateProgressBarsOnScroll() {
        $('.charity-progress-fill').each(function() {
            var $bar = $(this);
            var $container = $bar.closest('.charity-campaign-progress, .charity-campaign-item');
            
            if (isElementInViewport($container[0])) {
                var pct = $bar.data('progress');
                if (typeof pct !== 'undefined') {
                    $bar.css('width', '0%').animate({
                        width: pct + '%'
                    }, 1000, 'easeOutQuart');
                    $bar.data('animated', true);
                }
            }
        });
    }

    /**
     * Check if element is in viewport
     */
    function isElementInViewport(el) {
        var rect = el.getBoundingClientRect();
        return (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
            rect.right <= (window.innerWidth || document.documentElement.clientWidth)
        );
    }

    // Animate progress bars on scroll
    $(window).on('scroll', function() {
        $('.charity-progress-fill').each(function() {
            var $bar = $(this);
            if (!$bar.data('animated')) {
                var $container = $bar.closest('.charity-campaign-progress, .charity-campaign-item');
                if (isElementInViewport($container[0])) {
                    var pct = $bar.data('progress');
                    if (typeof pct !== 'undefined') {
                        $bar.css('width', '0%').animate({
                            width: pct + '%'
                        }, 1000, 'easeOutQuart');
                        $bar.data('animated', true);
                    }
                }
            }
        });
    });

    // Add easing function if not available
    if (!$.easing.easeOutQuart) {
        $.easing.easeOutQuart = function (x, t, b, c, d) {
            return -c * ((t=t/d-1)*t*t*t - 1) + b;
        };
    }

})(jQuery);