/**
 * Admin JavaScript for Charity Campaigns
 *
 * @package CCFS
 * @since 1.0.0
 */

(function($) {
    'use strict';

    $(document).ready(function() {
        // Campaign progress calculation
        calculateCampaignProgress();
        applyDynamicStyles();
        
        // Date validation
        validateDates();
        
        // Auto-format currency inputs
        formatCurrencyInputs();
        
        // Real-time updates
        $('#campaign_goal, #campaign_raised').on('input', function() {
            calculateCampaignProgress();
            updateExcessFundsDisplay();
        });
        $('#donation_amount').on('input', formatCurrencyInputs);
        
        // Sync raised amount button
        $('#sync-raised-amount').on('click', syncRaisedAmount);
    });
    function applyDynamicStyles() {
        // Apply dynamic colors for status indicators
        $('.status-indicator[data-color]').each(function(){
            var color = $(this).data('color');
            if (color) {
                $(this).css('background-color', color);
            }
        });

        // Apply progress fills based on data-progress
        $('.progress-fill[data-progress]').each(function(){
            var pct = parseFloat($(this).data('progress')) || 0;
            $(this).css('width', '0%').animate({ width: pct + '%' }, 800);
        });
    }

    /**
     * Calculate and display campaign progress
     */
    function calculateCampaignProgress() {
        var goal = parseFloat($('#campaign_goal').val()) || 0;
        var raised = parseFloat($('#campaign_raised').val()) || 0;
        
        if (goal > 0) {
            var percentage = Math.min((raised / goal) * 100, 100);

            // Create or update progress bar
            var progressHtml = '<div class="campaign-progress">' +
                '<div class="campaign-progress-bar" data-progress="' + percentage + '"></div>' +
                '</div>' +
                '<div class="campaign-progress-text">' +
                Math.round(percentage) + '% Complete (' +
                formatCurrency(raised) + ' / ' + formatCurrency(goal) + ')' +
                '</div>';

            // Ensure container exists at the top and update content
            var $container = $('.campaign-progress-container');
            if ($container.length === 0) {
                // If not present (older markup), insert at top of metabox
                var $formTable = $('.form-table').first();
                $('<div class="campaign-progress-container"></div>').insertBefore($formTable);
                $container = $('.campaign-progress-container');
            }

            $container.html(progressHtml);
            // animate
            var $bar = $container.find('.campaign-progress-bar');
            var pct = parseFloat($bar.data('progress')) || 0;
            $bar.css('width', '0%').animate({ width: pct + '%' }, 800);
        }
    }

    /**
     * Validate date fields
     */
    function validateDates() {
        $('#campaign_start_date, #campaign_end_date').on('change', function() {
            var startDate = new Date($('#campaign_start_date').val());
            var endDate = new Date($('#campaign_end_date').val());
            
            if (startDate && endDate && startDate > endDate) {
                alert('End date must be after start date.');
                $(this).val('');
            }
        });
    }

    /**
     * Format currency inputs
     */
    function formatCurrencyInputs() {
        $('input[type="number"][step="0.01"]').on('blur', function() {
            var value = parseFloat($(this).val());
            if (!isNaN(value)) {
                $(this).val(value.toFixed(2));
            }
        });
    }

    /**
     * Format currency for display
     */
    function formatCurrency(amount) {
        // Check if settings are available, otherwise use defaults
        var currency = (typeof charityCampaignsSettings !== 'undefined' && charityCampaignsSettings.defaultCurrency) ? charityCampaignsSettings.defaultCurrency : 'USD';
        var symbolPosition = (typeof charityCampaignsSettings !== 'undefined' && charityCampaignsSettings.currencySymbolPosition) ? charityCampaignsSettings.currencySymbolPosition : 'before';
        var decimalPlaces = (typeof charityCampaignsSettings !== 'undefined' && charityCampaignsSettings.decimalPlaces) ? charityCampaignsSettings.decimalPlaces : 2;
        
        var symbols = {
            'USD': '$',
            'EUR': '€',
            'GBP': '£',
            'CAD': 'C$',
            'AUD': 'A$',
            'JPY': '¥',
            'CHF': 'CHF',
            'CNY': '¥',
            'INR': '₹',
            'BRL': 'R$'
        };
        
        var symbol = symbols[currency] || currency + ' ';
        var formattedAmount = parseFloat(amount).toFixed(decimalPlaces).replace(/\B(?=(\d{3})+(?!\d))/g, ',');
        
        if (symbolPosition === 'after') {
            return formattedAmount + symbol;
        } else {
            return symbol + formattedAmount;
        }
    }

    /**
     * Auto-populate donation date with current date/time
     */
    $('#donation_date').on('focus', function() {
        if (!$(this).val()) {
            var now = new Date();
            var year = now.getFullYear();
            var month = String(now.getMonth() + 1).padStart(2, '0');
            var day = String(now.getDate()).padStart(2, '0');
            var hours = String(now.getHours()).padStart(2, '0');
            var minutes = String(now.getMinutes()).padStart(2, '0');
            
            $(this).val(year + '-' + month + '-' + day + 'T' + hours + ':' + minutes);
        }
    });

    /**
     * Campaign status change handler
     */
    $('#campaign_status').on('change', function() {
        var status = $(this).val();
        var statusClass = 'campaign-status-' + status;
        
        // Remove existing status classes
        $(this).removeClass('campaign-status-active campaign-status-completed campaign-status-paused campaign-status-cancelled');
        
        // Add new status class
        if (status) {
            $(this).addClass(statusClass);
        }
    });

    /**
     * Donation status change handler
     */
    $('#donation_status').on('change', function() {
        var status = $(this).val();
        var statusClass = 'donation-status-' + status;
        
        // Remove existing status classes
        $(this).removeClass('donation-status-pending donation-status-completed donation-status-failed donation-status-refunded');
        
        // Add new status class
        if (status) {
            $(this).addClass(statusClass);
        }
    });

    /**
     * Anonymous donation checkbox handler
     */
    $('#donation_anon').on('change', function() {
        if ($(this).is(':checked')) {
            $('#donation_name').closest('tr').hide();
            $('#donation_email').closest('tr').hide();
        } else {
            $('#donation_name').closest('tr').show();
            $('#donation_email').closest('tr').show();
        }
    });

    // Initialize anonymous donation state
    if ($('#donation_anon').is(':checked')) {
        $('#donation_name').closest('tr').hide();
        $('#donation_email').closest('tr').hide();
    }

    /**
     * Sync raised amount with calculated donations
     */
    function syncRaisedAmount() {
        var $button = $('#sync-raised-amount');
        var $raisedInput = $('#campaign_raised');
        var $description = $button.closest('p');
        
        // Get calculated amount from the description text
        var calculatedText = $description.find('strong').text();
        var calculatedAmount = calculatedText.match(/[\d,]+\.?\d*/);
        
        if (calculatedAmount) {
            var amount = calculatedAmount[0].replace(/,/g, '');
            $raisedInput.val(parseFloat(amount).toFixed(2));
            
            // Update excess funds display
            updateExcessFundsDisplay();
            
            calculateCampaignProgress();
            
            // Show success message
            $button.text('Synced!').addClass('button-primary').removeClass('button-secondary');
            setTimeout(function() {
                $button.text('Sync with Donations').removeClass('button-primary').addClass('button-secondary');
            }, 2000);
        }
    }

    /**
     * Update excess funds display based on current raised amount and goal
     */
    function updateExcessFundsDisplay() {
        var goal = parseFloat($('#campaign_goal').val()) || 0;
        var raised = parseFloat($('#campaign_raised').val()) || 0;
        var excessAmount = Math.max(raised - goal, 0);
        
        // Find the excess funds input field
        var $excessInput = $('input[readonly]').filter(function() {
            return $(this).closest('tr').find('label').text().indexOf('Excess Funds') !== -1;
        });
        
        if ($excessInput.length) {
            $excessInput.val(excessAmount.toFixed(2));
        }
    }

})(jQuery);

/**
 * Initialize charity calendar
 */
window.initCharityCalendar = function(events) {
    jQuery(document).ready(function($) {
        var currentDate = new Date();
        var currentMonth = currentDate.getMonth();
        var currentYear = currentDate.getFullYear();
        
        // Update month/year display
        function updateMonthYear() {
            var monthNames = [
                'January', 'February', 'March', 'April', 'May', 'June',
                'July', 'August', 'September', 'October', 'November', 'December'
            ];
            $('#current-month-year').text(monthNames[currentMonth] + ' ' + currentYear);
        }
        
        // Generate calendar
        function generateCalendar() {
            var firstDay = new Date(currentYear, currentMonth, 1);
            var lastDay = new Date(currentYear, currentMonth + 1, 0);
            var startDate = new Date(firstDay);
            startDate.setDate(startDate.getDate() - firstDay.getDay());
            
            var calendarHtml = '<div class="calendar-grid">';
            calendarHtml += '<div class="calendar-header-row">';
            calendarHtml += '<div class="calendar-day-header">Sun</div>';
            calendarHtml += '<div class="calendar-day-header">Mon</div>';
            calendarHtml += '<div class="calendar-day-header">Tue</div>';
            calendarHtml += '<div class="calendar-day-header">Wed</div>';
            calendarHtml += '<div class="calendar-day-header">Thu</div>';
            calendarHtml += '<div class="calendar-day-header">Fri</div>';
            calendarHtml += '<div class="calendar-day-header">Sat</div>';
            calendarHtml += '</div>';
            
            // Generate 6 weeks (42 days) to ensure full calendar
            for (var week = 0; week < 6; week++) {
                calendarHtml += '<div class="calendar-week">';
                for (var day = 0; day < 7; day++) {
                    var cellDate = new Date(startDate);
                    cellDate.setDate(startDate.getDate() + (week * 7) + day);
                    
                    var isCurrentMonth = cellDate.getMonth() === currentMonth;
                    var isToday = cellDate.toDateString() === new Date().toDateString();
                    var dateStr = cellDate.getFullYear() + '-' + 
                                 String(cellDate.getMonth() + 1).padStart(2, '0') + '-' + 
                                 String(cellDate.getDate()).padStart(2, '0');
                    
                    var dayEvents = events.filter(function(event) {
                        return event.date === dateStr;
                    });
                    
                    var cellClass = 'calendar-day';
                    if (!isCurrentMonth) cellClass += ' other-month';
                    if (isToday) cellClass += ' today';
                    if (dayEvents.length > 0) cellClass += ' has-events';
                    
                    calendarHtml += '<div class="' + cellClass + '" data-date="' + dateStr + '">';
                    calendarHtml += '<div class="day-number">' + cellDate.getDate() + '</div>';
                    
                    if (dayEvents.length > 0) {
                        calendarHtml += '<div class="day-events">';
                        dayEvents.forEach(function(event) {
                            calendarHtml += '<div class="event-dot" style="background-color: ' + event.color + '" title="' + event.title + '"></div>';
                        });
                        calendarHtml += '</div>';
                        
                        // Add event details below the dots
                        calendarHtml += '<div class="day-event-details">';
                        dayEvents.forEach(function(event) {
                            calendarHtml += '<div class="event-detail" style="border-left-color: ' + event.color + '">';
                            calendarHtml += event.title;
                            calendarHtml += '</div>';
                        });
                        calendarHtml += '</div>';
                    }
                    
                    calendarHtml += '</div>';
                }
                calendarHtml += '</div>';
            }
            
            calendarHtml += '</div>';
            $('#charity-calendar').html(calendarHtml);
        }
        
        // Navigation
        $('#prev-month').on('click', function() {
            currentMonth--;
            if (currentMonth < 0) {
                currentMonth = 11;
                currentYear--;
            }
            updateMonthYear();
            generateCalendar();
        });
        
        $('#next-month').on('click', function() {
            currentMonth++;
            if (currentMonth > 11) {
                currentMonth = 0;
                currentYear++;
            }
            updateMonthYear();
            generateCalendar();
        });
        
        // Initialize
        updateMonthYear();
        generateCalendar();
        
        // Event tooltips
        $(document).on('mouseenter', '.calendar-day.has-events', function() {
            var date = $(this).data('date');
            var dayEvents = events.filter(function(event) {
                return event.date === date;
            });
            
            if (dayEvents.length > 0) {
                var tooltip = '<div class="calendar-tooltip">';
                dayEvents.forEach(function(event) {
                    tooltip += '<div class="tooltip-event" style="border-left-color: ' + event.color + '">';
                    tooltip += event.title;
                    tooltip += '</div>';
                });
                tooltip += '</div>';
                
                $(this).append(tooltip);
            }
        });
        
        $(document).on('mouseleave', '.calendar-day.has-events', function() {
            $(this).find('.calendar-tooltip').remove();
        });
    });
};

/**
 * Initialize Charts for Reports Dashboard
 */
function initCharityCharts() {
    if (typeof Chart === 'undefined') {
        console.log('Chart.js not loaded');
        return;
    }
    
    // Wrap everything in jQuery to ensure $ is available
    jQuery(document).ready(function($) {
        // Chart instances
        var trendsChart, campaignsChart, statusChart;
        
        // Helper function to format currency
        function formatCurrency(value) {
            var symbol = charityCampaignsSettings.defaultCurrency || '$';
            var position = charityCampaignsSettings.currencySymbolPosition || 'before';
            if (position === 'after') {
                return value.toLocaleString() + symbol;
            } else {
                return symbol + value.toLocaleString();
            }
        }
        
        // Helper function to map array (for older browser compatibility)
        function mapArray(array, callback) {
            var result = [];
            for (var i = 0; i < array.length; i++) {
                result.push(callback(array[i], i));
            }
            return result;
        }
        
        // Chart tab functionality
        $('.chart-tab').on('click', function() {
        var chartType = $(this).data('chart');
        
        // Update active tab
        $('.chart-tab').removeClass('active');
        $(this).addClass('active');
        
        // Show/hide chart containers
        $('.chart-container').removeClass('active');
        $('#' + chartType + '-chart').addClass('active');
        
        // Initialize chart if not already done
        if (chartType === 'trends' && !trendsChart) {
            initTrendsChart();
        } else if (chartType === 'campaigns' && !campaignsChart) {
            initCampaignsChart();
        } else if (chartType === 'status' && !statusChart) {
            initStatusChart();
        }
    });
    
    // Initialize trends chart
    function initTrendsChart() {
        var canvas = document.getElementById('monthlyTrendsChart');
        if (!canvas) {
            console.error('Trends chart canvas element not found');
            return;
        }
        
        var ctx = canvas.getContext('2d');
        if (!ctx) {
            console.error('Failed to get canvas context for trends chart');
            return;
        }
        
        // Destroy existing chart if it exists
        if (trendsChart) {
            trendsChart.destroy();
        }
        
        trendsChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: mapArray(window.ccfsTrendsData || [], function(item) { return item.month; }),
                datasets: [{
                    label: 'Donation Amount',
                    data: mapArray(window.ccfsTrendsData || [], function(item) { return item.amount; }),
                    borderColor: '#667eea',
                    backgroundColor: 'rgba(102, 126, 234, 0.1)',
                    borderWidth: 3,
                    fill: true,
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return formatCurrency(value);
                            }
                        }
                    }
                }
            }
        });
    }
    
    // Initialize campaigns chart
    function initCampaignsChart() {
        var canvas = document.getElementById('campaignPerformanceChart');
        if (!canvas) return;
        
        var ctx = canvas.getContext('2d');
        if (!ctx) {
            console.error('Failed to get canvas context for campaigns chart');
            return;
        }
        
        // Destroy existing chart if it exists
        if (campaignsChart) {
            campaignsChart.destroy();
        }
        
        campaignsChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: mapArray(window.ccfsCampaignsData || [], function(item) { 
                    return item.title.length > 20 ? item.title.substring(0, 20) + '...' : item.title; 
                }),
                datasets: [{
                    label: 'Amount Raised',
                    data: mapArray(window.ccfsCampaignsData || [], function(item) { return item.raised; }),
                    backgroundColor: mapArray(window.ccfsCampaignsData || [], function(item) {
                        if (item.progress >= 100) return '#4caf50';
                        if (item.progress >= 75) return '#8bc34a';
                        if (item.progress >= 50) return '#ff9800';
                        return '#f44336';
                    }),
                    borderColor: '#fff',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return formatCurrency(value);
                            }
                        }
                    }
                }
            }
        });
    }
    
    // Initialize status chart
    function initStatusChart() {
        var canvas = document.getElementById('statusDistributionChart');
        if (!canvas) return;
        
        var ctx = canvas.getContext('2d');
        if (!ctx) {
            console.error('Failed to get canvas context for status chart');
            return;
        }
        
        // Destroy existing chart if it exists
        if (statusChart) {
            statusChart.destroy();
        }
        
        statusChart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: mapArray(window.ccfsStatusData || [], function(item) { return item.label; }),
                datasets: [{
                    data: mapArray(window.ccfsStatusData || [], function(item) { return item.amount; }),
                    backgroundColor: [
                        '#4caf50', // completed
                        '#ff9800', // pending
                        '#f44336', // failed
                        '#2196f3'  // refunded
                    ],
                    borderColor: '#fff',
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
        }
        
        // Initialize first chart after ensuring canvas elements exist
        setTimeout(function() {
            var canvas = document.getElementById('monthlyTrendsChart');
            if (canvas && typeof Chart !== 'undefined') {
                initTrendsChart();
            } else {
                console.error('Trends chart canvas not found or Chart.js not loaded');
            }
        }, 300);
        
	}); // End jQuery wrapper
}

/**
 * Initialize Dashboard Overview Calendar
 */
function initDashboardCalendar() {
	jQuery(document).ready(function($) {
		// Calendar data will be passed via localized script
		if (typeof window.charityCalendarEvents !== 'undefined') {
			initCharityCalendar(window.charityCalendarEvents);
		}
	});
}

/**
 * Initialize Dashboard Reports Charts
 */
function initDashboardCharts() {
	jQuery(document).ready(function($) {
		// Only initialize if on reports tab
		var urlParams = new URLSearchParams(window.location.search);
		var currentTab = urlParams.get('tab') || 'overview';
		
		if (currentTab !== 'reports') {
			return;
		}
		
		// Wait for Chart.js to be available
		var checkChartJs = setInterval(function() {
			if (typeof Chart !== 'undefined') {
				clearInterval(checkChartJs);
				
				// Check if chart data is available
				if (typeof window.ccfsTrendsData !== 'undefined' && 
					typeof window.ccfsCampaignsData !== 'undefined' && 
					typeof window.ccfsStatusData !== 'undefined') {
					
					// Initialize charts after a short delay to ensure DOM is ready
					setTimeout(function() {
						if (typeof initCharityCharts === 'function') {
							initCharityCharts();
						}
					}, 200);
				} else {
					console.error('Chart data not available:', {
						trends: typeof window.ccfsTrendsData,
						campaigns: typeof window.ccfsCampaignsData,
						status: typeof window.ccfsStatusData
					});
				}
			}
		}, 100);
		
		// Stop checking after 5 seconds
		setTimeout(function() {
			clearInterval(checkChartJs);
			if (typeof Chart === 'undefined') {
				console.error('Chart.js failed to load');
			}
		}, 5000);
	});
}

/**
 * Date filter function for Financial Summary
 */
function filterFinancialSummary(filter) {
	var currentUrl = new URL(window.location);
	currentUrl.searchParams.set('date_filter', filter);
	window.location.href = currentUrl.toString();
}

// Initialize dashboard components when DOM is ready
jQuery(document).ready(function($) {
	// Initialize calendar for overview tab
	initDashboardCalendar();
	
	// Initialize charts for reports tab
	initDashboardCharts();
});
