<?php
if (! defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * @class    CCF7_Mailchimp_Integration_Class
 * @category Class
 * @author   Nikhil Tiwari
 */
class CCF7_Mailchimp_Integration_Class
{
    protected static $_instance = null;
    public $glSettings = array();

    // Get singleton instance
    public static function get_instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    // Constructor
    public function __construct()
    {
        $this->hooks();
    }

    // Register hooks
    public function hooks()
    {
        // Enqueue necessary styles and scripts for admin pages
        add_action('admin_enqueue_scripts', array($this, 'ccf7mc_enqueue_styles_scripts'));

        // Add a custom Mailchimp tab to the CF7 form editor
        add_filter('wpcf7_editor_panels', array($this, 'ccf7mc_add_mailchimp_tab'));

        // Save Mailchimp settings when a CF7 form is saved
        add_action('wpcf7_save_contact_form', array($this, 'ccf7mc_save_mailchimp_settings'));

        // Handle the submission of CF7 forms to Mailchimp
        add_action('wpcf7_mail_sent', array($this, 'ccf7mc_cf7_to_mailchimp'));

        // Register AJAX action hooks for logged-in and non-logged-in users to verify the Mailchimp API key
        add_action('wp_ajax_verify_mailchimp_api_key', array($this, 'ccf7mc_verify_mailchimp_api_key_callback'));
        add_action('wp_ajax_verify_mailchimp_api_key', array($this, 'ccf7mc_verify_mailchimp_api_key_callback'));
    }

    /**
     * Callback function to verify the Mailchimp API key.
     * This function will be called via AJAX when a request is made.
     */
    public function ccf7mc_verify_mailchimp_api_key_callback()
    {
        // Verify the AJAX nonce for security
        check_ajax_referer('save_mailchimp_settings', 'nonce');

        // Retrieve the API key and form ID from the AJAX request   
        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;

        // Verify the API key using the Mailchimp API
        $is_verified = $this->ccf7mc_verify_mailchimp_api_key($api_key);

        // Check if the API key was successfully verified
        if ($is_verified) {
            // Update the post meta with the verified API key
            update_post_meta($post_id, '_mailchimp_api_key', $api_key);
            // Send a successful JSON response back to the AJAX request
            wp_send_json_success('API key verified successfully.');
        } else {
            // The API key verification failed
            wp_send_json_error('Invalid API key. Please try again.');
        }
    }

    /**
     * Send CF7 form data to Mailchimp when a form is submitted.
     *
     * @param object $contact_form The CF7 form object that was submitted.
     */
    public function ccf7mc_cf7_to_mailchimp($contact_form)
    {
        // Get the ID of the submitted form
        $form_id = $contact_form->id();

        // Get the current submission instance
        $submission = WPCF7_Submission::get_instance();

        // Check if a submission exists
        if ($submission) {
            // Get posted form data
            $form_data  = $submission->get_posted_data();

            // Retrieve Mailchimp API key from post meta
            $api_key = get_post_meta($form_id, '_mailchimp_api_key', true);

            // Prepare the form data to be sent to Mailchimp
            $fields_to_send = $this->ccf7mc_prepare_mailchimp_fields($form_id, $form_data, $api_key);

            // Send the data to Mailchimp
            $result = $this->ccf7mc_send_to_mailchimp($api_key, $fields_to_send);

            // Handle result if needed
            if ($result['status'] === 'error') {
                // Log error or take other action
                error_log($result['message']);
            }
        }
    }

    /**
     * Prepare form data for submission to Mailchimp.
     *
     * @param int $post_id The CF7 form ID.
     * @param array $form_data The submitted form data.
     * @param string $api_key The Mailchimp API key.
     * @return array The prepared data to send to Mailchimp.
     */
    public function ccf7mc_prepare_mailchimp_fields($post_id, $form_data, $api_key)
    {
        // Retrieve field mappings from post meta
        $mailchimp_field_mappings = get_post_meta($post_id, '_mailchimp_field_mappings', true) ?: [];

        // Fetch Mailchimp fields
        $mailchimp_response = $this->ccf7mc_mailchimp_api_response('/merge-fields', $api_key);
        $mailchimp_fields = isset($mailchimp_response['fields']) ? $mailchimp_response['fields'] : [];

        $fields = [];

        // Iterate through the Mailchimp fields
        foreach ($mailchimp_fields as $mailchimp_field) {
            $this->ccf7mc_process_mailchimp_field($mailchimp_field, $mailchimp_field_mappings, $form_data, $fields);
        }

        // Return the prepared fields
        return $fields;
    }

    /**
     * Process a single Mailchimp field and map it to CF7 form data.
     *
     * @param array $mailchimp_field The Mailchimp field to process.
     * @param array $mappings The field mappings between CF7 and Mailchimp.
     * @param array $form_data The form data submitted by the CF7 form.
     * @param array &$fields The array to hold prepared fields for Mailchimp.
     */
    private function ccf7mc_process_mailchimp_field($mailchimp_field, $mappings, $form_data, &$fields)
    {
        // Check if the Mailchimp tag exists in the mappings
        if (array_key_exists($mailchimp_field['tag'], $mappings)) {
            // Get the CF7 field mapping based on the Mailchimp tag
            $cf7_field_key = $mappings[$mailchimp_field['tag']];
            $cf7_field_clean_key = preg_replace('/\[.*?\s|\]/', '', $cf7_field_key);

            // Check if the form data has the cleaned field key
            if (isset($form_data[$cf7_field_clean_key])) {
                $field_value = $form_data[$cf7_field_clean_key];

                // Handle array values (e.g., checkboxes or multiple selections)
                if (is_array($field_value)) {
                    // Convert array to comma-separated string
                    $field_value = implode(', ', $field_value);
                }

                // Assign values dynamically to Mailchimp fields
                $this->ccf7mc_assign_field_value($mailchimp_field, $field_value, $form_data, $fields);
            }
        }
    }

    /**
     * Assign field values to the appropriate Mailchimp fields based on the Mailchimp tag.
     *
     * @param array $mailchimp_field The Mailchimp field being processed.
     * @param mixed $field_value The value from the CF7 form associated with the Mailchimp field.
     * @param array $form_data The form data submitted by the CF7 form.
     * @param array &$fields The array to hold prepared fields for Mailchimp.
     */
    private function ccf7mc_assign_field_value($mailchimp_field, $field_value, $form_data, &$fields)
    {
        if ($mailchimp_field['tag'] === 'EMAIL') {
            $fields['email_address'] = $field_value; // Directly assign email
        } elseif ($mailchimp_field['tag'] === 'ADDRESS') {
            // If the field is ADDRESS, break it into subfields
            $address_parts = $this->ccf7mc_prepare_address_fields($form_data, $field_value);
            $fields['merge_fields']['ADDRESS'] = $address_parts;
        } else {
            // For all other fields, assign them under merge_fields
            $fields['merge_fields'][$mailchimp_field['tag']] = $field_value;
        }
    }

    /**
     * Prepare address fields from the form data for Mailchimp.
     *
     * @param array $form_data The form data submitted by the CF7 form.
     * @return array An array of address fields structured for Mailchimp.
     */
    private function ccf7mc_prepare_address_fields($form_data, $field_value)
    {
        return [
            'addr1' => isset($field_value) ? $field_value : '',
            'addr2' => isset($form_data['address_line_2']) ? $form_data['address_line_2'] : '',
            'city' => isset($form_data['city']) ? $form_data['city'] : '',
            'state' => isset($form_data['State']) ? $form_data['State'] : '',
            'zip' => isset($form_data['Zip']) ? $form_data['Zip'] : '',
            'country' => isset($form_data['Country']) ? $form_data['Country'] : 'US' // Default to 'US' if not provided
        ];
    }

    /**
     * Send prepared fields to Mailchimp.
     *
     * @param string $api_key The Mailchimp API key.
     * @param array $fields The prepared fields to send.
     * @return array Result of the send operation.
     */
    public function ccf7mc_send_to_mailchimp($api_key, $fields)
    {

        $result =  $this->ccf7mc_send_cf7_data_to_mailchimp($api_key, $fields);
        return $result;
    }

    /**
     * Send CF7 data to Mailchimp API.
     *
     * @param string $api_key The Mailchimp API key.
     * @param array $fields The fields to send.
     * @param int|null $group_id Optional group ID for specific Mailchimp list.
     * @return array The result of the API request.
     */
    public function ccf7mc_send_cf7_data_to_mailchimp($api_key, $fields, $group_id = null)
    {
        // Extract the data center from the API key (API keys are in the format `xxxxxx-usX`)
        $data_center = substr($api_key, strpos($api_key, '-') + 1);

        $list_ids = $this->ccf7mc_mailchimp_api_lists($api_key);
        // Check if any lists were retrieved
        if (empty($list_ids['lists'])) {
            return array(
                'status' => 'error',
                'message' => 'No Mailchimp lists found.',
            );
        }

        // Define the Mailchimp API endpoint for adding a subscriber
        $url = 'https://' . $data_center . '.api.mailchimp.com/3.0/lists/' . $list_ids['lists'][0]['id'] . '/members?skip_merge_validation=true';

        // Prepare the payload for the API request
        $data = $this->ccf7mc_prepare_mailchimp_payload($fields);

        // error_log('encoded data is :' . print_r(json_encode($data), true));

        // Make the API request
        return $this->ccf7mc_execute_mailchimp_request($url, $api_key, $data);
    }

    /**
     * Prepare the payload for the Mailchimp API request.
     *
     * @param array $fields The fields containing the subscriber's data.
     * @return array The prepared payload for the API request.
     */
    private function ccf7mc_prepare_mailchimp_payload($fields)
    {
        return [
            'email_address' => $fields['email_address'], // Use email address from fields
            'status' => 'subscribed', // Subscription status
            'merge_fields' => isset($fields['merge_fields']) ? $fields['merge_fields'] : [], // Additional merge fields
        ];
    }

    /**
     * Execute the Mailchimp API request to add a subscriber.
     *
     * @param string $url The Mailchimp API URL.
     * @param string $api_key The API key for authenticating the request.
     * @param array $data The payload to send with the request.
     * @return array The response from the Mailchimp API.
     */
    private function ccf7mc_execute_mailchimp_request($url, $api_key, $data)
    {
        // Make the API request
        $response = wp_remote_post($url, [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode('user:' . $api_key),
                'Content-Type' => 'application/json'
            ],
            'body' => wp_json_encode($data),
        ]);

        // Check if the response is a WP_Error (network or HTTP errors)
        if (is_wp_error($response)) {
            return array(
                'status' => 'error',
                'message' => 'Request failed: ' . $response->get_error_message(),
            );
        }

        // Retrieve the response body
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        // Check the status code for success
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 200) {
            return array(
                'status' => 'success',
                'message' => 'Subscriber added successfully.',
                'data' => $data,
            );
        } else {
            return array(
                'status' => 'error',
                'message' => isset($data['detail']) ? $data['detail'] : 'An error occurred.',
                'data' => $data,
            );
        }
    }

    /**
     * Save Mailchimp settings when a CF7 form is saved.
     *
     * This function verifies the nonce for security, retrieves the API key,
     * and if valid, saves the field mappings submitted via the CF7 form settings.
     *
     * @param ContactForm $contact_form The CF7 form instance being processed.
     */
    public function ccf7mc_save_mailchimp_settings($contact_form)
    {
        // Verify nonce for security to prevent CSRF attacks
        if (!isset($_POST['mailchimp_settings_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mailchimp_settings_nonce'])), 'save_mailchimp_settings')) {
            return; // Invalid nonce
        }


        // Get the ID of the current form
        $post_id = $contact_form->id();

        // Get the ID of the current form
        $api_key = get_post_meta($post_id, '_mailchimp_api_key', true);

        // Verify the Mailchimp API key before proceeding to save settings
        if ($this->ccf7mc_verify_mailchimp_api_key($api_key)) {

            // Check if field mappings are submitted
            if (isset($_POST['mailchimp_settings'])) {
                // Sanitize and save field mappings
                $mappings = array_map('sanitize_text_field', wp_unslash($_POST['mailchimp_settings']));

                // Update the post meta with the sanitized mappings
                $updated_fields = update_post_meta($post_id, '_mailchimp_field_mappings', $mappings);

                //add params on url 
                if ($updated_fields) {
                    $cf7_version = get_plugin_data(WP_PLUGIN_DIR . '/contact-form-7/wp-contact-form-7.php')['Version'];
                    if ($cf7_version >= 6.0) {
                        $redirect_url = admin_url('admin.php?page=wpcf7&post=' . $post_id . '&action=edit&active-tab=mailchimp-integration-panel&section=field-mapping');
                        wp_redirect($redirect_url);
                        exit;
                    } else {
                        $redirect_url = admin_url('admin.php?page=wpcf7&post=' . $post_id . '&action=edit&active-tab=4&section=field-mapping');
                        wp_redirect($redirect_url);
                        exit;
                    }
                }
            }
        }
    }


    /**
     * Add Mailchimp settings tab to the Contact Form 7 edit panel
     *
     * @param array $panels The existing panels in the Contact Form 7 editor
     * @return array Updated panels including the Mailchimp tab
     */
    public function ccf7mc_add_mailchimp_tab($panels)
    {
        // Get the current Contact Form 7 form object
        $contact_form = WPCF7_ContactForm::get_current();

        // Check if the form has a post ID (meaning it's published or saved)
        if (!$contact_form->id()) {
            // If there is no post ID (form not published), return the panels without the Mailchimp tab
            return $panels;
        }
        // Define a new panel for Mailchimp settings
        $panels['mailchimp-integration-panel'] = [
            'title'    => __('Mailchimp Integration', 'centous-integration-for-contact-form-7-and-mailchimp'),
            'callback' => [$this, 'ccf7mc_render_mailchimp_panel_content'],
        ];

        // Create a new array to maintain the desired order
        $new_order = [];

        $current_index = 0;

        // Iterate through existing panels to build the new order
        foreach ($panels as $key => $panel) {
            if ($current_index === 4) {
                // Add Mailchimp tab before the current tab if we reached the fourth position
                $new_order['mailchimp-integration-panel'] = $panels['mailchimp-integration-panel'];
            }
            $new_order[$key] = $panel;
            $current_index++;
        }

        // If we have fewer than 4 tabs, just append it at the end
        if ($current_index < 4) {
            $new_order['mailchimp-integration-panel'] = $panels['mailchimp-integration-panel'];
        }

        return $new_order;
    }

    /**
     * Render the content for the Mailchimp Integration tab
     *
     * @param object $post The CF7 form post object.
     */
    public function ccf7mc_render_mailchimp_panel_content($post)
    {
        $data = array(
            'post' => $post
        );
        helper_ccf7mc_object()->ccf7mc_get_template('ccf7mc-api-key.php', 'cf7-mailchimp-integration-module', $data, false);
    }

    /**
     * Renders the Mailchimp settings section in the form editor.
     *
     * @param WP_Post $post The current post object.
     */
    public function ccf7mc_setting_section_mailchimp($post)
    {
        // Prepare data to be passed to the template
        $data = array(
            'post' => $post
        );
        // Load the Mailchimp settings template
        helper_ccf7mc_object()->ccf7mc_get_template('ccf7mc-api-setting-tabs.php', 'cf7-mailchimp-integration-module', $data, false);
    }

    /**
     * Verify the provided Mailchimp API key by pinging the Mailchimp API.
     *
     * @param string $api_key The Mailchimp API key to verify.
     * @return bool True if the API key is valid, false otherwise.
     */
    public function ccf7mc_verify_mailchimp_api_key($api_key)
    {
        // Mailchimp API URL for the ping endpoint
        $dc = substr($api_key, strpos($api_key, '-') + 1); // Get the data center from the API key
        $url = 'https://' . $dc . '.api.mailchimp.com/3.0/ping';

        // Set up the request headers including the API key
        $args = array(
            'headers' => array(
                'Authorization' => 'Basic ' . base64_encode('user:' . $api_key),
            ),
        );

        // Send the request
        $response = wp_remote_get($url, $args);

        // Check if the response is valid and the API key is working
        if (is_wp_error($response)) {
            return false;
        }

        $status_code = wp_remote_retrieve_response_code($response);

        // Assuming 200 is the success code for Mailchimp API
        return $status_code === 200;
    }

    /**
     * Fetch the Mailchimp lists associated with the provided API key.
     *
     * @param string $api_key The Mailchimp API key to use for authentication.
     * @return array An associative array containing the status and lists or error message.
     */
    public function ccf7mc_mailchimp_api_lists($api_key)
    {
        // Extract the data center from the API key
        $data_center = substr($api_key, strpos($api_key, '-') + 1);

        // Define the Mailchimp API endpoint for retrieving lists
        $url = 'https://' . $data_center . '.api.mailchimp.com/3.0/lists';

        // Make the API request
        $response = wp_remote_get($url, [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode('user:' . $api_key),
            ],
        ]);

        // Check if the response is a WP_Error
        if (is_wp_error($response)) {
            return array(
                'status' => 'error',
                'message' => 'Request failed: ' . $response->get_error_message(),
            );
        }

        // Retrieve the response body
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        // Check if the API returned any lists
        if (!empty($data['lists'])) {
            return array(
                'status' => 'success',
                'lists' => $data['lists'],
            );
        } else {
            return array(
                'status' => 'error',
                'message' => 'No lists found.',
            );
        }
    }

    /**
     * Retrieves the response from a specific Mailchimp API endpoint using the provided API key.
     *
     * @param string $endpoint The API endpoint to call (e.g., 'subscribers').
     * @param string $api_key The Mailchimp API key to use for authentication.
     * @return array Returns the decoded JSON response as an associative array, or an empty array on error.
     */
    public function ccf7mc_mailchimp_api_response($endpoint, $api_key)
    {
        // Extract the datacenter from the API key (API keys are in the format `xxxxxx-usX`)
        $data_center = substr($api_key, strpos($api_key, '-') + 1);

        $list_id = $this->ccf7mc_mailchimp_api_lists($api_key);

        // // Define the Mailchimp API endpoint for retrieving merge fields (list fields)
        $url = 'https://' . $data_center . '.api.mailchimp.com/3.0/lists/' . $list_id['lists'][0]['id'] . $endpoint . '';
        // return $url;

        // Make the API request
        $response = wp_remote_get($url, [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode('user:' . $api_key),
            ],
        ]);

        // Check for errors in the response
        if (is_wp_error($response)) {
            return [];
        }

        // Retrieve the body of the response
        $body = wp_remote_retrieve_body($response);

        // Decode the JSON response into an associative array
        $data = json_decode($body, true);

        // Prepare the email field
        $email_field = array(
            'name' => 'Email Address',
            'tag' => 'EMAIL',
            'type' => 'email',
        );

        // Initialize fields array with the email field
        $fields_with_email = array($email_field);

        // Check if the API returned any merge fields
        if (!empty($data['merge_fields'])) {
            $fields_with_email = array_merge($fields_with_email, $data['merge_fields']);
            return array(
                'status' => 'success',
                'fields' => $fields_with_email,
            );
        } else {
            return array(
                'status' => 'error',
                'message' => 'No merge fields found for the specified list.',
            );
        }
    }

    /**
     * Retrieves all shortcodes (fields) used in a Contact Form 7 form by its ID.
     *
     * @param int $post_id The ID of the Contact Form 7 post.
     * @return array An array of shortcodes representing the form fields.
     */
    public function ccf7mc_get_contact_from7_fields($post_id)
    {
        // Get the current form instance
        $form = WPCF7_ContactForm::get_instance($post_id);
        // Get the form content (HTML with shortcodes)
        $form_content = $form->prop('form');

        $cf7_shortcodes = [];


        // Regular expression to match all shortcodes (e.g., [text* your-name])
        if (preg_match_all('/\[(\w+)(\*?)\s+([^\]]+)\]/', $form_content, $matches, PREG_SET_ORDER)) {

            foreach ($matches as $shortcode) {
                $field_type = $shortcode[1]; // e.g., email, text, select, checkbox
                $is_required = $shortcode[2]; // If the field is required (denoted by '*')
                $field_details = $shortcode[3]; // Get the rest of the shortcode content

                // Split field details by space and take the first element as the field name
                $field_name = explode(' ', $field_details)[0]; // e.g., your-email, your-subject

                // Store the shortcode in an array
                $cf7_shortcodes[] = '[' . esc_attr($field_type . $is_required . ' ' . $field_name) . ']'; // e.g., [email* your-email]
            }
        }
        return $cf7_shortcodes;
    }

    /**
     * Renders the field mapping section in the form editor.
     *
     * @param WP_Post $post The current post object.
     */
    public function ccf7mc_field_mapping_section($post)
    {
        // Prepare data to be passed to the template
        $data = array(
            'post' => $post
        );
        // Load the field mapping settings template
        helper_ccf7mc_object()->ccf7mc_get_template('ccf7mc-field-mapping-page.php', 'cf7-mailchimp-integration-module', $data, false);
    }



    /**
     * Renders the documentation section for the Mailchimp integration.
     * This function is responsible for preparing data and loading the corresponding template.
     */
    public function ccf7mc_documentation_section()
    {
        // Prepare data to be passed to the template
        $data = array();

        // Load the field mapping settings template
        helper_ccf7mc_object()->ccf7mc_get_template('ccf7mc-documentation-page.php', 'cf7-mailchimp-integration-module', $data, false);
    }

    /**
     * Enqueue stylesheets and JavaScript files
     */
    public function ccf7mc_enqueue_styles_scripts()
    {
        // Check user capabilities to ensure they have access
        if (!current_user_can('manage_options')) {
            return;
        }
        // Register the style
        wp_register_style('cf7_mailchimp_style', CCF7MC_URL . 'modules/cf7-mailchimp-integration-module/css/style.css', array(), CCF7MC_VERSION, 'all');
        wp_enqueue_style('cf7_mailchimp_style');

        // Register the script
        wp_register_script('cf7_mailchimp_script', CCF7MC_URL . 'modules/cf7-mailchimp-integration-module/js/script.js', array('jquery'), CCF7MC_VERSION, true);
        wp_enqueue_script('cf7_mailchimp_script');

        // Pass Ajax URL to the JavaScript file
        wp_localize_script('cf7_mailchimp_script', 'wep_params', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('save_mailchimp_settings'),
            'cf7_version' => get_plugin_data(WP_PLUGIN_DIR . '/contact-form-7/wp-contact-form-7.php')['Version']
        ));
    }
}

// Function to get the singleton instance
function class_ccf7_mailchimp_Integration_object()
{
    return CCF7_Mailchimp_Integration_Class::get_instance();
}
// Initialize the class
class_ccf7_mailchimp_Integration_object();
